%% segmentGran.m - Supplementary File 4
% Load red image montage, process image, then segment granulocyte region
% from buffy coat (i.e., fluorescing region)
%
% Example case below provided for buffy coat in capillary tube
%
% Note: directories must be changed to correspond to your specific image
% file paths. Test images included in archive.
%
% Version: 1.0, 1 August 2016
% Author:  Max L. Balter
% Email:   balterm53=gmail*com, max.balter=rutgers*edu

clear all; close all; clc; warning off

%% Directories
code_dir               =   pwd;                                             % Set code directory
                                                   
cd('./2-Processed Images/Red')
red_image_dir          =   pwd;                                             % Set red image directory

cd('../../3-Segmented Images/Red')
results_dir        =   pwd;                                                 % Set processed image directory

cd('../Green')
load('boundaryData');                                                       % Extract lym/mon segmented boundary data
                           
%% Load Red Montage Images       
cd(red_image_dir);                                                          % Go to image directory
nfiles = length(dir('*png'));                                               % Search for images in directory
greenImIntensity{nfiles} = [];                                              % Initialize cell for green intensity profile data 
redBuffyArea = zeros(1,nfiles);                                             % Initialize empty vector for segmented area

dirOutput = dir('*.png');                                                   % Find all images in folder
cd(code_dir); 
fileNames = {dirOutput.name};                                               % Extract file names 

for c = 1 : nfiles                                                          % Loop through image files    
    cd(red_image_dir);
    I_g_red = imread(fileNames{c});                                         % Open red channel image file
    figure; imshow(I_g_red); title('Grayscale Image');                      % Show grayscale image 

    %% Black Out Buffy Coat, PLTs, and Plasma Layers
    s = size(I_g_red);                                                      % Extract size of red channel image
    mask_r = poly2mask(bn{c}(:,2),bn{c}(:,1), s(1),s(2));                   % Convert boundary to image mask

    amask = I_g_red;                                                        % Create base image mask from red channel image     
    amask(mask_r) = 0;                                                      % Mask out (in black) buffy coat from green image
    figure; imshow(amask)                                                   % Display initial masked image

    min_bn = round(min(bn{c}(:,1)) + 45);                                   % Find minimum boundary pixel location and adjust with gain value

    for i = min_bn : length(I_g_red(:,1));                                  % Loop through pixel locations to make all pixels below buffy coat black
        amask(i,:) = 0;
    end
    figure; imshow(amask)                                                   % Display final masked image
    
    %% Perform Image Pre-Processing
    cd(code_dir)                                                            % Go to code directory
    
    L = medfilt2(amask);                                                    % Apply a median filter to image
    figure; imshow(L);                                                      % Show image after contrast enhancement 
    title('Intensity values adjusted by a 2-D median filter');

    for i = 1:3
        M = ordfilt2(L,1,ones(2,2));                                        % 2D order statistic filter - 2x2 miniumum filter
    end
    figure; imshow(M); title('After minimum filtering')                     % Show image after filtering

    %% Otsu Global Threshold
    bw = im2bw(amask, graythresh(L));                                       % Apply global threshold based on Otsu's method
    figure; imshow(bw); title('Otsu global threshold')                      % Show image after segmentation

    bw_fill = imfill(bw,'holes');                                           % Fill holes of complement image
    figure; imshow(bw_fill); title('After filling holes')                   % Show image after filling in small holes

    %% Use Morphological Closing to Remove Small Pixel Objects
    d = strel('disk',1);                                                    % Create disk object for morphology operation    
    bw_2 = imopen(bw_fill,d);                                               % Morphological closing
    figure; imshow(bw_2); title('After morphological closing');             % Show image after closing gaps

    %% Remove Small Extraneous Objects Via Checking Area
    objects = bwconncomp(bw_2);                                             % Identify connected objects
    no_of_Obj = objects.NumObjects;                                         % Initiate object counter
    Obj_data = regionprops(objects, 'basic');                               % Extract geometrical data on segmented objects
    Obj_areas = [Obj_data.Area];                                            % Compute area of each object

    for k = 1:no_of_Obj
        if (Obj_areas(k)<10)                                                % Delete small objects based on area (*these values can be adjusted*)
            bw_2(objects.PixelIdxList{k}) = 0;
        end
    end

    objects = bwconncomp(bw_2);                                             % Identify connected objects after deleting small ones
    Obj_data = regionprops(objects, 'basic');                               % Extract geometrical data on objects after deleting small ones
    no_of_Obj = objects.NumObjects;                                         % Re-count objects in image
    figure; imshow(bw_2); title('After removing false objects');            % Show image after removing small objects
    cd(results_dir)                                                         % Go to results directory
    imwrite(bw_2, ['Segmented Image', int2str(c),'.png']);                  % Save segmented image 
    
    amask = I_g_red;                                                        % Create mask with original grayscale image
    amask(imcomplement(bw_2)) = 255;                                        % Final segmented image 
    figure; imshow(amask); title('The final segmented image');              % Show final segmented image
    
    %% Segemnted Image Overlayed on Original
    [B,L,N,A] = bwboundaries(bw_2);                                         % Obtain boundary data of connected objects
    I_n = I_g_red;                                                          % Get grayscale image
    figure; imshow(I_n); title('Region perimeter overlay');                 % Show segmented region overlayed on initial grayscale image 
    
    hold on
    for h = 1:length(B)
        boundary = B{h};                                                    % Loop through detected objects
        plot(boundary(:,2), boundary(:,1), 'r', 'LineWidth', 0.6)           % Overlay perimeter border on segmented objects
    end
    
    set(gca,'LooseInset',get(gca,'TightInset'));
    cd(results_dir)                                                         % Go to results directory
    saveas(gcf, ['Segmented overlay', int2str(c),'.jpg'])                   % Save final segmented image
    savefig(['Segmented overlay', int2str(c),'.fig']);                      % Save segmented image as a figure
    
    redBuffyArea(c) = sum(Obj_areas)                                        % Compute total buffy coat area (in pixels)
    
    cd(code_dir)                                                            % Return to code directory
end