%% segmentLym_Mon.m - Supplementary File 3
% Load green image montage, process image, then segment lymphocyte/
% monocyte region from buffy coat (i.e., fluorescing region)
%
% Example case below provided for buffy coat in capillary tube
%
% Note: directories must be changed to correspond to your specific image
% file paths. Test images included in archive.
%
% Version: 1.0, 1 August 2016
% Author:  Max L. Balter
% Email:   balterm53=gmail*com, max.balter=rutgers*edu

clear all; close all; clc; warning off

%% Directories
code_dir                 =   pwd;                                           % Set code directory
                                                   
cd('./2-Processed Images/Green')
green_image_dir          =   pwd;                                           % Set green image directory

cd('../../3-Segmented Images/Green')
results_dir             =   pwd;                                            % Set processed image directory
                           
%% Load Green Montage Images       
cd(green_image_dir)                                                         % Go to green image directory
nfiles = length(dir('*.png'));                                              % Search for images in directory
greenImIntensity{nfiles} = [];                                              % Initialize cell for green intensity profile data 
greenBuffyArea = zeros(1,nfiles);                                           % Initialize empty vector for segmented area
bn{nfiles} = [];                                                            % Initialize empty cell for segmented boundary_g data 

dirOutput = dir('*.png');                                                   % Find all images in folder
cd(code_dir); 
fileNames = sort_nat({dirOutput.name});                                     % Extract file names and sort based on number

for c = 1 : nfiles                                                          % Loop through image files    
    cd(green_image_dir);
    I_g_green = imread(fileNames{c});                                       % Open green channel image file    
    figure; imshow(I_g_green); title('Grayscale Image');                    % Show grayscale image 

    %% Perform Image Pre-Processing
    cd(code_dir)                                                            % Go to code directory
    [I_g_crop, rect] = imcrop(I_g_green);                                   % Manually crop image to buffy coat location for better segmentation results [xmin ymin width height]
    yCrop = rect(2);                                                        % Store ymin variable of bounding box
    xCrop = rect(1);                                                        % Store xmin variable of bounding box
    
    L = imadjust(I_g_crop);                                                 % Adjust image intensity values with a linear contrast stretching 
    figure; imshow(L);                                                      % Show image after constract enhancement 
    title('Intensity values adjusted by a linear contrast stretching');

    for i = 1:3
        M = ordfilt2(L,1,ones(3,3));                                        % 2D order statistic filter - 3x3 miniumum filter
    end
    figure; imshow(M); title('After minimum filtering')                     % Show image after filtering

    %% Otsu Global Threshold
    bw = im2bw(L, graythresh(M));                                           % Apply global threshold based on Otsu's method
    figure; imshow(bw); title('Otsu global threshold')                      % Show image after segmentation

    bw_fill = imfill(bw,'holes');                                           % Fill holes of complement image
    figure; imshow(bw_fill); title('After filling holes')                   % Show image after filling in small holes

    %% Use Morphological Closing to Remove Small Pixel Objects
    d = strel('disk',5);                                                    % Create disk object for morphology operation    
    bw_2 = imopen(bw_fill,d);                                               % Morphological closing
    figure; imshow(bw_2); title('After morphological closing');             % Show image after closing gaps

    %% Remove Small Extraneous Objects Via Checking Area
    objects = bwconncomp(bw_2);                                             % Identify connected objects
    no_of_Obj = objects.NumObjects;                                         % Initialize object counter
    Obj_data = regionprops(objects, 'basic');                               % Extract geometrical data on segmented objects
    Obj_areas = [Obj_data.Area];                                            % Compute area of each object

    for k = 1:no_of_Obj
        if (Obj_areas(k)<700) ||  (Obj_areas(k)>170000)                     % Delete small objects based on area (*these values can be adjusted*)
            bw_2(objects.PixelIdxList{k}) = 0;
        end
    end

    objects = bwconncomp(bw_2);                                             % Identify connected objects after deleting small ones
    Obj_data = regionprops(objects, 'basic');                               % Extract geometrical data on objects after deleting small ones
    no_of_Obj = objects.NumObjects;                                         % Re-count objects in image
    figure; imshow(bw_2); title('After removing false objects');            % Show image after removing small objects
    cd(results_dir)                                                         % Go to results directory
    imwrite(bw_2, ['Segmented Image', int2str(c),'.png']);                  % Save segmented image
       
    amask = I_g_crop;                                                       % Create mask with original grayscale image
    amask(imcomplement(bw_2)) = 255;                                        % Final segmented image 
    figure; imshow(amask); title('The final segmented image');              % Show final segmented image
    
    %% Segemnted Image Overlayed on Original
    [B,L,N,A] = bwboundaries(bw_2);                                         % Obtain boundary_g data of connected objects
    figure; imshow(I_g_crop); 
    title('Region perimeter overlay');                                      % Show segmented region overlayed on initial grayscale image 

    hold on
    boundary_g = B{1};                                                      % Label B as boundary_g for detected objects
    plot(boundary_g(:,2), boundary_g(:,1), 'r', 'LineWidth', 2)             % Overlay perimeter border on segmented objects
    hold off
    
    bn{c} = boundary_g;                                                     % Extract boundary_g info for each segmented region
    bn{c}(:,1) = bn{c}(:,1) + yCrop;                                        % Adjust boundary_g y location based on initial cropping
    bn{c}(:,2) = bn{c}(:,2) + xCrop;                                        % Adjust boundary_g x location based on initial cropping
    cd(results_dir)                                                         % Go to results directory
    
    set(gca,'LooseInset',get(gca,'TightInset'));                            % Delete borders around image
    saveas(gcf, ['Segmented overlay', int2str(c),'.jpg'])                   % Save final segmented image
    savefig(['Segmented overlay', int2str(c),'.fig']);                      % Save segmented image as a figure
    
    greenBuffyArea(c) = Obj_areas(1,1)                                      % Compute total buffy coat area (in pixels)
    
    cd(code_dir)                                                            % Return to code directory
end

cd(results_dir)                                                             % Return to results directory
save('greenBuffyArea.mat', 'greenBuffyArea');                               % Save lym/mon buffy area
save('boundaryData.mat', 'bn');                                             % Save lym/mon segmentation boundary data
cd(code_dir)                                                                % Return to code directory