

%     This MATLAB script calculates the growth rate of annual PV manufacturing as well as cumulative PV capacity over time
%     based on estimated costs and sustainable growth rates of the PV industry, assuming crystalline silicon (c-Si)
%     technology. Cost and sustainable growth models were modified from previous work by Powell et. al.
%     (D. M. Powell, R. Fu, K. Horowitz, P. A. Basore, M. Woodhouse, T. Buonassisi, Energy Environ. Sci. 2015, 8, 3395).
% 
%     These two models are integrated with a growth rate calculator and accessed from MATLAB directly to complete
%     calculations. They are written as Excel files, 'Cost model for climate.xlsx' and 'SustGR for climate.xlsx'.
%     Make sure both these files reside in the same directory as 'PVgrowth.m'.
% 
%     This script further introduces constraints to selling prices (and thus operating margins) based on demand.
% 
%     This script produces three figures:
%     (1) Growth rate of PV over time
%     (2) Sensitivity of PV growth, with respect to a particular model variable (absolute values)
%     (3) Sensitivity of PV growth, with respect to a particular model variable (relative change in %)
% 
%     For figures (2) and (3), model parameters to vary can be chosen through user prompts that appear when running the
%     script. Values can also be hard-coded in by altering their numeric values in section 1. Use Ctrl+F to search for the
%     numeric symbol '#' to find parameters that must be changed by hard-coding them directly inside this script.
% 
%     Please send any inquiries to the author: Jeremy R. Poindexter (jeremy.r.poindexter@gmail.com)
%     Created: September 29, 2015
%     Last modified: March 30, 2016


%% 1. DEFINITIONS: Define model parameters (along with default values) and prompt user for parameter to vary for sensitivity analysis.


%%% 1a. SUSTAINABLE GROWTH MODEL: Define 'Top level input' default parameters (#hard-coded):
%{
Format:
(Variable name)     = (value)    % (description) [unit]
%}
OpMargin            = 15.00305;  % Operating Margin [%]
PPE                 = 0.794478;  % Property, Plant, and Equipment ratio
DepreciationTime    = 7;         % Equipment depreciation period [years]
% % SustWorkingCapital  = 3;     % Net working capital period [months] (repeated variable)
% % SustTaxRate         = 27.9;  % Effective tax rate (repeated variable)
EquityOfCap         = 0.5;       % Equity portion of capital
DebtAvail           = 1;         % Defines whether additional debt is available (1 = yes, 0 = no)
DebtRate            = 0.046;     % Debt interest rate

Debt2Equity = 1/EquityOfCap - 1; % Debt-to-equity ratio (placeholder variable)


%%% 1b. COST MODEL: Define 'Top level input' default parameters (#hard-coded):
Efficiency          = 16;        % Module Efficiency [%]
Thickness           = 180;       % Cell thickness [micrometers]
IngotRecycled       = 33.8/100;  % Fraction of ingot recycled before brick stage
IngotNotRecycled    = 1.5/100;   % Fraction of ingot not recycled before brick stage
WireSawWidth        = 130;       % Wire saw cut width [micrometers]
SiPrice             = 23;        % Contract silicon prince [$/kilogram]
OpWage              = 14.5;      % Operator wage [$/hour]
TechWage            = 20.6;      % Technician wage [$/hour]
SupWage             = 27.3;      % Supervisor wage [$/hour]
EmplCostMultiplier  = 1.33;      % Employee cost multiplier (accounts for true employee cost)
Electricity         = 0.069;     % Cost of electricity [$/kWh]
Shipping            = 0;         % Cost of shipping [$/container]
Uptime              = 96;        % Factory uptime [%]
WACC                = 14/100;    % Weighted Average Cost of Capital [%]
PriceInflation      = 0/100;     % Expected price inflation rate
CostInflation       = 0/100;     % Expected cost inflation rate
CostTaxRate         = 27.9/100;  % Effective tax rate
SG_A                = 9.40/100;  % Sales, general and administration (fraction of revenue)
R_D                 = 1.10/100;  % Research and development (fraction of revenue)
CostWorkingCapital  = 3;         % Net working capital period [months]
Price               = 0.851423;  % Selling price [$/Wp]
Capex               = 1;         % Capex multiplier (affects all capex-related costs)
VariableCosts       = 1;         % Variable costs multiplier (affects all variable costs)

SustWorkingCapital  = CostWorkingCapital;   % repeated variable
SustTaxRate         = CostTaxRate*100;      % repeated variable
%{
Note that two variables -- working capital and tax rate -- appear in both the sustainable growth and cost models, which
is why they are repeated here. The script can handle changing either, though pay attention to differing units.
%}


%%% 1c. (Optional) Apply changes to the baseline model (i.e., for scenarios; can #hard-code these values by uncommenting):
%%% LINE-OF-SIGHT:
% % Efficiency          = 18;
% % VariableCosts       = 0.6;
% % Thickness           = 120;
% % Capex               = 0.6;
% % EquityOfCap         = 1/6;    % with debt (EquityOfCap = 1/6 means Debt2Equity = 5)
% % % % DebtRate            = 0.10;    % with more expensive debt


%%% ADVANCED CONCEPT 1 (high efficiency):
% % Efficiency          = 24;
% % VariableCosts       = 0.75;
% % Thickness           = 20;
% % WireSawWidth        = 20;
% % Capex               = 0.29;


%%% ADVANCED CONCEPT 2 (low variable costs):
% % Efficiency          = 18;
% % VariableCosts       = 0.5495;
% % Thickness           = 20;
% % WireSawWidth        = 20;
% % Capex               = 0.29;


%%% 1d. Define cell matrices for the name and values of each parameter (for later convenience):
ParamString = {...
    'Operating Margin', 'PP&E Ratio', 'Depreciation', 'Working Capital',...
    'Tax Rate', 'Debt-to-equity ratio', 'Debt Available', 'Debt Rate',...
    'Module Efficiency', 'Thickness', 'Ingot Recycled', 'Ingot Not Recycled', 'Wire Saw Width',...
    'Silicon price', 'Operator Wage', 'Technician Wage', 'Supervisor Wage', 'Employee Cost Multiplier',...
    'Electricity', 'Shipping', 'Uptime', 'WACC', 'Price Inflation', 'Cost Inflation',...
    'Tax Rate', 'SG&A', 'R&D', 'Working Capital Period', 'Price', 'Capex', 'Variable Costs'};

ActualParams = {...
    OpMargin PPE DepreciationTime SustWorkingCapital SustTaxRate Debt2Equity DebtAvail DebtRate...
    Efficiency Thickness IngotRecycled IngotNotRecycled WireSawWidth SiPrice OpWage TechWage SupWage...
    EmplCostMultiplier Electricity Shipping Uptime WACC PriceInflation CostInflation CostTaxRate SG_A R_D...
    CostWorkingCapital Price Capex VariableCosts};


%%% 1e. Prompt user for a parameter to vary:
disp('PV GROWTH MODEL DEFAULT PARAMETERS: current ''Top level inputs'' (labeled 1-31):')
disp('-------------------------------------------------------------------------------')
fprintf('%s \t%35s \t%2.5f \n','1)','Operating Margin [%]: ',OpMargin)
fprintf('%s \t%35s \t%1.4f \n','2)','PP&E ratio: ',PPE)
fprintf('%s \t%35s \t%2.1f \n','3)','Depreciation [yr]: ',DepreciationTime)
fprintf('%s \t%35s \t%2.1f \n','4)','Working Capital [months]: ',SustWorkingCapital)
fprintf('%s \t%35s \t%2.4f \n','5)','Tax Rate: ',SustTaxRate/100)
fprintf('%s \t%35s \t%2.2f \n','6)','Debt-to-equity ratio: ',Debt2Equity)
fprintf('%s \t%35s \t%d \n','7)','Debt Available? (1 = yes, 0 = no): ',DebtAvail)
fprintf('%s \t%35s \t%1.4f \n','8)','Debt Rate: ',DebtRate)
fprintf('%s \t%35s \t%2.2f \n','9)','Module Efficiency [%]: ',Efficiency)
fprintf('%s \t%31s \t%3.0f \n','10)','Thickness [microns]: ',Thickness)
fprintf('%s \t%31s \t%2.3f \n','11)','Ingot Recycled [%]: ', IngotRecycled)
fprintf('%s \t%31s \t%2.3f \n','12)','Ingot Not Recycled [%]: ', IngotNotRecycled)
fprintf('%s \t%31s \t%2.0f \n','13)','Wire Saw Width [microns]: ', WireSawWidth)
fprintf('%s \t%31s \t%2.2f \n','14)','Silicon price [$/kg]: ',SiPrice)
fprintf('%s \t%31s \t%2.2f \n','15)','Operator Wage [$/hr]: ',OpWage)
fprintf('%s \t%31s \t%2.2f \n','16)','Technician Wage [$/hr]: ',TechWage)
fprintf('%s \t%31s \t%2.2f \n','17)','Supervisor Wage [$/hr]: ',SupWage)
fprintf('%s \t%31s \t%1.2f \n','18)','Employee Cost Multiplier: ',EmplCostMultiplier)
fprintf('%s \t%31s \t%1.4f \n','19)','Electricity [$/kWh]: ',Electricity)
fprintf('%s \t%31s \t%1.1f \n','20)','Shipping [$/container]: ',Shipping)
fprintf('%s \t%31s \t%2.1f \n','21)','Uptime [%]: ',Uptime)
fprintf('%s \t%31s \t%2.3f \n','22)','WACC: ',WACC)
fprintf('%s \t%31s \t%1.3f \n','23)','Price Inflation: ',PriceInflation)
fprintf('%s \t%31s \t%1.3f \n','24)','Cost Inflation: ',CostInflation)
fprintf('%s \t%31s \t%2.4f \n','25)','Tax Rate: ',CostTaxRate)
fprintf('%s \t%31s \t%2.4f \n','26)','SG&A: ',SG_A)
fprintf('%s \t%31s \t%2.4f \n','27)','R&D: ',R_D)
fprintf('%s \t%31s \t%2.1f \n','28)','Working Capital [months]: ',CostWorkingCapital)
fprintf('%s \t%31s \t%2.2f \n','29)','Selling Price [$/W]: ',Price)
fprintf('%s \t%31s \t%2.2f \n','30)','Capex (multiplier): ',Capex)
fprintf('%s \t%31s \t%2.2f \n','31)','Variable Costs (multiplier): ',VariableCosts)

VarParam = input(sprintf(['\nWhich parameter would you like to vary?\nPlease enter a number from 1-31.'...
    '\nOr, to run a baseline calculation, enter ''0'': ']));

if VarParam < 0 || VarParam > 31
    error('ERROR: Please enter an integer between 0-31.')
end

% Always update the 'cost model' versions of tax rate and cost of working capital:
if VarParam == 4
    VarParam = 28;
elseif VarParam == 5 
    VarParam = 25;
end

% Treatment for baseline case (just use baseline value for operating margin):
if VarParam == 0
    ParamVector = OpMargin;
% Special treatment of "Is debt available?" case, which is either "yes" or "no":
elseif VarParam == 7
    disp('Performing two calculations: one with debt, one without.')
    ParamVector = [0 1];
else
    
    
    %%% 1f. Prompt user for minimum and maximum values of varied parameter, as well as the step size:
    ParamRange = zeros(1,2);
    
    disp('Next, select the parameter range (min/max) and number of calculations to perform.')
    ParamRange(1) = input(['Enter MINIMUM value for parameter ''' ParamString{VarParam} ''': ']);
    ParamRange(2) = input(['Enter MAXIMUM value for parameter ''' ParamString{VarParam} ''': ']);
    StepSize = input('Now, enter the step size (value to increment from the MIN to MAX value): ');
    % Set default value for 'StepSize' if it's unspecified:
    if isempty(StepSize)
        StepSize = (ParamRange(2) - ParamRange(1))/10;
    end
    % Define a vector to vary the user-specified parameter from minimum to maximum value:
    ParamVector = (ParamRange(1):StepSize:ParamRange(2))';
end

NumCalcs = length(ParamVector);


%% 2. COST MODEL: Perform scripting to calculate operating margin, PP&E ratio, and other outputs.


%%% 2a. Initialize vectors for model parameters:
% Define vectors for model parameters to be read by Excel.
ExcelDataIn = [OpMargin PPE DepreciationTime SustWorkingCapital SustTaxRate EquityOfCap DebtAvail DebtRate...
    Efficiency Thickness IngotRecycled IngotNotRecycled WireSawWidth SiPrice OpWage TechWage ...
    SupWage EmplCostMultiplier Electricity Shipping Uptime WACC PriceInflation CostInflation CostTaxRate ...
    SG_A R_D CostWorkingCapital Price Capex VariableCosts]';

VarCostsIn     = ones(17,1);
DepreciationIn = ones(4,1);

CostExcelDataOut = zeros(NumCalcs,10);
[PriceOut, VarCostsOut, FixedCostsOut] = deal(zeros(NumCalcs,1));


%%% 2b. Decide how to constrain the price:
    %{
    'KeepConstant' is a switch for constraining the price, operating margin, or PP&E ratio, which are different
    analyses. For the latter two, the price that keeps those variables constant is found and inputted into the cost
    model.
    %}
if VarParam == 29
    KeepConstant = 'A';
elseif VarParam == 1 || VarParam == 0
    KeepConstant = 'B';
elseif VarParam == 2
    KeepConstant = 'C';
else
    KeepConstant = upper(input(...
        'How would you like to constrain the price?: A) price, B) operating margin, or C) PP&E ratio?: ','s'));
end

wb1 = waitbar(0, 'Cost Model calculation progress:', 'name', 'COST MODEL');

for aa = 1:NumCalcs
    
    
    %%% 2c. Write data to cost model Excel file:
    if VarParam~= 0
        ExcelDataIn(VarParam) = ParamVector(aa);
    end
    xlswrite('Cost model for climate.xlsx',ExcelDataIn(3)*DepreciationIn,'C97:C100');   % Depreciation time
    xlswrite('Cost model for climate.xlsx',ExcelDataIn(9:29),'Sheet1','C8:C28');        % Main inputs
    xlswrite('Cost model for climate.xlsx',ExcelDataIn(30),'Sheet1','C102');            % Capex multiplier
    xlswrite('Cost model for climate.xlsx',ExcelDataIn(31)*VarCostsIn,'E44:E60');       % Variable costs multiplier(s)
    
    
    %%% 2d. Apply constraints to price, operating margin, or PP&E ratio when applicable:
    CapexCost           = xlsread('Cost model for climate.xlsx','Sheet1','C35');
    SumVariableCosts    = xlsread('Cost model for climate.xlsx','Sheet1','G61');
    DepreciationCost    = xlsread('Cost model for climate.xlsx','Sheet1','G62');
    
    switch KeepConstant
        case 'A'
                NewPrice = Price;
        case 'B'
            % Calculate new price based on operating margin (taking into account parameters that may be changing):
            if VarParam == 1 || VarParam == 0
                NewPrice = (SumVariableCosts + DepreciationCost) / (1 - ParamVector(aa)/100 - SG_A - R_D);
            elseif VarParam == 26;
                NewPrice = (SumVariableCosts + DepreciationCost) / (1 - OpMargin/100 - ParamVector(aa) - R_D);
            elseif VarParam == 27;
                NewPrice = (SumVariableCosts + DepreciationCost) / (1 - OpMargin/100 - SG_A - ParamVector(aa));
            else
                NewPrice = (SumVariableCosts + DepreciationCost) / (1 - OpMargin/100 - SG_A - R_D);
            end
        case 'C'
            % Calculate new price based on new PP&E ratio (which may be changing):
            if VarParam == 2
                NewPrice = CapexCost/ParamVector(aa);
            else
                NewPrice = CapexCost/PPE;
            end
    end
    % Update the price:
    xlswrite('Cost model for climate.xlsx',NewPrice,'Sheet1','C28');
    
    
    %%% 2e. Extract parameters from the cost model:
    CostExcelDataOut(aa,:)  = xlsread('Cost model for climate.xlsx','Sheet1','C31:C40');
    PriceOut(aa)            = xlsread('Cost model for climate.xlsx','Sheet1','C28');
    VarCostsOut(aa)         = xlsread('Cost model for climate.xlsx','Sheet1','G61');
    FixedCostsOut(aa)       = xlsread('Cost model for climate.xlsx','Sheet1','G65');
    
    waitbar(aa/NumCalcs)
end

delete(wb1)
% Optional display of Excel data parameters
% % disp(CostExcelDataOut);


%% 3. UNCONSTRAINED GROWTH: Run sustainable growth model calculations and calculate growth of PV without demand constraints.


%%% 3a. Define initial conditions and total time for growth of PV (# hard-coded):
InitialCapacity         = 213.335;        % Initial installed PV capacity (end of 2015) [GW]
InitialManufacturing    = 49.6556;        % Initial manufacturing capacity (end of 2015) [GW]
years                   = 15;             % Number of years over which to perform the sustainable growth rate calculation


%%% 3b. Initialize vectors for sustainable growth rate calculations:
SustGR                      = zeros(NumCalcs,1);          % Sustainable growth rate for annual manufacturing [%]
RealGR                      = zeros(NumCalcs,years);      % Real growth rate for annual manufacturing [%]

TotalInstalls               = zeros(NumCalcs,years);      % PV installed capacity [GW]
AnnualManufacturing         = zeros(NumCalcs,years);      % Annual manufacturing capacity [GW]

ConstrainedTotalInstalls    = zeros(NumCalcs,years);      % Constrained PV installed capacity [GW]
ConstrainedAnnualMan        = zeros(NumCalcs,years);      % Constrained annual manufacturing capacity [GW]


%%% 3c. Initialize some variables helpful for plotting:
FinalPVcapacity             = zeros(NumCalcs,1);
ConstrainedFinalPVcapacity  = zeros(NumCalcs,1);

wb2 = waitbar(0, 'Growth Model (unconstrained) calculation progress:', 'name', 'GROWTH MODEL: unconstrained');

for cc = 1:NumCalcs
    
    
    %%% 3d. Use outputs from cost model as inputs into sustainable growth model:
    if VarParam == 6
        % Convert from 'debt-to-equity ratio' to 'equity portion of capital':
        ExcelDataIn(VarParam) = 1/(1 + ParamVector(cc));
    elseif VarParam == 28
        % Update the working capital period for the sustainable growth model:
        ExcelDataIn(4) = ParamVector(cc);
    elseif VarParam == 25
        % Update the tax rate for the sustainable growth model:
        ExcelDataIn(5) = ParamVector(cc)*100;
    elseif VarParam ~= 0
        % Update other parameters (unless running a baseline case):
        ExcelDataIn(VarParam) = ParamVector(cc);
    end
    
    % Set new operating margin and PP&E ratio
    ExcelDataIn(1) = CostExcelDataOut(cc,10);   % Operating Margin
    ExcelDataIn(2) = CostExcelDataOut(cc,6);    % PP&E Ratio
    
    
    %%% 3e. Write to, then read data from the sustainable growth rate model Excel file:
    % Write data to the file:
    xlswrite('SustGR model for climate.xlsx',ExcelDataIn(1:8),'Sheet1','C7:C14');
    
    % Extract the sustainable growth rate:
    SustGR(cc) = xlsread('SustGR model for climate.xlsx','Sheet1','C46');
    
    % Define the real growth rate, which is time-dependent and constrained by demand:
    RealGR(cc,:) = SustGR(cc)*ones(years,1);
    
    
    %%% 3f. Calculate total installed capacity and annual manufacturing over time WITHOUT any demand constraints:
    for dd = 1:years
        
        if dd == 1
            % For the first year, use initial values:
            AnnualManufacturing(cc,dd) = InitialManufacturing;
            TotalInstalls(cc,dd) = InitialCapacity + AnnualManufacturing(cc,dd);
        else    
            
            % All other years: 
            AnnualManufacturing(cc,dd) = AnnualManufacturing(cc,dd-1)*(1 + SustGR(cc)/100);
            
            % Only add annual manufacturing if it's positive:
            if AnnualManufacturing(cc,dd) < 0
                AnnualManufacturing(cc,dd) = 0;
            end
            TotalInstalls(cc,dd) = TotalInstalls(cc,dd-1) + AnnualManufacturing(cc,dd);
        end
    end
    
    FinalPVcapacity(cc) = TotalInstalls(cc,end);

    
    %%% 3g. (Optional) Tracking progress of calculations:
% %     if VarParam ~=0
% %         fprintf('For %s = %2.2f, final PV capacities are %3d GW (unconstrained).\n',...
% %             ParamString{VarParam}, ParamVector(cc), round(TotalInstalls(cc,end)));
% %     end
        
    waitbar(cc/NumCalcs)
    
end
   
delete(wb2)

%% 4. DEMAND CONSTRAINED GROWTH: Calculate real growth of PV with demand constraints.
%{
With a demand constraint, the sustainable growth rate is limited based on capacity at a given price.
%}


%%% 4a. Define the price demand curve (# hard-coded):
InitialPrice = 3;
EndPrice     = 0.05;                    % lowest price [$/Wp]
PriceAdjust  = 0;
% % PriceAdjust = -0.10;            % Price adjustment (e.g., carbon tax) [$/Wp]
% % PriceAdjust = +0.10;            % Price adjustment (e.g., carbon subsidy) [$/Wp]
PriceMultiply = 1;
% % PriceMultiply = 0.75;           % Demand curve shift down
% % PriceMultiply = 1.25;           % Demand curve shift up

PriceDemand = zeros(1E6,2);
PriceDemand(:,2) = linspace(InitialPrice,EndPrice,1E6);
PriceDemand(:,1) = 197155/1000*((PriceDemand(:,2) + PriceAdjust)*PriceMultiply).^-2.735;    % demand curve in capacity [GW]


%%% 4b. Initialize variables for calculations::
TotalCapex           = zeros(NumCalcs,1);
ConstrainedPrice     = zeros(NumCalcs,years);
ConstrainedMargin    = zeros(NumCalcs,years);
ConstrainedPPE       = zeros(NumCalcs,years);
ConstrainedSustGR    = zeros(NumCalcs,years);

wb3 = waitbar(0, 'Growth Model (demand constrained) calculation progress:', 'name', 'GROWTH MODEL: demand constrained');

for ee = 1:NumCalcs
    
    
    %%% 4c. Find demand of PV based on current price:
    CurrentDemandIndex      = find(PriceOut(ee)    < PriceDemand(:,2),1,'last');
    MaxDemandIndex          = find(VarCostsOut(ee) < PriceDemand(:,2),1,'last');
    CurrentCapacityLimit    = PriceDemand(CurrentDemandIndex,1);
    MaxCapacityLimit        = PriceDemand(MaxDemandIndex,1);
      
    
    %%% 4d. Initialize the total capex and constrained price:
    TotalCapex(ee) = CostExcelDataOut(ee,5);
    ConstrainedPrice(ee,:) = PriceOut(ee);
    
    for ff = 1:years
        
        
        %%% 4e. Calculate total installed capacity and annual manufacturing over time WITH demand constraints:
        % For the first year, use initial values:
        if ff == 1
            if CurrentCapacityLimit < InitialCapacity
                % If initial price is too low for current demand, warn about unrealistic condition:
                warning(['Price of ''' num2str(PriceOut(ee))...
                    ''' in year 1 is above the price that corresponds to the demand for the initial capacity.'...
                    ' Model may give unrealistic results.'])
            end
            ConstrainedAnnualMan(ee,ff) = InitialManufacturing;
            ConstrainedTotalInstalls(ee,ff) = InitialCapacity + ConstrainedAnnualMan(ee,ff);
            
        % All other years:    
        else
            ConstrainedAnnualMan(ee,ff) = ConstrainedAnnualMan(ee,ff-1)*(1 + RealGR(ee,ff)/100);
            % Only add annual manufacturing if it's positive:
            if ConstrainedAnnualMan(cc,dd) < 0
                ConstrainedAnnualMan(cc,dd) = 0;
            end
            ConstrainedTotalInstalls(ee,ff) = ConstrainedTotalInstalls(ee,ff-1) + ConstrainedAnnualMan(ee,ff);
        end
        
        
        %%% 4f. Once the growth this year causes capacity to exceed demand,
        %%% recalculate manufacturing and total installations based on demand:
        if  ConstrainedTotalInstalls(ee,ff) > CurrentCapacityLimit
            
            DemandPriceIndex = find(PriceDemand(:,1) > ConstrainedTotalInstalls(ee,ff),1,'first');
            ConstrainedPrice(ee,ff) = PriceDemand(DemandPriceIndex,2);
        end
        
        
        %%% 4g. Based on new demand-constrained price, determine whether price hits variable cost, and whether maximum
        %%% capacities are reached (and thus whether annual manufacturing goes to zero):
        if ConstrainedPrice(ee,ff) <= VarCostsOut(ee)
            % Factories produce until the capacity limit is reached:
            if ff == 1
                % If this condition is triggered in year one, warn about unrealistic condition:
                warning(['Price of ''' num2str(PriceOut(ee))...
                    ''' in year 1 is below variable costs. Model may give unrealistic results.'])
                if (MaxCapacityLimit - InitialCapacity) > InitialManufacturing
                    % The factory will shut down next year, so update the capacity limit:
                    ConstrainedAnnualMan(ee,ff) = InitialManufacturing;
                    MaxCapacityLimit = InitialCapacity + InitialManufacturing;
                elseif (MaxCapacityLimit - InitialCapacity) < InitialManufacturing
                    ConstrainedAnnualMan(ee,ff) = MaxCapacityLimit - InitialCapacity;
                else
                    ConstrainedAnnualMan(ee,ff) = 0;
                    
                end
                ConstrainedTotalInstalls(ee,ff) = InitialCapacity + ConstrainedAnnualMan(ee,ff);
                
            else
                % Recalculate annual manufacturing from capacity cap (sell off remaining modules before shutdown):
                ConstrainedAnnualMan(ee,ff) = MaxCapacityLimit - ConstrainedTotalInstalls(ee,ff-1);
                
                % Only add annual manufacturing if it's positive:
                if ConstrainedAnnualMan(cc,dd) < 0
                    ConstrainedAnnualMan(cc,dd) = 0;
                end
                ConstrainedTotalInstalls(ee,ff) = ConstrainedTotalInstalls(ee,ff-1) + ConstrainedAnnualMan(ee,ff);           
                                
                % Recalculate the growth rate (condition: if growth rate in previous year was zero, avoid dividing by
                % zero this year)
                if ConstrainedAnnualMan(ee,ff-1) == 0
                    RealGR(ee,ff) = 0;
                else
                    RealGR(ee,ff) = (ConstrainedAnnualMan(ee,ff) - ConstrainedAnnualMan(ee,ff-1))/ConstrainedAnnualMan(ee,ff-1)*100;
                end
            end
        end
        
        
        %%% 4h. Recalculate the operating margin, PP&E ratio, and sustainable growth rate based on NEW constrained capacity
        %%% and price for next year (unless it's the last year):
        if ff < years
            
            % If currently changing a sustainable growth rate model parameter, remember not to forget about it:
            if VarParam == 6
                % Convert from 'debt-to-equity ratio' to 'equity portion of capital':
                ExcelDataIn(VarParam) = 1/(1 + ParamVector(ee));
            elseif VarParam == 28
                % Update the working capital period for the sustainable growth model:
                ExcelDataIn(4) = ParamVector(ee);
            elseif VarParam == 25
                % Update the tax rate for the sustainable growth model:
                ExcelDataIn(5) = ParamVector(ee)*100;
            elseif VarParam ~= 0
                ExcelDataIn(VarParam) = ParamVector(ee);
            end
            
            % Update the constrained margin and PP&E ratio:
            ConstrainedMargin(ee,ff) = ((ConstrainedPrice(ee,ff) - VarCostsOut(ee) - FixedCostsOut(ee))/ConstrainedPrice(ee,ff))*100;
            ConstrainedPPE(ee,ff) = TotalCapex(ee)/ConstrainedPrice(ee,ff);    % = Total Capex / Price
            
            ExcelDataIn(1) = ConstrainedMargin(ee,ff);
            ExcelDataIn(2) = ConstrainedPPE(ee,ff);
            
            % Write the updated values to the sustainable growth rate model:
            xlswrite('SustGR model for climate.xlsx',ExcelDataIn(1:8),'Sheet1','C7:C14');
            
            % Extract the demand-constrained sustainable growth rate for next year:
            ConstrainedSustGR(ee,ff) = xlsread('SustGR model for climate.xlsx','Sheet1','C46');
            RealGR(ee,ff+1) = ConstrainedSustGR(ee,ff);
        end
        
    end

    
    %%% 4i. (Optional) Tracking calculation progress:
% %     if VarParam ~= 0
% %         fprintf('For %s = %2.2f, final PV capacities are %3d GW (unconstrained) and %3d GW (constrained).\n',...
% %             ParamString{VarParam}, ParamVector(ee), round(TotalInstalls(ee,end)), round(ConstrainedTotalInstalls(ee,end)));
% %     end
    waitbar(ee/NumCalcs)
    
end

delete(wb3)


%% 5. PLOTTING AND FORMATTING.


%%% 5a. Define plot settings
if VarParam ~= 0
    Plotvariable = ActualParams{VarParam};
end

% Change default settings (can #hard-code in)
set(0,'DefaultLineLineWidth',1.5)
set(0,'DefaultFigureColor','w')
set(0,'DefaultAxesLineWidth',0.5)
set(0,'DefaultAxesFontName', 'Helvetica')
set(0,'DefaultLegendLineWidth',0.5)
set(0,'DefaultAxesFontSize',10)

if VarParam ~= 0
    f_array = cell(3,1);
else
    f_array = cell(1,1);
end
f_array{1} = figure('Name','Growth rate of PV over time','Color','w','Position',[500 50 600 700]);

plotcolors           = jet(NumCalcs);           %# can change the colormap for plotting here
legendlabel          = cell(NumCalcs,1);
legendentries        = zeros(NumCalcs,1);


%%% 5b. Plot the growth rate over time:
for ff = 1:NumCalcs
    ConstrainedFinalPVcapacity(ff) = ConstrainedTotalInstalls(ff,end);
    yearplot = 1:years;
    yearplot = yearplot + 2015;
    
    a1 = subplot(4,1,1:2);
    plot(yearplot,TotalInstalls(ff,:)/1000,'--','Color',plotcolors(ff,:));
    hold on
    legendentries(ff) = plot(yearplot,ConstrainedTotalInstalls(ff,:)/1000,'-','Color',plotcolors(ff,:));
    hold on
    
    a2 = subplot(4,1,3);
    plot(yearplot,AnnualManufacturing(ff,:)/1000,'--',...
        yearplot,ConstrainedAnnualMan(ff,:)/1000,'-','Color',plotcolors(ff,:));
    hold on
    
    a3 = subplot(4,1,4);
    plot(yearplot,RealGR(ff,:),'-','Color',plotcolors(ff,:))
    hold on
    
    legendlabel{ff} = num2str(ParamVector(ff));
end


%%% 5c. Apply formatting to figure:
axes(a1)
if VarParam ~= 0
    title(sprintf(['effect of ' ParamString{VarParam} '\n(dotted = no demand, solid = with demand constraints)']))
    legend(legendentries,legendlabel,'Location','Northwest')
else
    title(sprintf('baseline calculation\n(dotted = no demand, solid = with demand constraints)'))
end
grid on
ylabel(sprintf('PV installed capacity [TW]'))

axes(a2)
grid on
ylabel(sprintf('manufacturing\ncapacity [TW]'))

axes(a3)
grid on
ylabel(sprintf('actual\ngrowth rate [%%]'))
xlabel('years')


if VarParam ~= 0
    %%% 5d. Sensitivity of PV to different variables (using values as x-axis)
    f_array{2} = figure('Name','Sensitivity of PV growth (values)','color','w','Position',[200 100 600 600]);
    
    subplot(3,1,1);
    plot(ParamVector,SustGR,'--','Color','b');
    hold on
    grid on
    legend(ParamString{VarParam})
    ylabel(sprintf('unconstrained\nsustainable growth rate [%%]'))
    xlabel(num2str(ParamString{VarParam}))
    
    subplot(3,1,2)
    plot(ParamVector,FinalPVcapacity/1000,'--','Color','b');
    hold on
    grid on
    legend(ParamString{VarParam})
    ylabel(sprintf('unconstrained\ntotal PV capacity [TW]'))
    xlabel(num2str(ParamString{VarParam}))
    
    subplot(3,1,3)
    plot(ParamVector,ConstrainedFinalPVcapacity/1000,'-','Color','b');
    hold on
    grid on
    legend(ParamString{VarParam})
    ylabel(sprintf('constrained\ntotal PV capacity [TW]'))
    xlabel(num2str(ParamString{VarParam}))
    
    
    %%% 5e. Sensitivity of PV to different variables (using % change as x-axis)
    f_array{3} = figure('Name','Sensitivity of PV growth (% change)','color','w','Position',[800 100 600 600]);
    subplot(3,1,1)
    plot((ParamVector-Plotvariable)./Plotvariable*100,SustGR,'--','Color','b');
    hold on
    grid on
    legend(ParamString{VarParam})
    ylabel(sprintf('unconstrained\nsustainable growth rate [%%]'))
    xlabel(['percent change in ' num2str(ParamString{VarParam})])
    
    
    subplot(3,1,2)
    plot((ParamVector-Plotvariable)./Plotvariable*100,FinalPVcapacity/1000,'--','Color','b');
    hold on
    grid on
    legend(ParamString{VarParam})
    ylabel(sprintf('unconstrained\ntotal PV capacity [TW]'))
    xlabel(['percent change in ' num2str(ParamString{VarParam})])
    
    
    subplot(3,1,3)
    plot((ParamVector-Plotvariable)./Plotvariable*100,ConstrainedFinalPVcapacity/1000,'-','Color','b');
    hold on
    grid on
    legend(ParamString{VarParam})
    ylabel(sprintf('constrained\ntotal PV capacity [TW]'))
    xlabel(['percent change in ' num2str(ParamString{VarParam})])
    
    % % %%% 5f. Optional "sensitivity" output (calculates slope)
    % % SensitiveSlope =  ((SustGR(end) - SustGR(1))/SustGR(1)) / ((ParamVector(end) - ParamVector(1))/ParamVector(1));
    % % disp(['Sensitivity = ' num2str(SensitiveSlope)])
end


%% 6. (Optional) SAVE the workspace and figures.


%%% 6a. Prompt for saving data
saveTrigger = input('\nSave data and figures? (y/n): ','s');


%%% 6b. Save all current figures:
if strcmpi(saveTrigger,'y')
    ChooseDir = uigetdir;
    for gg = 1:length(f_array)
        fname = get(f_array{gg},'name');
        savepath = [ChooseDir '\' fname '.fig'];
        saveas(f_array{gg},savepath)
    end
    
    
    %%% 6c. Save the workspace into the current directory:
    clear a1 a2 a3 f_array PriceDemand     % Clear large and unimportant variables
    
    nowtime = clock;
    CurrentFolder = cd(ChooseDir);
    if VarParam ~= 0
        save([ParamString{VarParam} ' ' num2str(nowtime(2)) num2str(nowtime(3)) num2str(nowtime(4)) num2str(nowtime(5))])
    else
        save(['baseline ' num2str(nowtime(2)) num2str(nowtime(3)) num2str(nowtime(4)) num2str(nowtime(5))])
    end
    cd(CurrentFolder);
end
