# -*- coding: utf-8 -*-
"""# YaSoFo - Yet Another SOlar Fuels Optimizer #

Version 1.0 (19. Jan. 2017)

Author: Matthias May (2016-2017)

License: Creative Commons Attribution-NonCommercial-ShareAlike 4.0 International
http://creativecommons.org/licenses/by-nc-sa/4.0/

Updates and more info can be found under https://bitbucket.org/YaSoFo/yasofo

Tested with Python 3.5 with SiPy (www.scipy.org).
Further reference spectra can be downloaded e.g. from Fraunhofer ISE, see README.md.

Assumptions:

* Monolithic stack, i.e. current matching.
* Absorption is unity above the bandgap, 0 below.
* Constant reduction of open-circuit voltage per absorber due to recombination.
* One-dioade equation is matched with catalyst characteristic to obtain the operating current.
* For more details of the model, see publication below.

Please cite May et al., Sustainable Energy & Fuels (2017). DOI:10.1039/C6SE00083E
"""

import numpy as np
import matplotlib.pyplot as plt
import scipy.constants as phys_const
import scipy.special.lambertw as lambertw

# general functions

def spectrum(filename):
    """Import spectrum from etaOpt-Format and convert from W*m^-2*µm^-1] to
    photons*m^-2*(nm*s)^-1.
    """
    try:
        spectrum_orig = np.genfromtxt(filename)
        spectrum_new = np.array(spectrum_orig, copy=True)
        spectrum_new[:, 1] = spectrum_orig[:, 1]*spectrum_new[:, 0]*10**(-9)*\
        10**(-3)/(phys_const.h*phys_const.c)
        return spectrum_new
    except IOError:
        print('Error: Spectrum file ', filename, ' not found!')
        return None

# for convenience, import standard data from file am15g.dat (see README)
am15g = spectrum('am15g.dat')
# [exchange current density, Tafel slope, ohmic resistivity] exp. det.
IrO2 = [7.09071e-07, 0.036486402, 0.0]

def nm_to_ev(wavelength):
    """Convert wavelength in nm to electronvolts."""
    ev = phys_const.h*phys_const.c/(wavelength*phys_const.e*10**(-9))
    return ev

def ev_to_nm(ev):
    """Convert electronvolts to wavelength in nm."""
    wavelength = phys_const.h*phys_const.c/(ev*phys_const.e*10**(-9))
    return wavelength

def integrate_flux(data, upper, lower=0, electronvolts=True):
    """Integrate photon flux between two limits.
    Output: Current in A per cm^2. External quant. eff. is assumed to be unity.

    Parameters
    ----------
    data: numpy.ndarray
        the spectrum
    upper: float
        Upper integration limit
    lower: float
        lower integration limit
    electronvolts: bool
        Limits in electronvolts? If False in nm.
    """
    if lower == 0:
        lower = 0.0001 # avoid division by zero
    if electronvolts is False:
        upper_lambda = upper
        lower_lambda = lower
    else:
        lower_lambda = ev_to_nm(upper)
        upper_lambda = ev_to_nm(lower)
        if upper_lambda > data[-1, 0]:
            upper_lambda = data[-1, 0]
    upper_x_index = np.where(data[:, 0] >= upper_lambda)[0][0]
    lower_x_index = np.where(data[:, 0] > lower_lambda)[0][0]
    # Integrate photon flux by trapezoidal rule (as requested by etaopt Format)
    # Check if this is also true for your file format!
    photon_number = np.trapz(data[lower_x_index:upper_x_index, 1],
                             data[lower_x_index:upper_x_index, 0])
    current = photon_number*phys_const.e/10000
    return current

def match_IrO2_diode_current(deltaG, j_ph, U_oc, cat_para=IrO2, n=1.0, temp=300):
    """Matches the diode current to CV (+ohmic drop) of a catalyst.
       Default: Exp. det. IrO2 with no ohmic drop.
       Returns current where catalyst CV and diode current intersect.

       Parameters
       ----------
       deltaG: float
           Target DeltaG.
       j_ph: float
           Saturation photocurrent in A cm^-2.
       U_oc: float
           Open-circuit voltage in V.
       cat_para: list
           Catalyst parameters: [exchange current density, Tafel slope,
           ohmic resistivity]. Default: Exp. IrO2 with no ohmic resistivity.
       n: float
           Diode ideality factor. Default: 1.0
       temp: int
           Temperature. Default: 300K.
       """
    if deltaG > U_oc:
        current = 0.0
    else:
        #stepsize for voltage range 5mV
        U = np.arange(deltaG, U_oc + 0.01, 0.005)
        U_T = phys_const.k*temp/phys_const.e
        diode_current = -j_ph*(1-(np.exp(U/(n*U_T))-1)/((np.exp(U_oc/(n*U_T))-1)))
        if cat_para[2] < 1e-10: # no ohmic drop
            cat_current = -cat_para[0]*np.exp((U-deltaG)/cat_para[1])
        else:
            cat_current = -(cat_para[1]/cat_para[2])*np.real(lambertw(cat_para[0]*
                cat_para[2]*np.exp((U-deltaG)/cat_para[1])/cat_para[1]))
       # find the voltage, where the difference between catalyst current and
	  # one-diode current is smallest
        curr_element = np.where(abs(diode_current-cat_current) == min(abs(diode_current -
            cat_current)))[0][0]
        current = -diode_current[curr_element]
    return current

def mod_spectrum(filename, data=am15g, header=1):
    """Modify an existing spectrum by multiplying it with a filter from file.
    Returns modified spectrum.

    Parameters
    ----------
    filename: str
        Filename where the filter ist to be found. Filter file format:
        [wavelength, transmittance].
    data: numpy.ndarray
        The initial spectrum.
    header: int
        Number of header lines to ignore in filter file.
    """
    filter_function = np.genfromtxt(filename, skip_header=header)
    if max(filter_function[:, 1]) > 1.0:
        print('Warning: at least one value of filter greater 1!')
    if min(filter_function[:, 1]) < 0.:
        print('Warning: at least one value of filter negative!')
    spec = np.array(data, copy=True)
    transmittance = np.array(spec, copy=True)
    #interpolate the transmittance to the x-values of the spectrum
    transmittance[:, 1] = np.interp(transmittance[:, 0], filter_function[:, 0],
        filter_function[:, 1])
    spec[:, 1] = data[:, 1]*transmittance[:, 1]
    return spec

def mod_spectrum_by_abs(filename, thickness, data=am15g, header=5, reverse=True):
    """Modify an existing spectrum by the absorption e.g. of an electrolyte
    by the application of Beer's law.
    For absorbance of liquid water, see e.g. Pope and Fry, Appl. Opt. 36 (1997)
    (380-700 nm) or Kou et al, Appl. Opt. 32 (1993) (667-2500 nm).
    http://omlc.org/spectra/water has a nice compilation.

    Parameters
    ----------
    filename: str
        Filename where the tabulated absorbance is to be found. Format:
        [Wavelength (nm), absorption coefficient (1/cm)]
    thickness: float
        Thickness of the absorbing layer in cm.
    data: numpy.ndarray
        The spectrum.
    header: int
        Header lines of input file to ignore.
    reverse: bool
        Reverse oder? Values should be ordered from lowest to highest lambda.
    """
    spec = np.array(data, copy=True)
    abs_coefficient = np.genfromtxt(filename, skip_header=header)
    if reverse is True:
        abs_coefficient = abs_coefficient[::-1]
    transmittance = np.array(abs_coefficient, copy=True)
    transmittance_interp = np.array(spec, copy=True)
    # apply Beer's law
    transmittance[:,1] = np.exp(-abs_coefficient[:,1]*thickness)
    # interpolate (linearly!) the values of transmittance to x-axis of spectrum
    transmittance_interp[:, 1] = np.interp(transmittance_interp[:, 0],
        transmittance[:, 0], transmittance[:, 1])
    spec[:, 1] = data[:, 1]*transmittance_interp[:, 1]
    return spec

def mpp_single_diode_current(j_ph, U_oc, n=1.0, temp=300):
    """Finds the maximum power point by employing a single diode current.
    Returns resulting power, U_mpp, j_mpp.

    Parameters
    ----------
    U_oc: float
        Open-circuit voltage
    j_ph: float
        Photocurrent.
    n: float
        Diode ideality factor.
    temp: float
        Temperature in K.
    """
    U = np.arange(U_oc - 2.0, U_oc + 0.01, 0.001)
    U_T = phys_const.k*temp/phys_const.e
    # use one-diode equation, calculate power, and take max as MPP
    diode_current = -j_ph*(1-(np.exp(U/(n*U_T))-1)/((np.exp(U_oc/(n*U_T))-1)))
    power = -diode_current*U
    max_power = max(power)
    if U_oc <= 0.0:
        max_power = 0.0
    U_mpp = U[np.nonzero(power == max(power))[0][0]]
    j_mpp = diode_current[np.nonzero(power == max(power))[0][0]]
    return max_power, U_mpp, j_mpp

def plot_spectrum(data=am15g):
    """Plots the spectrum as photon flux over wavelength."""
    plt.plot(data[:,0], data[:,1])
    plt.xlabel('Wavelength / nm')
    plt.ylabel('Flux / Photons nm$^{-1}$m$^{-2}$s$^{-1}$')
    plt.show()

def show_output(power, best_comb):
    """Produces a simple plot of the results. Assumes that spectrum is
    normalised to 1000 W/m^2.

    Parameters
    ----------
    power: list
        Output from calculation, actual data.
    best_comb: list
        Output from calculation, best combination.
    """
    print('Highest STF efficiency:\n E0:', best_comb[0], '\n Bandgaps (eV): ',
          best_comb[1:-1], '\n STF (%): ', best_comb[-1]*1000)
    plt.plot(power[:, 0], power[:, 1]*1000)
    plt.xlabel('$E_0$ / eV')
    plt.ylabel('Max. efficiency / %')
    plt.show()

# single junction functions

def power_single_E0(data, E0, voltage_loss=0.4, cat_para=IrO2, ideality=1.0):
    """Calculates the power for a single junction with given E0, overpotental
       and voltage loss.
       Returns: Best combination. [band_gap, current, current*E0].

    Parameters
    ----------
    data: numpy.ndarray
        The spectrum.
    E0: float
        DeltaG.
    voltage_loss: float
        The photovoltage loss per junction compared to the bandgap in V.
    cat_para: list
        Catalyst parameters: [exchange current density, Tafel slope,
        ohmic resistivity]. Default: Exp. IrO2 with no ohmic resistivity.
    ideality: float
        Diode ideality factor.
    """
    current = 0
    single_junction = [0, 0, 0]
    # probe bandgaps in the range from E0 to E0 + 2 eV
    bandgap_range = np.arange(E0, E0 + 2, 0.01)
    for band_gap in bandgap_range:
        if band_gap < 2*voltage_loss:
            U_oc = band_gap/2
        else:
            U_oc = band_gap - voltage_loss
        photo_current = integrate_flux(data, 99, band_gap)
        current = match_IrO2_diode_current(E0, photo_current, U_oc, cat_para, ideality)
        if current > single_junction[1]:
            single_junction[1] = current
            single_junction[0] = band_gap
    single_junction[2] = single_junction[1]*E0
    return single_junction

def max_power_single_E0(data=am15g, min_E0=0.1, max_E0=2.0, stepsize=0.05,
                        voltage_loss=0.4, cat_para=IrO2, ideality=1.0, plotting=True):
    """Calculates maximum theoretical efficiencies for single junctions as a
    function of E0. Returns [E0, power, band_gap, current], best combination
    [E0, Eg, power(W/cm^2)].

    Parameters
    ----------
    data: numpy.ndarray
        The spectrum.
    min_E0: float
        Minimum E0 to be considered.
    max_E0: float
        Maximum E0 to be considered.
    stepsize: float
        Steysize in V by which E0 is scanned.
    voltage_loss: float
        The photovoltage loss per junction compared to the bandgap in V.
    cat_para:
        Catalyst parameters: [exchange current density, Tafel slope,
        ohmic resistivity]. Default: Exp. IrO2 with no ohmic resistivity.
    ideality: float
        Diode ideality factor.
    """
    E0_range = np.arange(min_E0, max_E0, stepsize)
    power = []
    best_comb = [0.0, 0.0, 0.0]
    for E0 in E0_range:
        temp_power = power_single_E0(data, E0=E0, voltage_loss=voltage_loss,
                                     cat_para=cat_para, ideality=ideality)
        power.append([E0, temp_power[2], temp_power[0], temp_power[1]])
        if temp_power[-1] > best_comb[-1]:
            best_comb = [E0, temp_power[1], temp_power[-1]]
    power = np.array(power)
    if plotting==True:
        show_output(power, best_comb)
    return power, best_comb

def calc_single_junction(data=am15g, power_E0=True, Eg=2.26, E0=1.23,
                         voltage_loss=0.4, cat_para=IrO2, ideality=1.0):
    """Calculates max. STF efficiency for a single junction with given bandgap
    and E0. Returns [power(W/cm^2), current(A/cm^2)].

    Parameters
    ----------
    data: numpy.ndarray
        The spectrum.
    power_E0: bool
        Is the power determined by current*E0? Otherwise: j*V_MPP.
    Eg: float
        Bandgap (eV).
    E0: float
        E0 to be considered.
    voltage_loss: float
        The photovoltage loss per junction compared to the bandgap in V.
    cat_para:
        Catalyst parameters: [exchange current density, Tafel slope,
        ohmic resistivity]. Default: Exp. IrO2 with no ohmic resistivity.
    ideality: float
        Diode ideality factor.
    """
    if Eg < 2*voltage_loss:
        U_oc = Eg/2
    else:
        U_oc = Eg - voltage_loss
    photo_current = integrate_flux(data, 99, Eg)
    if power_E0 is True:
        current = match_IrO2_diode_current(E0, photo_current, U_oc, cat_para, ideality)
        power = current*E0
        print('STF efficiency (%) with bandgap Eg=',Eg, 'eV for E0=', E0,
              'eV:', power*1000, ';\nCurrent (A/cm^2): ', current)
    else:
        [power, U_mpp, current] = mpp_single_diode_current(photo_current,
            U_oc, ideality, temp=300)
        print('Photovoltaic efficiency (%):', power*1000, ';\nCurrent (A/cm^2): ',
              current, '.\n')
    return power, current

#double junction functions

def find_max_double(data=am15g, power_E0=True, E0=0, voltage_loss=0.4,
                    allow_thinning=True, cat_para=IrO2, ideality=1.0,
                    only_print_results=False):
    """Find the maximum current for a double junction at a given E0.
    Returns [Eg_hi, Eg_lo, j] and [Eg_hi, Eg_lo, j, P].

    Parameters
    ----------
    data: numpy.ndarray
        The spectrum.
    power_E0: bool
        Is the power determined by current*E0? Otherwise: j*V_MPP.
    min_E0: float
        Minimum E0 to be considered.
    max_E0: float
        Maximum E0 to be considered.
    stepsize: float
        Steysize in V by which E0 is scanned.
    voltage_loss: float
        The photovoltage loss per junction compared to the bandgap in V.
    allow_thinning: bool
        Shall we allow the top absorber to be thinned to transmit light to
        subsequent cell and thereby minimise current mismatch?
    cat_para:
        Catalyst parameters: [exchange current density, Tafel slope,
        ohmic resistivity]. Default: Exp. IrO2 with no ohmic resistivity.
    ideality: float
        Diode ideality factor.
    """
    current = 0.0
    photo_current = [0.0, 0.0]
    power = 0.0
    # we store the overall best combination regarding the current/power here
    best_combination_current = [0.0, 0.0, 0.0]
    best_combination_power = [0.0, 0.0, 0.0, 0.0]
    # go through all the wavelengths for the upper cell
    for lower_lambda in data[1:-2, 0]:
        i = np.where(data[:, 0] > lower_lambda)[0][0]
        # bottom cell cannot have higher bandgap than top cell
        for higher_lambda in data[i:, 0]:
            higher_gap = nm_to_ev(lower_lambda)
            lower_gap = nm_to_ev(higher_lambda)
            if lower_gap < 2*voltage_loss:
                U_oc = higher_gap + lower_gap/2 - voltage_loss
            else:
                U_oc = higher_gap + lower_gap - 2*voltage_loss
            # bottom cell
            photo_current[1] = integrate_flux(data, higher_lambda,
                lower_lambda, electronvolts=False)
            # top cell
            photo_current[0] = integrate_flux(data, lower_lambda, data[0, 0],
                electronvolts=False)
            # current matching condition
            min_photo_current = min([photo_current[0], photo_current[1]])
            if (allow_thinning is True) and (photo_current[0] > photo_current[1]):
                # make top cell thinner to transmit light to bottom cell
                average_current = 0.5*(photo_current[0] + photo_current[1])
                min_photo_current = average_current
            if power_E0 is False:
                [power, U_mpp, current] = mpp_single_diode_current(min_photo_current,
                    U_oc, ideality, temp=300)
            else:
                current = match_IrO2_diode_current(E0, min_photo_current, U_oc,
                                                      cat_para, ideality)
                if current > best_combination_current[2]:
                    best_combination_current = [higher_gap, lower_gap, current]
                if U_oc < E0:
                    power = 0.0
                else:
                    power = current*E0
            if power > (best_combination_power[3]):
                    best_combination_power = [higher_gap, lower_gap, current, power]
    if only_print_results is True:
        if power_E0 is True:
            print('Highest STF efficiency at E0=', E0,'eV :', best_combination_power[-1]*1000,
                '%.\n Bandgaps (eV): ', best_combination_power[0:2])
        else:
            print('Highest photovoltaic efficiency of :', best_combination_power[-1]*1000,
                '%.\n Bandgaps (eV): ', best_combination_power[0:2])
        return None
    return best_combination_current, best_combination_power

def max_power_double_E0(data=am15g, min_E0=1.0, max_E0=3.0, stepsize=0.05,
                        voltage_loss=0.4, allow_thinning=True, cat_para=IrO2,
                        ideality=1.0, plotting=True):
    """Calculates max. powers for double junctions as a function of E0. Returns
       [E0, power, Eg1, Eg2, current].

    Parameters
    ----------
    data: numpy.ndarray
        The spectrum.
    min_E0: float
        Minimum E0 to be considered.
    max_E0: float
        Maximum E0 to be considered.
    stepsize: float
        Steysize in V by which E0 is scanned.
    voltage_loss: float
        The photovoltage loss per junction compared to the bandgap in V.
    allow_thinning: bool
        Shall we allow the top absorber to be thinned to transmit light to
        subsequent cell and thereby minimise current mismatch?
    cat_para:
        Catalyst parameters: [exchange current density, Tafel slope,
        ohmic resistivity]. Default: Exp. IrO2 with no ohmic resistivity.
    ideality: float
        Diode ideality factor.
    plotting: bool
        Plot results at the end?
    """
    E0_range = np.arange(min_E0, max_E0, stepsize)
    power = []
    best_comb = [0.0, 0.0, 0.0, 0.0]
    for E0 in E0_range:
        print(E0, flush=True)
        temp_power = find_max_double(data, True, E0, voltage_loss,
                                     allow_thinning, cat_para, ideality)
        if temp_power[1][3] > best_comb[3]:
            best_comb = [E0, temp_power[1][0], temp_power[1][1], temp_power[1][3]]
        power.append([E0, temp_power[1][3], temp_power[1][0], temp_power[1][1],
                      temp_power[1][2]])
    power = np.array(power)
    print('Finished!')
    if plotting is True:
        show_output(power, best_comb)
    return power, best_comb

def best_gaps_double(data=am15g, power_E0=True, E0=1.23, voltage_loss=0.4,
                     allow_thinning=True, cat_para=IrO2, ideality=1, plotting=True):
    """Calculate 2d plot of efficiency vs. gaps for a double junction. Returns
    [gap axis, efficiency].

    Parameters
    ----------
    data: numpy.ndarray
        The spectrum.
    power_E0: bool
        Is the power determined by current*E0? Otherwise: j*V_MPP.
    min_E0: float
        Minimum E0 to be considered.
    max_E0: float
        Maximum E0 to be considered.
    stepsize: float
        Steysize in V by which E0 is scanned.
    voltage_loss: float
        The photovoltage loss per junction compared to the bandgap in V.
    allow_thinning: bool
        Shall we allow the top absorber to be thinned to transmit light to
        subsequent cell and thereby minimise current mismatch?
    cat_para:
        Catalyst parameters: [exchange current density, Tafel slope,
        ohmic resistivity]. Default: Exp. IrO2 with no ohmic resistivity.
    ideality: float
        Diode ideality factor.
    plotting: bool
        Plot results at the end?
    """
    current = 0.0
    photo_current = [0.0, 0.0] # for [higher_gap,lower_gap]
    efficiency = np.zeros([len(data), len(data)])
    gap_axis = nm_to_ev(data[:, 0])
    for lower_lambda in data[1:-2, 0]:
        i = np.where(data[:, 0] > lower_lambda)[0][0]
        for higher_lambda in data[i:, 0]:
            j = np.where(data[:, 0] == higher_lambda)[0][0]
            higher_gap = nm_to_ev(lower_lambda)
            lower_gap = nm_to_ev(higher_lambda)
            if lower_gap < 2*voltage_loss:
                U_oc = higher_gap + lower_gap/2 - voltage_loss
            else:
                U_oc = higher_gap + lower_gap - 2*voltage_loss
            photo_current[1] = integrate_flux(data, higher_lambda, lower_lambda,
                electronvolts=False)
            photo_current[0] = integrate_flux(data, lower_lambda, data[0, 0],
                electronvolts=False)
            min_photo_current = min([photo_current[0], photo_current[1]])
            if (allow_thinning is True) and (photo_current[0] > photo_current[1]):
                average_current = photo_current[1] + 0.5*(photo_current[0] - 
                    photo_current[1])
                min_photo_current = average_current
            if power_E0 is True:
                current = match_IrO2_diode_current(E0, min_photo_current, U_oc,
                                                   cat_para, ideality)
                if U_oc < E0:
                    efficiency[i, j] = 0
                else:
                    efficiency[i, j] = current*E0
            else:
                mpp = mpp_single_diode_current(min_photo_current, U_oc,
                                               ideality, temp=300)
                efficiency[i, j] = mpp[0]
    if plotting is True:
        fig, axes = plt.subplots(nrows=1,ncols=1)
        p1 = axes.pcolor(gap_axis, gap_axis, efficiency*1000, cmap=plt.cm.jet,
                         vmin=abs(efficiency*1000).min(), vmax=abs(efficiency*1000).max())
        axes.set_xlabel("Bottom cell bandgap / eV", fontweight='bold', fontsize=14)
        axes.set_ylabel("Top cell bandgap / eV", fontweight='bold', fontsize=14)
        axes.tick_params(color='red')
        fig.subplots_adjust(right=0.85)
        cbar_ax = fig.add_axes([0.9, 0.15, 0.03, 0.7])
        cb=fig.colorbar(p1, cax=cbar_ax)
        if power_E0 is True:
            cb.set_label('STF / %')
        else:
            cb.set_label('Photovolt. eff. / %')
        plt.show()
    return(gap_axis, efficiency)

def calc_double_junction(data=am15g, power_E0=True, Eg1=2.26, Eg2=1.12, E0=1.23,
                         voltage_loss=0.4, allow_thinning=True, cat_para=IrO2,
                         ideality=1.0):
    """Calculates max. STF efficiency for a double junction with two given
    bandgaps and E0. Returns [power(W/cm^2), current(A/cm^2)].

    Parameters
    ----------
    data: numpy.ndarray
        The spectrum.
    power_E0: bool
        Is the power determined by current*E0? Otherwise: j*V_MPP.
    Eg1: float
        Top cell bandgap (eV).
    Eg2: float
        Bottom cell bandgap (eV).
    E0: float
        E0 to be considered.
    voltage_loss: float
        The photovoltage loss per junction compared to the bandgap in V.
    allow_thinning: bool
        Shall we allow the top absorber to be thinned to transmit light to
        subsequent cell and thereby minimise current mismatch?
    cat_para:
        Catalyst parameters: [exchange current density, Tafel slope,
        ohmic resistivity]. Default: Exp. IrO2 with no ohmic resistivity.
    ideality: float
        Diode ideality factor.
    """
    photo_current = [0.0, 0.0]
    if Eg2 < 2*voltage_loss:
        U_oc = Eg1 + Eg2/2 - voltage_loss
    else:
        U_oc = Eg1 + Eg2 - 2*voltage_loss
    photo_current[0] = integrate_flux(data, 99, Eg1)
    photo_current[1] = integrate_flux(data, Eg1, Eg2)
    min_photo_current = min([photo_current[0], photo_current[1]])
    if (allow_thinning is True) and (photo_current[0] > photo_current[1]):
        average_current = 0.5*(photo_current[0] + photo_current[1])
        min_photo_current = average_current
    if power_E0 is True:
        current = match_IrO2_diode_current(E0, min_photo_current, U_oc, cat_para, ideality)
        power = current*E0
        print('STF efficiency (%) with bandgaps Eg1, Eg2 =', Eg1, ',', Eg2,
        'eV for E0=', E0, 'eV:', power*1000, ';\nCurrent (A/cm^2): ', current)
    else:
        [power, U_mpp, current] = mpp_single_diode_current(min_photo_current,
            U_oc, ideality, temp=300)
        print('Photovoltaic efficiency (%) with bandgaps Eg1, Eg2 =', Eg1, ',',
              Eg2, 'eV:', power*1000, ';\nCurrent (A/cm^2): ', current, '.\n')
    return power, current

#triple junction functions

def max_power_triple_E0(data=am15g, min_E0=1.0, max_E0=3.0, stepsize=0.05,
                        voltage_loss=0.4, allow_thinning=True, cat_para=IrO2,
                        ideality=1.0, homotop=False, plotting=True):
    """Calculates powers for three junctions. Returns [E0, P, Gap1, Gap2,
       Gap3, current] and best combination [E0, Eg1, Eg2, power(W/cm^2)].

    Parameters
    ----------
    data: numpy.ndarray
        The spectrum.
    min_E0: float
        Minimum E0 to be considered.
    max_E0: float
        Maximum E0 to be considered.
    stepsize: float
        Steysize in V by which E0 is scanned.
    voltage_loss: float
        The photovoltage loss per junction compared to the bandgap in V.
    allow_thinning: bool
        Shall we allow the top absorber to be thinned to transmit light to
        subsequent cell and thereby minimise current mismatch?
    cat_para:
        Catalyst parameters: [exchange current density, Tafel slope,
        ohmic resistivity]. Default: Exp. IrO2 with no ohmic resistivity.
    ideality: float
        Diode ideality factor.
    homotop: bool
        If true, the top two absorbers have the same bandgap, i.e. creating a
        double junction of the very same material.
    plotting: bool
        Plot results at the end?
    """
    E0_range = np.arange(min_E0, max_E0, stepsize)
    power = []
    best_comb = [0, 0, 0, 0, 0]
    for E0 in E0_range:
        print(E0, flush=True)
        if homotop is True:
            temp_power_red = find_max_triple_homotop(data, True, E0,
                                voltage_loss, cat_para, ideality)
            temp_power = [temp_power_red[0], [temp_power_red[1][0], temp_power_red[1][0],
            temp_power_red[1][1], temp_power_red[1][2], temp_power_red[1][3]]]
        else:
            temp_power = find_max_triple(data, True, E0, voltage_loss,
                                         allow_thinning, cat_para, ideality)
        if temp_power[1][-1] > best_comb[-1]:
            best_comb = [E0, temp_power[1][0], temp_power[1][1], \
                temp_power[1][2], temp_power[1][-1]]
        power.append([E0, temp_power[1][-1], temp_power[1][0], temp_power[1][1],
            temp_power[1][2], temp_power[1][3]])
    power = np.array(power)
    print('Finished!')
    if plotting is True:
        show_output(power, best_comb)
    return power, best_comb

def find_max_triple(data=am15g, power_E0=True, E0=0, voltage_loss=0.4,
                    allow_thinning=True, cat_para=IrO2, ideality=1, only_print_results=False):
    """Find the maximum current for a triple junction. Returns [Eg_hi(eV),
    Eg_med, Eg_lo(eV),j(mAcm-2)] and [Eg_hi(eV), Eg_med,Eg_lo(eV),j(mAcm-2), P]
    allow_thinning: allow the upper cell to be thinned to transmit light to
    lower cell.

    Parameters
    ----------
    data: numpy.ndarray
        The spectrum.
    power_E0: bool
        Is the power determined by current*E0? Otherwise: j*V_MPP.
    min_E0: float
        Minimum E0 to be considered.
    max_E0: float
        Maximum E0 to be considered.
    stepsize: float
        Steysize in V by which E0 is scanned.
    voltage_loss: float
        The photovoltage loss per junction compared to the bandgap in V.
    allow_thinning: bool
        Shall we allow the top absorber to be thinned to transmit light to
        subsequent cell and thereby minimise current mismatch?
    cat_para:
        Catalyst parameters: [exchange current density, Tafel slope,
        ohmic resistivity]. Default: Exp. IrO2 with no ohmic resistivity.
    ideality: float
        Diode ideality factor.
    """
    current = 0.0
    photo_current = [0, 0, 0]
    power = 0.0
    best_combination_current = [0, 0, 0, 0]
    best_combination_power = [0, 0, 0, 0]
    for lower_lambda in data[2:-2, 0]:
        i = np.where(data[:, 0] > lower_lambda)[0][0]
        for medium_lambda in data[i:-3, 0]:
            j = np.where(data[:, 0] > medium_lambda)[0][0]
            for higher_lambda in data[j:-4, 0]:
                higher_gap = nm_to_ev(lower_lambda)
                medium_gap = nm_to_ev(medium_lambda)
                lower_gap = nm_to_ev(higher_lambda)
                if lower_gap < 2*voltage_loss:
                    U_oc = higher_gap + lower_gap/2 + medium_gap - 2*voltage_loss
                else:
                    U_oc = higher_gap + lower_gap + medium_gap - 3*voltage_loss
                # to speed things up:
                if (power_E0 is True) and (U_oc < E0):
                    power = 0.0
                else:
                    photo_current[2] = integrate_flux(data, higher_lambda,
                        medium_lambda, electronvolts=False)
                    photo_current[1] = integrate_flux(data, medium_lambda,
                        lower_lambda, electronvolts=False)
                    photo_current[0] = integrate_flux(data, lower_lambda,
                        data[0, 0], electronvolts=False)
                    if (allow_thinning is True) and (photo_current[0] >
                            photo_current[1]) and (photo_current[0] > 0.0015)\
                            and (photo_current[1] > 0.0015):
                        average_current = photo_current[1] + 0.5*(photo_current[0] -
                            photo_current[1])
                        photo_current[0] = average_current
                        photo_current[1] = average_current
                    min_photo_current = min([photo_current[0], photo_current[1],
                                             photo_current[2]])
                    if power_E0 is False:
                        [power, U_mpp, current] = mpp_single_diode_current(min_photo_current,
                            U_oc, ideality, temp=300)
                    else:
                        current = match_IrO2_diode_current(E0, min_photo_current,
                            U_oc, cat_para, ideality)
                        power = current*E0
                    if current > best_combination_current[-1]:
                        best_combination_current = [higher_gap, medium_gap,
                            lower_gap, current]
                    if power > (best_combination_power[-1]):
                        best_combination_power = [higher_gap, medium_gap,
                            lower_gap, current, power]
    if only_print_results is True:
        if power_E0 is True:
            print('Highest STF efficiency at E0=', E0,'eV :', best_combination_power[-1]*1000,
                '%.\n Bandgaps (eV): ', best_combination_power[0:3])
        else:
            print('Highest photovoltaic efficiency of :', best_combination_power[-1]*1000,
                '%.\n Bandgaps (eV): ', best_combination_power[0:3])
        return None
    return best_combination_current, best_combination_power

def find_max_triple_homotop(data, power_E0=True, E0=0, voltage_loss=0.4,
                            cat_para=IrO2, ideality=1, only_print_results=False):
    """Find the maximum current for a triple junction, but top two gaps are the
    same. Returns [Eg_hi, Eg_med, Eg_lo, j(mAcm-2)] and
    [Eg_hi, Eg_med, Eg_lo, j(mAcm-2), P].

    Parameters
    ----------
    data: numpy.ndarray
        The spectrum.
    power_E0: bool
        Is the power determined by current*E0? Otherwise: j*V_MPP.
    E0: float
        E0 to be considered.
    voltage_loss: float
        The photovoltage loss per junction compared to the bandgap in V.
    cat_para:
        Catalyst parameters: [exchange current density, Tafel slope,
        ohmic resistivity]. Default: Exp. IrO2 with no ohmic resistivity.
    ideality: float
        Diode ideality factor.
    """
    current = 0.0
    photo_current = [0, 0]
    power = 0.0
    best_combination_current = [0, 0, 0]
    best_combination_power = [0, 0, 0, 0]
    for lower_lambda in data[1:-2, 0]:
        i = np.where(data[:, 0] > lower_lambda)[0][0]
        for higher_lambda in data[i:, 0]:
		# the two top cells have the same bandgap
            higher_gap = nm_to_ev(lower_lambda)
            lower_gap = nm_to_ev(higher_lambda)
            if lower_gap < 2*voltage_loss:
                U_oc = 2*higher_gap + lower_gap/2 - 2*voltage_loss
            else:
                U_oc = 2*higher_gap + lower_gap - 3*voltage_loss
            photo_current[1] = integrate_flux(data, higher_lambda, lower_lambda,
                electronvolts=False)
            photo_current[0] = integrate_flux(data, lower_lambda, data[0, 0],
                electronvolts=False)/2
            min_photo_current = min([photo_current[0], photo_current[1]])
            if power_E0 is False:
                [power, U_mpp, current] = mpp_single_diode_current(min_photo_current,
                    U_oc, ideality, temp=300)
            else:
                current = match_IrO2_diode_current(E0, min_photo_current, U_oc,
                                                   cat_para, ideality)
                if U_oc < E0:
                    power = 0.0
                else:
                    power = current*E0
            if current > best_combination_current[2]:
                best_combination_current = [higher_gap, lower_gap, current]
            if power > (best_combination_power[3]):
                best_combination_power = [higher_gap, lower_gap, current, power]
    if only_print_results is True:
        if power_E0 is True:
            print('Highest STF efficiency at E0=', E0,'eV :',
                best_combination_power[-1]*1000, '%.\n Bandgaps (eV): ',
                [best_combination_power[0], best_combination_power[0],
                best_combination_power[1]])
        else:
            print('Highest photovoltaic efficiency of :',
                best_combination_power[-1]*1000, '%.\n Bandgaps (eV): ',
                [best_combination_power[0], best_combination_power[0],
                best_combination_power[1]])
        return None
    return best_combination_current, best_combination_power

def calc_triple_junction(data=am15g, power_E0=True, Eg1=2.26, Eg2=1.12, Eg3=0.67,
                         E0=1.23, voltage_loss=0.4, allow_thinning=True,
                         cat_para=IrO2, ideality=1.0):
    """Calculates max. STF efficiency for a triple junction with three given
    bandgaps and E0. Returns [power(W/cm^2), current(A/cm^2)].

    Parameters
    ----------
    data: numpy.ndarray
        The spectrum.
    power_E0: bool
        Is the power determined by current*E0? Otherwise: j*V_MPP.
    Eg1: float
        Top cell bandgap (eV).
    Eg2: float
        Middle cell bandgap (eV).
    Eg3: float
        Bottom cell bandgap (eV).
    E0: float
        E0 to be considered.
    voltage_loss: float
        The photovoltage loss per junction compared to the bandgap in V.
    allow_thinning: bool
        Shall we allow the top absorber to be thinned to transmit light to
        subsequent cell and thereby minimise current mismatch?
    cat_para:
        Catalyst parameters: [exchange current density, Tafel slope,
        ohmic resistivity]. Default: Exp. IrO2 with no ohmic resistivity.
    ideality: float
        Diode ideality factor.
    """
    photo_current = [0.0, 0.0, 0.0]
    if Eg3 < 2*voltage_loss:
        U_oc = Eg1 + Eg2 + Eg3/2 - 2*voltage_loss
    else:
        U_oc = Eg1 + Eg2 + Eg3 - 3*voltage_loss
    photo_current[0] = integrate_flux(data, 99, Eg1)
    photo_current[1] = integrate_flux(data, Eg1, Eg2)
    photo_current[2] = integrate_flux(data, Eg2, Eg3)
    min_photo_current = min([photo_current[0], photo_current[1], photo_current[2]])
    if (allow_thinning is True) and (photo_current[0] > photo_current[1]):
        average_current = 0.5*(photo_current[0] + photo_current[1])
        min_photo_current = min(average_current, photo_current[2])
    if power_E0 is True:
        current = match_IrO2_diode_current(E0, min_photo_current, U_oc,
                                           cat_para, ideality)
        power = current*E0
        print('STF efficiency (%) with bandgaps Eg1, Eg2, Eg3 =', Eg1, ',',
              Eg2, ',', Eg3, 'eV for E0=', E0, 'eV:', power*1000, ';\nCurrent (A/cm^2): ', current)
    else:
        [power, U_mpp, current] = mpp_single_diode_current(min_photo_current,
            U_oc, ideality, temp=300)
        print('Photovoltaic efficiency (%) with bandgaps Eg1, Eg2, Eg3 =', Eg1,
        ',', Eg2, ',', Eg3, 'eV:', power*1000, ';\nCurrent (A/cm^2): ', current, '.\n')
    return power, current

#quadruple junction functions

def max_power_quadruple_E0(data=am15g, min_E0=1.0, max_E0=3.0, stepsize=0.05,
                           voltage_loss=0.4, allow_thinning=True, cat_para=IrO2,
                           ideality=1.0, tempfile='tmp_quad.dat',
                           plotting=True):
    """Calculates powers for 4 junctions. Returns [E0, Power, Eg1, Eg2, Eg3,
    Eg4, current].

    Parameters
    ----------
    data: numpy.ndarray
        The spectrum.
    power_E0: bool
        Is the power determined by current*E0? Otherwise: j*V_MPP.
    min_E0: float
        Minimum E0 to be considered.
    max_E0: float
        Maximum E0 to be considered.
    stepsize: float
        Steysize in V by which E0 is scanned.
    voltage_loss: float
        The photovoltage loss per junction compared to the bandgap in V.
    allow_thinning: bool
        Shall we allow the top absorber to be thinned to transmit light to
        subsequent cell and thereby minimise current mismatch?
    cat_para:
        Catalyst parameters: [exchange current density, Tafel slope,
        ohmic resistivity]. Default: Exp. IrO2 with no ohmic resistivity.
    ideality: float
        Diode ideality factor.
    tempfile: str
        File for storing temporary results during calculation. Not needed, but
        recommended due to long calc.
    plotting: bool
        Plot results at the end?
    """
    E0_range = np.arange(min_E0, max_E0, stepsize)
    power = []
    best_comb = [0, 0, 0, 0, 0]
    for E0 in E0_range:
        print(E0, flush=True)
        temp_power = find_max_quadruple(data, True, E0, voltage_loss,
                                        allow_thinning, cat_para, ideality)
        if temp_power[1][-1] > best_comb[-1]:
            best_comb = [E0, temp_power[1][0], temp_power[1][1], temp_power[1][2],
                temp_power[1][3], temp_power[1][4], temp_power[1][-1]]
        power.append([E0, temp_power[1][-1], temp_power[1][0], temp_power[1][1],
            temp_power[1][2], temp_power[1][3], temp_power[1][4]])
        if tempfile != '':
            np.savetxt(tempfile, power, header='tempfile_quadruple_calc')
    power = np.array(power)
    print('Finished!')
    if plotting is True:
        show_output(power, best_comb)
    return power, best_comb

def find_max_quadruple(data=am15g, power_E0=True, E0=0, voltage_loss=0.4,
                       allow_thinning=True, cat_para=IrO2, ideality=1):
    """Find the maximum current for a quadruple junction. Returns
    [Eg_highest, Eg_hi, Eg_med, Eg_lo,j(mAcm-2)] and
    [Eg_highest, Eg_hi, Eg_lo, j(mAcm-2), P].

    Parameters
    ----------
    data: numpy.ndarray
        The spectrum.
    power_E0: bool
        Is the power determined by current*E0? Otherwise: j*V_MPP.
    E0: float
        E0 to be considered.
    voltage_loss: float
        The photovoltage loss per junction compared to the bandgap in V.
    allow_thinning: bool
        Shall we allow the top absorber to be thinned to transmit light to
        subsequent cell and thereby minimise current mismatch?
    cat_para:
        Catalyst parameters: [exchange current density, Tafel slope,
        ohmic resistivity]. Default: Exp. IrO2 with no ohmic resistivity.
    ideality: float
        Diode ideality factor.
    """
    current = 0.0
    photo_current = [0, 0, 0, 0]
    power = 0.0
    best_combination_current = [0, 0, 0, 0, 0]
    best_combination_power = [0, 0, 0, 0, 0]
    for lowest_lambda in data[:-4, 0]:
        k = np.where(data[:, 0] > lowest_lambda)[0][0]
        for lower_lambda in data[k:-3, 0]:
            i = np.where(data[:, 0] > lower_lambda)[0][0]
            for medium_lambda in data[i:-2, 0]:
                j = np.where(data[:, 0] > medium_lambda)[0][0]
                for higher_lambda in data[j:, 0]:
                    highest_gap = nm_to_ev(lowest_lambda)
                    higher_gap = nm_to_ev(lower_lambda)
                    medium_gap = nm_to_ev(medium_lambda)
                    lower_gap = nm_to_ev(higher_lambda)
                    if lower_gap < 2*voltage_loss:
                        U_oc = highest_gap + higher_gap + medium_gap + lower_gap/2 - 3*voltage_loss
                    else:
                        U_oc = highest_gap + higher_gap + medium_gap + lower_gap - 4*voltage_loss
                    if (power_E0 is True) and (U_oc < E0):
                        power = 0.0
                    else:
                        photo_current[3] = integrate_flux(data, higher_lambda,
                            medium_lambda, electronvolts=False)
                        photo_current[2] = integrate_flux(data, medium_lambda,
                            lower_lambda, electronvolts=False)
                        photo_current[1] = integrate_flux(data, lower_lambda,
                            lowest_lambda, electronvolts=False)
                        photo_current[0] = integrate_flux(data, lowest_lambda,
                            data[0, 0], electronvolts=False)
                        if (allow_thinning is True) and (photo_current[0] > photo_current[1]):
                            average_current = photo_current[1]+0.5*(photo_current[0] -
                                photo_current[1])
                            photo_current[0] = average_current
                            photo_current[1] = average_current
                        min_photo_current = min([photo_current[0], photo_current[1],
                                        photo_current[2], photo_current[3]])
                        if power_E0 is False:
                            [power, U_mpp, current] = mpp_single_diode_current(min_photo_current,
                                U_oc, ideality, temp=300)
                        else:
                            current = match_IrO2_diode_current(E0,
                                min_photo_current, U_oc, cat_para, ideality)
                            power = current*E0
                        if current > best_combination_current[-1]:
                            best_combination_current = [highest_gap,
                                    higher_gap, medium_gap, lower_gap, current]
                        if power > (best_combination_power[-1]):
                            best_combination_power = [highest_gap, higher_gap,
                                        medium_gap, lower_gap, current, power]
    return best_combination_current, best_combination_power

def calc_quadruple_junction(data=am15g, power_E0=True, Eg1=1.9, Eg2=1.4,
                            Eg3=1.0, Eg4=0.5, E0=1.23, voltage_loss=0.4,
                            allow_thinning=True, cat_para=IrO2, ideality=1.0):
    """Calculates max. STF efficiency for a quadruple junction with four given
    bandgaps and E0. Returns [power(W/cm^2), current(A/cm^2)].

    Parameters
    ----------
    data: numpy.ndarray
        The spectrum.
    power_E0: bool
        Is the power determined by current*E0? Otherwise: j*V_MPP.
    Eg1: float
        Top cell bandgap (eV).
    Eg2: float
        Second cell bandgap (eV).
    Eg3: float
        Third cell bandgap (eV).
    Eg4: float
        Bottom cell bandgap (eV).
    E0: float
        E0 to be considered.
    voltage_loss: float
        The photovoltage loss per junction compared to the bandgap in V.
    allow_thinning: bool
        Shall we allow the top absorber to be thinned to transmit light to
        subsequent cell and thereby minimise current mismatch?
    cat_para:
        Catalyst parameters: [exchange current density, Tafel slope,
        ohmic resistivity]. Default: Exp. IrO2 with no ohmic resistivity.
    ideality: float
        Diode ideality factor.
    """
    photo_current = [0.0, 0.0, 0.0, 0.0]
    if Eg4 < 2*voltage_loss:
        U_oc = Eg1 + Eg2 + + Eg3 + Eg4/2 - 3*voltage_loss
    else:
        U_oc = Eg1 + Eg2 + Eg3 + Eg4 - 4*voltage_loss
    photo_current[0] = integrate_flux(data, 99, Eg1)
    photo_current[1] = integrate_flux(data, Eg1, Eg2)
    photo_current[2] = integrate_flux(data, Eg2, Eg3)
    photo_current[3] = integrate_flux(data, Eg3, Eg4)
    min_photo_current = min([photo_current[0], photo_current[1],
                             photo_current[2], photo_current[3]])
    if (allow_thinning is True) and (photo_current[0] > photo_current[1]):
        average_current = 0.5*(photo_current[0] + photo_current[1])
        min_photo_current = min(average_current, photo_current[2], photo_current[3])
    if power_E0 is True:
        current = match_IrO2_diode_current(E0, min_photo_current, U_oc, cat_para, ideality)
        power = current*E0
        print('STF efficiency (%) with bandgaps Eg1, Eg2, Eg3, Eg4 =', Eg1,
              ',', Eg2, ',', Eg3, ',', Eg4, 'eV for E0=', E0, 'eV:', power*1000,
              ';\nCurrent (A/cm^2): ', current)
    else:
        [power, U_mpp, current] = mpp_single_diode_current(min_photo_current,
            U_oc, ideality, temp=300)
        print('Photovoltaic efficiency (%) with bandgaps Eg1, Eg2, Eg3, Eg4 =',
              Eg1, ',', Eg2, ',', Eg3, ',', Eg4, 'eV:', power*1000,
              ';\nCurrent (A/cm^2): ', current, '.\n')
    return power, current

# demo routines

def demo_single_junction(data=am15g):
    """Demo for single junction with AM 1.5G spectrum and IrO2 CE with 1 Ohm
    resistivity.
    """
    print('You should see a plot soon.')
    max_power_single_E0(data, cat_para=[IrO2[0], IrO2[1], 1], plotting=True)
    plt.show()

def demo_double_junction(data=am15g, stepsize=0.2):
    """Demo for double junction with AM 1.5G spectrum and IrO2 CE with 1 Ohm
    resistivity.
    """
    print('Calculating DeltaG from 1.0 to 3.0:\n', flush=True)
    print('You should see a plot after the end of the calculation.')
    max_power_double_E0(data, stepsize=stepsize, cat_para=[IrO2[0],
        IrO2[1], 1], plotting=True)    

def demo_triple_junction(data=am15g, stepsize=0.5):
    """Demo for triple junction with AM 1.5G spectrum and IrO2 CE with 1 Ohm
    resistivity.
    """
    print('Calculating DeltaG from 2.0 to 4.0:\n(This will take some time.)\n',
          flush=True)
    print('You should see a plot after the end of the calculation.')
    max_power_triple_E0(data, min_E0=2.0, max_E0=4.0, stepsize=stepsize,
                        cat_para=[IrO2[0], IrO2[1], 1], plotting=True)

def demo_quadruple_junction(data=am15g, stepsize=1.0):
    """Demo for quadruple junction with AM 1.5G spectrum and IrO2 CE with 1 Ohm
    resistivity.
    """
    print('Calculating DeltaG from 3.0 to 5.0:\nThis will take a loong time '
    '(up to hours)!\n', flush=True)
    print('You should see a plot after the end of the calculation.')
    max_power_quadruple_E0(am15g, min_E0=3.0, max_E0=5.0, stepsize=stepsize,
                           cat_para=[IrO2[0], IrO2[1], 1], plotting=False)
