c     Fortran routine for the H2O-CO potential
c     Kalugina et al. 2017
c     The output is the potential at a distance supplied by the user.
c     The current default for the spherical angles is theta=theta'=phi=phi'=0
c
c     A. Faure Oct 2017

      program potmap
      implicit none
      integer ndim, i, j
      parameter(ndim=1000)
      integer icntrl, nlam, lam(4,ndim), p1, q1, p2, p
      character dir*80
      real*8 R, COEF(ndim),
     $     thx, phx, thpx, phpx, tnormed, y

c-------------------------------- INITIALIZATION
      icntrl=-1
      dir='./H2O-CO-FIT233/'
      r=0
      do i=1,ndim
         coef(i)=0
      enddo
      call h2oco_5d(icntrl, dir, ndim, nlam, lam, r, coef)

c-------------------------------- PES EVALUATION

      icntrl=0

      write(6,*) 'Distance (au) and spherical angles (degrees) ?'
      read(5,*) r, thx, phx, thpx, phpx

         call h2oco_5d(icntrl, dir, ndim, nlam, lam, r, coef)
       
         y=0

         do i=1,nlam
            p1=lam(1,i)
            q1=lam(2,i)
            p2=lam(3,i)
            p =lam(4,i)
            y=y+coef(i)*tnormed(p1,q1,p2,p,thx,phx,thpx,phpx)
         enddo

         write(6,'(5f10.3,f15.6)') r, thx, phx, thpx, phpx, y

      end

      subroutine h2oco_5d(ICNTRL, DIR, NDIM, NLAM, LAM, RD, P)
c
c -------------------------------------------------------------------
c Five dimensional PES expansion for H2O-CO
c Kalugina et al, 2017
c -------------------------------------------------------------------
c
c  ARGUMENT LIST
c
c  ICNTRL      (input)   -1  mandatory initialization
c                         0  to evaluate P(R)
c  DIR         (input)   slash/ terminated directory containing data   (-1)
c  NDIM        (input)   dimension of LAM(4,*) and P(*)
c  NLAM        (output)  number of terms of the expansion 
c  LAM(4,NDIM) (output)  (4,NDIM) p1, q1, p2, p expansion terms        (-1)
c  RD          (input)   intermolecular distance (a.u.)                (0)
c  P(NDIM)     (output)  coefficients for angular expansion in cm-1    (0)
c              properly interpolated or extrapolated at distance R 
c              assuming normalized expansion terms (beware the normalization
c              error in PMG94's paper).
c              These coefficients come in the ordering indicated in LAM. 
c              This ordering reflects the list of terms given in the
c              input file mesh.dat and can thus be easily changed. 
c              Terms can easily be removed or added in mesh.dat provided
c              they form a subset of the original 149 term expansion in
c              files ref.DDDD.dat. 
c
c     (-1)  relevant for initialization only (ICNTRL=-1)
c     (0)   relevant for PES evaluation only (ICNTRL=0)
c
c  RELATED FILES
c
c     mesh.dat, ref.DDDD.dat, cal.DDDD.dat and vib.DDDD.dat in
c     directory DIR.
c
      implicit none
      integer icntrl, ndim, nlam, lam(4,ndim), np1
      character*(*) dir
      real*8 rd, p(ndim)
      integer ndx, ntx, nttx
      parameter (ndx=40, ntx=1000, nttx=1000)
      integer ndist, id, Ngeom, Nfunc, Ngeom0, Nfunc0, i, it, nt, k, k2,
     $     l0(4,ntx), l(4,ntx), nex
      integer ind, ind2
      real*8 fdist, Dist(ndx),
     $     Sinv(ndx), RMS(ndx), ERR(ndx)
      real*8 coef(ndx,ntx),
     $     cc(ndx,nttx), c1(ndx,nttx), c2(ndx,nttx), c3(ndx,nttx)
      integer ntt, itt, in, ll(4,nttx), indll(nttx)
      logical chk(ntx)
      integer n0000, kleft, kright
      real*8 a(nttx), alpha(nttx), dleft, dright, hinvl, hinvr
      real*8 c(nttx), b(nttx),
     $     v0000, ccinv0000, fact
      character cfit*20, comment*132, name*132
      integer le
      real*8 h, r, rinv, aniso, tres, thres
      real*8 f, u, fu, Pi, Pis2
c     funct(u)=(1-cos(u*Pi))/2 ; f(u)=funct(funct(u))
      f(u)=(1-cos((1-cos(u*Pi))*Pis2))*0.5d0
      save

      if (ICNTRL.eq.0) goto 1000

c-------------------------------------------------------------------
c Initialization (ICNTRL=-1)
c-------------------------------------------------------------------
      if (ICNTRL.ne.-1) stop 'invalid ICNTRL, expected -1'
      Pi=acos(-1.0d0)
      Pis2=Pi/2

c Set directory containing the fitted and mesh data (slash terminated)

      le=len_trim(dir)

c Open mesh.dat

      name=dir(:le)//'mesh.dat'
      open(unit=1, file=name, status='old')

      read(1,*) comment
      write(6,*) '___________________________________________________'
      write(6,*)
      write(6,*)
     $     'H2O-CO 5D PES, Version 1.0, May 2016.'
      write(6,*)
      write(6,*) comment(1:len_trim(comment))
      write(6,*) '___________________________________________________'


c-------------------------------------------------------------------
c  Reference PES
c-------------------------------------------------------------------

c Read fitted data in ref.DDDD.dat files

      read(1,*) ndist
      write(6,*)
      write(6,*) '********************** CCSD(T)-F12 reference'
      write(6,*)
      write(6,*) 'Number of distances', ndist
      if (ndist.gt.ndx) stop 'increase dx'

      do id=1,ndist

         read(1,*) fdist, cfit
         write(6,'(f8.3,1x,2a)') fdist, dir(:le), cfit

         name=dir(:le)//cfit
         open(unit=2, file=name, status='old')

         read(2,*)
         read(2,*)
         read(2,*) Dist(id), Ngeom, Nfunc, Sinv(id), RMS(id), ERR(id)
         if (dist(id).ne.fdist)
     $        stop 'inconsistent distance in ref.DDDD.dat'
         if (id.eq.1) then
            Ngeom0=Ngeom
            Nfunc0=Nfunc
            nt=Nfunc
            write(6,*) 'Ngeom = ', Ngeom, '    Nfunc = ', Nfunc
            if (Nfunc.gt.ntx) stop 'increase ntx'
         else
            if (Ngeom0.ne.Ngeom) stop 'inconsistent Ngeom'
            if (Nfunc0.ne.Nfunc) stop 'inconsistent Nfunc'
            if (Dist(id).le.Dist(id-1)) stop 'non monotonic Dist'
         endif
         
         do it=1,nt
            read(2,*) (l(i,it),i=1,4), coef(id,it)
         enddo

         if (id.eq.1) then
            do it=1,nt
               do i=1,4
                  l0(i,it)=l(i,it)
               enddo
            enddo
         else
            k=0
            do it=1,nt
               do i=1,4
                  k=k+abs(l0(i,it)-l(i,it))
               enddo
            enddo
            if (k.ne.0) stop 'inconsistent expansion terms'
         endif

         close(unit=2)

      enddo

      write(6,*)
      write(6,*) ndist, ' Distances'
      write(6,*)
      write(6,*)
     $     '   Dist  Ngeom Nfunc  Sinv       RMS         ERR'
      do id=1,ndist
         write(6,'(f8.3,2i6,f8.3,2f12.6)')
     $        Dist(id), Ngeom, nt, Sinv(id), RMS(id), ERR(id)
      enddo

      write(6,*)
      write(6,*) 'Original terms from ref.DDDD.dat files', nt
      write(6,*)
      write(6,'(3(i4,2x,4i3,3x))') (it, (l(i,it), i=1,4), it=1,nt)

c Select final expansion terms and corresponding pointers to original data

      read(1,*) ntt
      write(6,*)
      write(6,*) 'final expansion terms', ntt
      write(6,*)
      if (ntt.gt.nttx) stop 'increase nttx'
      read(1,*) ((ll(i,itt), i=1,4), itt=1,ntt)
      write(6,'(3(i4,2x,4i3,3x))') (itt, (ll(i,itt), i=1,4), itt=1,ntt)

      do it=1,ntx
         chk(it)=.false.
      enddo
      do itt=1,nttx
         indll(itt)=0
      enddo
      do itt=1,ntt
         do it=1,nt
            if (       l(1,it).eq.ll(1,itt)
     $           .and. l(2,it).eq.ll(2,itt)
     $           .and. l(3,it).eq.ll(3,itt)
     $           .and. l(4,it).eq.ll(4,itt) ) then
               if (chk(it)) then
                  write(6,*) 'l =', (l(i,it),i=1,4)
                  write(6,*) 'll=', (ll(i,itt),i=1,4)
                  write(6,*) 'itt=', itt, '     it=', it
                  stop 'error -- chk(it) already set'
               else
                  indll(itt)=it
                  chk(it)=.true.
               endif
            endif
         enddo
      enddo

      write(6,*)
      write(6,*) 'Pointers to original fit expansion terms'
      write(6,*)
      write(6,'(5(i4,a,i3,3x))') (itt, ' ->', indll(itt), itt=1,ntt)
      do itt=1,ntt
         if (indll(itt).eq.0) stop 'some invalid pointer(s)'
      enddo
      write(6,*)
      write(6,*) 'Copy original expansion coeffs to final cc matrix'
      do itt=1,ntt
         it=indll(itt)
         do id=1,ndist
            cc(id,itt)=coef(id,it)
         enddo
      enddo

      n0000=0
      do itt=1,ntt
         if (       ll(1,itt).eq.0
     $        .and. ll(2,itt).eq.0
     $        .and. ll(3,itt).eq.0
     $        .and. ll(4,itt).eq.0 ) then
            n0000=itt
         endif
      enddo
      if (n0000.eq.0) stop 'n0000 term missing in expansion'

c Retrieve and set up long range terms as C/R**beta

      write(6,*)
      write(6,*) 'Set up long range extrapolation'
      write(6,*) 'All terms above thres at 20 and 30 au are'
     $     ,' extrapolated'
      write(6,*)
      write(6,*) ' #term          term            C                beta'

c The threshold thres is defined as 10% of v_0000 at 30 au
c All terms above thres at 20 and 30 au are extrapolated by a power law
c obtained from values at 20 and 30 au.
c
c Additional conditions to extrapolate: 
c 2) the signs of cc(ndist-1,itt) and cc(ndist,itt) must be the same
c 3) the magnitude of the coeffs must decrease with increasing dist
c
      do itt=1,ntt
         b(itt)=0
         c(itt)=0
      enddo
      thres=0.10d0*abs(cc(ndist,1))
      nex=0
      do itt=1, ntt
c         if (abs(cc(ndist-1,itt)).ge.thres .and. abs(cc(ndist,itt)).ge
c     $        .thres) then
         if (abs(cc(ndist-1,itt)).ge.thres .and. abs(cc(ndist,itt)).ge
     $        .thres .and. cc(ndist-1,itt)/(cc(ndist,itt)).gt.0 .and.
     $        abs(cc(ndist-1,itt)).gt.abs(cc(ndist,itt))) then
            nex=nex+1
            b(itt)=log(cc(ndist-1,itt)/cc(ndist,itt))
     $           /log(dist(ndist)/dist(ndist-1))
            c(itt)=cc(ndist,itt)*dist(ndist)**b(itt)
            write(6,'(i6,4i6,1pe16.6,0pf12.6)')
     $           itt, (ll(i,itt),i=1,4), c(itt), b(itt)
         endif
      enddo

      write(6,*)
      write(6,*) nex, ' terms extrapolated beyond 30 au'

c Initialize extrapolation at short range

      write(6,*)
c      write(6,*) 'Monitor exponential short range extrapolation'
      tres=5000
c      write(6,'(1x,a,1pe12.2,a,0pf8.3)') 'Select terms above TRES =',
c     $     tres, '  at R=', dist(1)
      do itt=1,ntt
         alpha(itt)=log(cc(1,itt)/cc(2,itt))/(dist(2)-dist(1))
         a(itt)=cc(1,itt)*exp(alpha(itt)*dist(1))
         if (cc(1,itt).gt.tres) then
c            write(6,'(i4,4x,4i3,1pe14.3,0pf12.3)')
c     $           itt, (ll(i,itt),i=1,4), a(itt), alpha(itt)
         endif
      enddo
c      write(6,*)
c      write(6,*)'    Dist      Isotropic   Sigma(Aniso)    Ratio'
      r=0
      do 
         aniso=0
         do itt=1,ntt
            if (itt.ne.n0000 .and. cc(1,itt).gt.tres) then
               aniso=aniso+abs(a(itt))*exp(-alpha(itt)*r)
            endif
         enddo
         v0000=a(n0000)*exp(-alpha(n0000)*r)
c         write(6,'(f10.3, 1p2e14.3, 0pf10.2)')
c     $        r, v0000, aniso, aniso/v0000
         r=r+0.1d0
         if (r.gt.dist(3)+0.0001) exit
      enddo
c      write(6,*)
c     $     '  ==> Anisotropic terms grow faster than isotropic one;'
c      write(6,*)
c     $     '  ==> blind extrapolation of anisotropic terms is DANGEROUS'
c      write(6,*)
c     $     '  ==> proportional extrapolation more secure'

c set up smooth transition domains using step function f
      read(1,*) fact
      dleft=dist(1)
      kleft=2
      hinvl=1/(dist(kleft+1)-dist(1))
      dright=dist(ndist)
      kright=ndist-1
      hinvr=1/(dist(ndist)-dist(kright))
      write(6,*)
      write(6,*) 'Set up smooth transition domains...'
      write(6,'(1x,a,f10.3 )')
     $     'Anisotropy fraction kept at shorter range', fact
      write(6,'(1x,a,f10.3,a,f10.3)')
     $     'Left  domain', dist(1),  '  -> ', dist(kleft+1)
      write(6,'(1x,a,f10.3,a,f10.3)')
     $     'Right domain', dist(kright), '  -> ', dist(ndist)


c In order to avoid the ondulation of the spline in the last
c interval (where the interaction energy is small), a single
c additional point obtained from the long-range extrapolation at
c r=dist(ndist)+1 (ie 16 au) is added to cc.
c
      np1=ndist+1
      dist(np1)=dist(ndist)+1
      do itt=1,ntt
         cc(np1,itt)=c(itt)*(1/dist(np1))**b(itt)
      enddo

c Initialize spline coefficients for original data

      write(6,*)
      write(6,*) 'Set up spline coefficients...'
      do itt=1,ntt
         call cubspl (np1,dist,
     $        cc(1,itt),c1(1,itt),c2(1,itt),c3(1,itt),0,0)
c scale c2 and c3 to obtain directly taylor coefficients
c optionally one may put a unit conversion there
c be clever enough in this case to take into account also the unit 
c conversions in the short and long range extraoplations...
         do id=1,np1
!            cc(id,itt) = cc(id,itt)
!            c1(id,itt) = c1(id,itt)
            c2(id,itt) = c2(id,itt) / 2.d0
            c3(id,itt) = c3(id,itt) / 6.d0
         enddo
      enddo

c-------------------------------------------------------------------

c return proper data
      if (ntt.gt.ndim) stop 'increase NDIM'
      nlam=ntt
      do itt=1,ntt
         do i=1,4
            lam(i,itt)=ll(i,itt)
         enddo
      enddo
      close(unit=1)
      write(6,*)
      write(6,*) 'Initialization done.'
      write(6,*)
      return      


c-------------------------------------------------------------------
c Interpolation or extrapolation of the PES expansion (ICNTRL=0)
c-------------------------------------------------------------------

 1000 if (ntt.gt.ndim) stop 'invalid NDIM'
      nlam=ntt
      r=rd

c------------------------------------------ Reference PES

c short range extrapolation
c fact=0   -->   squeeze completely anisotropy for RD <= dleft
c fact=1   -->   keep anisotropy for RD <= dleft in a proportional way:
c the extrapolation of the isotropic term is used for all terms with a 
c scaling factor computed at r=dist(2) (ie 3.25 au)

      if (r.lt.dleft) then
         v0000=a(n0000)*exp(-alpha(n0000)*r)
         ccinv0000=1/cc(2,n0000)
         do itt=1,ntt
            if (itt.eq.n0000) then
               p(itt)=v0000
            else
               p(itt)=fact*v0000*cc(2,itt)*ccinv0000
            endif
         enddo

c long range extrapolation
      else if (r.gt.dright) then
         do itt=1,ntt
            p(itt)=0
         enddo
         Rinv=1/r
 
         do itt=1,ntt
            if (abs(cc(ndist-1,itt)).ge.thres
     $           .and. abs(cc(ndist,itt)).ge.thres) then
               p(itt)=c(itt)*Rinv**b(itt)
            endif
         enddo

c spline domain
      else
         call splget (ndist,dist,r,k)
         h = r - dist(k)
         do itt=1,ntt
c remember y2 and y3 have been scaled for optimisation
            p(itt) = cc(k,itt)+h*
     $           (c1(k,itt)+h*(c2(k,itt)+h*c3(k,itt)))
         enddo

c branch smooth step function for first interval
         if (k.le.kleft) then
            fu=f((r-dist(1))*hinvl)
            v0000=a(n0000)*exp(-alpha(n0000)*r)         
            ccinv0000=1/cc(2,n0000)
            do itt=1,ntt
               if (itt.eq.n0000) then
                  p(itt)=fu*p(itt)+(1-fu)*v0000
               else
                  p(itt)=fu*p(itt)
     $                 +(1-fu)*fact*v0000*cc(2,itt)*ccinv0000
               endif
            enddo
            
c branch smooth step function for last interval
         else if (k.ge.kright) then
            fu=f((r-dist(kright))*hinvr)
            do itt=1,ntt
               p(itt)=(1-fu)*p(itt)
            enddo
            Rinv=1/r

            do itt=1,ntt
               if (abs(cc(ndist-1,itt)).ge.thres
     $              .and. abs(cc(ndist,itt)).ge.thres) then
                  p(itt)=p(itt)+fu*c(itt)*Rinv**b(itt)
               endif
            enddo

         endif
      endif

 2000 end
