(* Content-type: application/vnd.wolfram.mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 11.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       158,          7]
NotebookDataLength[    285076,       6233]
NotebookOptionsPosition[    269745,       5867]
NotebookOutlinePosition[    270179,       5884]
CellTagsIndexPosition[    270136,       5881]
WindowFrame->Normal*)

(* Beginning of Notebook Content *)
Notebook[{
Cell[BoxData[
 StyleBox[
  RowBox[{
  "Higher", " ", "order", " ", "perturbation", " ", "theory", " ", "for", " ",
    "the", " ", "buckling", " ", "of", " ", "a", " ", "growing", " ", 
   "compressible", " ", "neoHookean", " ", "layer", " ", "adhered", " ", "to",
    "  ", "a", " ", "compressible", " ", "neoHookean", " ", "substrate"}], 
  "Title"]], "Input",
 CellChangeTimes->{{3.748157106127223*^9, 3.7481572393084497`*^9}, {
  3.748157280213007*^9, 3.748157298711442*^9}, {3.748157349817041*^9, 
  3.7481573833723717`*^9}, {3.7481574974242907`*^9, 3.748157498273388*^9}, {
  3.748364692459585*^9, 3.748364699433175*^9}, {3.7493707327882504`*^9, 
  3.749370739045542*^9}},
 TextAlignment->Center],

Cell[TextData[StyleBox["\[LineSeparator]Notebook by Nontawit Cheewaruangroj \
and John S Biggins", "Author"]], "Text",
 CellChangeTimes->{3.748364707364526*^9},
 TextAlignment->Center],

Cell[TextData[StyleBox["How to use this notebook ", "Section"]], "Text",
 CellChangeTimes->{{3.7481578035150757`*^9, 3.748157808410305*^9}}],

Cell["\<\
This notebook is intended to be read/used in conjunction with the paper \
\[OpenCurlyDoubleQuote]Pattern selection when a layer buckles on a soft \
substrate.\[CloseCurlyDoubleQuote] This notebook also uses helper functions \
written by the authors, which are contained in the accompanying Mathematica \
notebook Pert_helper _functions.nb, which must be in the same folder as this \
notebook. \
\>", "Text",
 CellChangeTimes->{{3.748157822772893*^9, 3.7481579853128366`*^9}, {
  3.748158317923436*^9, 3.748158318538887*^9}}],

Cell["\<\
If you run the \"input\" cells of this notebook in order (from top to bottom) \
the notebook will verify the first, second and third order perturbation \
fields for stripe, square and hexagonal patterns, and compute the \
accompanying energy coefficients. You can either do this whilst reading \
through, running each input cell manually, or run the entire notebook in one \
command by selecting Evaluation/Evaluate-notebook. Fair warning: running the \
entire notebook may take around 10 minutes.\
\>", "Text",
 CellChangeTimes->{{3.748157822772893*^9, 3.748157944657064*^9}, {
  3.7481579932965107`*^9, 3.748158065558572*^9}, {3.748196265301153*^9, 
  3.7481962945792093`*^9}, {3.75378745760856*^9, 3.7537874600653906`*^9}}],

Cell["\<\
Unlike the incompressible system, the algebra here is very complicated. In \
some calculation, we have to resort to solve the system with numerical \
parameters.\
\>", "Text",
 CellChangeTimes->{{3.7481583089030447`*^9, 3.748158308907529*^9}, {
  3.749459566643412*^9, 3.7494596310438995`*^9}}],

Cell["", "Text"],

Cell[CellGroupData[{

Cell[TextData[StyleBox["Helper functions", "Section"]], "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.},
 CellChangeTimes->{{3.748157764841991*^9, 3.7481577803021307`*^9}, 
   3.7487821236144075`*^9}],

Cell["\<\
This notebook uses the standard \[OpenCurlyDoubleQuote]Variational methods\
\[CloseCurlyDoubleQuote] package, to calculate the equilibrium equations \
directly from the energy via calculus of variations.\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.},
 CellChangeTimes->{{3.748158093464594*^9, 3.7481581320612297`*^9}, 
   3.748782123615381*^9}],

Cell[BoxData[
 RowBox[{"Needs", "[", "\"\<VariationalMethods`\>\"", "]"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.},
 CellChangeTimes->{{3.748157589976445*^9, 3.748157632245802*^9}, {
   3.748158159112691*^9, 3.748158159791313*^9}, 3.748782123615381*^9}],

Cell["\<\
This notebook also uses helper functions written by the authors, which are \
contained in the accompanying Mathematica notebook pert_helper_functions.nb, \
which must be in the same folder as this notebook. \
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.},
 CellChangeTimes->{{3.748158147401422*^9, 3.748158155335235*^9}, {
   3.748158225718522*^9, 3.748158274427133*^9}, {3.748158353542948*^9, 
   3.748158353703319*^9}, 3.748195912982089*^9, {3.7481959460371103`*^9, 
   3.748195946317108*^9}, 3.748782123615381*^9}],

Cell[BoxData[
 RowBox[{"NotebookEvaluate", "[", 
  RowBox[{"FileNameJoin", "[", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"NotebookDirectory", "[", "]"}], ",", 
     "\"\<pert_helper_functions.nb\>\""}], "}"}], "]"}], "]"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.},
 CellChangeTimes->{{3.748158213150754*^9, 3.748158219797617*^9}, {
   3.7481941325489397`*^9, 3.748194136035887*^9}, 3.748195107133922*^9, {
   3.748195421557001*^9, 3.7481954825069237`*^9}, 3.748195517246182*^9, {
   3.74819562767171*^9, 3.748195632607148*^9}, {3.7481957842483664`*^9, 
   3.74819582426667*^9}, 3.748782123615381*^9}],

Cell["\<\
These helper functions include two new simplify commands, TrigSimplify and \
ExpSimplify, which are quicker than the regular Mathematica Simplify command \
at dealing with expressions that are sums of fourier components, or sums of \
exponential terms. They work by splitting the expression into \
fourier/exponential components, simplifying each component separately, then \
adding the, back together again.\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.},
 CellChangeTimes->{{3.748158294534396*^9, 3.748158329875403*^9}, {
   3.748158385934456*^9, 3.748158530969438*^9}, 3.748263803954225*^9, 
   3.748782123616377*^9}],

Cell["\<\
Finally, we load a second Mathematica notebook, which contains the full \
algebraic forms of the constant of integrations from the boundart condtions, \
which we validate in this notebook.\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.},
 CellChangeTimes->{{3.74819596044608*^9, 3.748196016626272*^9}, 
   3.748782123616377*^9, {3.7494520174298267`*^9, 3.7494520316739316`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"NotebookEvaluate", "[", 
    RowBox[{"FileNameJoin", "[", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"NotebookDirectory", "[", "]"}], ",", 
       "\"\<compressible_pert_solutions.nb\>\""}], "}"}], "]"}], "]"}], ";"}],
   " "}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.},
 CellChangeTimes->{{3.748158213150754*^9, 3.748158219797617*^9}, {
   3.7481941325489397`*^9, 3.748194136035887*^9}, 3.748195107133922*^9, {
   3.748195421557001*^9, 3.7481954825069237`*^9}, 3.748195517246182*^9, {
   3.74819562767171*^9, 3.748195632607148*^9}, {3.7481957842483664`*^9, 
   3.74819582426667*^9}, {3.748195982630145*^9, 3.748195984068635*^9}, {
   3.7481968983523693`*^9, 3.748196910168001*^9}, 3.748782123616377*^9, {
   3.7494049891114655`*^9, 3.749404993877717*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[StyleBox["Defining the elastic energy ", "Section"]], "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.7481585477387753`*^9, 3.7481585735823603`*^9}, {
   3.748158867345829*^9, 3.7481588758818827`*^9}, {3.748263214336389*^9, 
   3.748263215129652*^9}, {3.7482636671820307`*^9, 3.748263668106621*^9}, 
   3.7487821424858932`*^9}],

Cell["\<\
We start by defining the displacement in the layer  u[x,y,z] as a vector \
displacement field in the layer, with x-y-z components ux, uy and,uz.\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.748263262110486*^9, 3.748263275531332*^9}, 
   3.7487821424858932`*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"u", "[", 
   RowBox[{"x_", ",", "y_", ",", "z_"}], "]"}], " ", ":=", " ", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"ux", "[", 
     RowBox[{"x", ",", "y", ",", "z"}], "]"}], ",", 
    RowBox[{"uy", "[", 
     RowBox[{"x", ",", "y", ",", "z"}], "]"}], ",", 
    RowBox[{"uz", "[", 
     RowBox[{"x", ",", "y", ",", "z"}], "]"}]}], "}"}]}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{
  3.7481585932803793`*^9, 3.748158691560307*^9, {3.7481955716264753`*^9, 
   3.748195575602419*^9}, 3.7487821424858932`*^9}],

Cell["\<\
Unlike in the paper, in Mathematica it is not convenient to define piecewise \
functions to describe the different behavior of the layer and substrate, so \
we instead use different variables in the layer and substrate, using the \
convention that substrate variables are distinguished by an S appended to the \
variable name. For example, we first define uS[x,y,z] is the a vector \
displacement field in the substrate, with x-y-z components uxS, uyS and,uzS. \
To avoid too much repetition we define a substitution rule that replaces the \
common layer fields in an expression with their substrate counterparts. \
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.748263292506982*^9, 3.748263338374407*^9}, {
   3.7482633913666887`*^9, 3.748263395085491*^9}, {3.7482634613960323`*^9, 
   3.748263486651383*^9}, 3.7487821424858932`*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"SubstrateFields", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"uz", "\[Rule]", "uzS"}], ",", 
     RowBox[{"uy", "\[Rule]", "uyS"}], ",", 
     RowBox[{"ux", "\[Rule]", "uxS"}], ",", 
     RowBox[{"P", "\[Rule]", "PS"}], ",", 
     RowBox[{"\[Mu]1", "\[Rule]", "\[Mu]2"}], ",", 
     RowBox[{"g", "\[Rule]", "1"}], ",", 
     RowBox[{"\[Gamma]", "\[Rule]", "1"}], ",", 
     RowBox[{"fx", "\[Rule]", "fxS"}], ",", 
     RowBox[{"fz", "\[Rule]", "fzS"}], ",", 
     RowBox[{"fp", "\[Rule]", "fpS"}]}], "}"}]}], ";"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.7482633825687637`*^9, 3.748263385225129*^9}, {
   3.748263452555109*^9, 3.7482634528935003`*^9}, 3.748263812463072*^9, 
   3.7487821424858932`*^9, {3.7493709014652996`*^9, 3.749370904052367*^9}, 
   3.7494038940962286`*^9}],

Cell["The displacement in the substrate is thus defined as ", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.748159360806814*^9, 3.7481593690050898`*^9}, 
   3.7487821424858932`*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"uS", "[", 
   RowBox[{"x_", ",", "y_", ",", "z_"}], "]"}], " ", ":=", " ", 
  RowBox[{
   RowBox[{"u", "[", 
    RowBox[{"x", ",", "y", ",", "z"}], "]"}], "/.", 
   "SubstrateFields"}]}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{
  3.7481585932803793`*^9, 3.748158691560307*^9, {3.748159375366708*^9, 
   3.748159381068968*^9}, {3.7481943922204533`*^9, 3.748194396540489*^9}, {
   3.74826349633512*^9, 3.748263501972477*^9}, 3.7487821424858932`*^9}],

Cell[BoxData[
 RowBox[{"uS", "[", 
  RowBox[{"x", ",", "y", ",", "z"}], "]"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{
  3.7481585932803793`*^9, 3.748158691560307*^9, {3.748159375366708*^9, 
   3.748159381068968*^9}, {3.7481943922204533`*^9, 3.748194396540489*^9}, {
   3.74826349633512*^9, 3.748263545272531*^9}, 3.7487821424858932`*^9}],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"uxS", "[", 
    RowBox[{"x", ",", "y", ",", "z"}], "]"}], ",", 
   RowBox[{"uyS", "[", 
    RowBox[{"x", ",", "y", ",", "z"}], "]"}], ",", 
   RowBox[{"uzS", "[", 
    RowBox[{"x", ",", "y", ",", "z"}], "]"}]}], "}"}]], "Output",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{
  3.748263545688115*^9, 3.748263815713512*^9, 3.748264215273073*^9, 
   3.74826464617157*^9, 3.748264996489705*^9, 3.7482651158157597`*^9, 
   3.748265231615069*^9, 3.748265441693152*^9, 3.7482656176804037`*^9, 
   3.748266727219419*^9, 3.748267446003017*^9, 3.748358477639948*^9, 
   3.748362089463801*^9, 3.748363714317644*^9, 3.748364432624249*^9, 
   3.748364730217554*^9, 3.7483648325553493`*^9, 3.7483649683656607`*^9, 
   3.7483654390385427`*^9, 3.7483656457419662`*^9, 3.7483658652691298`*^9, 
   3.7483678957754803`*^9, 3.748368477006544*^9, 3.748368837736546*^9, 
   3.7483693813441973`*^9, 3.748370070179399*^9, 3.748371270376502*^9, 
   3.74837140350008*^9, 3.7487617748069644`*^9, {3.748782142486891*^9, 
   3.7487821443170533`*^9}, 3.748784703203154*^9, 3.748788156063622*^9, 
   3.7487887340276933`*^9, 3.748790781897951*^9, 3.748876096673195*^9, 
   3.74887640958061*^9, 3.7488791280751395`*^9, 3.7488939511566176`*^9, 
   3.748895871540728*^9, 3.748944405507182*^9, 3.748945639036533*^9, 
   3.7489475137871847`*^9, 3.748957548281746*^9, 3.7489642557115855`*^9, 
   3.7491987287177677`*^9, 3.749403014970085*^9, 3.7494520433040257`*^9, 
   3.749452876711939*^9, 3.7494535654106827`*^9, 3.749455207702671*^9, 
   3.7494670110772204`*^9, 3.749468661662424*^9, 3.7494996139496803`*^9, 
   3.74953352683332*^9, 3.7498149834155445`*^9, 3.749815023315222*^9, 
   3.7499696041069775`*^9, 3.7499705826507797`*^9, 3.750143087590087*^9, 
   3.7501939950530615`*^9, 3.7504435330069714`*^9, 3.750446631578229*^9, 
   3.7504477991299176`*^9, 3.7537816635185003`*^9, 3.753782862788084*^9, 
   3.753785810674659*^9, 3.7537867961586*^9, 3.753787088726141*^9}],

Cell["As in the paper,  the deformation gradient is defined as:", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.7481587064971523`*^9, 3.748158716990864*^9}, 
   3.748782142486891*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"F", " ", "=", " ", 
   RowBox[{
    RowBox[{"IdentityMatrix", "[", "3", "]"}], " ", "+", " ", 
    RowBox[{"Grad", "[", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"x", ",", "y", ",", "z"}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], "]"}]}]}], ";"}]], "Input",\

 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.748158727332*^9, 3.748158743908091*^9}, 
   3.748782142486891*^9}],

Cell["\<\
If we display F in matrix form, it has the expected components:\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.748158795434462*^9, 3.748158807880468*^9}, 
   3.748782142486891*^9}],

Cell[BoxData[
 RowBox[{"F", "//", "MatrixForm"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.748158758467133*^9, 3.7481587610096483`*^9}, 
   3.748782142486891*^9}],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      RowBox[{"1", "+", 
       RowBox[{
        SuperscriptBox["ux", 
         TagBox[
          RowBox[{"(", 
           RowBox[{"1", ",", "0", ",", "0"}], ")"}],
          Derivative],
         MultilineFunction->None], "[", 
        RowBox[{"x", ",", "y", ",", "z"}], "]"}]}], 
      RowBox[{
       SuperscriptBox["ux", 
        TagBox[
         RowBox[{"(", 
          RowBox[{"0", ",", "1", ",", "0"}], ")"}],
         Derivative],
        MultilineFunction->None], "[", 
       RowBox[{"x", ",", "y", ",", "z"}], "]"}], 
      RowBox[{
       SuperscriptBox["ux", 
        TagBox[
         RowBox[{"(", 
          RowBox[{"0", ",", "0", ",", "1"}], ")"}],
         Derivative],
        MultilineFunction->None], "[", 
       RowBox[{"x", ",", "y", ",", "z"}], "]"}]},
     {
      RowBox[{
       SuperscriptBox["uy", 
        TagBox[
         RowBox[{"(", 
          RowBox[{"1", ",", "0", ",", "0"}], ")"}],
         Derivative],
        MultilineFunction->None], "[", 
       RowBox[{"x", ",", "y", ",", "z"}], "]"}], 
      RowBox[{"1", "+", 
       RowBox[{
        SuperscriptBox["uy", 
         TagBox[
          RowBox[{"(", 
           RowBox[{"0", ",", "1", ",", "0"}], ")"}],
          Derivative],
         MultilineFunction->None], "[", 
        RowBox[{"x", ",", "y", ",", "z"}], "]"}]}], 
      RowBox[{
       SuperscriptBox["uy", 
        TagBox[
         RowBox[{"(", 
          RowBox[{"0", ",", "0", ",", "1"}], ")"}],
         Derivative],
        MultilineFunction->None], "[", 
       RowBox[{"x", ",", "y", ",", "z"}], "]"}]},
     {
      RowBox[{
       SuperscriptBox["uz", 
        TagBox[
         RowBox[{"(", 
          RowBox[{"1", ",", "0", ",", "0"}], ")"}],
         Derivative],
        MultilineFunction->None], "[", 
       RowBox[{"x", ",", "y", ",", "z"}], "]"}], 
      RowBox[{
       SuperscriptBox["uz", 
        TagBox[
         RowBox[{"(", 
          RowBox[{"0", ",", "1", ",", "0"}], ")"}],
         Derivative],
        MultilineFunction->None], "[", 
       RowBox[{"x", ",", "y", ",", "z"}], "]"}], 
      RowBox[{"1", "+", 
       RowBox[{
        SuperscriptBox["uz", 
         TagBox[
          RowBox[{"(", 
           RowBox[{"0", ",", "0", ",", "1"}], ")"}],
          Derivative],
         MultilineFunction->None], "[", 
        RowBox[{"x", ",", "y", ",", "z"}], "]"}]}]}
    },
    GridBoxAlignment->{
     "Columns" -> {{Center}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{
  3.748158761558833*^9, 3.748159434430613*^9, 3.7482642154665117`*^9, 
   3.748264646350333*^9, 3.74826499735951*^9, 3.7482651160286913`*^9, 
   3.748265231811203*^9, 3.7482654419099483`*^9, 3.748265617905122*^9, 
   3.7482667274369717`*^9, 3.748267446239506*^9, 3.748358477788678*^9, 
   3.748362089545095*^9, 3.748363714396614*^9, 3.748364432709051*^9, 
   3.748364730299684*^9, 3.748364832663312*^9, 3.748364968440797*^9, 
   3.7483654391315203`*^9, 3.748365645809317*^9, 3.748365865369316*^9, 
   3.748367895873979*^9, 3.748368477089388*^9, 3.748368837838067*^9, 
   3.748369381432191*^9, 3.7483700702626743`*^9, 3.7483712704753857`*^9, 
   3.748371403717348*^9, 3.7487617790117145`*^9, {3.748782142486891*^9, 
   3.7487821443688555`*^9}, 3.748784703234404*^9, 3.748788156105542*^9, 
   3.7487887340725718`*^9, 3.7487907819597845`*^9, 3.748876096707106*^9, 
   3.748876409627485*^9, 3.7488791281439524`*^9, 3.748893951221187*^9, 
   3.7488958715719724`*^9, 3.7489444055540447`*^9, 3.748945639067809*^9, 
   3.748947513820125*^9, 3.7489575483216333`*^9, 3.7489642557425356`*^9, 
   3.74919872874899*^9, 3.7494030176669254`*^9, 3.7494520433352585`*^9, 
   3.7494528767708125`*^9, 3.7494535654606028`*^9, 3.7494546201792984`*^9, 
   3.749455207749562*^9, 3.749467011113137*^9, 3.7494686617023177`*^9, 
   3.74949961401155*^9, 3.7495335268752394`*^9, 3.7498149836498675`*^9, 
   3.7498150233620825`*^9, 3.7499696041527376`*^9, 3.749970582697651*^9, 
   3.7501430876581154`*^9, 3.7501939951089106`*^9, 3.750443533093829*^9, 
   3.7504466316650815`*^9, 3.7504477992016926`*^9, 3.7537816635673676`*^9, 
   3.7537828628220177`*^9, 3.7537858107185106`*^9, 3.753786796204507*^9, 
   3.7537870887580547`*^9}],

Cell["Similarly, we define the growth tensor as ", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.7481587064971523`*^9, 3.748158716990864*^9}, {
   3.748158827729438*^9, 3.748158836423505*^9}, 3.748782142486891*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"G", " ", "=", " ", 
   RowBox[{"DiagonalMatrix", "[", 
    RowBox[{"{", 
     RowBox[{"g", ",", "g", ",", 
      FractionBox["1", 
       SuperscriptBox["g", "2"]]}], "}"}], "]"}]}], ";"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.748158727332*^9, 3.748158743908091*^9}, 
   3.748158841571538*^9, 3.748782142486891*^9}],

Cell["\<\
If we display G in matrix form, it has the expected components:\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.748158795434462*^9, 3.748158807880468*^9}, {
   3.7481588442311907`*^9, 3.7481588477269373`*^9}, 3.748782142486891*^9}],

Cell[BoxData[
 RowBox[{"G", "//", "MatrixForm"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.748158758467133*^9, 3.7481587610096483`*^9}, {
   3.748158849110083*^9, 3.748158849543005*^9}, 3.748782142486891*^9}],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {"g", "0", "0"},
     {"0", "g", "0"},
     {"0", "0", 
      FractionBox["1", 
       SuperscriptBox["g", "2"]]}
    },
    GridBoxAlignment->{
     "Columns" -> {{Center}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{
  3.748158761558833*^9, 3.748158852755788*^9, 3.748159437433158*^9, 
   3.748263824018909*^9, 3.748264215805725*^9, 3.748264646513412*^9, 
   3.748264997562785*^9, 3.7482651162466516`*^9, 3.748265231995348*^9, 
   3.748265442124243*^9, 3.7482656181215343`*^9, 3.74826672765707*^9, 
   3.7482674464391623`*^9, 3.7483584779402227`*^9, 3.748362089629294*^9, 
   3.748363714473106*^9, 3.7483644327922163`*^9, 3.748364730382368*^9, 
   3.748364832764303*^9, 3.748364968531913*^9, 3.748365439214426*^9, 
   3.7483656464239273`*^9, 3.748365865540492*^9, 3.748367895974348*^9, 
   3.748368477172717*^9, 3.7483688379394493`*^9, 3.7483693815321903`*^9, 
   3.7483700703531322`*^9, 3.7483712705783167`*^9, 3.748371403834971*^9, 
   3.7487617842706437`*^9, {3.748782142486891*^9, 3.7487821443997726`*^9}, 
   3.748784703265626*^9, 3.7487881561583967`*^9, 3.748788734113464*^9, 
   3.7487907819956865`*^9, 3.7488760967369967`*^9, 3.7488764096603956`*^9, 
   3.7488791281928225`*^9, 3.748893951254071*^9, 3.7488958716032357`*^9, 
   3.748944405569671*^9, 3.7489456390990505`*^9, 3.7489475138560257`*^9, 
   3.74895754835751*^9, 3.748964255780403*^9, 3.749198728782507*^9, 
   3.749452043388635*^9, 3.749452876825635*^9, 3.749453565504432*^9, 
   3.7494552077651453`*^9, 3.749467011148032*^9, 3.7494686617452283`*^9, 
   3.749499614073363*^9, 3.7495335269091244`*^9, 3.749814983915428*^9, 
   3.7498150234245625`*^9, 3.7499696041996064`*^9, 3.7499705827380257`*^9, 
   3.7501430877220883`*^9, 3.75019399516177*^9, 3.75044353315434*^9, 
   3.750446631778308*^9, 3.7504477992670493`*^9, 3.7537816635942855`*^9, 
   3.7537828628479395`*^9, 3.753785810762398*^9, 3.753786796262385*^9, 
   3.7537870888269114`*^9}],

Cell["\<\
Finally,  we can calculate the energy density, including the Lagrange \
multiplier term for incompressibility P[x,y,z]. For the layer we have:\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.748158795434462*^9, 3.748158807880468*^9}, {
   3.7481588442311907`*^9, 3.7481588477269373`*^9}, {3.748158892826709*^9, 
   3.748158940097808*^9}, {3.748159028555542*^9, 3.748159032993976*^9}, 
   3.748782142487891*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"BG", "=", 
   RowBox[{"F", ".", " ", 
    RowBox[{"Inverse", "[", "G", "]"}], ".", " ", 
    RowBox[{"Inverse", "[", 
     RowBox[{"Transpose", "[", "G", "]"}], "]"}], ".", " ", 
    RowBox[{"Transpose", "[", "F", "]"}]}]}], ";", 
  RowBox[{"J", "=", 
   RowBox[{"Det", "[", "F", "]"}]}], ";"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.749371208709732*^9, 3.7493712179669743`*^9}, 
   3.749452042730871*^9, 3.7494526121144257`*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"W", "=", "  ", 
   RowBox[{
    FractionBox["1", "2"], " ", "\[Mu]1", " ", 
    RowBox[{"(", 
     RowBox[{
      FractionBox[
       RowBox[{"Tr", "[", "BG", "]"}], 
       SuperscriptBox["J", 
        RowBox[{"2", "/", "3"}]]], "-", "3", "+", 
      RowBox[{"K", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"J", "-", "1"}], ")"}], "2"]}]}], ")"}]}]}], ";"}]], "Input",\

 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.7481589205795193`*^9, 3.7481589283160753`*^9}, 
   3.7481590457953243`*^9, 3.748159420606257*^9, {3.7482635945743217`*^9, 
   3.748263595485778*^9}, 3.748782142487891*^9, {3.749370926718769*^9, 
   3.7493709294145308`*^9}, {3.749371243680197*^9, 3.749371251459389*^9}, 
   3.749452632613575*^9}],

Cell["and for the substrate we have ", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.748159036794915*^9, 3.748159042857609*^9}, 
   3.748782142487891*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"WS", "=", " ", 
   RowBox[{"W", "/.", "SubstrateFields"}]}], ";"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.74815906494503*^9, 3.748159079513112*^9}, {
   3.748159411277789*^9, 3.748159442849318*^9}, {3.748263605647221*^9, 
   3.7482636093279133`*^9}, 3.748263834339552*^9, 3.748782142487891*^9}],

Cell[TextData[{
 "The total elastic energy, which we need to minimize is then ",
 Cell[BoxData[
  RowBox[{" ", 
   RowBox[{"E", "=", 
    RowBox[{"<", 
     RowBox[{
      RowBox[{
       SubsuperscriptBox["\[Integral]", "0", "a"], 
       RowBox[{"W", " ", 
        RowBox[{"\[DifferentialD]", "z"}]}]}], "+", 
      RowBox[{
       SubsuperscriptBox["\[Integral]", 
        RowBox[{"-", "\[Infinity]"}], "0"], 
       RowBox[{"WL", 
        RowBox[{"\[DifferentialD]", "z"}]}]}]}], ">"}]}]}]], "Input",
  CellChangeTimes->{{3.7481596044512787`*^9, 3.748159618171105*^9}}]
}], "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.7481594993347054`*^9, 3.748159509811297*^9}, {
   3.748159542637512*^9, 3.748159543269404*^9}, {3.748159625204307*^9, 
   3.748159635693334*^9}, 3.748782142487891*^9}]
}, Open  ]],

Cell[BoxData[""], "Input",
 CellChangeTimes->{{3.748158727332*^9, 3.748158743908091*^9}, 
   3.748158841571538*^9, {3.748159931466012*^9, 3.74816004871554*^9}, {
   3.748160228196992*^9, 3.7481602298423862`*^9}, {3.748263051930909*^9, 
   3.7482630826898746`*^9}, 3.748263252957427*^9}],

Cell[CellGroupData[{

Cell[TextData[StyleBox["Equations of equilibrium", "Section"]], "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.},
 CellChangeTimes->{{3.748159461353683*^9, 3.74815946512779*^9}, 
   3.748782150345865*^9}],

Cell["From the main text, we introduce the PK1 stress:", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.},
 CellChangeTimes->{{3.748159912245761*^9, 3.748159921094906*^9}, 
   3.748782150345865*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"\[Sigma]", " ", "=", " ", 
   RowBox[{"\[Mu]1", "  ", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"(", 
        FractionBox[
         RowBox[{"BG", "-", 
          RowBox[{
           RowBox[{"IdentityMatrix", "[", "3", "]"}], 
           RowBox[{
            RowBox[{"Tr", "[", "BG", "]"}], "/", "3"}]}]}], 
         SuperscriptBox["J", 
          RowBox[{"2", "/", "3"}]]], ")"}], "+", 
       RowBox[{"K", " ", "J", " ", 
        RowBox[{"(", 
         RowBox[{"J", "-", "1"}], ")"}], 
        RowBox[{"IdentityMatrix", "[", "3", "]"}]}]}], ")"}], ".", 
     RowBox[{"Inverse", "[", 
      RowBox[{"Transpose", "[", "F", "]"}], "]"}]}]}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"\[Sigma]S", " ", "=", " ", 
   RowBox[{"\[Sigma]", "/.", "SubstrateFields"}]}], ";"}]}], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.},
 CellChangeTimes->{{3.748158727332*^9, 3.748158743908091*^9}, 
   3.748158841571538*^9, {3.748159931466012*^9, 3.74816004871554*^9}, {
   3.748160228196992*^9, 3.7481602298423862`*^9}, {3.748263847307515*^9, 
   3.748263868825564*^9}, 3.7487821503468614`*^9, {3.749371153842516*^9, 
   3.7493712313073177`*^9}, {3.7493712632189355`*^9, 
   3.7493712874122615`*^9}, {3.749452614848092*^9, 3.7494526164478426`*^9}, 
   3.7494545403904624`*^9, 3.7494548181687107`*^9, {3.7494551712279997`*^9, 
   3.749455171436723*^9}}],

Cell["\<\
We  minimize E with respect to variations in displacement and pressure, using \
Euler-Lagrange equations. The \[OpenCurlyDoubleQuote]Variational methods\
\[CloseCurlyDoubleQuote] package is too slow with this energy density, but we \
can write the Euler-Lagrange equations explicitly. Here, ELx, ELy, ELz refer \
to variation with respect to ux, uy, uz\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.},
 CellChangeTimes->{{3.7481594707573547`*^9, 3.748159492909875*^9}, {
   3.7481596477118273`*^9, 3.7481597019004793`*^9}, {3.7481597925987*^9, 
   3.7481598386164017`*^9}, 3.7487821503468614`*^9, {3.749370959075198*^9, 
   3.749370972843397*^9}, {3.749403039077284*^9, 3.7494031143848577`*^9}, {
   3.7498125293624835`*^9, 3.7498125342363434`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"ELx", "=", 
   RowBox[{
    RowBox[{"D", "[", 
     RowBox[{"W", ",", 
      RowBox[{"ux", "[", 
       RowBox[{"x", ",", "y", ",", "z"}], "]"}]}], "]"}], "-", 
    RowBox[{"D", "[", 
     RowBox[{
      RowBox[{"D", "[", 
       RowBox[{"W", ",", 
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"ux", "[", 
           RowBox[{"x", ",", "y", ",", "z"}], "]"}], ",", "x"}], "]"}]}], 
       "]"}], ",", "x"}], "]"}], "-", 
    RowBox[{"D", "[", 
     RowBox[{
      RowBox[{"D", "[", 
       RowBox[{"W", ",", 
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"ux", "[", 
           RowBox[{"x", ",", "y", ",", "z"}], "]"}], ",", "z"}], "]"}]}], 
       "]"}], ",", "z"}], "]"}], "-", 
    RowBox[{"D", "[", 
     RowBox[{
      RowBox[{"D", "[", 
       RowBox[{"W", ",", 
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"ux", "[", 
           RowBox[{"x", ",", "y", ",", "z"}], "]"}], ",", "y"}], "]"}]}], 
       "]"}], ",", "y"}], "]"}]}]}], ";"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.},
 CellChangeTimes->{{3.682408718815958*^9, 3.6824088213211393`*^9}, {
   3.6824088827667246`*^9, 3.6824088884242473`*^9}, 3.6895851791041374`*^9, 
   3.735908310211338*^9, 3.7476591494549203`*^9, {3.747663126425218*^9, 
   3.7476631266486206`*^9}, {3.7476663046623793`*^9, 3.747666316461813*^9}},
 ExpressionUUID -> "1de607fd-934b-4aa1-8396-910b3c728638"],

Cell[BoxData[
 RowBox[{
  RowBox[{"ELy", "=", 
   RowBox[{
    RowBox[{"D", "[", 
     RowBox[{"W", ",", 
      RowBox[{"uy", "[", 
       RowBox[{"x", ",", "y", ",", "z"}], "]"}]}], "]"}], "-", 
    RowBox[{"D", "[", 
     RowBox[{
      RowBox[{"D", "[", 
       RowBox[{"W", ",", 
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"uy", "[", 
           RowBox[{"x", ",", "y", ",", "z"}], "]"}], ",", "x"}], "]"}]}], 
       "]"}], ",", "x"}], "]"}], "-", 
    RowBox[{"D", "[", 
     RowBox[{
      RowBox[{"D", "[", 
       RowBox[{"W", ",", 
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"uy", "[", 
           RowBox[{"x", ",", "y", ",", "z"}], "]"}], ",", "z"}], "]"}]}], 
       "]"}], ",", "z"}], "]"}], "-", 
    RowBox[{"D", "[", 
     RowBox[{
      RowBox[{"D", "[", 
       RowBox[{"W", ",", 
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"uy", "[", 
           RowBox[{"x", ",", "y", ",", "z"}], "]"}], ",", "y"}], "]"}]}], 
       "]"}], ",", "y"}], "]"}]}]}], ";"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.},
 CellChangeTimes->{{3.6824088294933696`*^9, 3.6824088363784966`*^9}, {
   3.6824088695520253`*^9, 3.682408879669493*^9}, 3.6895851791041374`*^9, {
   3.735907761494424*^9, 3.73590777392456*^9}, 3.7359083102116127`*^9, 
   3.74765915095275*^9, {3.7476631271213746`*^9, 3.747663127184187*^9}, {
   3.7476663055739393`*^9, 3.7476663172866273`*^9}},
 ExpressionUUID -> "2f0329a3-5fcb-496d-a69f-e34e4a4ecd5e"],

Cell[BoxData[
 RowBox[{
  RowBox[{"ELz", "=", 
   RowBox[{
    RowBox[{"D", "[", 
     RowBox[{"W", ",", 
      RowBox[{"uz", "[", 
       RowBox[{"x", ",", "y", ",", "z"}], "]"}]}], "]"}], "-", 
    RowBox[{"D", "[", 
     RowBox[{
      RowBox[{"D", "[", 
       RowBox[{"W", ",", 
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"uz", "[", 
           RowBox[{"x", ",", "y", ",", "z"}], "]"}], ",", "x"}], "]"}]}], 
       "]"}], ",", "x"}], "]"}], "-", 
    RowBox[{"D", "[", 
     RowBox[{
      RowBox[{"D", "[", 
       RowBox[{"W", ",", 
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"uz", "[", 
           RowBox[{"x", ",", "y", ",", "z"}], "]"}], ",", "z"}], "]"}]}], 
       "]"}], ",", "z"}], "]"}], "-", 
    RowBox[{"D", "[", 
     RowBox[{
      RowBox[{"D", "[", 
       RowBox[{"W", ",", 
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"uz", "[", 
           RowBox[{"x", ",", "y", ",", "z"}], "]"}], ",", "y"}], "]"}]}], 
       "]"}], ",", "y"}], "]"}]}]}], ";"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.},
 CellChangeTimes->{
  3.735907752193316*^9, 3.73590831021187*^9, 3.7476591529497414`*^9, {
   3.7476631276968155`*^9, 3.747663127729727*^9}, {3.7476663065254173`*^9, 
   3.747666318550249*^9}},
 ExpressionUUID -> "846c022b-4157-44eb-b562-4f34a2b56a7b"],

Cell[BoxData[{
 RowBox[{
  RowBox[{"EL", " ", "=", 
   RowBox[{"{", 
    RowBox[{"ELx", ",", "ELy", ",", "ELz"}], "}"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"ELS", "=", 
   RowBox[{"EL", "/.", "SubstrateFields"}]}], ";"}]}], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.},
 CellChangeTimes->{{3.748159737349607*^9, 3.748159777026465*^9}, {
   3.7481598632020063`*^9, 3.7481598810558777`*^9}, 3.7487821503468614`*^9, {
   3.7493709877595015`*^9, 3.7493710275320935`*^9}, 3.749403034903448*^9}],

Cell["\<\
We join these two list together into a single large list of all eight  bulk \
equations\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.},
 CellChangeTimes->{{3.748267170216387*^9, 3.748267196294633*^9}, 
   3.7487821503468614`*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"BulkEquations", "=", 
   RowBox[{"Join", "[", 
    RowBox[{"EL", ",", "ELS"}], "]"}]}], ";"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.},
 CellChangeTimes->{{3.748159737349607*^9, 3.748159777026465*^9}, {
   3.7481598632020063`*^9, 3.7481598810558777`*^9}, {3.748267210423747*^9, 
   3.748267232269842*^9}, 3.7487821503468614`*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"BoundaryConditionSurfaceStress", ":=", 
   RowBox[{
    RowBox[{
     RowBox[{"\[Sigma]", ".", 
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "+", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", 
       RowBox[{"2", " ", "L", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"uz", "[", 
           RowBox[{"x", ",", "y", ",", "z1"}], "]"}], "-", 
          RowBox[{
           RowBox[{"Integrate", "[", 
            RowBox[{
             RowBox[{"uz", "[", 
              RowBox[{"x", ",", "y", ",", "z1"}], "]"}], ",", 
             RowBox[{"{", 
              RowBox[{"x", ",", "0", ",", "xmax"}], "}"}], ",", 
             RowBox[{"{", 
              RowBox[{"y", ",", "0", ",", "ymax"}], "}"}]}], "]"}], "/", 
           RowBox[{"(", 
            RowBox[{"xmax", "*", "ymax"}], ")"}]}]}], ")"}]}]}], "}"}]}], "/.", 
    RowBox[{"{", 
     RowBox[{"z", "\[Rule]", " ", "z1"}], "}"}]}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"BoundaryConditionDisplacement", "=", 
   RowBox[{
    RowBox[{"u", "[", 
     RowBox[{"x", ",", "y", ",", "z0"}], "]"}], "-", 
    RowBox[{"uS", "[", 
     RowBox[{"x", ",", "y", ",", "z0"}], "]"}]}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"BoundaryConditionsInterfaceStress", "=", 
   RowBox[{
    RowBox[{
     RowBox[{"\[Sigma]", ".", 
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "-", 
     RowBox[{"\[Sigma]S", ".", 
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "1"}], "}"}]}]}], "/.", 
    RowBox[{"{", 
     RowBox[{"z", "\[Rule]", " ", "z0"}], "}"}]}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"BoundaryConditionsAmplitude", ":=", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"1", "/", "\[CurlyEpsilon]"}], ")"}], 
    RowBox[{"(", 
     RowBox[{
      RowBox[{
       RowBox[{"Integrate", "[", 
        RowBox[{
         SuperscriptBox[
          RowBox[{"(", 
           RowBox[{
            RowBox[{"uz", "[", 
             RowBox[{"x", ",", "y", ",", "z1"}], "]"}], "-", 
            RowBox[{
             RowBox[{"Integrate", "[", 
              RowBox[{
               RowBox[{"uz", "[", 
                RowBox[{"x", ",", "y", ",", "z1"}], "]"}], ",", 
               RowBox[{"{", 
                RowBox[{"x", ",", "0", ",", "xmax"}], "}"}], ",", 
               RowBox[{"{", 
                RowBox[{"y", ",", "0", ",", "ymax"}], "}"}]}], "]"}], "/", 
             RowBox[{"(", 
              RowBox[{"xmax", "*", "ymax"}], ")"}]}]}], ")"}], "2"], ",", 
         RowBox[{"{", 
          RowBox[{"x", ",", "0", ",", "xmax"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"y", ",", "0", ",", "ymax"}], "}"}]}], "]"}], "/", 
       RowBox[{"(", 
        RowBox[{"xmax", "*", "ymax"}], ")"}]}], "-", 
      RowBox[{"\[CurlyEpsilon]", "^", "2"}]}], ")"}]}]}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.},
 CellChangeTimes->{{3.716620573616859*^9, 3.7166206460982933`*^9}, {
   3.7166665325178995`*^9, 3.716666537893193*^9}, {3.7166666610035477`*^9, 
   3.716666664499898*^9}, {3.716667247192238*^9, 3.7166672962699986`*^9}, {
   3.7166698439897795`*^9, 3.716669852672717*^9}, {3.7166699436832447`*^9, 
   3.716669964491853*^9}, {3.7166705739286613`*^9, 3.7166705784487686`*^9}, {
   3.7166709965282393`*^9, 3.7166710122606955`*^9}, {3.716671165258074*^9, 
   3.7166711744841456`*^9}, 3.7170405284753494`*^9, {3.7198214941631956`*^9, 
   3.719821497491223*^9}, {3.719821833896503*^9, 3.71982187590659*^9}, {
   3.719825602201521*^9, 3.719825603929378*^9}, 3.734338433142171*^9, {
   3.7343416068751993`*^9, 3.7343416429229355`*^9}, {3.734524027035774*^9, 
   3.734524044987177*^9}, {3.7348032223222446`*^9, 3.7348032397050104`*^9}, {
   3.7483596691754847`*^9, 3.748359675223789*^9}, {3.7483621048490763`*^9, 
   3.7483621092926903`*^9}, {3.748362206482781*^9, 3.748362339641395*^9}, {
   3.7483624068271103`*^9, 3.748362426864996*^9}, {3.7483626830627604`*^9, 
   3.748362710905182*^9}, {3.7483627829528646`*^9, 3.7483628059333477`*^9}, {
   3.74836436962538*^9, 3.748364399607009*^9}, {3.7483649291164217`*^9, 
   3.748364943985876*^9}, 3.748585348073351*^9, 3.7487821503468614`*^9, {
   3.7494531448497887`*^9, 3.74945318430024*^9}, 3.7494532503261614`*^9, {
   3.7494534012380695`*^9, 3.7494534549382052`*^9}},
 ExpressionUUID -> "576cfcb5-ac56-451a-b7a7-277ed325b1cc"],

Cell[BoxData[
 RowBox[{
  RowBox[{"BoundaryConditions", ":=", 
   RowBox[{"Join", "[", 
    RowBox[{
    "BoundaryConditionSurfaceStress", ",", "BoundaryConditionDisplacement", 
     ",", "BoundaryConditionsInterfaceStress", ",", 
     RowBox[{"{", "BoundaryConditionsAmplitude", "}"}]}], "]"}]}], 
  ";"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.},
 CellChangeTimes->{{3.748159737349607*^9, 3.748159777026465*^9}, {
   3.7481598632020063`*^9, 3.7481598810558777`*^9}, {3.748267210423747*^9, 
   3.748267232269842*^9}, {3.748362353152483*^9, 3.748362372874612*^9}, {
   3.748362430936763*^9, 3.7483624385081987`*^9}, {3.748363159362442*^9, 
   3.748363162500799*^9}, 3.7487821503468614`*^9, 3.749453457177657*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[StyleBox["Stripe Pattern", "Section"]], "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.7483712732950583`*^9, 3.7483712754201603`*^9}, 
   3.7487821767591968`*^9}],

Cell["Displacement fields", "Subsubsection",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.734858096251944*^9, 3.734858108803898*^9}, 
   3.7487821767591968`*^9},
 TextAlignment->Left,
 ExpressionUUID -> "1b3bfbbc-60b3-4b40-9a2f-a856d7813b9c"],

Cell["\<\
As discussed in the main paper, the form of u(x,y,z) for the stripe pattern \
in the thin slab is expected to be of the form:\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.7161206305567656`*^9, 3.716120654414336*^9}, {
   3.719825444630498*^9, 3.7198254490843143`*^9}, {3.7343365435634594`*^9, 
   3.7343365530569415`*^9}, {3.7481926611831303`*^9, 3.7481926774792223`*^9}, 
   3.7487821767591968`*^9},
 ExpressionUUID -> "f5a52526-2b6d-407b-9f04-21d090a88149"],

Cell[BoxData[
 RowBox[{
  RowBox[{"uz", "[", 
   RowBox[{"x_", ",", "y_", ",", "z_"}], "]"}], ":=", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"\[Gamma]", "-", "1"}], ")"}], "z"}], "+", 
   RowBox[{"\[CurlyEpsilon]", " ", 
    RowBox[{"Cos", "[", 
     RowBox[{"k", " ", "x"}], "]"}], " ", 
    RowBox[{
     RowBox[{"fz", "[", 
      RowBox[{"1", ",", "1"}], "]"}], "[", "z", "]"}]}]}]}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.6657713232344956`*^9, 3.665771343381418*^9}, {
   3.665771418266056*^9, 3.665771443269935*^9}, {3.6657714854748755`*^9, 
   3.6657715331802435`*^9}, {3.665771716847458*^9, 3.665771720446126*^9}, {
   3.6657718139064054`*^9, 3.6657718741793876`*^9}, {3.665772091294901*^9, 
   3.665772215307926*^9}, {3.665772247964039*^9, 3.665772263610958*^9}, {
   3.665772485446594*^9, 3.6657725184863443`*^9}, 3.6657729166750526`*^9, {
   3.66577426017365*^9, 3.6657742603906803`*^9}, 3.6658296584374185`*^9, {
   3.6672460521798906`*^9, 3.667246110757153*^9}, 3.667301860838393*^9, {
   3.7058179970092716`*^9, 3.705818048247526*^9}, {3.705818285589779*^9, 
   3.7058183237035446`*^9}, 3.705818449364562*^9, {3.71612068835028*^9, 
   3.7161206885199823`*^9}, 3.7161210204739447`*^9, 3.7345160841540394`*^9, 
   3.748244002237558*^9, {3.7487821619637814`*^9, 3.7487821767601943`*^9}, {
   3.7493800296529713`*^9, 3.749380032969139*^9}, 3.749402701254895*^9, {
   3.7494045120502443`*^9, 3.749404533873191*^9}, {3.7494530652031155`*^9, 
   3.749453065473361*^9}},
 ExpressionUUID -> "56930096-319d-4d45-b5d5-c082ff58afa2"],

Cell[BoxData[
 RowBox[{
  RowBox[{"ux", "[", 
   RowBox[{"x_", ",", "y_", ",", "z_"}], "]"}], ":=", 
  RowBox[{"\[CurlyEpsilon]", " ", 
   RowBox[{"Sin", "[", 
    RowBox[{"k", " ", "x"}], "]"}], " ", 
   RowBox[{
    RowBox[{"fx", "[", 
     RowBox[{"1", ",", "1"}], "]"}], "[", "z", "]"}]}]}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.665774263326209*^9, 3.66577427385541*^9}, {
   3.6657743236327744`*^9, 3.665774371967226*^9}, {3.665774467469146*^9, 
   3.6657745171664667`*^9}, {3.6657746780929537`*^9, 
   3.6657747272032385`*^9}, {3.6657757781517873`*^9, 3.665775808017256*^9}, 
   3.6657759580813837`*^9, 3.6657760065598683`*^9, 3.6658296584374185`*^9, 
   3.6660107266892014`*^9, 3.666012204886698*^9, {3.666012513519569*^9, 
   3.6660125247351093`*^9}, {3.66601278685109*^9, 3.666012806026108*^9}, {
   3.666012972519032*^9, 3.666012979006877*^9}, {3.666013887366681*^9, 
   3.666013928053396*^9}, 3.666013989868081*^9, {3.6660147320221443`*^9, 
   3.666014756277513*^9}, {3.666014819652697*^9, 3.6660148425158243`*^9}, {
   3.666015002745376*^9, 3.666015011136634*^9}, 3.666015703700157*^9, 
   3.666015744531355*^9, 3.667246053897068*^9, {3.6672461691966453`*^9, 
   3.6672461754305334`*^9}, {3.705818060293044*^9, 3.7058180743009596`*^9}, {
   3.705818289204507*^9, 3.705818297352764*^9}, {3.705818346823226*^9, 
   3.705818381764964*^9}, 3.7058184518271837`*^9, {3.705819379045109*^9, 
   3.7058193899392233`*^9}, {3.716120693874295*^9, 3.716120694038066*^9}, 
   3.71612103748448*^9, {3.71666379795361*^9, 3.7166638046447215`*^9}, 
   3.7345160870401254`*^9, {3.7347802139114037`*^9, 3.734780215949613*^9}, 
   3.7482440160956907`*^9, {3.7487821619637814`*^9, 3.7487821767601943`*^9}, 
   3.749402703697362*^9, {3.7494045149804163`*^9, 3.7494045422577934`*^9}, {
   3.749453066894868*^9, 3.7494530671227765`*^9}},
 ExpressionUUID -> "40dad8a7-b736-4d12-af4b-30060025e83a"],

Cell[BoxData[
 RowBox[{
  RowBox[{"uy", "[", 
   RowBox[{"x_", ",", "y_", ",", "z_"}], "]"}], ":=", "0"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.6657759391218367`*^9, 3.665776057713159*^9}, 
   3.6658296584379063`*^9, 3.6660107291851473`*^9, 3.666012209694578*^9, {
   3.666012528999206*^9, 3.666012537142894*^9}, {3.666012999870729*^9, 
   3.666013019358135*^9}, {3.666013072317521*^9, 3.666013074029131*^9}, {
   3.6660131551716967`*^9, 3.6660131691793137`*^9}, {3.6660139204933443`*^9, 
   3.666013926621202*^9}, 3.666013992463925*^9, {3.6660147384939632`*^9, 
   3.666014757597476*^9}, {3.666014822508614*^9, 3.6660148479637136`*^9}, {
   3.6660150080811462`*^9, 3.666015018784547*^9}, {3.666015633085493*^9, 
   3.666015653788924*^9}, 3.666015704827859*^9, 3.666015746131331*^9, 
   3.6672460553753724`*^9, 3.66724612564708*^9, {3.7161206968910065`*^9, 
   3.7161206970825825`*^9}, {3.7487821619637814`*^9, 3.7487821767601943`*^9}},
 
 ExpressionUUID -> "37011bf4-9d75-4895-8b92-c683ff6c780b"],

Cell["and in the substrate", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.7343365589552717`*^9, 3.7343365626007757`*^9}, 
   3.7487821767601943`*^9},
 ExpressionUUID -> "ceb842ef-d180-4cf0-a344-86b9f7dc9cc3"],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"uxS", "[", 
    RowBox[{"x_", ",", "y_", ",", "z_"}], "]"}], " ", ":=", 
   RowBox[{
    RowBox[{"ux", "[", 
     RowBox[{"x", ",", "y", ",", "z"}], "]"}], "/.", "SubstrateFields"}]}], 
  ";"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.653378489608873*^9, 3.6533785312774076`*^9}, {
   3.653378984925927*^9, 3.65337900502431*^9}, {3.65338276511313*^9, 
   3.6533828063939*^9}, 3.653382854167697*^9, {3.653383745662682*^9, 
   3.653383782981093*^9}, {3.653383840692298*^9, 3.6533838456198883`*^9}, {
   3.653384249995995*^9, 3.6533843123629847`*^9}, {3.653386202863043*^9, 
   3.6533862584998693`*^9}, {3.653390737705628*^9, 3.653390739937315*^9}, {
   3.653390902013857*^9, 3.65339090337775*^9}, {3.653390940309018*^9, 
   3.653390940712945*^9}, 3.653391190623646*^9, {3.6533912495193567`*^9, 
   3.653391282789853*^9}, 3.653634747776924*^9, 3.653637435127521*^9, 
   3.6536375108258133`*^9, {3.6536379695531263`*^9, 3.653637970546637*^9}, {
   3.653638004896541*^9, 3.653638005288282*^9}, {3.6536381345973387`*^9, 
   3.653638138828762*^9}, {3.6536424525112953`*^9, 3.653642456086172*^9}, {
   3.6536466374026337`*^9, 3.653646655164146*^9}, {3.653651776708181*^9, 
   3.653651784372064*^9}, {3.653651967105109*^9, 3.653651968536775*^9}, {
   3.65365202098381*^9, 3.6536520477108297`*^9}, {3.6544258621826887`*^9, 
   3.65442587450977*^9}, {3.654431134252376*^9, 3.654431145011314*^9}, 
   3.6544318188854713`*^9, {3.6544319448675013`*^9, 3.654431962603526*^9}, {
   3.654432465720717*^9, 3.654432471511896*^9}, {3.654432647083828*^9, 
   3.654432652452134*^9}, {3.654432766721627*^9, 3.654432776369487*^9}, {
   3.654432814024529*^9, 3.6544328205363894`*^9}, {3.654432959118018*^9, 
   3.654432964701429*^9}, {3.654433096708247*^9, 3.654433104705942*^9}, {
   3.654433218424775*^9, 3.654433220887821*^9}, {3.6544332764712877`*^9, 
   3.6544332772706413`*^9}, {3.654433470843918*^9, 3.654433480682355*^9}, {
   3.654433735581304*^9, 3.6544337625649023`*^9}, {3.654433903273954*^9, 
   3.654433910361353*^9}, {3.654433999919984*^9, 3.6544340151513968`*^9}, {
   3.654434209837215*^9, 3.6544342138123627`*^9}, {3.654434319016773*^9, 
   3.6544343431604*^9}, {3.654434533861601*^9, 3.654434538244244*^9}, {
   3.654434589083997*^9, 3.6544345900926113`*^9}, 3.6544348550385513`*^9, {
   3.6544350995764933`*^9, 3.6544351006003923`*^9}, {3.654435170048601*^9, 
   3.6544351740790243`*^9}, {3.654435232855584*^9, 3.654435236309228*^9}, {
   3.65443548767491*^9, 3.654435489263926*^9}, 3.654435612109846*^9, 
   3.6544356557016687`*^9, 3.654435765363168*^9, {3.6544358781764812`*^9, 
   3.654435881271947*^9}, {3.654435925103607*^9, 3.654435927151387*^9}, 
   3.654436023741592*^9, {3.654436101331933*^9, 3.654436139826598*^9}, {
   3.654436195178339*^9, 3.6544362263610697`*^9}, {3.654600142957807*^9, 
   3.654600143893454*^9}, 3.654600174442533*^9, {3.654601230294147*^9, 
   3.654601246720536*^9}, 3.6546031280197973`*^9, 3.654603331190138*^9, {
   3.654603409846778*^9, 3.654603410797145*^9}, {3.654607368657799*^9, 
   3.6546073984854603`*^9}, 3.654944865923127*^9, 3.654944959384799*^9, {
   3.6549456872439747`*^9, 3.6549457067261343`*^9}, {3.654948243623043*^9, 
   3.654948272841017*^9}, 3.654948381915938*^9, {3.6549484811720543`*^9, 
   3.654948481446971*^9}, {3.654949843398978*^9, 3.6549498892377996`*^9}, {
   3.65495003633182*^9, 3.654950041850161*^9}, {3.655119452157836*^9, 
   3.655119458179923*^9}, 3.6551195417469053`*^9, {3.6551199819944477`*^9, 
   3.65512002344252*^9}, {3.6551226931248283`*^9, 3.655122697436098*^9}, {
   3.655449270237895*^9, 3.6554493688669243`*^9}, {3.6554523971960616`*^9, 
   3.6554524221045923`*^9}, {3.655539312869871*^9, 3.655539327941757*^9}, 
   3.655540209700885*^9, {3.656235442696848*^9, 3.656235448360507*^9}, {
   3.656242241814143*^9, 3.656242299502399*^9}, {3.6562423503894672`*^9, 
   3.656242354437297*^9}, {3.656242431156152*^9, 3.656242436570652*^9}, {
   3.6562442575626163`*^9, 3.656244283730896*^9}, 3.656322116209444*^9, {
   3.656324411648356*^9, 3.656324412527975*^9}, {3.65632860940414*^9, 
   3.65632862648352*^9}, {3.656331744076589*^9, 3.65633176621229*^9}, {
   3.656331803163847*^9, 3.656331855067171*^9}, {3.6563323005370827`*^9, 
   3.656332337090822*^9}, {3.6563325306464443`*^9, 3.656332588333802*^9}, {
   3.656336033484117*^9, 3.656336096057795*^9}, {3.6564018134846015`*^9, 
   3.65640181834159*^9}, {3.6564020186226864`*^9, 3.6564020385035753`*^9}, {
   3.6564020807025366`*^9, 3.656402113699897*^9}, {3.656402189561782*^9, 
   3.6564022851846113`*^9}, {3.6564023423213015`*^9, 
   3.6564023829871492`*^9}, {3.6564024147071257`*^9, 3.656402430196471*^9}, {
   3.656402512787984*^9, 3.65640251391029*^9}, 3.6564025646841536`*^9, {
   3.6564026110136905`*^9, 3.6564026319503975`*^9}, {3.6564044250173144`*^9, 
   3.6564044300125113`*^9}, {3.65640446768567*^9, 3.656404490823603*^9}, {
   3.6564060240181212`*^9, 3.656406036030825*^9}, {3.65640611179447*^9, 
   3.6564061130183363`*^9}, {3.656406145692709*^9, 3.656406190104973*^9}, 
   3.6564091094866962`*^9, {3.6564097325831666`*^9, 3.6564098157265*^9}, {
   3.6570301059500732`*^9, 3.657030110037575*^9}, {3.657354761327952*^9, 
   3.65735476350546*^9}, 3.6649119715204744`*^9, {3.66514571166147*^9, 
   3.6651457456776066`*^9}, 3.6651461491061306`*^9, {3.665146376692958*^9, 
   3.665146384909066*^9}, {3.665165810870386*^9, 3.6651658112532315`*^9}, {
   3.665176365125589*^9, 3.665176398747633*^9}, {3.6652262011487246`*^9, 
   3.6652262027787337`*^9}, {3.665300636031836*^9, 3.6653006541165915`*^9}, {
   3.6653010448729863`*^9, 3.66530104813066*^9}, {3.6653012653576336`*^9, 
   3.6653012660088434`*^9}, 3.6653092065011587`*^9, {3.6653247721505795`*^9, 
   3.665324772613239*^9}, {3.6653914647859087`*^9, 3.665391525419199*^9}, {
   3.665392575525505*^9, 3.6653925763305845`*^9}, 3.6653934786372986`*^9, {
   3.665433338210455*^9, 3.6654333410946145`*^9}, {3.6658296132314606`*^9, 
   3.6658296398754435`*^9}, {3.6670365339806337`*^9, 
   3.6670365344652624`*^9}, {3.667549811196762*^9, 3.667549823541642*^9}, 
   3.6708461501379952`*^9, {3.671515747845851*^9, 3.67151574800206*^9}, 
   3.7339012212388983`*^9, 3.748263123339094*^9, {3.74826427511895*^9, 
   3.748264275689365*^9}, {3.7487821619637814`*^9, 3.7487821767601943`*^9}},
 ExpressionUUID -> "961a37e2-31ff-4366-af70-d4df4d9c7878"],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"uyS", "[", 
    RowBox[{"x_", ",", "y_", ",", "z_"}], "]"}], " ", ":=", 
   RowBox[{
    RowBox[{"uy", "[", 
     RowBox[{"x", ",", "y", ",", "z"}], "]"}], "/.", "SubstrateFields"}]}], 
  ";"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.653646656864345*^9, 3.653646666201375*^9}, {
   3.6544258790946913`*^9, 3.6544258819273167`*^9}, {3.654431134883913*^9, 
   3.654431148657559*^9}, 3.654431819769801*^9, {3.654431967483809*^9, 
   3.6544319777410393`*^9}, {3.654432474728406*^9, 3.654432501759036*^9}, {
   3.654432656380678*^9, 3.654432660348859*^9}, {3.654432966669482*^9, 
   3.6544329700770397`*^9}, {3.654433101419043*^9, 3.654433110930039*^9}, {
   3.6544332812877903`*^9, 3.65443328461489*^9}, {3.654433474835051*^9, 
   3.654433477234099*^9}, {3.654433754766362*^9, 3.654433759645101*^9}, {
   3.654433896290979*^9, 3.654433910753422*^9}, {3.654434220261179*^9, 
   3.654434224250766*^9}, {3.654434319752801*^9, 3.654434351944413*^9}, 
   3.654434399543202*^9, {3.654434535749795*^9, 3.654434546805729*^9}, 
   3.654434596040084*^9, 3.654434856477832*^9, {3.654435103905141*^9, 
   3.654435104577302*^9}, {3.654435176407405*^9, 3.65443518226292*^9}, {
   3.654435237250764*^9, 3.6544352385893707`*^9}, {3.654435492097211*^9, 
   3.654435492888328*^9}, 3.654435613318428*^9, 3.654435724500189*^9, {
   3.654435766779348*^9, 3.654435768314274*^9}, {3.6544359675596333`*^9, 
   3.654435968414906*^9}, 3.654436016173621*^9, {3.654436140634481*^9, 
   3.654436143467098*^9}, {3.6544362039776278`*^9, 3.654436228369001*^9}, 
   3.6546001761650763`*^9, {3.654601231667083*^9, 3.654601258872344*^9}, {
   3.654603142435042*^9, 3.6546031450094852`*^9}, {3.654603418312087*^9, 
   3.654603418948114*^9}, {3.654607370182158*^9, 3.654607416756566*^9}, {
   3.654852246040944*^9, 3.6548522598968573`*^9}, {3.654852314008891*^9, 
   3.654852314144066*^9}, {3.654852732552444*^9, 3.654852736568254*^9}, 
   3.654944860955061*^9, 3.654944962121249*^9, {3.6549456905310926`*^9, 
   3.654945715457518*^9}, {3.6549482817625732`*^9, 3.654948284038377*^9}, 
   3.654948385700252*^9, {3.654948483465907*^9, 3.654948485490468*^9}, {
   3.6549498488876667`*^9, 3.654949890598049*^9}, {3.654950043228282*^9, 
   3.6549500515465508`*^9}, {3.655116306100341*^9, 3.6551163106042843`*^9}, {
   3.655116445886464*^9, 3.655116470392399*^9}, {3.655117198129714*^9, 
   3.655117215969509*^9}, {3.6551174415973787`*^9, 3.65511746020441*^9}, 
   3.6551183824662323`*^9, {3.655119453797896*^9, 3.655119459572075*^9}, 
   3.655119543328308*^9, {3.655120028300734*^9, 3.655120038329707*^9}, {
   3.655122702278792*^9, 3.655122703700597*^9}, {3.655449373300736*^9, 
   3.655449397635953*^9}, {3.6554521762627583`*^9, 3.6554521913682113`*^9}, {
   3.655452394013936*^9, 3.655452429502508*^9}, {3.655539331485566*^9, 
   3.655539364165963*^9}, {3.6555398130531387`*^9, 3.655539829747613*^9}, 
   3.655540212661139*^9, 3.6562354564256163`*^9, {3.656242312482699*^9, 
   3.656242316230077*^9}, {3.656244288251334*^9, 3.6562443066184893`*^9}, {
   3.656322120841134*^9, 3.656322121611712*^9}, {3.656322351157514*^9, 
   3.656322356989119*^9}, {3.656324416608314*^9, 3.65632442102378*^9}, {
   3.656328610387838*^9, 3.656328627427506*^9}, {3.6563317457803183`*^9, 
   3.656331859586472*^9}, {3.656332607007971*^9, 3.6563326192843523`*^9}, {
   3.6563361151247063`*^9, 3.656336124937318*^9}, {3.6564018246350327`*^9, 
   3.6564018318525796`*^9}, 3.6564020469767513`*^9, {3.656402083369752*^9, 
   3.6564020849698887`*^9}, {3.656402308529245*^9, 3.656402394087489*^9}, {
   3.656402642862974*^9, 3.656402714402313*^9}, 3.656402756846137*^9, {
   3.656404427394226*^9, 3.6564044893855834`*^9}, {3.6564059844045863`*^9, 
   3.6564060168452473`*^9}, {3.6564061141501384`*^9, 
   3.6564061156622105`*^9}, {3.656406159067218*^9, 3.6564062011502333`*^9}, 
   3.656409113788725*^9, {3.656409823793915*^9, 3.656409838026557*^9}, {
   3.657030116383741*^9, 3.6570301244534273`*^9}, {3.657354765729664*^9, 
   3.6573547678040657`*^9}, 3.664911968549234*^9, {3.6651463730200953`*^9, 
   3.665146387680727*^9}, {3.6651658157814565`*^9, 3.665165816470908*^9}, {
   3.665176366732342*^9, 3.6651764046392145`*^9}, {3.665226205860623*^9, 
   3.6652262085959606`*^9}, {3.665226276977391*^9, 3.6652262773792405`*^9}, 
   3.6653092065011587`*^9, {3.665433338210455*^9, 3.6654333410946145`*^9}, {
   3.665829616987405*^9, 3.6658296431942353`*^9}, 3.667036538137045*^9, {
   3.667549813188735*^9, 3.667549824333385*^9}, 3.6708461506140385`*^9, {
   3.6715157490333633`*^9, 3.6715157490958643`*^9}, 3.7339012212388983`*^9, {
   3.734780218998967*^9, 3.734780220293178*^9}, 3.748263124091325*^9, {
   3.748264278266306*^9, 3.748264278635758*^9}, {3.7487821619637814`*^9, 
   3.7487821767601943`*^9}},
 ExpressionUUID -> "8c7abfee-ce24-49c0-ab8a-7e8fc1b916d3"],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"uzS", "[", 
    RowBox[{"x_", ",", "y_", ",", "z_"}], "]"}], " ", ":=", " ", 
   RowBox[{
    RowBox[{"uz", "[", 
     RowBox[{"x", ",", "y", ",", "z"}], "]"}], "/.", "SubstrateFields"}]}], 
  ";"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.653646668681004*^9, 3.653646699145215*^9}, {
   3.653651962025899*^9, 3.653651976480847*^9}, {3.6536520249427347`*^9, 
   3.6536520498153677`*^9}, {3.654425813951067*^9, 3.65442581783171*^9}, {
   3.6544304676650343`*^9, 3.654430509489046*^9}, {3.6544310802697268`*^9, 
   3.6544310827164516`*^9}, 3.65443182115392*^9, 3.6544319867863283`*^9, {
   3.65443254753474*^9, 3.654432582821368*^9}, {3.654432866977209*^9, 
   3.654432881535369*^9}, 3.654433017380834*^9, {3.654433190081985*^9, 
   3.654433195392448*^9}, {3.654433487330469*^9, 3.654433494186371*^9}, {
   3.6544341672446613`*^9, 3.654434200156747*^9}, 3.654434321264763*^9, {
   3.654434360176662*^9, 3.654434365016451*^9}, 3.6544344341359653`*^9, {
   3.65443453959233*^9, 3.6544345431130257`*^9}, {3.654434600411928*^9, 
   3.654434604923008*^9}, {3.654434857493829*^9, 3.654434858742721*^9}, {
   3.654435035130908*^9, 3.654435038849515*^9}, {3.654599015572548*^9, 
   3.6545990907637367`*^9}, {3.6545991246198483`*^9, 3.654599127658218*^9}, {
   3.65459917444527*^9, 3.654599203160955*^9}, {3.654599245304274*^9, 
   3.6545992522253647`*^9}, {3.654599699287374*^9, 3.654599716207759*^9}, {
   3.654600711180138*^9, 3.654600743266433*^9}, {3.65460082516348*^9, 
   3.654600868136133*^9}, 3.654853362060231*^9, {3.654941705651705*^9, 
   3.6549417213311157`*^9}, 3.654944852283134*^9, 3.654944965477054*^9, {
   3.654945718026718*^9, 3.654945727097561*^9}, {3.6551194182370853`*^9, 
   3.655119440972431*^9}, {3.655120052006782*^9, 3.655120068768977*^9}, {
   3.655122705902609*^9, 3.655122708773796*^9}, {3.655539392228231*^9, 
   3.655539402211912*^9}, 3.656235457329543*^9, {3.656236096013527*^9, 
   3.656236100300889*^9}, {3.6562367023385983`*^9, 3.65623674575331*^9}, {
   3.656237279744659*^9, 3.6562372927583103`*^9}, {3.656237328119171*^9, 
   3.6562373542553053`*^9}, {3.656237386622595*^9, 3.65623739033451*^9}, 
   3.656238918650964*^9, {3.656239409234419*^9, 3.656239471432931*^9}, {
   3.656239528657618*^9, 3.656239542936122*^9}, {3.6562400713854923`*^9, 
   3.656240149613399*^9}, 3.656242215808754*^9, 3.656244243667077*^9, 
   3.6563220677301397`*^9, {3.6563286122161283`*^9, 3.656328635755446*^9}, {
   3.6564027290294943`*^9, 3.656402738891775*^9}, {3.656404280569064*^9, 
   3.65640434942456*^9}, {3.6564059781954913`*^9, 3.6564059785893593`*^9}, {
   3.656413116052379*^9, 3.656413121967552*^9}, {3.6564157307394524`*^9, 
   3.656415738118362*^9}, {3.6649119643670444`*^9, 3.664911965078226*^9}, {
   3.665146363592942*^9, 3.665146389961588*^9}, 3.6651464604984417`*^9, {
   3.6651658195361357`*^9, 3.6651658213485804`*^9}, {3.665166008082565*^9, 
   3.6651660085779257`*^9}, {3.6651763692147284`*^9, 
   3.6651764086038017`*^9}, {3.6652262116202974`*^9, 3.6652262128931503`*^9}, 
   3.665226281228283*^9, {3.6653006401102057`*^9, 3.6653006575867515`*^9}, {
   3.665301050955241*^9, 3.6653010542312765`*^9}, {3.6653012680230923`*^9, 
   3.665301268390334*^9}, {3.6653022793879952`*^9, 3.6653022810010614`*^9}, 
   3.6653092065011587`*^9, 3.6653247739741764`*^9, {3.6653914664078445`*^9, 
   3.665391526571164*^9}, {3.665433338210455*^9, 3.6654333410946145`*^9}, {
   3.6658296215607457`*^9, 3.6658296439636726`*^9}, 3.6670365395473323`*^9, {
   3.667549815692732*^9, 3.6675498251974363`*^9}, 3.670846151338025*^9, 
   3.7339012212388983`*^9, 3.74826312492128*^9, {3.748264280802253*^9, 
   3.748264281146008*^9}, {3.7487821619637814`*^9, 3.7487821767601943`*^9}},
 ExpressionUUID -> "811cca09-e39f-418f-9481-2f476967a27c"],

Cell[TextData[{
 "As in the main paper, ",
 StyleBox["fx[m,n][z] ", "Output"],
 "refers to the nth fourier term in the nth order in \[CurlyEpsilon] for ux, \
etc, which is a function of z."
}], "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.7166711408488564`*^9, 3.7166711591302385`*^9}, {
   3.717343550334776*^9, 3.7173435553189697`*^9}, 3.717405806531722*^9, {
   3.7201699417103024`*^9, 3.7201699536755857`*^9}, {3.748192683134962*^9, 
   3.748192723039256*^9}, {3.74826292775035*^9, 3.748262987975998*^9}, 
   3.7487821767601943`*^9, {3.749403158337298*^9, 3.7494031583422832`*^9}},
 ExpressionUUID -> "4ca55bf9-035a-4d4d-9323-a1eeaafb8812"],

Cell["The constant \[Gamma] can be found via", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.7166711408488564`*^9, 3.7166711591302385`*^9}, {
   3.717343550334776*^9, 3.7173435553189697`*^9}, 3.717405806531722*^9, {
   3.7201699417103024`*^9, 3.7201699536755857`*^9}, {3.748192683134962*^9, 
   3.748192723039256*^9}, {3.74826292775035*^9, 3.748262987975998*^9}, 
   3.7487821767601943`*^9, {3.749403159563052*^9, 3.749403171215852*^9}},
 ExpressionUUID -> "4ca55bf9-035a-4d4d-9323-a1eeaafb8812"]
}, Open  ]],

Cell[BoxData[
 RowBox[{
  RowBox[{"gamma", "[", 
   RowBox[{
    RowBox[{"g_", "?", "NumericQ"}], ",", 
    RowBox[{"K_", "?", "NumericQ"}]}], "]"}], ":=", 
  RowBox[{"Chop", "[", 
   RowBox[{
    RowBox[{"Values", "[", 
     RowBox[{"FindRoot", "[", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"K", 
          RowBox[{"(", " ", 
           RowBox[{"\[Gamma]", "-", "1"}], ")"}]}], " ", "+", 
         FractionBox[
          RowBox[{"2", " ", 
           RowBox[{"(", 
            RowBox[{
             RowBox[{"-", "1"}], "+", 
             RowBox[{
              SuperscriptBox["g", "6"], " ", 
              SuperscriptBox["\[Gamma]", "2"]}]}], ")"}], " "}], 
          RowBox[{"3", " ", 
           SuperscriptBox["g", "2"], " ", 
           SuperscriptBox["\[Gamma]", 
            RowBox[{"5", "/", "3"}]]}]]}], "\[Equal]", "0"}], ",", 
       RowBox[{"{", 
        RowBox[{"\[Gamma]", ",", "0.05"}], "}"}]}], "]"}], "]"}], "[", 
    RowBox[{"[", "1", "]"}], "]"}], "]"}]}]], "Input",
 CellChangeTimes->{{3.7476588246475377`*^9, 3.74765884378458*^9}, {
   3.7476787436139*^9, 3.747678745294374*^9}, {3.7476789117912183`*^9, 
   3.7476789178220854`*^9}, {3.7476815377786083`*^9, 3.747681539138942*^9}, {
   3.749413967415736*^9, 3.749413969461957*^9}, {3.749455407638543*^9, 
   3.749455416704508*^9}, {3.7494554819977636`*^9, 3.7494554821363926`*^9}, 
   3.7494646452267823`*^9}],

Cell[CellGroupData[{

Cell[TextData[StyleBox["Surface stress", "Subsection"]], "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.748192884714484*^9, 3.748192886896722*^9}, 
   3.7487821767601943`*^9}],

Cell["\<\
We also write the Lagrange multiplier for the surface amplitude as a series \
in \
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.748192725130557*^9, 3.7481927471727858`*^9}, 
   3.7487821767601943`*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"L", "=", 
   RowBox[{"L0", "+", 
    RowBox[{"\[CurlyEpsilon]", " ", "L1"}], " ", "+", 
    RowBox[{
     SuperscriptBox["\[CurlyEpsilon]", "2"], " ", "L2"}]}]}], " ", 
  ";"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.6651762408646874`*^9, 3.665176284308905*^9}, {
   3.6652272035076323`*^9, 3.6652272055495043`*^9}, 3.6653092065011587`*^9, {
   3.665433338210455*^9, 3.6654333410946145`*^9}, {3.669689820988999*^9, 
   3.669689828501999*^9}, {3.669691223963415*^9, 3.6696912248537703`*^9}, 
   3.7058486877397184`*^9, 3.7164901243745613`*^9, {3.734516110546652*^9, 
   3.7345161144425993`*^9}, {3.748782161964778*^9, 3.7487821767611914`*^9}},
 ExpressionUUID -> "157f94d4-a345-4842-ba8b-e758cc27a061"],

Cell[TextData[StyleBox["Calculation of the energy coefficients", \
"Subsection"]], "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.748193033814044*^9, 3.7481930409641542`*^9}, 
   3.7487821767611914`*^9}],

Cell["\<\
To calculate the energy coefficients, we first expand the energy density as a \
series in \[CurlyEpsilon].\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.748261801266735*^9, 3.7482618033912163`*^9}, 
   3.7487821767611914`*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"W1", "=", 
   RowBox[{"SeriesCoefficient", "[", 
    RowBox[{"W", ",", 
     RowBox[{"{", 
      RowBox[{"\[CurlyEpsilon]", ",", "0", ",", "1"}], "}"}]}], "]"}]}], ";", 
  
  RowBox[{"W2", "=", 
   RowBox[{"SeriesCoefficient", "[", 
    RowBox[{"W", ",", 
     RowBox[{"{", 
      RowBox[{"\[CurlyEpsilon]", ",", "0", ",", "2"}], "}"}]}], "]"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"W1S2", ",", "W2S"}], "}"}], "=", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"W1", ",", "W2"}], "}"}], "/.", "SubstrateFields"}]}], 
  ";"}]}], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.6657713232344956`*^9, 3.665771343381418*^9}, {
   3.665771418266056*^9, 3.665771443269935*^9}, {3.6657714854748755`*^9, 
   3.6657715331802435`*^9}, {3.665771716847458*^9, 3.665771720446126*^9}, {
   3.6657718139064054`*^9, 3.6657718741793876`*^9}, {3.665772091294901*^9, 
   3.665772215307926*^9}, {3.665772247964039*^9, 3.665772263610958*^9}, {
   3.665772485446594*^9, 3.6657725184863443`*^9}, 3.6657729166750526`*^9, {
   3.66577426017365*^9, 3.6657742603906803`*^9}, 3.6658296584374185`*^9, {
   3.6672460521798906`*^9, 3.667246110757153*^9}, 3.667301860838393*^9, {
   3.7058179970092716`*^9, 3.705818048247526*^9}, {3.705818285589779*^9, 
   3.7058183237035446`*^9}, 3.705818449364562*^9, {3.71612068835028*^9, 
   3.7161206885199823`*^9}, 3.7161210204739447`*^9, 3.7345160841540394`*^9, 
   3.748244002237558*^9, 3.74824513592642*^9, 3.748245199162204*^9, {
   3.748260944110375*^9, 3.7482609500335903`*^9}, 3.7482611528669987`*^9, {
   3.7482612607979317`*^9, 3.748261267619701*^9}, {3.7482613798230553`*^9, 
   3.7482613948351583`*^9}, {3.7482615037390327`*^9, 3.7482615540063*^9}, {
   3.748261810434952*^9, 3.748261852402668*^9}, {3.748262405354032*^9, 
   3.748262405520245*^9}, {3.7483637331750517`*^9, 3.748363879931884*^9}, {
   3.748782161964778*^9, 3.7487821767611914`*^9}, {3.749401952955367*^9, 
   3.7494019621278367`*^9}, {3.749453055169674*^9, 3.749453060542202*^9}},
 ExpressionUUID -> "56930096-319d-4d45-b5d5-c082ff58afa2"],

Cell[TextData[StyleBox["In-plane averaging", "Subsection"]], "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.7481928957942*^9, 3.7481929037282*^9}, 
   3.7487821767611914`*^9}],

Cell["\<\
For the stripe pattern, to take in-plane averages we integrate over an \
in-plane square with verticies (0,0) and (xmay, ymas) where:\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.7166710645876665`*^9, 3.7166710770508313`*^9}, {
   3.716671134840579*^9, 3.716671135971878*^9}, {3.7198254746813087`*^9, 
   3.719825475217451*^9}, {3.7348015038859835`*^9, 3.7348015088925223`*^9}, {
   3.734801559159957*^9, 3.7348015650851173`*^9}, {3.748192754780232*^9, 
   3.748192838370331*^9}, 3.7487821767611914`*^9},
 ExpressionUUID -> "09bd2c59-97fe-406b-8035-c128314c4e86"],

Cell[BoxData[
 RowBox[{
  RowBox[{"xmax", "=", 
   RowBox[{"2", 
    RowBox[{"\[Pi]", "/", "k"}]}]}], ";", 
  RowBox[{"ymax", "=", 
   RowBox[{"2", 
    RowBox[{"\[Pi]", "/", "k"}]}]}], ";"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{
  3.719933909813597*^9, {3.734776818484407*^9, 3.73477681962022*^9}, 
   3.7487821767611914`*^9},
 ExpressionUUID -> "5855c714-aa51-498c-95ad-fb5bc38951a6"],

Cell["\<\
We first average the W coefficients over this in-plane unit cell:\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.748261748545721*^9, 3.748261754737146*^9}, {
   3.748261885770821*^9, 3.748261903569084*^9}, 3.7487821767611914`*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"W1av", "=", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"1", "/", 
      RowBox[{"(", 
       RowBox[{"xmax", " ", "ymax"}], ")"}]}], ")"}], 
    RowBox[{"Integrate", "[", 
     RowBox[{"W1", ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "0", ",", "xmax"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"y", ",", "0", ",", "ymax"}], "}"}]}], "]"}]}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"W2av", "=", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"1", "/", 
      RowBox[{"(", 
       RowBox[{"xmax", " ", "ymax"}], ")"}]}], ")"}], 
    RowBox[{"Integrate", "[", 
     RowBox[{"W2", ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "0", ",", "xmax"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"y", ",", "0", ",", "ymax"}], "}"}]}], "]"}]}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"W1Sav", ",", "W2Sav"}], "}"}], "=", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"W1av", ",", "W2av"}], "}"}], "/.", "SubstrateFields"}]}], 
  ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.6657713232344956`*^9, 3.665771343381418*^9}, {
   3.665771418266056*^9, 3.665771443269935*^9}, {3.6657714854748755`*^9, 
   3.6657715331802435`*^9}, {3.665771716847458*^9, 3.665771720446126*^9}, {
   3.6657718139064054`*^9, 3.6657718741793876`*^9}, {3.665772091294901*^9, 
   3.665772215307926*^9}, {3.665772247964039*^9, 3.665772263610958*^9}, {
   3.665772485446594*^9, 3.6657725184863443`*^9}, 3.6657729166750526`*^9, {
   3.66577426017365*^9, 3.6657742603906803`*^9}, 3.6658296584374185`*^9, {
   3.6672460521798906`*^9, 3.667246110757153*^9}, 3.667301860838393*^9, {
   3.7058179970092716`*^9, 3.705818048247526*^9}, {3.705818285589779*^9, 
   3.7058183237035446`*^9}, 3.705818449364562*^9, {3.71612068835028*^9, 
   3.7161206885199823`*^9}, 3.7161210204739447`*^9, 3.7345160841540394`*^9, 
   3.748244002237558*^9, 3.74824513592642*^9, 3.748245199162204*^9, {
   3.748260944110375*^9, 3.7482609500335903`*^9}, 3.7482611528669987`*^9, {
   3.7482612607979317`*^9, 3.748261267619701*^9}, {3.7482613798230553`*^9, 
   3.7482613948351583`*^9}, {3.7482615037390327`*^9, 3.7482615540063*^9}, {
   3.748261810434952*^9, 3.748261852402668*^9}, {3.748261923192767*^9, 
   3.748261997419652*^9}, {3.748262036893588*^9, 3.7482620405479507`*^9}, {
   3.748363304860167*^9, 3.748363305950943*^9}, {3.748363884084024*^9, 
   3.7483639296474533`*^9}, 3.74836397218202*^9, {3.748782161964778*^9, 
   3.7487821767611914`*^9}, {3.749404628753442*^9, 3.74940463319357*^9}, {
   3.749453049001571*^9, 3.749453052455569*^9}},
 ExpressionUUID -> "56930096-319d-4d45-b5d5-c082ff58afa2"],

Cell[TextData[StyleBox["Validation the solution satisfies the bulk equations \
at first order", "Subsection"]], "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.7481928957942*^9, 3.7481929037282*^9}, {
   3.7481929666078777`*^9, 3.748192979141858*^9}, 3.7487821767611914`*^9, {
   3.749402525790184*^9, 3.749402529979978*^9}}],

Cell["\<\
We use the inbuilt Mathematica \[OpenCurlyDoubleQuote]Series\
\[CloseCurlyDoubleQuote]  to expand the bulk equations to third order as a \
series in  \[CurlyEpsilon]\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.748267286562029*^9, 3.74826730455931*^9}, {
   3.748358066566401*^9, 3.74835810807306*^9}, 3.7487821767611914`*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"BulkEquationSeries", "=", 
   RowBox[{"Series", "[", 
    RowBox[{"BulkEquations", ",", 
     RowBox[{"{", 
      RowBox[{"\[CurlyEpsilon]", ",", "0", ",", "1"}], "}"}]}], "]"}]}], 
  ";"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.748244164508481*^9, 3.748244176032332*^9}, {
   3.748263945014106*^9, 3.748264022185272*^9}, 3.7482645170562563`*^9, {
   3.748265025803114*^9, 3.748265063629121*^9}, {3.7482663672032747`*^9, 
   3.7482664252577047`*^9}, {3.748266531005845*^9, 3.748266532199972*^9}, {
   3.748266616190357*^9, 3.74826663949277*^9}, 3.748266896189138*^9, {
   3.7482673152855577`*^9, 3.7482673161669197`*^9}, {3.7482678090557737`*^9, 
   3.748267809614543*^9}, {3.74826789081393*^9, 3.748267895061193*^9}, {
   3.748267927612143*^9, 3.74826794725991*^9}, {3.748268583368083*^9, 
   3.7482685998720217`*^9}, {3.748268645047811*^9, 3.7482686485923843`*^9}, {
   3.7482688459580517`*^9, 3.7482688772805*^9}, {3.7483581446122227`*^9, 
   3.748358196022498*^9}, {3.748359076251601*^9, 3.748359096792377*^9}, 
   3.7487821767611914`*^9, 3.749402543955596*^9}]
}, Open  ]],

Cell["The first solution order, as in the main text, is given by", "Text",
 CellChangeTimes->{{3.7494029577971563`*^9, 3.7494029759165387`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"fsolutionstripe", "=", 
   RowBox[{
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"fz", "[", 
         RowBox[{"1", ",", "1"}], "]"}], "\[Rule]", 
        RowBox[{"Function", "[", 
         RowBox[{"z", ",", 
          RowBox[{
           RowBox[{
            SuperscriptBox["\[ExponentialE]", 
             RowBox[{"k", " ", "G1", " ", "z"}]], " ", 
            RowBox[{"A", "[", 
             RowBox[{"1", ",", "1"}], "]"}]}], "+", 
           RowBox[{
            SuperscriptBox["\[ExponentialE]", 
             RowBox[{
              RowBox[{"-", "k"}], " ", "G1", " ", "z"}]], " ", 
            RowBox[{"A", "[", 
             RowBox[{"1", ",", "2"}], "]"}]}], "+", 
           RowBox[{
            SuperscriptBox["\[ExponentialE]", 
             RowBox[{"k", " ", "G2", " ", "z"}]], " ", 
            RowBox[{"A", "[", 
             RowBox[{"1", ",", "3"}], "]"}]}], "+", 
           RowBox[{
            SuperscriptBox["\[ExponentialE]", 
             RowBox[{
              RowBox[{"-", "k"}], " ", "G2", " ", "z"}]], " ", 
            RowBox[{"A", "[", 
             RowBox[{"1", ",", "4"}], "]"}]}]}]}], "]"}]}], ",", 
       RowBox[{
        RowBox[{"fx", "[", 
         RowBox[{"1", ",", "1"}], "]"}], "\[Rule]", 
        RowBox[{"Function", "[", 
         RowBox[{"z", ",", 
          RowBox[{
           FractionBox["1", 
            RowBox[{"G1", " ", "g2", " ", "G2"}]], 
           SuperscriptBox["\[ExponentialE]", 
            RowBox[{
             RowBox[{"-", 
              RowBox[{"(", 
               RowBox[{"G1", "+", "G2"}], ")"}]}], " ", "k", " ", "z"}]], " ", 
           RowBox[{"(", 
            RowBox[{
             RowBox[{
              RowBox[{"-", 
               SuperscriptBox["\[ExponentialE]", 
                RowBox[{
                 RowBox[{"(", 
                  RowBox[{
                   RowBox[{"2", " ", "G1"}], "+", "G2"}], ")"}], " ", "k", 
                 " ", "z"}]]}], " ", "G2", " ", 
              RowBox[{"(", 
               RowBox[{
                RowBox[{
                 SuperscriptBox["G1", "2"], " ", "g3"}], "-", 
                RowBox[{"9", " ", 
                 SuperscriptBox["\[Gamma]", "2"]}]}], ")"}], " ", 
              RowBox[{"A", "[", 
               RowBox[{"1", ",", "1"}], "]"}]}], "+", 
             RowBox[{
              SuperscriptBox["\[ExponentialE]", 
               RowBox[{"G2", " ", "k", " ", "z"}]], " ", "G2", " ", 
              RowBox[{"(", 
               RowBox[{
                RowBox[{
                 SuperscriptBox["G1", "2"], " ", "g3"}], "-", 
                RowBox[{"9", " ", 
                 SuperscriptBox["\[Gamma]", "2"]}]}], ")"}], " ", 
              RowBox[{"A", "[", 
               RowBox[{"1", ",", "2"}], "]"}]}], "-", 
             RowBox[{
              SuperscriptBox["\[ExponentialE]", 
               RowBox[{
                RowBox[{"(", 
                 RowBox[{"G1", "+", 
                  RowBox[{"2", " ", "G2"}]}], ")"}], " ", "k", " ", "z"}]], 
              " ", "G1", " ", 
              RowBox[{"(", 
               RowBox[{
                RowBox[{
                 SuperscriptBox["G2", "2"], " ", "g3"}], "-", 
                RowBox[{"9", " ", 
                 SuperscriptBox["\[Gamma]", "2"]}]}], ")"}], " ", 
              RowBox[{"A", "[", 
               RowBox[{"1", ",", "3"}], "]"}]}], "+", 
             RowBox[{
              SuperscriptBox["\[ExponentialE]", 
               RowBox[{"G1", " ", "k", " ", "z"}]], " ", "G1", " ", 
              RowBox[{"(", 
               RowBox[{
                RowBox[{
                 SuperscriptBox["G2", "2"], " ", "g3"}], "-", 
                RowBox[{"9", " ", 
                 SuperscriptBox["\[Gamma]", "2"]}]}], ")"}], " ", 
              RowBox[{"A", "[", 
               RowBox[{"1", ",", "4"}], "]"}]}]}], ")"}]}]}], "]"}]}], ",", 
       RowBox[{
        RowBox[{"fzS", "[", 
         RowBox[{"1", ",", "1"}], "]"}], "\[Rule]", 
        RowBox[{"Function", "[", 
         RowBox[{"z", ",", 
          RowBox[{
           SuperscriptBox["\[ExponentialE]", 
            RowBox[{"k", " ", "z"}]], " ", 
           RowBox[{"(", 
            RowBox[{
             RowBox[{"B", "[", 
              RowBox[{"1", ",", "3"}], "]"}], "+", 
             RowBox[{"z", " ", 
              RowBox[{"B", "[", 
               RowBox[{"1", ",", "4"}], "]"}]}]}], ")"}]}]}], "]"}]}], ",", 
       RowBox[{
        RowBox[{"fxS", "[", 
         RowBox[{"1", ",", "1"}], "]"}], "\[Rule]", 
        RowBox[{"Function", "[", 
         RowBox[{"z", ",", 
          RowBox[{"-", 
           FractionBox[
            RowBox[{
             SuperscriptBox["\[ExponentialE]", 
              RowBox[{"k", " ", "z"}]], " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{
                RowBox[{"(", 
                 RowBox[{"7", "+", 
                  RowBox[{"3", "K"}]}], ")"}], " ", 
                RowBox[{"B", "[", 
                 RowBox[{"1", ",", "4"}], "]"}]}], "+", 
               RowBox[{
                RowBox[{"(", 
                 RowBox[{"1", "+", 
                  RowBox[{"3", "K"}]}], ")"}], " ", "k", " ", 
                RowBox[{"(", 
                 RowBox[{
                  RowBox[{"B", "[", 
                   RowBox[{"1", ",", "3"}], "]"}], "+", 
                  RowBox[{"z", " ", 
                   RowBox[{"B", "[", 
                    RowBox[{"1", ",", "4"}], "]"}]}]}], ")"}]}]}], ")"}]}], 
            RowBox[{
             RowBox[{"(", 
              RowBox[{"1", "+", 
               RowBox[{"3", " ", "K"}]}], ")"}], " ", "k"}]]}]}], "]"}]}]}], 
      "}"}], "/.", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"G1", "\[Rule]", 
        SqrtBox[
         FractionBox[
          RowBox[{"g4", "+", 
           SqrtBox[
            RowBox[{
             SuperscriptBox["g4", "2"], "-", 
             RowBox[{"4", "g1", " ", "g3", " ", 
              SuperscriptBox["g", "6"]}]}]]}], 
          RowBox[{"2", "g3", " ", 
           SuperscriptBox["g", "6"]}]]]}], ",", 
       RowBox[{"G2", "\[Rule]", 
        SqrtBox[
         FractionBox[
          RowBox[{"g4", "-", 
           SqrtBox[
            RowBox[{
             SuperscriptBox["g4", "2"], "-", 
             RowBox[{"4", "g1", " ", "g3", " ", 
              SuperscriptBox["g", "6"]}]}]]}], 
          RowBox[{"2", "g3", " ", 
           SuperscriptBox["g", "6"]}]]]}]}], "}"}]}], "/.", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"g1", "\[Rule]", 
       RowBox[{
        SuperscriptBox["\[Gamma]", "2"], 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"9", "K", " ", 
           SuperscriptBox["\[Gamma]", 
            RowBox[{"8", "/", "3"}]], 
           SuperscriptBox["g", "2"]}], "+", 
          RowBox[{"5", 
           SuperscriptBox["\[Gamma]", "2"], 
           SuperscriptBox["g", "6"]}], "+", "7"}], ")"}]}]}], ",", 
      RowBox[{"g2", "\[Rule]", 
       RowBox[{"\[Gamma]", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"9", "K", " ", 
           SuperscriptBox["\[Gamma]", 
            RowBox[{"8", "/", "3"}]], 
           SuperscriptBox["g", "2"]}], "-", 
          RowBox[{
           SuperscriptBox["g", "6"], " ", 
           SuperscriptBox["\[Gamma]", "2"]}], "+", "4"}], ")"}]}]}], ",", 
      RowBox[{"g3", "->", 
       RowBox[{"(", 
        RowBox[{"10", "+", 
         RowBox[{"2", " ", 
          SuperscriptBox["g", "6"], " ", 
          SuperscriptBox["\[Gamma]", "2"]}], "+", 
         RowBox[{"9", " ", "K", " ", 
          SuperscriptBox["g", "2"], " ", 
          SuperscriptBox["\[Gamma]", 
           RowBox[{"8", "/", "3"}]]}]}], ")"}]}], ",", 
      RowBox[{"g4", "->", 
       RowBox[{"(", 
        RowBox[{"6", "+", 
         RowBox[{"17", " ", 
          SuperscriptBox["g", "6"], " ", 
          SuperscriptBox[
           RowBox[{"(", "\[Gamma]", ")"}], "2"]}], "+", 
         RowBox[{"9", " ", "K", " ", 
          SuperscriptBox["g", "2"], " ", 
          SuperscriptBox[
           RowBox[{"(", "\[Gamma]", ")"}], 
           RowBox[{"8", "/", "3"}]]}], "+", 
         RowBox[{
          SuperscriptBox["g", "12"], " ", 
          SuperscriptBox[
           RowBox[{"(", "\[Gamma]", ")"}], "4"]}], "+", 
         RowBox[{"9", "K", "\[IndentingNewLine]", " ", 
          SuperscriptBox["g", "8"], " ", 
          SuperscriptBox[
           RowBox[{"(", "\[Gamma]", ")"}], 
           RowBox[{"14", "/", "3"}]]}]}], ")"}]}]}], "}"}]}]}], 
  ";"}]], "Input",
 CellChangeTimes->{{3.7345163670083504`*^9, 3.734516432645673*^9}, {
   3.7348035388823757`*^9, 3.7348035810039244`*^9}, {3.7476573408684344`*^9, 
   3.747657363747212*^9}, {3.747657460970975*^9, 3.747657461579859*^9}, {
   3.7476584985774083`*^9, 3.7476585907686467`*^9}, {3.747658677074243*^9, 
   3.74765868485036*^9}, {3.7476590823837767`*^9, 3.747659086336222*^9}, {
   3.747659785501252*^9, 3.747659823704348*^9}, {3.7476604188708005`*^9, 
   3.7476604497122955`*^9}, {3.7476617311334243`*^9, 3.747661792813361*^9}, 
   3.7476630390595255`*^9, {3.7476689801250725`*^9, 3.747668988501643*^9}, {
   3.747669659371147*^9, 3.747669674227427*^9}, {3.747671074385416*^9, 
   3.7476710803534265`*^9}, {3.7476737161914306`*^9, 3.747673717931717*^9}, {
   3.749402941411828*^9, 3.7494029428180666`*^9}, {3.749403221707799*^9, 
   3.7494032366189165`*^9}, {3.7494033247271194`*^9, 3.749403327683213*^9}, {
   3.749403389117565*^9, 3.749403401029278*^9}, {3.7494035300627894`*^9, 
   3.7494035305671473`*^9}, {3.7494039346218*^9, 3.749403936931622*^9}, {
   3.749404373628255*^9, 3.749404427725545*^9}},
 ExpressionUUID -> "6a85fd6d-1531-4683-9aa0-f92a6398b5bb"],

Cell["\<\
Substituting the solution into the bulk equations simplifies to zero\
\>", "Text",
 CellChangeTimes->{{3.7494029824530544`*^9, 3.7494030008308983`*^9}, 
   3.749403212032677*^9}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{"BulkEquationSeries", "/.", "fsolutionstripe"}], "]"}]], "Input",
 CellChangeTimes->{{3.749402736362962*^9, 3.7494027553282356`*^9}, {
  3.749402933687427*^9, 3.74940294644038*^9}}],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "2"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 2, 2, 1],
    Editable->False], ",", "0", ",", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "2"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 2, 2, 1],
    Editable->False], ",", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "2"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 2, 2, 1],
    Editable->False], ",", "0", ",", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "2"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 2, 2, 1],
    Editable->False]}], "}"}]], "Output",
 CellChangeTimes->{3.749403212692913*^9, 3.749403282356449*^9, 
  3.74940337713314*^9, 3.7494034748225527`*^9, 3.749403585358035*^9, 
  3.7494039364130106`*^9, 3.749403992793208*^9, 3.749404236409747*^9, 
  3.7494044499391623`*^9, 3.7537828869723787`*^9, 3.7537868206777315`*^9, 
  3.7537871128230705`*^9}]
}, Open  ]],

Cell["", "Text",
 CellChangeTimes->{{3.7494027312626038`*^9, 3.7494027319776945`*^9}}],

Cell[CellGroupData[{

Cell[TextData[StyleBox["Validation the solution satisfies the boundary \
conditions", "Subsection"]], "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.748192982991539*^9, 3.748192988325735*^9}, 
   3.748782176762218*^9}],

Cell["\<\
The full solutions for the constants of integration (and L0) are obtained \
from FirstOrderConstantsStripe (from \
\[OpenCurlyDoubleQuote]compressible_pert_solutions.nb\[CloseCurlyDoubleQuote])\
.\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.7482607612792273`*^9, 3.748260875123485*^9}, {
   3.7482657965084953`*^9, 3.7482657975804243`*^9}, 3.748782176762218*^9, {
   3.74940494972581*^9, 3.74940496767879*^9}, {3.749405008802796*^9, 
   3.749405035040638*^9}, {3.7494553464947186`*^9, 3.7494553490630884`*^9}}],

Cell[" For example:", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.7482608804202003`*^9, 3.748260931153734*^9}, {
   3.748260970268648*^9, 3.748260985377118*^9}, 3.748782176763212*^9, 
   3.749405028992793*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"A", "[", 
   RowBox[{"1", ",", "1"}], "]"}], "/.", 
  "FirstOrderConstantsStripe"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.6657713232344956`*^9, 3.665771343381418*^9}, {
   3.665771418266056*^9, 3.665771443269935*^9}, {3.6657714854748755`*^9, 
   3.6657715331802435`*^9}, {3.665771716847458*^9, 3.665771720446126*^9}, {
   3.6657718139064054`*^9, 3.6657718741793876`*^9}, {3.665772091294901*^9, 
   3.665772215307926*^9}, {3.665772247964039*^9, 3.665772263610958*^9}, {
   3.665772485446594*^9, 3.6657725184863443`*^9}, 3.6657729166750526`*^9, {
   3.66577426017365*^9, 3.6657742603906803`*^9}, 3.6658296584374185`*^9, {
   3.6672460521798906`*^9, 3.667246110757153*^9}, 3.667301860838393*^9, {
   3.7058179970092716`*^9, 3.705818048247526*^9}, {3.705818285589779*^9, 
   3.7058183237035446`*^9}, 3.705818449364562*^9, {3.71612068835028*^9, 
   3.7161206885199823`*^9}, 3.7161210204739447`*^9, 3.7345160841540394`*^9, 
   3.748244002237558*^9, 3.74824513592642*^9, 3.748245199162204*^9, {
   3.748260944110375*^9, 3.7482609500335903`*^9}, {3.748782161964778*^9, 
   3.748782176763212*^9}},
 ExpressionUUID -> "56930096-319d-4d45-b5d5-c082ff58afa2"],

Cell[BoxData[
 InterpretationBox[
  TagBox[
   FrameBox[GridBox[{
      {
       ItemBox[
        TagBox[
         RowBox[{"-", 
          FractionBox[
           RowBox[{
            RowBox[{
             RowBox[{"(", 
              RowBox[{
               RowBox[{"-", 
                FractionBox[
                 RowBox[{
                  SuperscriptBox["\[ExponentialE]", 
                   RowBox[{"-", 
                    FractionBox[
                    RowBox[{"k", " ", 
                    SqrtBox[
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]]}], 
                    SqrtBox["2"]]}]], " ", 
                  RowBox[{"(", 
                   RowBox[{
                    RowBox[{
                    SuperscriptBox["\[ExponentialE]", 
                    RowBox[{"-", 
                    FractionBox[
                    RowBox[{"k", " ", 
                    SqrtBox[
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{"\" \[CenterEllipsis]\"", #, 
                    "\"\[CenterEllipsis] \""}], Background -> GrayLevel[0.75],
                     BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]]}], 
                    SqrtBox["2"]]}]], " ", "k", " ", "K", " ", 
                    RowBox[{"(", 
                    RowBox[{
                    RowBox[{"-", "1"}], "+", "\[Gamma]"}], ")"}], " ", 
                    "\[Mu]1"}], "-", 
                    FractionBox[
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]], "+", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], "+", 
                    FractionBox[
                    RowBox[{
                    SqrtBox["2"], " ", 
                    TemplateBox[{"5"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], " ", "\[Mu]1"}], 
                    RowBox[{
                    SuperscriptBox["\[Gamma]", 
                    RowBox[{"5", "/", "3"}]], " ", 
                    RowBox[{"(", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], ")"}], " ", 
                    SqrtBox[
                    FractionBox[
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{"\" \[CenterEllipsis]\"", #, 
                    "\"\[CenterEllipsis] \""}], Background -> GrayLevel[0.75],
                     BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{"\" \[CenterEllipsis]\"", #, 
                    "\"\[CenterEllipsis] \""}], Background -> GrayLevel[0.75],
                     BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]]]}]]}], ")"}]}], 
                 SqrtBox["2"]]}], "+", 
               FractionBox[
                RowBox[{
                 SuperscriptBox["\[ExponentialE]", 
                  RowBox[{"-", 
                   FractionBox[
                    RowBox[{"k", " ", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]}], 
                    SqrtBox["2"]]}]], " ", 
                 RowBox[{"(", 
                  RowBox[{
                   TemplateBox[{"4"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], "+", 
                   TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]}], ")"}]}], 
                SqrtBox["2"]]}], ")"}], " ", 
             RowBox[{"(", 
              TemplateBox[{"1"},
               "OutputSizeLimit`Skeleton",
               DisplayFunction->(FrameBox[
                 
                 RowBox[{
                  "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                 Background -> GrayLevel[0.75], 
                 BaseStyle -> {
                  "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                   ShowStringCharacters -> False}, BaselinePosition -> 
                 Baseline, ContentPadding -> False, 
                 FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                 RoundingRadius -> 7]& )], ")"}]}], "-", 
            RowBox[{
             RowBox[{"(", 
              RowBox[{
               RowBox[{
                SuperscriptBox["\[ExponentialE]", 
                 RowBox[{"-", 
                  FractionBox[
                   RowBox[{"k", " ", 
                    SqrtBox[
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]]}], 
                   SqrtBox["2"]]}]], " ", "k", " ", "K", " ", 
                RowBox[{"(", 
                 RowBox[{
                  RowBox[{"-", "1"}], "+", "\[Gamma]"}], ")"}], " ", 
                "\[Mu]1"}], "-", 
               FractionBox[
                TemplateBox[{"1"},
                 "OutputSizeLimit`Skeleton",
                 DisplayFunction->(FrameBox[
                   
                   RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                   Background -> GrayLevel[0.75], 
                   BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                   Baseline, ContentPadding -> False, 
                   FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                   RoundingRadius -> 7]& )], 
                TemplateBox[{"1"},
                 "OutputSizeLimit`Skeleton",
                 DisplayFunction->(FrameBox[
                   
                   RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                   Background -> GrayLevel[0.75], 
                   BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                   Baseline, ContentPadding -> False, 
                   FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                   RoundingRadius -> 7]& )]], "+", 
               FractionBox[
                TemplateBox[{"1"},
                 "OutputSizeLimit`Skeleton",
                 DisplayFunction->(FrameBox[
                   
                   RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                   Background -> GrayLevel[0.75], 
                   BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                   Baseline, ContentPadding -> False, 
                   FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                   RoundingRadius -> 7]& )], 
                TemplateBox[{"1"},
                 "OutputSizeLimit`Skeleton",
                 DisplayFunction->(FrameBox[
                   
                   RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                   Background -> GrayLevel[0.75], 
                   BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                   Baseline, ContentPadding -> False, 
                   FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                   RoundingRadius -> 7]& )]], "+", 
               FractionBox[
                RowBox[{
                 SqrtBox["2"], " ", 
                 TemplateBox[{"5"},
                  "OutputSizeLimit`Skeleton",
                  DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], " ", "\[Mu]1"}], 
                RowBox[{
                 SuperscriptBox["\[Gamma]", 
                  RowBox[{"5", "/", "3"}]], " ", 
                 RowBox[{"(", 
                  TemplateBox[{"1"},
                   "OutputSizeLimit`Skeleton",
                   DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], ")"}], " ", 
                 SqrtBox[
                  FractionBox[
                   TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], 
                   TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]]]}]]}], ")"}], " ", 
             TemplateBox[{"1"},
              "OutputSizeLimit`Skeleton",
              DisplayFunction->(FrameBox[
                
                RowBox[{
                 "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                Background -> GrayLevel[0.75], 
                BaseStyle -> {
                 "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                  ShowStringCharacters -> False}, BaselinePosition -> 
                Baseline, ContentPadding -> False, 
                FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                RoundingRadius -> 7]& )]}]}], 
           RowBox[{
            RowBox[{
             RowBox[{"(", 
              RowBox[{
               FractionBox[
                RowBox[{
                 SuperscriptBox["\[ExponentialE]", 
                  FractionBox[
                   RowBox[{"k", " ", 
                    SqrtBox[
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]]}], 
                   SqrtBox["2"]]], " ", 
                 RowBox[{"(", 
                  RowBox[{
                   RowBox[{
                    SuperscriptBox["\[ExponentialE]", 
                    RowBox[{"-", 
                    FractionBox[
                    RowBox[{"k", " ", 
                    SqrtBox[
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{"\" \[CenterEllipsis]\"", #, 
                    "\"\[CenterEllipsis] \""}], Background -> GrayLevel[0.75],
                     BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]]}], 
                    SqrtBox["2"]]}]], " ", "k", " ", "K", " ", 
                    RowBox[{"(", 
                    RowBox[{
                    RowBox[{"-", "1"}], "+", "\[Gamma]"}], ")"}], " ", 
                    "\[Mu]1"}], "-", 
                   FractionBox[
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]], "+", 
                   TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], "+", 
                   FractionBox[
                    RowBox[{
                    SqrtBox["2"], " ", 
                    TemplateBox[{"5"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], " ", "\[Mu]1"}], 
                    RowBox[{
                    SuperscriptBox["\[Gamma]", 
                    RowBox[{"5", "/", "3"}]], " ", 
                    RowBox[{"(", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], ")"}], " ", 
                    SqrtBox[
                    FractionBox[
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]]]}]]}], ")"}]}], 
                SqrtBox["2"]], "-", 
               FractionBox[
                RowBox[{
                 SuperscriptBox["\[ExponentialE]", 
                  RowBox[{"-", 
                   FractionBox[
                    RowBox[{"k", " ", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]}], 
                    SqrtBox["2"]]}]], " ", 
                 RowBox[{"(", 
                  TemplateBox[{"1"},
                   "OutputSizeLimit`Skeleton",
                   DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], ")"}]}], 
                SqrtBox["2"]]}], ")"}], " ", 
             RowBox[{"(", 
              TemplateBox[{"1"},
               "OutputSizeLimit`Skeleton",
               DisplayFunction->(FrameBox[
                 
                 RowBox[{
                  "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                 Background -> GrayLevel[0.75], 
                 BaseStyle -> {
                  "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                   ShowStringCharacters -> False}, BaselinePosition -> 
                 Baseline, ContentPadding -> False, 
                 FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                 RoundingRadius -> 7]& )], ")"}]}], "+", 
            RowBox[{
             RowBox[{"(", 
              RowBox[{
               RowBox[{"-", 
                FractionBox[
                 RowBox[{
                  SuperscriptBox["\[ExponentialE]", 
                   RowBox[{"-", 
                    FractionBox[
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], 
                    SqrtBox["2"]]}]], " ", 
                  RowBox[{"(", 
                   TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], ")"}]}], 
                 SqrtBox["2"]]}], "+", 
               FractionBox[
                RowBox[{
                 SuperscriptBox["\[ExponentialE]", 
                  RowBox[{"-", 
                   FractionBox[
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], 
                    SqrtBox["2"]]}]], " ", 
                 RowBox[{"(", 
                  TemplateBox[{"1"},
                   "OutputSizeLimit`Skeleton",
                   DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], ")"}]}], 
                SqrtBox["2"]]}], ")"}], " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{
                RowBox[{"(", 
                 RowBox[{
                  RowBox[{"-", 
                   FractionBox[
                    RowBox[{
                    SuperscriptBox["\[ExponentialE]", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]], " ", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]}], 
                    SqrtBox["2"]]}], "+", 
                  FractionBox[
                   TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], 
                   SqrtBox["2"]]}], ")"}], " ", 
                RowBox[{"(", 
                 RowBox[{
                  RowBox[{"-", 
                   FractionBox[
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], 
                    SqrtBox["2"]]}], "+", 
                  TemplateBox[{"1"},
                   "OutputSizeLimit`Skeleton",
                   DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]}], ")"}]}], "-", 
               TemplateBox[{"1"},
                "OutputSizeLimit`Skeleton",
                DisplayFunction->(FrameBox[
                  
                  RowBox[{
                   "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                  Background -> GrayLevel[0.75], 
                  BaseStyle -> {
                   "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                  Baseline, ContentPadding -> False, 
                  FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                  RoundingRadius -> 7]& )]}], ")"}]}]}]]}],
         Short[#, 5]& ],
        BaseStyle->{Deployed -> False},
        StripOnInput->False]},
      {GridBox[{
         {
          TagBox[
           TooltipBox[
            StyleBox[
             StyleBox[
              DynamicBox[ToBoxes[
                FEPrivate`FrontEndResource[
                "FEStrings", "sizeBriefExplanation"], StandardForm],
               ImageSizeCache->{65., {3., 9.}}],
              StripOnInput->False,
              DynamicUpdating->True], "SuggestionsBarCategory",
             StripOnInput->False],
            StyleBox[
             DynamicBox[
              ToBoxes[
               FEPrivate`FrontEndResource["FEStrings", "sizeExplanation"], 
               StandardForm]], DynamicUpdating -> True, StripOnInput -> 
             False]],
           Annotation[#, 
            Style[
             Dynamic[
              FEPrivate`FrontEndResource["FEStrings", "sizeExplanation"]], 
             DynamicUpdating -> True], "Tooltip"]& ], 
          ButtonBox[
           PaneSelectorBox[{False->
            StyleBox[
             StyleBox[
              DynamicBox[ToBoxes[
                FEPrivate`FrontEndResource["FEStrings", "sizeShowLess"], 
                StandardForm],
               ImageSizeCache->{52., {1., 9.}}],
              StripOnInput->False,
              DynamicUpdating->True], "SuggestionsBarControl",
             StripOnInput->False], True->
            StyleBox[
             StyleBox[
              DynamicBox[ToBoxes[
                FEPrivate`FrontEndResource["FEStrings", "sizeShowLess"], 
                StandardForm]],
              StripOnInput->False,
              DynamicUpdating->True], "SuggestionsBarControlActive",
             StripOnInput->False]}, Dynamic[
             CurrentValue["MouseOver"]],
            Alignment->Center,
            FrameMargins->0,
            ImageSize->{Automatic, 25}],
           Appearance->None,
           
           ButtonFunction:>OutputSizeLimit`ButtonFunction[
            OutputSizeLimit`Defer, 47, 31106859009397539697, 5/2],
           Enabled->True,
           Evaluator->Automatic,
           Method->"Queued"], 
          ButtonBox[
           PaneSelectorBox[{False->
            StyleBox[
             StyleBox[
              DynamicBox[ToBoxes[
                FEPrivate`FrontEndResource["FEStrings", "sizeShowMore"], 
                StandardForm],
               ImageSizeCache->{62., {1., 9.}}],
              StripOnInput->False,
              DynamicUpdating->True], "SuggestionsBarControl",
             StripOnInput->False], True->
            StyleBox[
             StyleBox[
              DynamicBox[ToBoxes[
                FEPrivate`FrontEndResource["FEStrings", "sizeShowMore"], 
                StandardForm]],
              StripOnInput->False,
              DynamicUpdating->True], "SuggestionsBarControlActive",
             StripOnInput->False]}, Dynamic[
             CurrentValue["MouseOver"]],
            Alignment->Center,
            FrameMargins->0,
            ImageSize->{Automatic, 25}],
           Appearance->None,
           
           ButtonFunction:>OutputSizeLimit`ButtonFunction[
            OutputSizeLimit`Defer, 47, 31106859009397539697, 5 2],
           Enabled->True,
           Evaluator->Automatic,
           Method->"Queued"], 
          ButtonBox[
           PaneSelectorBox[{False->
            StyleBox[
             StyleBox[
              DynamicBox[ToBoxes[
                FEPrivate`FrontEndResource["FEStrings", "sizeShowAll"], 
                StandardForm],
               ImageSizeCache->{44., {1., 9.}}],
              StripOnInput->False,
              DynamicUpdating->True], "SuggestionsBarControl",
             StripOnInput->False], True->
            StyleBox[
             StyleBox[
              DynamicBox[ToBoxes[
                FEPrivate`FrontEndResource["FEStrings", "sizeShowAll"], 
                StandardForm]],
              StripOnInput->False,
              DynamicUpdating->True], "SuggestionsBarControlActive",
             StripOnInput->False]}, Dynamic[
             CurrentValue["MouseOver"]],
            Alignment->Center,
            FrameMargins->0,
            ImageSize->{Automatic, 25}],
           Appearance->None,
           
           ButtonFunction:>OutputSizeLimit`ButtonFunction[
            OutputSizeLimit`Defer, 47, 31106859009397539697, Infinity],
           Enabled->True,
           Evaluator->Automatic,
           Method->"Queued"], 
          ButtonBox[
           PaneSelectorBox[{False->
            StyleBox[
             StyleBox[
              DynamicBox[ToBoxes[
                FEPrivate`FrontEndResource["FEStrings", "sizeChangeLimit"], 
                StandardForm],
               ImageSizeCache->{78., {1., 9.}}],
              StripOnInput->False,
              DynamicUpdating->True], "SuggestionsBarControl",
             StripOnInput->False], True->
            StyleBox[
             StyleBox[
              DynamicBox[ToBoxes[
                FEPrivate`FrontEndResource["FEStrings", "sizeChangeLimit"], 
                StandardForm]],
              StripOnInput->False,
              DynamicUpdating->True], "SuggestionsBarControlActive",
             StripOnInput->False]}, Dynamic[
             CurrentValue["MouseOver"]],
            Alignment->Center,
            FrameMargins->0,
            ImageSize->{Automatic, 25}],
           Appearance->None,
           ButtonFunction:>FrontEndExecute[{
              FrontEnd`SetOptions[
              FrontEnd`$FrontEnd, 
               FrontEnd`PreferencesSettings -> {"Page" -> "Evaluation"}], 
              FrontEnd`FrontEndToken["PreferencesDialog"]}],
           Evaluator->None,
           Method->"Preemptive"]}
        },
        AutoDelete->False,
        FrameStyle->GrayLevel[0.85],
        GridBoxDividers->{"Columns" -> {False, {True}}},
        GridBoxItemSize->{"Columns" -> {{Automatic}}, "Rows" -> {{Automatic}}},
        GridBoxSpacings->{"Columns" -> {{2}}}]}
     },
     DefaultBaseStyle->"Column",
     GridBoxAlignment->{
      "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
       "RowsIndexed" -> {}},
     GridBoxDividers->{
      "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {{False}}, 
       "RowsIndexed" -> {}},
     GridBoxItemSize->{
      "Columns" -> {{Automatic}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, 
       "RowsIndexed" -> {}},
     GridBoxSpacings->{"Columns" -> {
         Offset[0.27999999999999997`], {
          Offset[0.5599999999999999]}, 
         Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
         Offset[0.2], 
         Offset[1.2], {
          Offset[0.4]}, 
         Offset[0.2]}, "RowsIndexed" -> {}}],
    Background->RGBColor[0.9657, 0.9753, 0.9802],
    FrameMargins->{{12, 12}, {0, 15}},
    FrameStyle->GrayLevel[0.85],
    RoundingRadius->5,
    StripOnInput->False],
   Deploy,
   DefaultBaseStyle->"Deploy"],
  Out[47]]], "Output",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{3.748260951225204*^9, 3.748363017860961*^9, 
  3.74836452913402*^9, 3.748365064394639*^9, 3.7483663728484793`*^9, 
  3.748367993301462*^9, 3.748368573734091*^9, 3.748368934061269*^9, 
  3.748369478943746*^9, 3.748370167978312*^9, 3.7487623028065624`*^9, 
  3.748782176763212*^9, 3.74941379621758*^9, 3.753782887306598*^9, 
  3.7537868209524117`*^9, 3.7537871130989017`*^9}],

Cell["\<\
The second order constants are in terms of the first order ones, and the \
third order constants are in terms of the first and second order ones:\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.7482610014395227`*^9, 3.748261036909691*^9}, 
   3.748782176763212*^9}],

Cell["\<\
Mathematica is not able to validate that these solutions solve the boundary \
conditions in a reasonable amount of time, so we instead conduct this \
validation numerically at a high-precision numerical test point: \
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.7482610837996073`*^9, 3.7482611632713633`*^9}, 
   3.748782176763212*^9}],

Cell[BoxData[
 RowBox[{"TestPoint", "=", 
  RowBox[{"SetPrecision", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"\[Mu]1", "\[Rule]", " ", "3"}], ",", 
      RowBox[{"\[Mu]2", "\[Rule]", " ", "1"}], ",", "  ", 
      RowBox[{"g", "\[Rule]", " ", "1.4"}], ",", " ", 
      RowBox[{"k", "\[Rule]", " ", "5"}], ",", 
      RowBox[{"K", "\[Rule]", "4"}], ",", 
      RowBox[{"\[Gamma]", "\[Rule]", 
       RowBox[{"gamma", "[", 
        RowBox[{"1.4", ",", "4"}], "]"}]}], ",", " ", 
      RowBox[{"z0", "\[Rule]", "0"}], ",", 
      RowBox[{"z1", "\[Rule]", "1"}]}], "}"}], ",", "50"}], "]"}]}]], "Input",\

 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.6657713232344956`*^9, 3.665771343381418*^9}, {
   3.665771418266056*^9, 3.665771443269935*^9}, {3.6657714854748755`*^9, 
   3.6657715331802435`*^9}, {3.665771716847458*^9, 3.665771720446126*^9}, {
   3.6657718139064054`*^9, 3.6657718741793876`*^9}, {3.665772091294901*^9, 
   3.665772215307926*^9}, {3.665772247964039*^9, 3.665772263610958*^9}, {
   3.665772485446594*^9, 3.6657725184863443`*^9}, 3.6657729166750526`*^9, {
   3.66577426017365*^9, 3.6657742603906803`*^9}, 3.6658296584374185`*^9, {
   3.6672460521798906`*^9, 3.667246110757153*^9}, 3.667301860838393*^9, {
   3.7058179970092716`*^9, 3.705818048247526*^9}, {3.705818285589779*^9, 
   3.7058183237035446`*^9}, 3.705818449364562*^9, {3.71612068835028*^9, 
   3.7161206885199823`*^9}, 3.7161210204739447`*^9, 3.7345160841540394`*^9, 
   3.748244002237558*^9, 3.74824513592642*^9, 3.748245199162204*^9, {
   3.748260944110375*^9, 3.7482609500335903`*^9}, 3.7482611528669987`*^9, {
   3.748367111887246*^9, 3.7483671121796923`*^9}, {3.748782161964778*^9, 
   3.748782176763212*^9}, {3.74941381885165*^9, 3.749413853495726*^9}, {
   3.7494139386881266`*^9, 3.7494139441712112`*^9}, {3.7494139769367146`*^9, 
   3.7494139777179265`*^9}, {3.7494554341868095`*^9, 
   3.7494554369497414`*^9}, {3.7494554778336916`*^9, 3.749455478080347*^9}},
 ExpressionUUID -> "56930096-319d-4d45-b5d5-c082ff58afa2"],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"\[Mu]1", "\[Rule]", "3.`50."}], ",", 
   RowBox[{"\[Mu]2", "\[Rule]", "1.`50."}], ",", 
   RowBox[{
   "g", "\[Rule]", 
    "1.399999999999999911182158029987476766109466552734375`50."}], ",", 
   RowBox[{"k", "\[Rule]", "5.`50."}], ",", 
   RowBox[{"K", "\[Rule]", "4.`50."}], ",", 
   RowBox[{
   "\[Gamma]", "\[Rule]", 
    "0.63270045913251038438573914390872232615947723388671875`50."}], ",", 
   RowBox[{"z0", "\[Rule]", "0"}], ",", 
   RowBox[{"z1", "\[Rule]", "1.`50."}]}], "}"}]], "Output",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{
  3.74835928266391*^9, 3.7483630202645607`*^9, 3.7483640992877703`*^9, 
   3.748364529203432*^9, 3.7483650644576797`*^9, 3.748366375786837*^9, 
   3.7483670795186777`*^9, 3.748367112856585*^9, 3.748367993387788*^9, 
   3.748368573802745*^9, 3.748368934137578*^9, 3.7483694790276117`*^9, 
   3.748370168053019*^9, 3.7487623072167625`*^9, 3.748782176763212*^9, 
   3.749413875771596*^9, 3.749413983622615*^9, 3.749452241721936*^9, 
   3.7494529305522904`*^9, 3.7494535941728926`*^9, 3.749455353290141*^9, {
   3.74945542102245*^9, 3.7494554376029215`*^9}, 3.7537828873923697`*^9, 
   3.7537868210850267`*^9, 3.7537871131771927`*^9}]
}, Open  ]],

Cell["\<\
At this TestPoint, we can numerically evaluate the constants at high precision\
\>", "Text",
 CellChangeTimes->{{3.749452374876814*^9, 3.7494523764376106`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FirstOrderConstantsStripe", "/.", "TestPoint"}]], "Input",
 CellChangeTimes->{3.7494139148031096`*^9}],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
   "L0", "\[Rule]", 
    "4.7470233935091770861997200661808245602659871266344660116749`43.\
04968427163076"}], ",", 
   RowBox[{
    RowBox[{"A", "[", 
     RowBox[{"1", ",", "1"}], "]"}], "\[Rule]", 
    RowBox[{
    "-", "0.0919085546101528958257576886502213076493379589550584663128`45.\
54640156069921"}]}], ",", 
   RowBox[{
    RowBox[{"A", "[", 
     RowBox[{"1", ",", "2"}], "]"}], "\[Rule]", 
    RowBox[{
    "-", "1.5666185658230589162185065042876998552309789166775297166037`43.\
972742654926265"}]}], ",", 
   RowBox[{
    RowBox[{"A", "[", 
     RowBox[{"1", ",", "3"}], "]"}], "\[Rule]", 
    "0.4099185191002708916295407521447723821264727741652711663971`44.\
98242700579917"}], ",", 
   RowBox[{
    RowBox[{"A", "[", 
     RowBox[{"1", ",", "4"}], "]"}], "\[Rule]", 
    "2.542222702526484443308522983900971399824799269497398462837`43.\
980998032433384"}], ",", 
   RowBox[{
    RowBox[{"B", "[", 
     RowBox[{"1", ",", "3"}], "]"}], "\[Rule]", 
    "1.2936141011935435228937995431078226190709551680300814463115`44.\
13510797611144"}], ",", 
   RowBox[{
    RowBox[{"B", "[", 
     RowBox[{"1", ",", "4"}], "]"}], "\[Rule]", 
    RowBox[{
    "-", "1.5366442469385035264905562055015137884221799414032960202134`44.\
07884745964118"}]}]}], "}"}]], "Output",
 CellChangeTimes->{3.749452406384392*^9, 3.74945246967677*^9, 
  3.7494529369087534`*^9, 3.7494536009655457`*^9, 3.7537828922067614`*^9, 
  3.753786826105092*^9, 3.7537871177569294`*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Finally, we look at the boundary conditions to first order in epsilon, \
substitute in the fields and the constants and evaluate at the test point, to \
show that they are satisfied up to machine precision. The Chop command rounds \
numbers smaller than 10^-10 to zero.\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.7482613237444077`*^9, 3.748261354007496*^9}, {
   3.748261408839826*^9, 3.748261423343892*^9}, {3.748367234219556*^9, 
   3.748367248653973*^9}, 3.7487821767642107`*^9, {3.7494144280192327`*^9, 
   3.7494144322666454`*^9}, 3.749455498721511*^9}],

Cell[BoxData[
 RowBox[{"Chop", "[", 
  RowBox[{
   RowBox[{"FullSimplify", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"Series", "[", 
        RowBox[{"BoundaryConditions", ",", 
         RowBox[{"{", 
          RowBox[{"\[CurlyEpsilon]", ",", "0", ",", "1"}], "}"}]}], "]"}], "/.",
        "fsolutionstripe"}], "/.", "FirstOrderConstantsStripe"}], "/.", 
     "TestPoint"}], "]"}], ",", 
   RowBox[{"10", "^", 
    RowBox[{"-", "10"}]}]}], "]"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.6657713232344956`*^9, 3.665771343381418*^9}, {
   3.665771418266056*^9, 3.665771443269935*^9}, {3.6657714854748755`*^9, 
   3.6657715331802435`*^9}, {3.665771716847458*^9, 3.665771720446126*^9}, {
   3.6657718139064054`*^9, 3.6657718741793876`*^9}, {3.665772091294901*^9, 
   3.665772215307926*^9}, {3.665772247964039*^9, 3.665772263610958*^9}, {
   3.665772485446594*^9, 3.6657725184863443`*^9}, 3.6657729166750526`*^9, {
   3.66577426017365*^9, 3.6657742603906803`*^9}, 3.6658296584374185`*^9, {
   3.6672460521798906`*^9, 3.667246110757153*^9}, 3.667301860838393*^9, {
   3.7058179970092716`*^9, 3.705818048247526*^9}, {3.705818285589779*^9, 
   3.7058183237035446`*^9}, 3.705818449364562*^9, {3.71612068835028*^9, 
   3.7161206885199823`*^9}, 3.7161210204739447`*^9, 3.7345160841540394`*^9, 
   3.748244002237558*^9, 3.74824513592642*^9, 3.748245199162204*^9, {
   3.748260944110375*^9, 3.7482609500335903`*^9}, 3.7482611528669987`*^9, {
   3.7482612607979317`*^9, 3.748261267619701*^9}, {3.7482613798230553`*^9, 
   3.7482613948351583`*^9}, {3.7482615037390327`*^9, 3.7482615540063*^9}, 
   3.748363049102429*^9, {3.7483671689589977`*^9, 3.7483671715347033`*^9}, {
   3.748367220011492*^9, 3.7483672221377087`*^9}, {3.7483675709932613`*^9, 
   3.7483675787994757`*^9}, {3.748782161964778*^9, 3.7487821767642107`*^9}, 
   3.749414425382658*^9, {3.749451966229314*^9, 3.7494519784641676`*^9}, 
   3.7494529569485383`*^9, {3.7494554949382534`*^9, 3.7494554951078105`*^9}},
 ExpressionUUID -> "56930096-319d-4d45-b5d5-c082ff58afa2"]
}, Open  ]],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "2"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 2, 2, 1],
    Editable->False], ",", "0", ",", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "2"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 2, 2, 1],
    Editable->False], ",", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "2"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 2, 2, 1],
    Editable->False], ",", "0", ",", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "2"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 2, 2, 1],
    Editable->False], ",", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "2"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 2, 2, 1],
    Editable->False], ",", "0", ",", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "2"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 2, 2, 1],
    Editable->False], ",", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "2"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 2, 2, 1],
    Editable->False]}], "}"}]], "Output",
 CellChangeTimes->{3.749452986885996*^9, 3.7494531082453766`*^9, 
  3.7494536252680645`*^9, 3.749455384040872*^9, 3.749455466479368*^9, 
  3.7494555249879785`*^9, 3.7537829152530003`*^9, 3.753786850443565*^9, 
  3.753787140249118*^9}],

Cell["\<\
The user is invited to validate the boundary conditions at alternative \
test-points by modifying \[OpenCurlyDoubleQuote]TestPoint\
\[CloseCurlyDoubleQuote] above.\
\>", "Text",
 CellChangeTimes->{{3.7482613237444077`*^9, 3.748261354007496*^9}, {
   3.748261408839826*^9, 3.7482614927761793`*^9}, 3.7487821767642107`*^9, 
   3.753782004693363*^9}],

Cell[TextData[StyleBox["Threshold for instability", "Subsection"]], "Text",
 CellChangeTimes->{{3.74836666158285*^9, 3.748366669568527*^9}, 
   3.7487821767642107`*^9, 3.753782004693363*^9}],

Cell["\<\
Instability occurs spontaneously at the point where L0=0, which corresponds \
to the following conditions, which is too complicated to display\
\>", "Text",
 CellChangeTimes->{{3.7483666730568247`*^9, 3.748366714346525*^9}, 
   3.7487821767642107`*^9, {3.7494555793399534`*^9, 3.749455585584717*^9}, {
   3.749455624184799*^9, 3.749455624426186*^9}, 3.753782004693363*^9}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{
    RowBox[{"L0", "/.", "FirstOrderConstantsStripe"}], "/.", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"\[Mu]2", "\[Rule]", " ", "1"}], ",", " ", 
      RowBox[{"\[Mu]1", "\[Rule]", " ", "\[Eta]"}]}], "}"}]}], ")"}], 
  "\[Equal]", "0"}]], "Input",
 CellChangeTimes->{{3.6657713232344956`*^9, 3.665771343381418*^9}, {
   3.665771418266056*^9, 3.665771443269935*^9}, {3.6657714854748755`*^9, 
   3.6657715331802435`*^9}, {3.665771716847458*^9, 3.665771720446126*^9}, {
   3.6657718139064054`*^9, 3.6657718741793876`*^9}, {3.665772091294901*^9, 
   3.665772215307926*^9}, {3.665772247964039*^9, 3.665772263610958*^9}, {
   3.665772485446594*^9, 3.6657725184863443`*^9}, 3.6657729166750526`*^9, {
   3.66577426017365*^9, 3.6657742603906803`*^9}, 3.6658296584374185`*^9, {
   3.6672460521798906`*^9, 3.667246110757153*^9}, 3.667301860838393*^9, {
   3.7058179970092716`*^9, 3.705818048247526*^9}, {3.705818285589779*^9, 
   3.7058183237035446`*^9}, 3.705818449364562*^9, {3.71612068835028*^9, 
   3.7161206885199823`*^9}, 3.7161210204739447`*^9, 3.7345160841540394`*^9, 
   3.748244002237558*^9, 3.74824513592642*^9, 3.748245199162204*^9, {
   3.748260944110375*^9, 3.7482609500335903`*^9}, 3.7482611528669987`*^9, {
   3.7482612607979317`*^9, 3.748261267619701*^9}, {3.7482613798230553`*^9, 
   3.7482613948351583`*^9}, {3.7482615037390327`*^9, 3.7482615540063*^9}, 
   3.748363049102429*^9, {3.748366427727549*^9, 3.748366436312138*^9}, {
   3.748366475923089*^9, 3.7483664991077414`*^9}, {3.748366540926355*^9, 
   3.748366591864334*^9}, {3.748366706525754*^9, 3.748366706896644*^9}, {
   3.748782161964778*^9, 3.7487821767642107`*^9}, {3.7494555654419394`*^9, 
   3.7494555706062126`*^9}, {3.7494556163863616`*^9, 3.749455617351404*^9}, 
   3.753782004693363*^9},
 ExpressionUUID -> "56930096-319d-4d45-b5d5-c082ff58afa2"],

Cell[BoxData[
 InterpretationBox[
  TagBox[
   FrameBox[GridBox[{
      {
       ItemBox[
        TagBox[
         RowBox[{
          RowBox[{"-", 
           FractionBox[
            RowBox[{"\[Eta]", " ", 
             RowBox[{"(", 
              RowBox[{
               TemplateBox[{"1"},
                "OutputSizeLimit`Skeleton",
                DisplayFunction->(FrameBox[
                  
                  RowBox[{
                   "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                  Background -> GrayLevel[0.75], 
                  BaseStyle -> {
                   "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                  Baseline, ContentPadding -> False, 
                  FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                  RoundingRadius -> 7]& )], "+", 
               RowBox[{"2", " ", 
                SqrtBox["2"], " ", 
                SuperscriptBox["g", "12"], " ", "k", " ", 
                SuperscriptBox["\[Gamma]", "2"], " ", 
                RowBox[{"(", 
                 RowBox[{"4", "-", 
                  RowBox[{
                   SuperscriptBox["g", "6"], " ", 
                   SuperscriptBox["\[Gamma]", "2"]}], "+", 
                  RowBox[{"9", " ", 
                   SuperscriptBox["g", "2"], " ", "K", " ", 
                   SuperscriptBox["\[Gamma]", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]]}]}], ")"}], " ", 
                SqrtBox[
                 FractionBox[
                  RowBox[{"6", "+", 
                   TemplateBox[{"6"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]}], 
                  RowBox[{
                   SuperscriptBox["g", "6"], " ", 
                   RowBox[{"(", 
                    RowBox[{"10", "+", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], "+", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]}], ")"}]}]]], " ", 
                SqrtBox[
                 FractionBox[
                  RowBox[{"6", "+", 
                   TemplateBox[{"4"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], "+", 
                   SqrtBox[
                    RowBox[{
                    SuperscriptBox[
                    RowBox[{"(", 
                    RowBox[{
                    RowBox[{"-", "1"}], "+", 
                    RowBox[{
                    SuperscriptBox["g", "6"], " ", 
                    SuperscriptBox["\[Gamma]", "2"]}]}], ")"}], "2"], " ", 
                    RowBox[{"(", 
                    RowBox[{"36", "-", 
                    RowBox[{"4", " ", 
                    SuperscriptBox["g", "6"], " ", 
                    SuperscriptBox["\[Gamma]", "2"]}], "+", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], "+", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], "+", 
                    RowBox[{"18", " ", 
                    SuperscriptBox["g", "8"], " ", "K", " ", 
                    SuperscriptBox["\[Gamma]", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{"\" \[CenterEllipsis]\"", #, 
                    "\"\[CenterEllipsis] \""}], Background -> GrayLevel[0.75],
                     BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]]}], "+", 
                    RowBox[{"81", " ", 
                    SuperscriptBox["g", "4"], " ", 
                    SuperscriptBox["K", "2"], " ", 
                    SuperscriptBox["\[Gamma]", 
                    RowBox[{"16", "/", "3"}]]}]}], ")"}]}]]}], 
                  RowBox[{
                   SuperscriptBox["g", "6"], " ", 
                   RowBox[{"(", 
                    RowBox[{"10", "+", 
                    RowBox[{"2", " ", 
                    SuperscriptBox["g", "6"], " ", 
                    SuperscriptBox["\[Gamma]", "2"]}], "+", 
                    RowBox[{"9", " ", 
                    SuperscriptBox["g", "2"], " ", "K", " ", 
                    SuperscriptBox["\[Gamma]", 
                    RowBox[{"8", "/", "3"}]]}]}], ")"}]}]]], " ", 
                RowBox[{"(", 
                 TemplateBox[{"1"},
                  "OutputSizeLimit`Skeleton",
                  DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], ")"}]}], "+", 
               TemplateBox[{"3"},
                "OutputSizeLimit`Skeleton",
                DisplayFunction->(FrameBox[
                  
                  RowBox[{
                   "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                  Background -> GrayLevel[0.75], 
                  BaseStyle -> {
                   "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                  Baseline, ContentPadding -> False, 
                  FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                  RoundingRadius -> 7]& )], "+", 
               TemplateBox[{"1"},
                "OutputSizeLimit`Skeleton",
                DisplayFunction->(FrameBox[
                  
                  RowBox[{
                   "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                  Background -> GrayLevel[0.75], 
                  BaseStyle -> {
                   "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                  Baseline, ContentPadding -> False, 
                  FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                  RoundingRadius -> 7]& )], "-", 
               RowBox[{"18", " ", 
                SqrtBox["2"], " ", 
                SuperscriptBox["\[ExponentialE]", 
                 RowBox[{"-", 
                  FractionBox[
                   RowBox[{"k", " ", 
                    RowBox[{"(", 
                    RowBox[{
                    SqrtBox[
                    FractionBox[
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{"\" \[CenterEllipsis]\"", #, 
                    "\"\[CenterEllipsis] \""}], Background -> GrayLevel[0.75],
                     BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], 
                    RowBox[{
                    SuperscriptBox["g", "6"], " ", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{"\" \[CenterEllipsis]\"", #, 
                    "\"\[CenterEllipsis] \""}], Background -> GrayLevel[0.75],
                     BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]}]]], "+", 
                    SqrtBox[
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]]}], ")"}]}], 
                   SqrtBox["2"]]}]], " ", 
                SuperscriptBox["g", "2"], " ", "k", " ", "K", " ", 
                SuperscriptBox["\[Gamma]", 
                 RowBox[{"8", "/", "3"}]], " ", 
                RowBox[{"(", 
                 TemplateBox[{"1"},
                  "OutputSizeLimit`Skeleton",
                  DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], ")"}]}]}], ")"}]}], 
            RowBox[{"36", " ", 
             SuperscriptBox["g", "8"], " ", 
             TemplateBox[{"3"},
              "OutputSizeLimit`Skeleton",
              DisplayFunction->(FrameBox[
                
                RowBox[{
                 "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                Background -> GrayLevel[0.75], 
                BaseStyle -> {
                 "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                  ShowStringCharacters -> False}, BaselinePosition -> 
                Baseline, ContentPadding -> False, 
                FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                RoundingRadius -> 7]& )], " ", 
             SqrtBox[
              TemplateBox[{"1"},
               "OutputSizeLimit`Skeleton",
               DisplayFunction->(FrameBox[
                 
                 RowBox[{
                  "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                 Background -> GrayLevel[0.75], 
                 BaseStyle -> {
                  "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                   ShowStringCharacters -> False}, BaselinePosition -> 
                 Baseline, ContentPadding -> False, 
                 FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                 RoundingRadius -> 7]& )]], " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"-", 
                FractionBox[
                 RowBox[{
                  SuperscriptBox["\[ExponentialE]", 
                   RowBox[{"k", " ", 
                    SqrtBox[
                    FractionBox[
                    RowBox[{"6", "+", 
                    TemplateBox[{"4"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], "+", 
                    SqrtBox[
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{"\" \[CenterEllipsis]\"", #, 
                    "\"\[CenterEllipsis] \""}], Background -> GrayLevel[0.75],
                     BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]]}], 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]]]}]], " ", 
                  RowBox[{"(", 
                   RowBox[{
                    RowBox[{
                    RowBox[{"(", 
                    RowBox[{
                    RowBox[{"-", 
                    FractionBox[
                    RowBox[{
                    SuperscriptBox["\[ExponentialE]", 
                    RowBox[{
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{"\" \[CenterEllipsis]\"", #, 
                    "\"\[CenterEllipsis] \""}], Background -> GrayLevel[0.75],
                     BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{"\" \[CenterEllipsis]\"", #, 
                    "\"\[CenterEllipsis] \""}], Background -> GrayLevel[0.75],
                     BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]}]], " ", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{"\" \[CenterEllipsis]\"", #, 
                    "\"\[CenterEllipsis] \""}], Background -> GrayLevel[0.75],
                     BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]}], 
                    SqrtBox["2"]]}], "+", 
                    FractionBox[
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{"\" \[CenterEllipsis]\"", #, 
                    "\"\[CenterEllipsis] \""}], Background -> GrayLevel[0.75],
                     BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], 
                    SqrtBox["2"]]}], ")"}], " ", 
                    RowBox[{"(", 
                    RowBox[{
                    RowBox[{
                    RowBox[{"(", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{"\" \[CenterEllipsis]\"", #, 
                    "\"\[CenterEllipsis] \""}], Background -> GrayLevel[0.75],
                     BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], ")"}], " ", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{"\" \[CenterEllipsis]\"", #, 
                    "\"\[CenterEllipsis] \""}], Background -> GrayLevel[0.75],
                     BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]}], "+", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]}], ")"}]}], "-", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]}], ")"}]}], 
                 RowBox[{
                  RowBox[{
                   RowBox[{"(", 
                    RowBox[{
                    FractionBox[
                    RowBox[{
                    SuperscriptBox["\[ExponentialE]", 
                    FractionBox[
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{"\" \[CenterEllipsis]\"", #, 
                    "\"\[CenterEllipsis] \""}], Background -> GrayLevel[0.75],
                     BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], 
                    SqrtBox["2"]]], " ", 
                    RowBox[{"(", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{"\" \[CenterEllipsis]\"", #, 
                    "\"\[CenterEllipsis] \""}], Background -> GrayLevel[0.75],
                     BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], ")"}]}], 
                    SqrtBox["2"]], "-", 
                    FractionBox[
                    RowBox[{
                    SuperscriptBox["\[ExponentialE]", 
                    RowBox[{"-", 
                    FractionBox[
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{"\" \[CenterEllipsis]\"", #, 
                    "\"\[CenterEllipsis] \""}], Background -> GrayLevel[0.75],
                     BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], 
                    SqrtBox["2"]]}]], " ", 
                    RowBox[{"(", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{"\" \[CenterEllipsis]\"", #, 
                    "\"\[CenterEllipsis] \""}], Background -> GrayLevel[0.75],
                     BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], ")"}]}], 
                    SqrtBox["2"]]}], ")"}], " ", 
                   RowBox[{"(", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], ")"}]}], "+", 
                  RowBox[{
                   RowBox[{"(", 
                    RowBox[{
                    RowBox[{"-", 
                    FractionBox[
                    RowBox[{
                    SuperscriptBox["\[ExponentialE]", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{"\" \[CenterEllipsis]\"", #, 
                    "\"\[CenterEllipsis] \""}], Background -> GrayLevel[0.75],
                     BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]], " ", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{"\" \[CenterEllipsis]\"", #, 
                    "\"\[CenterEllipsis] \""}], Background -> GrayLevel[0.75],
                     BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]}], 
                    SqrtBox["2"]]}], "+", 
                    FractionBox[
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], 
                    SqrtBox["2"]]}], ")"}], " ", 
                   RowBox[{"(", 
                    RowBox[{
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], "+", 
                    TemplateBox[{"1"},
                    "OutputSizeLimit`Skeleton",
                    DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]}], ")"}]}]}]]}], "+", 
               TemplateBox[{"2"},
                "OutputSizeLimit`Skeleton",
                DisplayFunction->(FrameBox[
                  
                  RowBox[{
                   "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                  Background -> GrayLevel[0.75], 
                  BaseStyle -> {
                   "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                  Baseline, ContentPadding -> False, 
                  FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                  RoundingRadius -> 7]& )], "+", 
               RowBox[{
                SuperscriptBox["\[ExponentialE]", 
                 TemplateBox[{"1"},
                  "OutputSizeLimit`Skeleton",
                  DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )]], " ", 
                RowBox[{"(", 
                 TemplateBox[{"1"},
                  "OutputSizeLimit`Skeleton",
                  DisplayFunction->(FrameBox[
                    
                    RowBox[{
                    "\" \[CenterEllipsis]\"", #, "\"\[CenterEllipsis] \""}], 
                    Background -> GrayLevel[0.75], 
                    BaseStyle -> {
                    "Deploy", FontColor -> GrayLevel[1], FontSize -> Smaller, 
                    ShowStringCharacters -> False}, BaselinePosition -> 
                    Baseline, ContentPadding -> False, 
                    FrameMargins -> {{1, 1}, {1, 2}}, FrameStyle -> None, 
                    RoundingRadius -> 7]& )], ")"}]}]}], ")"}]}]]}], 
          "\[Equal]", "0"}],
         Short[#, 5]& ],
        BaseStyle->{Deployed -> False},
        StripOnInput->False]},
      {GridBox[{
         {
          TagBox[
           TooltipBox[
            StyleBox[
             StyleBox[
              DynamicBox[ToBoxes[
                FEPrivate`FrontEndResource[
                "FEStrings", "sizeBriefExplanation"], StandardForm],
               ImageSizeCache->{65., {3., 9.}}],
              StripOnInput->False,
              DynamicUpdating->True], "SuggestionsBarCategory",
             StripOnInput->False],
            StyleBox[
             DynamicBox[
              ToBoxes[
               FEPrivate`FrontEndResource["FEStrings", "sizeExplanation"], 
               StandardForm]], DynamicUpdating -> True, StripOnInput -> 
             False]],
           Annotation[#, 
            Style[
             Dynamic[
              FEPrivate`FrontEndResource["FEStrings", "sizeExplanation"]], 
             DynamicUpdating -> True], "Tooltip"]& ], 
          ButtonBox[
           PaneSelectorBox[{False->
            StyleBox[
             StyleBox[
              DynamicBox[ToBoxes[
                FEPrivate`FrontEndResource["FEStrings", "sizeShowLess"], 
                StandardForm],
               ImageSizeCache->{52., {1., 9.}}],
              StripOnInput->False,
              DynamicUpdating->True], "SuggestionsBarControl",
             StripOnInput->False], True->
            StyleBox[
             StyleBox[
              DynamicBox[ToBoxes[
                FEPrivate`FrontEndResource["FEStrings", "sizeShowLess"], 
                StandardForm]],
              StripOnInput->False,
              DynamicUpdating->True], "SuggestionsBarControlActive",
             StripOnInput->False]}, Dynamic[
             CurrentValue["MouseOver"]],
            Alignment->Center,
            FrameMargins->0,
            ImageSize->{Automatic, 25}],
           Appearance->None,
           
           ButtonFunction:>OutputSizeLimit`ButtonFunction[
            OutputSizeLimit`Defer, 51, 31106859009397539697, 5/2],
           Enabled->True,
           Evaluator->Automatic,
           Method->"Queued"], 
          ButtonBox[
           PaneSelectorBox[{False->
            StyleBox[
             StyleBox[
              DynamicBox[ToBoxes[
                FEPrivate`FrontEndResource["FEStrings", "sizeShowMore"], 
                StandardForm],
               ImageSizeCache->{62., {1., 9.}}],
              StripOnInput->False,
              DynamicUpdating->True], "SuggestionsBarControl",
             StripOnInput->False], True->
            StyleBox[
             StyleBox[
              DynamicBox[ToBoxes[
                FEPrivate`FrontEndResource["FEStrings", "sizeShowMore"], 
                StandardForm]],
              StripOnInput->False,
              DynamicUpdating->True], "SuggestionsBarControlActive",
             StripOnInput->False]}, Dynamic[
             CurrentValue["MouseOver"]],
            Alignment->Center,
            FrameMargins->0,
            ImageSize->{Automatic, 25}],
           Appearance->None,
           
           ButtonFunction:>OutputSizeLimit`ButtonFunction[
            OutputSizeLimit`Defer, 51, 31106859009397539697, 5 2],
           Enabled->True,
           Evaluator->Automatic,
           Method->"Queued"], 
          ButtonBox[
           PaneSelectorBox[{False->
            StyleBox[
             StyleBox[
              DynamicBox[ToBoxes[
                FEPrivate`FrontEndResource["FEStrings", "sizeShowAll"], 
                StandardForm],
               ImageSizeCache->{44., {1., 9.}}],
              StripOnInput->False,
              DynamicUpdating->True], "SuggestionsBarControl",
             StripOnInput->False], True->
            StyleBox[
             StyleBox[
              DynamicBox[ToBoxes[
                FEPrivate`FrontEndResource["FEStrings", "sizeShowAll"], 
                StandardForm]],
              StripOnInput->False,
              DynamicUpdating->True], "SuggestionsBarControlActive",
             StripOnInput->False]}, Dynamic[
             CurrentValue["MouseOver"]],
            Alignment->Center,
            FrameMargins->0,
            ImageSize->{Automatic, 25}],
           Appearance->None,
           
           ButtonFunction:>OutputSizeLimit`ButtonFunction[
            OutputSizeLimit`Defer, 51, 31106859009397539697, Infinity],
           Enabled->True,
           Evaluator->Automatic,
           Method->"Queued"], 
          ButtonBox[
           PaneSelectorBox[{False->
            StyleBox[
             StyleBox[
              DynamicBox[ToBoxes[
                FEPrivate`FrontEndResource["FEStrings", "sizeChangeLimit"], 
                StandardForm],
               ImageSizeCache->{78., {1., 9.}}],
              StripOnInput->False,
              DynamicUpdating->True], "SuggestionsBarControl",
             StripOnInput->False], True->
            StyleBox[
             StyleBox[
              DynamicBox[ToBoxes[
                FEPrivate`FrontEndResource["FEStrings", "sizeChangeLimit"], 
                StandardForm]],
              StripOnInput->False,
              DynamicUpdating->True], "SuggestionsBarControlActive",
             StripOnInput->False]}, Dynamic[
             CurrentValue["MouseOver"]],
            Alignment->Center,
            FrameMargins->0,
            ImageSize->{Automatic, 25}],
           Appearance->None,
           ButtonFunction:>FrontEndExecute[{
              FrontEnd`SetOptions[
              FrontEnd`$FrontEnd, 
               FrontEnd`PreferencesSettings -> {"Page" -> "Evaluation"}], 
              FrontEnd`FrontEndToken["PreferencesDialog"]}],
           Evaluator->None,
           Method->"Preemptive"]}
        },
        AutoDelete->False,
        FrameStyle->GrayLevel[0.85],
        GridBoxDividers->{"Columns" -> {False, {True}}},
        GridBoxItemSize->{"Columns" -> {{Automatic}}, "Rows" -> {{Automatic}}},
        GridBoxSpacings->{"Columns" -> {{2}}}]}
     },
     DefaultBaseStyle->"Column",
     GridBoxAlignment->{
      "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
       "RowsIndexed" -> {}},
     GridBoxDividers->{
      "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {{False}}, 
       "RowsIndexed" -> {}},
     GridBoxItemSize->{
      "Columns" -> {{Automatic}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, 
       "RowsIndexed" -> {}},
     GridBoxSpacings->{"Columns" -> {
         Offset[0.27999999999999997`], {
          Offset[0.5599999999999999]}, 
         Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
         Offset[0.2], 
         Offset[1.2], {
          Offset[0.4]}, 
         Offset[0.2]}, "RowsIndexed" -> {}}],
    Background->RGBColor[0.9657, 0.9753, 0.9802],
    FrameMargins->{{12, 12}, {0, 15}},
    FrameStyle->GrayLevel[0.85],
    RoundingRadius->5,
    StripOnInput->False],
   Deploy,
   DefaultBaseStyle->"Deploy"],
  Out[51]]], "Output",
 CellChangeTimes->{{3.749455599093499*^9, 3.7494556266421537`*^9}, 
   3.7537829228406763`*^9, 3.753786858384491*^9, 3.7537871479558945`*^9}]
}, Open  ]],

Cell["\<\
This equation does not have an analytic solution, but can be solved \
numerically for g* at any value of k and \[Eta]. We have written a helper \
function CharacterizeInstabilityCompressible[\[Eta],K] which finds \
numerically (but at high precision) the k* and g* for the first point of \
instability at a given value of \[Eta]. Uncomment this following line and \
evaluate.This might take a while (30 minutes).\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.7483666730568247`*^9, 3.748366848273715*^9}, {
   3.748366907912177*^9, 3.748366914683483*^9}, 3.7487821767642107`*^9, 
   3.7494556565278883`*^9, {3.749457969304774*^9, 3.7494579700088906`*^9}, {
   3.7494593914073153`*^9, 3.7494594179732103`*^9}, 3.753782812272453*^9, {
   3.753787013702876*^9, 3.75378703270002*^9}}],

Cell[BoxData[
 RowBox[{"(*", 
  RowBox[{"Threshold3", "=", " ", 
   RowBox[{
    RowBox[{"CharacterizeInstabilityCompressible", "[", 
     RowBox[{"3", ",", "3"}], "]"}], "//", "Quiet"}]}], "*)"}]], "Input",
 CellChangeTimes->{{3.749455937057074*^9, 3.7494559376276827`*^9}, {
  3.749456074398494*^9, 3.7494560797899785`*^9}, {3.7494645411729746`*^9, 
  3.749464547757373*^9}, {3.7537824820422497`*^9, 3.753782488232666*^9}, {
  3.7537866375664477`*^9, 3.7537866399915543`*^9}, {3.75378700470078*^9, 
  3.7537870076015644`*^9}}],

Cell["Or take this value if you do not want to wait", "Text",
 CellChangeTimes->{{3.7537869826689997`*^9, 3.7537869950507407`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Threshold3", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"g", "\[Rule]", "1.2094508496552547`"}], ",", 
     RowBox[{"k", "\[Rule]", "1.3369970510001397`"}], ",", 
     RowBox[{"\[Mu]2", "\[Rule]", "1"}], ",", 
     RowBox[{"\[Mu]1", "\[Rule]", "3"}], ",", 
     RowBox[{"K", "\[Rule]", "3"}], ",", 
     RowBox[{"\[Gamma]", "->", "0.7872741904711991`"}]}], "}"}]}], 
  ";"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.6657713232344956`*^9, 3.665771343381418*^9}, {
   3.665771418266056*^9, 3.665771443269935*^9}, {3.6657714854748755`*^9, 
   3.6657715331802435`*^9}, {3.665771716847458*^9, 3.665771720446126*^9}, {
   3.6657718139064054`*^9, 3.6657718741793876`*^9}, {3.665772091294901*^9, 
   3.665772215307926*^9}, {3.665772247964039*^9, 3.665772263610958*^9}, {
   3.665772485446594*^9, 3.6657725184863443`*^9}, 3.6657729166750526`*^9, {
   3.66577426017365*^9, 3.6657742603906803`*^9}, 3.6658296584374185`*^9, {
   3.6672460521798906`*^9, 3.667246110757153*^9}, 3.667301860838393*^9, {
   3.7058179970092716`*^9, 3.705818048247526*^9}, {3.705818285589779*^9, 
   3.7058183237035446`*^9}, 3.705818449364562*^9, {3.71612068835028*^9, 
   3.7161206885199823`*^9}, 3.7161210204739447`*^9, 3.7345160841540394`*^9, 
   3.748244002237558*^9, 3.74824513592642*^9, 3.748245199162204*^9, {
   3.748260944110375*^9, 3.7482609500335903`*^9}, 3.7482611528669987`*^9, {
   3.7482612607979317`*^9, 3.748261267619701*^9}, {3.7482613798230553`*^9, 
   3.7482613948351583`*^9}, {3.7482615037390327`*^9, 3.7482615540063*^9}, 
   3.748363049102429*^9, {3.748366427727549*^9, 3.748366436312138*^9}, {
   3.748366475923089*^9, 3.7483664991077414`*^9}, {3.748366540926355*^9, 
   3.748366591864334*^9}, {3.748366706525754*^9, 3.748366706896644*^9}, {
   3.748366863463852*^9, 3.7483668668954573`*^9}, {3.748782161964778*^9, 
   3.7487821767642107`*^9}, {3.749455643776758*^9, 3.7494556452551403`*^9}, {
   3.749456119214484*^9, 3.7494561234171705`*^9}, {3.7494579670617733`*^9, 
   3.7494579674327803`*^9}, {3.749464553414239*^9, 3.7494646032698636`*^9}, 
   3.749464687194419*^9},
 ExpressionUUID -> "56930096-319d-4d45-b5d5-c082ff58afa2"],

Cell["\<\
We can validate that this point does indeed correspond to L0=0.\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.7483666730568247`*^9, 3.748366848273715*^9}, {
   3.748366907912177*^9, 3.748366939643381*^9}, 3.748782176766178*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"L0", "/.", "FirstOrderConstantsStripe"}], "/.", 
  "Threshold3"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.6657713232344956`*^9, 3.665771343381418*^9}, {
   3.665771418266056*^9, 3.665771443269935*^9}, {3.6657714854748755`*^9, 
   3.6657715331802435`*^9}, {3.665771716847458*^9, 3.665771720446126*^9}, {
   3.6657718139064054`*^9, 3.6657718741793876`*^9}, {3.665772091294901*^9, 
   3.665772215307926*^9}, {3.665772247964039*^9, 3.665772263610958*^9}, {
   3.665772485446594*^9, 3.6657725184863443`*^9}, 3.6657729166750526`*^9, {
   3.66577426017365*^9, 3.6657742603906803`*^9}, 3.6658296584374185`*^9, {
   3.6672460521798906`*^9, 3.667246110757153*^9}, 3.667301860838393*^9, {
   3.7058179970092716`*^9, 3.705818048247526*^9}, {3.705818285589779*^9, 
   3.7058183237035446`*^9}, 3.705818449364562*^9, {3.71612068835028*^9, 
   3.7161206885199823`*^9}, 3.7161210204739447`*^9, 3.7345160841540394`*^9, 
   3.748244002237558*^9, 3.74824513592642*^9, 3.748245199162204*^9, {
   3.748260944110375*^9, 3.7482609500335903`*^9}, 3.7482611528669987`*^9, {
   3.7482612607979317`*^9, 3.748261267619701*^9}, {3.7482613798230553`*^9, 
   3.7482613948351583`*^9}, {3.7482615037390327`*^9, 3.7482615540063*^9}, 
   3.748363049102429*^9, {3.748366427727549*^9, 3.748366436312138*^9}, {
   3.748366475923089*^9, 3.7483664991077414`*^9}, {3.748366540926355*^9, 
   3.748366591864334*^9}, {3.748366706525754*^9, 3.748366706896644*^9}, {
   3.748366880429227*^9, 3.748366892535075*^9}, {3.748782161964778*^9, 
   3.748782176766178*^9}, 3.749456130193714*^9, {3.7494647236191454`*^9, 
   3.74946475795133*^9}},
 ExpressionUUID -> "56930096-319d-4d45-b5d5-c082ff58afa2"],

Cell[BoxData[
 RowBox[{"-", "4.292836632443186`*^-14"}]], "Output",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{3.7483668935893517`*^9, 3.748367745014412*^9, 
  3.74836810758436*^9, 3.748368724399867*^9, 3.7483690949662523`*^9, 
  3.748369632630701*^9, 3.748370325296257*^9, 3.7487625343553905`*^9, 
  3.748782176766178*^9, 3.7494645683482685`*^9, 3.7494646111867156`*^9, 
  3.749464692233024*^9, 3.7494647327237883`*^9, 3.7537829248323164`*^9, 
  3.753787149947946*^9}],

Cell["\<\
As the solution is complicated, integrating the energy to find E2 can take a \
very long time. Instead, we will do the integration numerically. Here, we \
show it for \[Eta] =3 and K =3\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.},
 CellChangeTimes->{{3.7483666730568247`*^9, 3.748366848273715*^9}, {
   3.748366907912177*^9, 3.748366939643381*^9}, {3.7483682698590727`*^9, 
   3.748368290906556*^9}, 3.7487821767671757`*^9, {3.7494577056216955`*^9, 
   3.7494577723681345`*^9}, {3.749457952390045*^9, 3.749457978055368*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"E2", "=", 
  RowBox[{
   RowBox[{"NIntegrate", "[", 
    RowBox[{
     RowBox[{"ReleaseHold", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"W2av", "/.", "fsolutionstripe"}], "/.", 
        "FirstOrderConstantsStripe"}], "/.", "Threshold3"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"z", ",", "0", ",", "1"}], "}"}]}], "]"}], "+", 
   RowBox[{"NIntegrate", "[", 
    RowBox[{
     RowBox[{"ReleaseHold", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"W2Sav", "/.", "fsolutionstripe"}], "/.", 
        "FirstOrderConstantsStripe"}], "/.", "Threshold3"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"z", ",", 
       RowBox[{"-", "\[Infinity]"}], ",", "0"}], "}"}]}], "]"}]}]}]], "Input",\

 CellChangeTimes->{{3.7480176039539375`*^9, 3.748017607542214*^9}, 
   3.7480177471340256`*^9, {3.748017878744255*^9, 3.7480178981702194`*^9}, {
   3.7480180082642827`*^9, 3.74801801666995*^9}, {3.7480181489018383`*^9, 
   3.748018177887064*^9}, {3.7480182200852275`*^9, 3.748018250326195*^9}, {
   3.7480183019973974`*^9, 3.748018303934718*^9}, {3.748018418805282*^9, 
   3.7480184213492627`*^9}, {3.7480184961732473`*^9, 3.748018510077756*^9}, {
   3.7480185746538258`*^9, 3.748018603053455*^9}, {3.748018689189012*^9, 
   3.748018719998295*^9}, {3.748018751892922*^9, 3.7480188195489163`*^9}, {
   3.7480189311413097`*^9, 3.748018953893408*^9}, {3.748019073188226*^9, 
   3.748019075515997*^9}, {3.7494578548598795`*^9, 3.7494579443634844`*^9}, {
   3.749457987552996*^9, 3.7494579898398757`*^9}, {3.749460351496496*^9, 
   3.7494603518286333`*^9}, {3.7494647342098303`*^9, 3.749464762436333*^9}}],

Cell[BoxData["1.056932319443149`*^-13"], "Output",
 CellChangeTimes->{3.753782933063329*^9, 3.7537871583908424`*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[StyleBox["Hexagonal pattern", "Section"]], "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{3.748782196939209*^9}],

Cell["Displacement fields", "Subsubsection",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.734858096251944*^9, 3.734858108803898*^9}, 
   3.748782196940205*^9},
 TextAlignment->Left,
 ExpressionUUID -> "1b3bfbbc-60b3-4b40-9a2f-a856d7813b9c"],

Cell["\<\
As discussed in the main paper, the form of u(x,y,z) for the hexagonal \
pattern in the thin slab is expected to be of the form:\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.7161206305567656`*^9, 3.716120654414336*^9}, {
   3.719825444630498*^9, 3.7198254490843143`*^9}, {3.7343365435634594`*^9, 
   3.7343365530569415`*^9}, {3.7481926611831303`*^9, 
   3.7481926774792223`*^9}, {3.7482452333005257`*^9, 3.748245235635438*^9}, 
   3.748782196940205*^9},
 ExpressionUUID -> "f5a52526-2b6d-407b-9f04-21d090a88149"],

Cell[BoxData[
 RowBox[{
  RowBox[{"uz", "[", 
   RowBox[{"x_", ",", "y_", ",", "z_"}], "]"}], ":=", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{"\[Gamma]", "-", "1"}], ")"}], "z"}], "+", 
    RowBox[{"\[CurlyEpsilon]", " ", 
     FractionBox[
      RowBox[{"(", 
       RowBox[{
        RowBox[{"Cos", "[", 
         RowBox[{"k", " ", "x"}], "]"}], "+", 
        RowBox[{"Cos", "[", 
         RowBox[{"k", " ", "x1"}], "]"}], "+", 
        RowBox[{"Cos", "[", 
         RowBox[{"k", " ", "x2"}], "]"}]}], ")"}], 
      SqrtBox["3"]], " ", 
     RowBox[{
      RowBox[{"fz", "[", 
       RowBox[{"1", ",", "1"}], "]"}], "[", "z", "]"}]}], "+", 
    RowBox[{
     SuperscriptBox["\[CurlyEpsilon]", "2"], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{
        RowBox[{"fz", "[", 
         RowBox[{"2", ",", "0"}], "]"}], "[", "z", "]"}], "+", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{"Cos", "[", 
           RowBox[{"2", " ", "k", " ", "x"}], "]"}], "+", 
          RowBox[{"Cos", "[", 
           RowBox[{"2", " ", "k", " ", "x1"}], "]"}], "+", 
          RowBox[{"Cos", "[", 
           RowBox[{"2", " ", "k", " ", "x2"}], "]"}]}], ")"}], " ", 
        RowBox[{
         RowBox[{"fz", "[", 
          RowBox[{"2", ",", "1"}], "]"}], "[", "z", "]"}]}], "+", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{"Cos", "[", 
           RowBox[{"k", " ", "x"}], "]"}], "+", 
          RowBox[{"Cos", "[", 
           RowBox[{"k", " ", "x1"}], "]"}], "+", 
          RowBox[{"Cos", "[", 
           RowBox[{"k", " ", "x2"}], "]"}]}], ")"}], " ", 
        RowBox[{
         RowBox[{"fz", "[", 
          RowBox[{"2", ",", "2"}], "]"}], "[", "z", "]"}]}], "+", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{"Cos", "[", 
           RowBox[{
            RowBox[{"k", " ", "x"}], "+", 
            RowBox[{"k", " ", "x1"}]}], "]"}], "+", 
          RowBox[{"Cos", "[", 
           RowBox[{
            RowBox[{"k", " ", "x"}], "-", 
            RowBox[{"k", " ", "x2"}]}], "]"}], "+", 
          RowBox[{"Cos", "[", 
           RowBox[{
            RowBox[{"k", " ", "x1"}], "+", 
            RowBox[{"k", " ", "x2"}]}], "]"}]}], ")"}], " ", 
        RowBox[{
         RowBox[{"fz", "[", 
          RowBox[{"2", ",", "3"}], "]"}], "[", "z", "]"}]}]}], ")"}]}]}], "/.", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x1", "\[Rule]", " ", 
      RowBox[{
       FractionBox["x", "2"], "+", 
       RowBox[{
        FractionBox[
         SqrtBox["3"], "2"], "y"}]}]}], ",", 
     RowBox[{"x2", "\[Rule]", " ", 
      RowBox[{
       RowBox[{"-", 
        FractionBox["x", "2"]}], "+", 
       RowBox[{
        FractionBox[
         SqrtBox["3"], "2"], "y"}]}]}]}], "}"}]}]}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.6657713232344956`*^9, 3.665771343381418*^9}, {
   3.665771418266056*^9, 3.665771443269935*^9}, {3.6657714854748755`*^9, 
   3.6657715331802435`*^9}, {3.665771716847458*^9, 3.665771720446126*^9}, {
   3.6657718139064054`*^9, 3.6657718741793876`*^9}, {3.665772091294901*^9, 
   3.665772215307926*^9}, {3.665772247964039*^9, 3.665772263610958*^9}, {
   3.665772485446594*^9, 3.6657725184863443`*^9}, 3.6657729166750526`*^9, {
   3.66577426017365*^9, 3.6657742603906803`*^9}, 3.6658296584374185`*^9, {
   3.6672460521798906`*^9, 3.667246110757153*^9}, 3.667301860838393*^9, {
   3.7058179970092716`*^9, 3.705818048247526*^9}, {3.705818285589779*^9, 
   3.7058183237035446`*^9}, 3.705818449364562*^9, {3.71612068835028*^9, 
   3.7161206885199823`*^9}, 3.7161210204739447`*^9, 3.7345160841540394`*^9, 
   3.748244002237558*^9, 3.74824513592642*^9, 3.748245199162204*^9, {
   3.748780010470646*^9, 3.7487800120527086`*^9}, {3.748781914084138*^9, 
   3.748781914297538*^9}, {3.7487819459333086`*^9, 3.7487819481186695`*^9}, 
   3.748782196940205*^9, {3.748788268565628*^9, 3.748788279497382*^9}, 
   3.7494583589275575`*^9, {3.7494583947916317`*^9, 3.749458398881691*^9}, {
   3.7494900293700852`*^9, 3.749490034354744*^9}, {3.7501431000981007`*^9, 
   3.7501431009821367`*^9}},
 ExpressionUUID -> "56930096-319d-4d45-b5d5-c082ff58afa2"],

Cell[BoxData[
 RowBox[{
  RowBox[{"ux", "[", 
   RowBox[{"x_", ",", "y_", ",", "z_"}], "]"}], ":=", 
  RowBox[{
   RowBox[{
    RowBox[{"\[CurlyEpsilon]", " ", 
     FractionBox[
      RowBox[{"(", 
       RowBox[{
        RowBox[{"Sin", "[", 
         RowBox[{"k", " ", "x"}], "]"}], "+", 
        RowBox[{
         FractionBox["1", "2"], " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"Sin", "[", 
            RowBox[{"k", " ", "x1"}], "]"}], "-", 
           RowBox[{"Sin", "[", 
            RowBox[{"k", " ", "x2"}], "]"}]}], ")"}]}]}], ")"}], 
      SqrtBox["3"]], " ", 
     RowBox[{
      RowBox[{"fx", "[", 
       RowBox[{"1", ",", "1"}], "]"}], "[", "z", "]"}]}], "+", 
    RowBox[{
     SuperscriptBox["\[CurlyEpsilon]", "2"], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{"2", " ", 
           RowBox[{"Sin", "[", 
            RowBox[{"2", " ", "k", " ", "x"}], "]"}]}], "+", 
          RowBox[{"Sin", "[", 
           RowBox[{"2", " ", "k", " ", "x1"}], "]"}], "-", 
          RowBox[{"Sin", "[", 
           RowBox[{"2", " ", "k", " ", "x2"}], "]"}]}], ")"}], " ", 
        RowBox[{
         RowBox[{"fx", "[", 
          RowBox[{"2", ",", "1"}], "]"}], "[", "z", "]"}]}], "+", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{"2", " ", 
           RowBox[{"Sin", "[", 
            RowBox[{"k", " ", "x"}], "]"}]}], "+", 
          RowBox[{"Sin", "[", 
           RowBox[{"k", " ", "x1"}], "]"}], "-", 
          RowBox[{"Sin", "[", 
           RowBox[{"k", " ", "x2"}], "]"}]}], ")"}], " ", 
        RowBox[{
         RowBox[{"fx", "[", 
          RowBox[{"2", ",", "2"}], "]"}], "[", "z", "]"}]}], "+", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{"3", " ", 
           RowBox[{"Sin", "[", 
            RowBox[{
             RowBox[{"k", " ", "x"}], "+", 
             RowBox[{"k", " ", "x1"}]}], "]"}]}], "+", 
          RowBox[{"3", " ", 
           RowBox[{"Sin", "[", 
            RowBox[{
             RowBox[{"k", " ", "x"}], "-", 
             RowBox[{"k", " ", "x2"}]}], "]"}]}]}], ")"}], " ", 
        RowBox[{
         RowBox[{"fx", "[", 
          RowBox[{"2", ",", "3"}], "]"}], "[", "z", "]"}]}]}], ")"}]}]}], "/.", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x1", "\[Rule]", " ", 
      RowBox[{
       FractionBox["x", "2"], "+", 
       RowBox[{
        FractionBox[
         SqrtBox["3"], "2"], "y"}]}]}], ",", 
     RowBox[{"x2", "\[Rule]", " ", 
      RowBox[{
       RowBox[{"-", 
        FractionBox["x", "2"]}], "+", 
       RowBox[{
        FractionBox[
         SqrtBox["3"], "2"], "y"}]}]}]}], "}"}]}]}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.665774263326209*^9, 3.66577427385541*^9}, {
   3.6657743236327744`*^9, 3.665774371967226*^9}, {3.665774467469146*^9, 
   3.6657745171664667`*^9}, {3.6657746780929537`*^9, 
   3.6657747272032385`*^9}, {3.6657757781517873`*^9, 3.665775808017256*^9}, 
   3.6657759580813837`*^9, 3.6657760065598683`*^9, 3.6658296584374185`*^9, 
   3.6660107266892014`*^9, 3.666012204886698*^9, {3.666012513519569*^9, 
   3.6660125247351093`*^9}, {3.66601278685109*^9, 3.666012806026108*^9}, {
   3.666012972519032*^9, 3.666012979006877*^9}, {3.666013887366681*^9, 
   3.666013928053396*^9}, 3.666013989868081*^9, {3.6660147320221443`*^9, 
   3.666014756277513*^9}, {3.666014819652697*^9, 3.6660148425158243`*^9}, {
   3.666015002745376*^9, 3.666015011136634*^9}, 3.666015703700157*^9, 
   3.666015744531355*^9, 3.667246053897068*^9, {3.6672461691966453`*^9, 
   3.6672461754305334`*^9}, {3.705818060293044*^9, 3.7058180743009596`*^9}, {
   3.705818289204507*^9, 3.705818297352764*^9}, {3.705818346823226*^9, 
   3.705818381764964*^9}, 3.7058184518271837`*^9, {3.705819379045109*^9, 
   3.7058193899392233`*^9}, {3.716120693874295*^9, 3.716120694038066*^9}, 
   3.71612103748448*^9, {3.71666379795361*^9, 3.7166638046447215`*^9}, 
   3.7345160870401254`*^9, {3.7347802139114037`*^9, 3.734780215949613*^9}, 
   3.7482440160956907`*^9, 3.748245155719054*^9, 3.748245198121097*^9, {
   3.748780017585449*^9, 3.748780022184403*^9}, {3.7487819283017635`*^9, 
   3.748781939897328*^9}, 3.748782196940205*^9, {3.748788283472746*^9, 
   3.748788288882307*^9}, {3.749458367501625*^9, 3.7494583682805424`*^9}},
 ExpressionUUID -> "40dad8a7-b736-4d12-af4b-30060025e83a"],

Cell[BoxData[
 RowBox[{
  RowBox[{"uy", "[", 
   RowBox[{"x_", ",", "y_", ",", "z_"}], "]"}], ":=", 
  RowBox[{
   RowBox[{
    RowBox[{
     FractionBox["1", "2"], "  ", "\[CurlyEpsilon]", " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"Sin", "[", 
        RowBox[{"k", " ", "x1"}], "]"}], "+", 
       RowBox[{"Sin", "[", 
        RowBox[{"k", " ", "x2"}], "]"}]}], ")"}], " ", 
     RowBox[{
      RowBox[{"fx", "[", 
       RowBox[{"1", ",", "1"}], "]"}], "[", "z", "]"}]}], "+", 
    RowBox[{
     SuperscriptBox["\[CurlyEpsilon]", "2"], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{
        SqrtBox["3"], " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"Sin", "[", 
           RowBox[{"2", " ", "k", " ", "x1"}], "]"}], "+", 
          RowBox[{"Sin", "[", 
           RowBox[{"2", " ", "k", " ", "x2"}], "]"}]}], ")"}], " ", 
        RowBox[{
         RowBox[{"fx", "[", 
          RowBox[{"2", ",", "1"}], "]"}], "[", "z", "]"}]}], "+", 
       RowBox[{
        SqrtBox["3"], " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"Sin", "[", 
           RowBox[{"k", " ", "x1"}], "]"}], "+", 
          RowBox[{"Sin", "[", 
           RowBox[{"k", " ", "x2"}], "]"}]}], ")"}], " ", 
        RowBox[{
         RowBox[{"fx", "[", 
          RowBox[{"2", ",", "2"}], "]"}], "[", "z", "]"}]}], "+", 
       RowBox[{
        SqrtBox["3"], " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"Sin", "[", 
           RowBox[{"k", " ", 
            RowBox[{"(", 
             RowBox[{"x", "+", "x1"}], ")"}]}], "]"}], "-", 
          RowBox[{"Sin", "[", 
           RowBox[{"k", " ", 
            RowBox[{"(", 
             RowBox[{"x", "-", "x2"}], ")"}]}], "]"}], "+", 
          RowBox[{"2", " ", 
           RowBox[{"Sin", "[", 
            RowBox[{"k", " ", 
             RowBox[{"(", 
              RowBox[{"x1", "+", "x2"}], ")"}]}], "]"}]}]}], ")"}], " ", 
        RowBox[{
         RowBox[{"fx", "[", 
          RowBox[{"2", ",", "3"}], "]"}], "[", "z", "]"}]}]}], ")"}]}]}], "/.", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x1", "\[Rule]", " ", 
      RowBox[{
       FractionBox["x", "2"], "+", 
       RowBox[{
        FractionBox[
         SqrtBox["3"], "2"], "y"}]}]}], ",", 
     RowBox[{"x2", "\[Rule]", " ", 
      RowBox[{
       RowBox[{"-", 
        FractionBox["x", "2"]}], "+", 
       RowBox[{
        FractionBox[
         SqrtBox["3"], "2"], "y"}]}]}]}], "}"}]}]}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.6657759391218367`*^9, 3.665776057713159*^9}, 
   3.6658296584379063`*^9, 3.6660107291851473`*^9, 3.666012209694578*^9, {
   3.666012528999206*^9, 3.666012537142894*^9}, {3.666012999870729*^9, 
   3.666013019358135*^9}, {3.666013072317521*^9, 3.666013074029131*^9}, {
   3.6660131551716967`*^9, 3.6660131691793137`*^9}, {3.6660139204933443`*^9, 
   3.666013926621202*^9}, 3.666013992463925*^9, {3.6660147384939632`*^9, 
   3.666014757597476*^9}, {3.666014822508614*^9, 3.6660148479637136`*^9}, {
   3.6660150080811462`*^9, 3.666015018784547*^9}, {3.666015633085493*^9, 
   3.666015653788924*^9}, 3.666015704827859*^9, 3.666015746131331*^9, 
   3.6672460553753724`*^9, 3.66724612564708*^9, {3.7161206968910065`*^9, 
   3.7161206970825825`*^9}, {3.7482451738278837`*^9, 3.748245196360682*^9}, 
   3.7487800522733364`*^9, {3.7487819218896985`*^9, 3.74878195810406*^9}, 
   3.748782196940205*^9, {3.74878829251655*^9, 3.7487882992335787`*^9}, 
   3.749458372627913*^9},
 ExpressionUUID -> "37011bf4-9d75-4895-8b92-c683ff6c780b"],

Cell["and in the substrate", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.7343365589552717`*^9, 3.7343365626007757`*^9}, 
   3.748782196941202*^9, 3.748788952347237*^9},
 ExpressionUUID -> "ceb842ef-d180-4cf0-a344-86b9f7dc9cc3"],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"uxS", "[", 
    RowBox[{"x_", ",", "y_", ",", "z_"}], "]"}], " ", ":=", 
   RowBox[{
    RowBox[{"ux", "[", 
     RowBox[{"x", ",", "y", ",", "z"}], "]"}], "/.", "SubstrateFields"}]}], 
  ";"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.653378489608873*^9, 3.6533785312774076`*^9}, {
   3.653378984925927*^9, 3.65337900502431*^9}, {3.65338276511313*^9, 
   3.6533828063939*^9}, 3.653382854167697*^9, {3.653383745662682*^9, 
   3.653383782981093*^9}, {3.653383840692298*^9, 3.6533838456198883`*^9}, {
   3.653384249995995*^9, 3.6533843123629847`*^9}, {3.653386202863043*^9, 
   3.6533862584998693`*^9}, {3.653390737705628*^9, 3.653390739937315*^9}, {
   3.653390902013857*^9, 3.65339090337775*^9}, {3.653390940309018*^9, 
   3.653390940712945*^9}, 3.653391190623646*^9, {3.6533912495193567`*^9, 
   3.653391282789853*^9}, 3.653634747776924*^9, 3.653637435127521*^9, 
   3.6536375108258133`*^9, {3.6536379695531263`*^9, 3.653637970546637*^9}, {
   3.653638004896541*^9, 3.653638005288282*^9}, {3.6536381345973387`*^9, 
   3.653638138828762*^9}, {3.6536424525112953`*^9, 3.653642456086172*^9}, {
   3.6536466374026337`*^9, 3.653646655164146*^9}, {3.653651776708181*^9, 
   3.653651784372064*^9}, {3.653651967105109*^9, 3.653651968536775*^9}, {
   3.65365202098381*^9, 3.6536520477108297`*^9}, {3.6544258621826887`*^9, 
   3.65442587450977*^9}, {3.654431134252376*^9, 3.654431145011314*^9}, 
   3.6544318188854713`*^9, {3.6544319448675013`*^9, 3.654431962603526*^9}, {
   3.654432465720717*^9, 3.654432471511896*^9}, {3.654432647083828*^9, 
   3.654432652452134*^9}, {3.654432766721627*^9, 3.654432776369487*^9}, {
   3.654432814024529*^9, 3.6544328205363894`*^9}, {3.654432959118018*^9, 
   3.654432964701429*^9}, {3.654433096708247*^9, 3.654433104705942*^9}, {
   3.654433218424775*^9, 3.654433220887821*^9}, {3.6544332764712877`*^9, 
   3.6544332772706413`*^9}, {3.654433470843918*^9, 3.654433480682355*^9}, {
   3.654433735581304*^9, 3.6544337625649023`*^9}, {3.654433903273954*^9, 
   3.654433910361353*^9}, {3.654433999919984*^9, 3.6544340151513968`*^9}, {
   3.654434209837215*^9, 3.6544342138123627`*^9}, {3.654434319016773*^9, 
   3.6544343431604*^9}, {3.654434533861601*^9, 3.654434538244244*^9}, {
   3.654434589083997*^9, 3.6544345900926113`*^9}, 3.6544348550385513`*^9, {
   3.6544350995764933`*^9, 3.6544351006003923`*^9}, {3.654435170048601*^9, 
   3.6544351740790243`*^9}, {3.654435232855584*^9, 3.654435236309228*^9}, {
   3.65443548767491*^9, 3.654435489263926*^9}, 3.654435612109846*^9, 
   3.6544356557016687`*^9, 3.654435765363168*^9, {3.6544358781764812`*^9, 
   3.654435881271947*^9}, {3.654435925103607*^9, 3.654435927151387*^9}, 
   3.654436023741592*^9, {3.654436101331933*^9, 3.654436139826598*^9}, {
   3.654436195178339*^9, 3.6544362263610697`*^9}, {3.654600142957807*^9, 
   3.654600143893454*^9}, 3.654600174442533*^9, {3.654601230294147*^9, 
   3.654601246720536*^9}, 3.6546031280197973`*^9, 3.654603331190138*^9, {
   3.654603409846778*^9, 3.654603410797145*^9}, {3.654607368657799*^9, 
   3.6546073984854603`*^9}, 3.654944865923127*^9, 3.654944959384799*^9, {
   3.6549456872439747`*^9, 3.6549457067261343`*^9}, {3.654948243623043*^9, 
   3.654948272841017*^9}, 3.654948381915938*^9, {3.6549484811720543`*^9, 
   3.654948481446971*^9}, {3.654949843398978*^9, 3.6549498892377996`*^9}, {
   3.65495003633182*^9, 3.654950041850161*^9}, {3.655119452157836*^9, 
   3.655119458179923*^9}, 3.6551195417469053`*^9, {3.6551199819944477`*^9, 
   3.65512002344252*^9}, {3.6551226931248283`*^9, 3.655122697436098*^9}, {
   3.655449270237895*^9, 3.6554493688669243`*^9}, {3.6554523971960616`*^9, 
   3.6554524221045923`*^9}, {3.655539312869871*^9, 3.655539327941757*^9}, 
   3.655540209700885*^9, {3.656235442696848*^9, 3.656235448360507*^9}, {
   3.656242241814143*^9, 3.656242299502399*^9}, {3.6562423503894672`*^9, 
   3.656242354437297*^9}, {3.656242431156152*^9, 3.656242436570652*^9}, {
   3.6562442575626163`*^9, 3.656244283730896*^9}, 3.656322116209444*^9, {
   3.656324411648356*^9, 3.656324412527975*^9}, {3.65632860940414*^9, 
   3.65632862648352*^9}, {3.656331744076589*^9, 3.65633176621229*^9}, {
   3.656331803163847*^9, 3.656331855067171*^9}, {3.6563323005370827`*^9, 
   3.656332337090822*^9}, {3.6563325306464443`*^9, 3.656332588333802*^9}, {
   3.656336033484117*^9, 3.656336096057795*^9}, {3.6564018134846015`*^9, 
   3.65640181834159*^9}, {3.6564020186226864`*^9, 3.6564020385035753`*^9}, {
   3.6564020807025366`*^9, 3.656402113699897*^9}, {3.656402189561782*^9, 
   3.6564022851846113`*^9}, {3.6564023423213015`*^9, 
   3.6564023829871492`*^9}, {3.6564024147071257`*^9, 3.656402430196471*^9}, {
   3.656402512787984*^9, 3.65640251391029*^9}, 3.6564025646841536`*^9, {
   3.6564026110136905`*^9, 3.6564026319503975`*^9}, {3.6564044250173144`*^9, 
   3.6564044300125113`*^9}, {3.65640446768567*^9, 3.656404490823603*^9}, {
   3.6564060240181212`*^9, 3.656406036030825*^9}, {3.65640611179447*^9, 
   3.6564061130183363`*^9}, {3.656406145692709*^9, 3.656406190104973*^9}, 
   3.6564091094866962`*^9, {3.6564097325831666`*^9, 3.6564098157265*^9}, {
   3.6570301059500732`*^9, 3.657030110037575*^9}, {3.657354761327952*^9, 
   3.65735476350546*^9}, 3.6649119715204744`*^9, {3.66514571166147*^9, 
   3.6651457456776066`*^9}, 3.6651461491061306`*^9, {3.665146376692958*^9, 
   3.665146384909066*^9}, {3.665165810870386*^9, 3.6651658112532315`*^9}, {
   3.665176365125589*^9, 3.665176398747633*^9}, {3.6652262011487246`*^9, 
   3.6652262027787337`*^9}, {3.665300636031836*^9, 3.6653006541165915`*^9}, {
   3.6653010448729863`*^9, 3.66530104813066*^9}, {3.6653012653576336`*^9, 
   3.6653012660088434`*^9}, 3.6653092065011587`*^9, {3.6653247721505795`*^9, 
   3.665324772613239*^9}, {3.6653914647859087`*^9, 3.665391525419199*^9}, {
   3.665392575525505*^9, 3.6653925763305845`*^9}, 3.6653934786372986`*^9, {
   3.665433338210455*^9, 3.6654333410946145`*^9}, {3.6658296132314606`*^9, 
   3.6658296398754435`*^9}, {3.6670365339806337`*^9, 
   3.6670365344652624`*^9}, {3.667549811196762*^9, 3.667549823541642*^9}, 
   3.6708461501379952`*^9, {3.671515747845851*^9, 3.67151574800206*^9}, 
   3.7339012212388983`*^9, 3.748263123339094*^9, {3.74826427511895*^9, 
   3.748264275689365*^9}, 3.7487821969441943`*^9, 3.748788952347237*^9},
 ExpressionUUID -> "961a37e2-31ff-4366-af70-d4df4d9c7878"],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"uyS", "[", 
    RowBox[{"x_", ",", "y_", ",", "z_"}], "]"}], " ", ":=", 
   RowBox[{
    RowBox[{"uy", "[", 
     RowBox[{"x", ",", "y", ",", "z"}], "]"}], "/.", "SubstrateFields"}]}], 
  ";"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.653646656864345*^9, 3.653646666201375*^9}, {
   3.6544258790946913`*^9, 3.6544258819273167`*^9}, {3.654431134883913*^9, 
   3.654431148657559*^9}, 3.654431819769801*^9, {3.654431967483809*^9, 
   3.6544319777410393`*^9}, {3.654432474728406*^9, 3.654432501759036*^9}, {
   3.654432656380678*^9, 3.654432660348859*^9}, {3.654432966669482*^9, 
   3.6544329700770397`*^9}, {3.654433101419043*^9, 3.654433110930039*^9}, {
   3.6544332812877903`*^9, 3.65443328461489*^9}, {3.654433474835051*^9, 
   3.654433477234099*^9}, {3.654433754766362*^9, 3.654433759645101*^9}, {
   3.654433896290979*^9, 3.654433910753422*^9}, {3.654434220261179*^9, 
   3.654434224250766*^9}, {3.654434319752801*^9, 3.654434351944413*^9}, 
   3.654434399543202*^9, {3.654434535749795*^9, 3.654434546805729*^9}, 
   3.654434596040084*^9, 3.654434856477832*^9, {3.654435103905141*^9, 
   3.654435104577302*^9}, {3.654435176407405*^9, 3.65443518226292*^9}, {
   3.654435237250764*^9, 3.6544352385893707`*^9}, {3.654435492097211*^9, 
   3.654435492888328*^9}, 3.654435613318428*^9, 3.654435724500189*^9, {
   3.654435766779348*^9, 3.654435768314274*^9}, {3.6544359675596333`*^9, 
   3.654435968414906*^9}, 3.654436016173621*^9, {3.654436140634481*^9, 
   3.654436143467098*^9}, {3.6544362039776278`*^9, 3.654436228369001*^9}, 
   3.6546001761650763`*^9, {3.654601231667083*^9, 3.654601258872344*^9}, {
   3.654603142435042*^9, 3.6546031450094852`*^9}, {3.654603418312087*^9, 
   3.654603418948114*^9}, {3.654607370182158*^9, 3.654607416756566*^9}, {
   3.654852246040944*^9, 3.6548522598968573`*^9}, {3.654852314008891*^9, 
   3.654852314144066*^9}, {3.654852732552444*^9, 3.654852736568254*^9}, 
   3.654944860955061*^9, 3.654944962121249*^9, {3.6549456905310926`*^9, 
   3.654945715457518*^9}, {3.6549482817625732`*^9, 3.654948284038377*^9}, 
   3.654948385700252*^9, {3.654948483465907*^9, 3.654948485490468*^9}, {
   3.6549498488876667`*^9, 3.654949890598049*^9}, {3.654950043228282*^9, 
   3.6549500515465508`*^9}, {3.655116306100341*^9, 3.6551163106042843`*^9}, {
   3.655116445886464*^9, 3.655116470392399*^9}, {3.655117198129714*^9, 
   3.655117215969509*^9}, {3.6551174415973787`*^9, 3.65511746020441*^9}, 
   3.6551183824662323`*^9, {3.655119453797896*^9, 3.655119459572075*^9}, 
   3.655119543328308*^9, {3.655120028300734*^9, 3.655120038329707*^9}, {
   3.655122702278792*^9, 3.655122703700597*^9}, {3.655449373300736*^9, 
   3.655449397635953*^9}, {3.6554521762627583`*^9, 3.6554521913682113`*^9}, {
   3.655452394013936*^9, 3.655452429502508*^9}, {3.655539331485566*^9, 
   3.655539364165963*^9}, {3.6555398130531387`*^9, 3.655539829747613*^9}, 
   3.655540212661139*^9, 3.6562354564256163`*^9, {3.656242312482699*^9, 
   3.656242316230077*^9}, {3.656244288251334*^9, 3.6562443066184893`*^9}, {
   3.656322120841134*^9, 3.656322121611712*^9}, {3.656322351157514*^9, 
   3.656322356989119*^9}, {3.656324416608314*^9, 3.65632442102378*^9}, {
   3.656328610387838*^9, 3.656328627427506*^9}, {3.6563317457803183`*^9, 
   3.656331859586472*^9}, {3.656332607007971*^9, 3.6563326192843523`*^9}, {
   3.6563361151247063`*^9, 3.656336124937318*^9}, {3.6564018246350327`*^9, 
   3.6564018318525796`*^9}, 3.6564020469767513`*^9, {3.656402083369752*^9, 
   3.6564020849698887`*^9}, {3.656402308529245*^9, 3.656402394087489*^9}, {
   3.656402642862974*^9, 3.656402714402313*^9}, 3.656402756846137*^9, {
   3.656404427394226*^9, 3.6564044893855834`*^9}, {3.6564059844045863`*^9, 
   3.6564060168452473`*^9}, {3.6564061141501384`*^9, 
   3.6564061156622105`*^9}, {3.656406159067218*^9, 3.6564062011502333`*^9}, 
   3.656409113788725*^9, {3.656409823793915*^9, 3.656409838026557*^9}, {
   3.657030116383741*^9, 3.6570301244534273`*^9}, {3.657354765729664*^9, 
   3.6573547678040657`*^9}, 3.664911968549234*^9, {3.6651463730200953`*^9, 
   3.665146387680727*^9}, {3.6651658157814565`*^9, 3.665165816470908*^9}, {
   3.665176366732342*^9, 3.6651764046392145`*^9}, {3.665226205860623*^9, 
   3.6652262085959606`*^9}, {3.665226276977391*^9, 3.6652262773792405`*^9}, 
   3.6653092065011587`*^9, {3.665433338210455*^9, 3.6654333410946145`*^9}, {
   3.665829616987405*^9, 3.6658296431942353`*^9}, 3.667036538137045*^9, {
   3.667549813188735*^9, 3.667549824333385*^9}, 3.6708461506140385`*^9, {
   3.6715157490333633`*^9, 3.6715157490958643`*^9}, 3.7339012212388983`*^9, {
   3.734780218998967*^9, 3.734780220293178*^9}, 3.748263124091325*^9, {
   3.748264278266306*^9, 3.748264278635758*^9}, 3.7487821969471855`*^9, 
   3.748788952347237*^9},
 ExpressionUUID -> "8c7abfee-ce24-49c0-ab8a-7e8fc1b916d3"],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"uzS", "[", 
    RowBox[{"x_", ",", "y_", ",", "z_"}], "]"}], " ", ":=", " ", 
   RowBox[{
    RowBox[{"uz", "[", 
     RowBox[{"x", ",", "y", ",", "z"}], "]"}], "/.", "SubstrateFields"}]}], 
  ";"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.653646668681004*^9, 3.653646699145215*^9}, {
   3.653651962025899*^9, 3.653651976480847*^9}, {3.6536520249427347`*^9, 
   3.6536520498153677`*^9}, {3.654425813951067*^9, 3.65442581783171*^9}, {
   3.6544304676650343`*^9, 3.654430509489046*^9}, {3.6544310802697268`*^9, 
   3.6544310827164516`*^9}, 3.65443182115392*^9, 3.6544319867863283`*^9, {
   3.65443254753474*^9, 3.654432582821368*^9}, {3.654432866977209*^9, 
   3.654432881535369*^9}, 3.654433017380834*^9, {3.654433190081985*^9, 
   3.654433195392448*^9}, {3.654433487330469*^9, 3.654433494186371*^9}, {
   3.6544341672446613`*^9, 3.654434200156747*^9}, 3.654434321264763*^9, {
   3.654434360176662*^9, 3.654434365016451*^9}, 3.6544344341359653`*^9, {
   3.65443453959233*^9, 3.6544345431130257`*^9}, {3.654434600411928*^9, 
   3.654434604923008*^9}, {3.654434857493829*^9, 3.654434858742721*^9}, {
   3.654435035130908*^9, 3.654435038849515*^9}, {3.654599015572548*^9, 
   3.6545990907637367`*^9}, {3.6545991246198483`*^9, 3.654599127658218*^9}, {
   3.65459917444527*^9, 3.654599203160955*^9}, {3.654599245304274*^9, 
   3.6545992522253647`*^9}, {3.654599699287374*^9, 3.654599716207759*^9}, {
   3.654600711180138*^9, 3.654600743266433*^9}, {3.65460082516348*^9, 
   3.654600868136133*^9}, 3.654853362060231*^9, {3.654941705651705*^9, 
   3.6549417213311157`*^9}, 3.654944852283134*^9, 3.654944965477054*^9, {
   3.654945718026718*^9, 3.654945727097561*^9}, {3.6551194182370853`*^9, 
   3.655119440972431*^9}, {3.655120052006782*^9, 3.655120068768977*^9}, {
   3.655122705902609*^9, 3.655122708773796*^9}, {3.655539392228231*^9, 
   3.655539402211912*^9}, 3.656235457329543*^9, {3.656236096013527*^9, 
   3.656236100300889*^9}, {3.6562367023385983`*^9, 3.65623674575331*^9}, {
   3.656237279744659*^9, 3.6562372927583103`*^9}, {3.656237328119171*^9, 
   3.6562373542553053`*^9}, {3.656237386622595*^9, 3.65623739033451*^9}, 
   3.656238918650964*^9, {3.656239409234419*^9, 3.656239471432931*^9}, {
   3.656239528657618*^9, 3.656239542936122*^9}, {3.6562400713854923`*^9, 
   3.656240149613399*^9}, 3.656242215808754*^9, 3.656244243667077*^9, 
   3.6563220677301397`*^9, {3.6563286122161283`*^9, 3.656328635755446*^9}, {
   3.6564027290294943`*^9, 3.656402738891775*^9}, {3.656404280569064*^9, 
   3.65640434942456*^9}, {3.6564059781954913`*^9, 3.6564059785893593`*^9}, {
   3.656413116052379*^9, 3.656413121967552*^9}, {3.6564157307394524`*^9, 
   3.656415738118362*^9}, {3.6649119643670444`*^9, 3.664911965078226*^9}, {
   3.665146363592942*^9, 3.665146389961588*^9}, 3.6651464604984417`*^9, {
   3.6651658195361357`*^9, 3.6651658213485804`*^9}, {3.665166008082565*^9, 
   3.6651660085779257`*^9}, {3.6651763692147284`*^9, 
   3.6651764086038017`*^9}, {3.6652262116202974`*^9, 3.6652262128931503`*^9}, 
   3.665226281228283*^9, {3.6653006401102057`*^9, 3.6653006575867515`*^9}, {
   3.665301050955241*^9, 3.6653010542312765`*^9}, {3.6653012680230923`*^9, 
   3.665301268390334*^9}, {3.6653022793879952`*^9, 3.6653022810010614`*^9}, 
   3.6653092065011587`*^9, 3.6653247739741764`*^9, {3.6653914664078445`*^9, 
   3.665391526571164*^9}, {3.665433338210455*^9, 3.6654333410946145`*^9}, {
   3.6658296215607457`*^9, 3.6658296439636726`*^9}, 3.6670365395473323`*^9, {
   3.667549815692732*^9, 3.6675498251974363`*^9}, 3.670846151338025*^9, 
   3.7339012212388983`*^9, 3.74826312492128*^9, {3.748264280802253*^9, 
   3.748264281146008*^9}, 3.748782196948182*^9, 3.748788952347237*^9},
 ExpressionUUID -> "811cca09-e39f-418f-9481-2f476967a27c"],

Cell[TextData[StyleBox["Surface stress", "Subsection"]], "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.748192884714484*^9, 3.748192886896722*^9}, 
   3.748782196951205*^9, 3.748788952348234*^9}],

Cell["\<\
We also write the Lagrange multiplier for the surface amplitude as a series \
in \
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.748192725130557*^9, 3.7481927471727858`*^9}, 
   3.748782196951205*^9, 3.748788952348234*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"L", "=", 
   RowBox[{"L0", "+", 
    RowBox[{"\[CurlyEpsilon]", " ", "L1"}], " ", "+", 
    RowBox[{
     SuperscriptBox["\[CurlyEpsilon]", "2"], " ", "L2"}]}]}], " ", 
  ";"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.6651762408646874`*^9, 3.665176284308905*^9}, {
   3.6652272035076323`*^9, 3.6652272055495043`*^9}, 3.6653092065011587`*^9, {
   3.665433338210455*^9, 3.6654333410946145`*^9}, {3.669689820988999*^9, 
   3.669689828501999*^9}, {3.669691223963415*^9, 3.6696912248537703`*^9}, 
   3.7058486877397184`*^9, 3.7164901243745613`*^9, {3.734516110546652*^9, 
   3.7345161144425993`*^9}, 3.7487821969522*^9, 3.748788952348234*^9},
 ExpressionUUID -> "157f94d4-a345-4842-ba8b-e758cc27a061"],

Cell[TextData[StyleBox["Calculation of the energy coefficients", \
"Subsection"]], "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.748193033814044*^9, 3.7481930409641542`*^9}, 
   3.7487821969531994`*^9, 3.748788952348234*^9}],

Cell["\<\
To calculate the energy coefficients, we first expand the energy density as a \
series in \[CurlyEpsilon].\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.748261801266735*^9, 3.7482618033912163`*^9}, 
   3.7487821969531994`*^9, 3.748788952348234*^9}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"W1", "=", 
   RowBox[{"SeriesCoefficient", "[", 
    RowBox[{"W", ",", 
     RowBox[{"{", 
      RowBox[{"\[CurlyEpsilon]", ",", "0", ",", "1"}], "}"}]}], "]"}]}], ";", 
  
  RowBox[{"W2", "=", 
   RowBox[{"SeriesCoefficient", "[", 
    RowBox[{"W", ",", 
     RowBox[{"{", 
      RowBox[{"\[CurlyEpsilon]", ",", "0", ",", "2"}], "}"}]}], "]"}]}], ";", 
  
  RowBox[{"W3", "=", 
   RowBox[{"SeriesCoefficient", "[", 
    RowBox[{"W", ",", 
     RowBox[{"{", 
      RowBox[{"\[CurlyEpsilon]", ",", "0", ",", "3"}], "}"}]}], "]"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"W1S2", ",", "W2S", ",", "W3S"}], "}"}], "=", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"W1", ",", "W2", ",", "W3"}], "}"}], "/.", "SubstrateFields"}]}],
   ";"}]}], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.6657713232344956`*^9, 3.665771343381418*^9}, {
   3.665771418266056*^9, 3.665771443269935*^9}, {3.6657714854748755`*^9, 
   3.6657715331802435`*^9}, {3.665771716847458*^9, 3.665771720446126*^9}, {
   3.6657718139064054`*^9, 3.6657718741793876`*^9}, {3.665772091294901*^9, 
   3.665772215307926*^9}, {3.665772247964039*^9, 3.665772263610958*^9}, {
   3.665772485446594*^9, 3.6657725184863443`*^9}, 3.6657729166750526`*^9, {
   3.66577426017365*^9, 3.6657742603906803`*^9}, 3.6658296584374185`*^9, {
   3.6672460521798906`*^9, 3.667246110757153*^9}, 3.667301860838393*^9, {
   3.7058179970092716`*^9, 3.705818048247526*^9}, {3.705818285589779*^9, 
   3.7058183237035446`*^9}, 3.705818449364562*^9, {3.71612068835028*^9, 
   3.7161206885199823`*^9}, 3.7161210204739447`*^9, 3.7345160841540394`*^9, 
   3.748244002237558*^9, 3.74824513592642*^9, 3.748245199162204*^9, {
   3.748260944110375*^9, 3.7482609500335903`*^9}, 3.7482611528669987`*^9, {
   3.7482612607979317`*^9, 3.748261267619701*^9}, {3.7482613798230553`*^9, 
   3.7482613948351583`*^9}, {3.7482615037390327`*^9, 3.7482615540063*^9}, {
   3.748261810434952*^9, 3.748261852402668*^9}, {3.748262405354032*^9, 
   3.748262405520245*^9}, {3.7483637331750517`*^9, 3.748363879931884*^9}, 
   3.7487821969551697`*^9, 3.748788952348234*^9, {3.7494584086136613`*^9, 
   3.749458416903517*^9}},
 ExpressionUUID -> "56930096-319d-4d45-b5d5-c082ff58afa2"],

Cell[TextData[StyleBox["In-plane averaging", "Subsection"]], "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.7481928957942*^9, 3.7481929037282*^9}, 
   3.748782196956193*^9, 3.748788952348234*^9}],

Cell["\<\
For the stripe pattern, to take in-plane averages we integrate over an \
in-plane square with verticies (0,0) and (xmay, ymas) where:\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.7166710645876665`*^9, 3.7166710770508313`*^9}, {
   3.716671134840579*^9, 3.716671135971878*^9}, {3.7198254746813087`*^9, 
   3.719825475217451*^9}, {3.7348015038859835`*^9, 3.7348015088925223`*^9}, {
   3.734801559159957*^9, 3.7348015650851173`*^9}, {3.748192754780232*^9, 
   3.748192838370331*^9}, 3.748782196957158*^9, 3.748788952348234*^9},
 ExpressionUUID -> "09bd2c59-97fe-406b-8035-c128314c4e86"],

Cell[BoxData[
 RowBox[{
  RowBox[{"xmax", "=", 
   RowBox[{"4", 
    RowBox[{"\[Pi]", "/", "k"}]}]}], ";", 
  RowBox[{"ymax", "=", 
   RowBox[{"4", 
    RowBox[{"\[Pi]", "/", 
     RowBox[{"(", 
      RowBox[{
       SqrtBox["3"], "k"}], ")"}]}]}]}], ";"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{
  3.719933909813597*^9, {3.734776818484407*^9, 3.73477681962022*^9}, 
   3.748370890827632*^9, 3.7487821969581842`*^9, 3.748788952348234*^9},
 ExpressionUUID -> "5855c714-aa51-498c-95ad-fb5bc38951a6"],

Cell["\<\
We first average the W coefficients over this in-plane unit cell: (may take a \
few minutes)\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.748261748545721*^9, 3.748261754737146*^9}, {
   3.748261885770821*^9, 3.748261903569084*^9}, 3.7487821969581842`*^9, 
   3.748788952348234*^9, {3.7494672195285726`*^9, 3.7494672263375874`*^9}, {
   3.749467332912554*^9, 3.7494673363094397`*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"W1av", "=", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"1", "/", 
      RowBox[{"(", 
       RowBox[{"xmax", " ", "ymax"}], ")"}]}], ")"}], 
    RowBox[{"Integrate", "[", 
     RowBox[{
      RowBox[{"Expand", "[", "W1", "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "0", ",", "xmax"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"y", ",", "0", ",", "ymax"}], "}"}]}], "]"}]}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"W2av", "=", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"1", "/", 
      RowBox[{"(", 
       RowBox[{"xmax", " ", "ymax"}], ")"}]}], ")"}], 
    RowBox[{"Integrate", "[", 
     RowBox[{
      RowBox[{"Expand", "[", "W2", "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "0", ",", "xmax"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"y", ",", "0", ",", "ymax"}], "}"}]}], "]"}]}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"W3av", "=", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"1", "/", 
      RowBox[{"(", 
       RowBox[{"xmax", " ", "ymax"}], ")"}]}], ")"}], 
    RowBox[{"Integrate", "[", 
     RowBox[{
      RowBox[{"Expand", "[", "W3", "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "0", ",", "xmax"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"y", ",", "0", ",", "ymax"}], "}"}]}], "]"}]}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"W1Sav", ",", "W2Sav", ",", "W3Sav"}], "}"}], "=", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"W1av", ",", "W2av", ",", "W3av"}], "}"}], "/.", 
    "SubstrateFields"}]}], ";"}], "\[IndentingNewLine]"}], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.6657713232344956`*^9, 3.665771343381418*^9}, {
   3.665771418266056*^9, 3.665771443269935*^9}, {3.6657714854748755`*^9, 
   3.6657715331802435`*^9}, {3.665771716847458*^9, 3.665771720446126*^9}, {
   3.6657718139064054`*^9, 3.6657718741793876`*^9}, {3.665772091294901*^9, 
   3.665772215307926*^9}, {3.665772247964039*^9, 3.665772263610958*^9}, {
   3.665772485446594*^9, 3.6657725184863443`*^9}, 3.6657729166750526`*^9, {
   3.66577426017365*^9, 3.6657742603906803`*^9}, 3.6658296584374185`*^9, {
   3.6672460521798906`*^9, 3.667246110757153*^9}, 3.667301860838393*^9, {
   3.7058179970092716`*^9, 3.705818048247526*^9}, {3.705818285589779*^9, 
   3.7058183237035446`*^9}, 3.705818449364562*^9, {3.71612068835028*^9, 
   3.7161206885199823`*^9}, 3.7161210204739447`*^9, 3.7345160841540394`*^9, 
   3.748244002237558*^9, 3.74824513592642*^9, 3.748245199162204*^9, {
   3.748260944110375*^9, 3.7482609500335903`*^9}, 3.7482611528669987`*^9, {
   3.7482612607979317`*^9, 3.748261267619701*^9}, {3.7482613798230553`*^9, 
   3.7482613948351583`*^9}, {3.7482615037390327`*^9, 3.7482615540063*^9}, {
   3.748261810434952*^9, 3.748261852402668*^9}, {3.748261923192767*^9, 
   3.748261997419652*^9}, {3.748262036893588*^9, 3.7482620405479507`*^9}, {
   3.748363304860167*^9, 3.748363305950943*^9}, {3.748363884084024*^9, 
   3.7483639296474533`*^9}, 3.74836397218202*^9, 3.748782196963142*^9, 
   3.7487889523492317`*^9, {3.7489523302958765`*^9, 3.74895235539946*^9}, {
   3.7494584100159388`*^9, 3.7494584134916425`*^9}},
 ExpressionUUID -> "56930096-319d-4d45-b5d5-c082ff58afa2"],

Cell[TextData[StyleBox["Validation the solution satisfies the bulk \
equations", "Subsection"]], "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.7481928957942*^9, 3.7481929037282*^9}, {
   3.7481929666078777`*^9, 3.748192979141858*^9}, 3.748782196963142*^9, 
   3.7487889523492317`*^9}],

Cell["\<\
.We use the inbuilt Mathematica \[OpenCurlyDoubleQuote]Series\
\[CloseCurlyDoubleQuote]  to expand the bulk equations to second order as a \
series in  \[CurlyEpsilon]\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.748267286562029*^9, 3.74826730455931*^9}, {
   3.748358066566401*^9, 3.74835810807306*^9}, 3.74878219696414*^9, 
   3.7487885413311186`*^9, 3.7487889523492317`*^9, {3.749491946354383*^9, 
   3.7494919470515137`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"BulkEquationSeries", "=", 
   RowBox[{"Series", "[", 
    RowBox[{"BulkEquations", ",", 
     RowBox[{"{", 
      RowBox[{"\[CurlyEpsilon]", ",", "0", ",", "2"}], "}"}]}], "]"}]}], 
  ";"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.748244164508481*^9, 3.748244176032332*^9}, {
   3.748263945014106*^9, 3.748264022185272*^9}, 3.7482645170562563`*^9, {
   3.748265025803114*^9, 3.748265063629121*^9}, {3.7482663672032747`*^9, 
   3.7482664252577047`*^9}, {3.748266531005845*^9, 3.748266532199972*^9}, {
   3.748266616190357*^9, 3.74826663949277*^9}, 3.748266896189138*^9, {
   3.7482673152855577`*^9, 3.7482673161669197`*^9}, {3.7482678090557737`*^9, 
   3.748267809614543*^9}, {3.74826789081393*^9, 3.748267895061193*^9}, {
   3.748267927612143*^9, 3.74826794725991*^9}, {3.748268583368083*^9, 
   3.7482685998720217`*^9}, {3.748268645047811*^9, 3.7482686485923843`*^9}, {
   3.7482688459580517`*^9, 3.7482688772805*^9}, {3.7483581446122227`*^9, 
   3.748358196022498*^9}, {3.748359076251601*^9, 3.748359096792377*^9}, {
   3.748763367207853*^9, 3.7487633681848793`*^9}, 3.7487821969651365`*^9, 
   3.7487889523492317`*^9, 3.7494584319682245`*^9}],

Cell["\<\
We are not able to solve for the field at second order analytically as in the \
incompressible system. Instead, we will solve them by assigning all the \
parameters with numerical values, then solve them analytically. This is done \
by a function GenerateSolution[g , k, \[Eta]  K]. \
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.7166710645876665`*^9, 3.7166710770508313`*^9}, {
   3.716671134840579*^9, 3.716671135971878*^9}, {3.7198254746813087`*^9, 
   3.719825475217451*^9}, {3.7348015038859835`*^9, 3.7348015088925223`*^9}, {
   3.734801559159957*^9, 3.7348015650851173`*^9}, {3.748192754780232*^9, 
   3.748192838370331*^9}, {3.748197133874509*^9, 3.748197229470049*^9}, {
   3.748197378441827*^9, 3.748197390169345*^9}, {3.748265901484597*^9, 
   3.748265904740265*^9}, {3.748265956999032*^9, 3.7482659702719507`*^9}, 
   3.748774746791428*^9, 3.748782197231453*^9, {3.7487888048911004`*^9, 
   3.7487888059802055`*^9}, 3.7487889523492317`*^9, {3.7494584374285903`*^9, 
   3.749458479354499*^9}, {3.7494586247964354`*^9, 3.7494587526838503`*^9}, {
   3.749458803085863*^9, 3.7494588114405174`*^9}, {3.749471694159129*^9, 
   3.7494716941621137`*^9}},
 ExpressionUUID -> "09bd2c59-97fe-406b-8035-c128314c4e86"],

Cell["\<\
For example, for \[Eta] = 3 and K = 3, the critical growth and wavenumber are \
g=1.20945,k=1.337. We will use these values to solve for the displacement \
fields, boundary conditions and energy coefficients\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.7166710645876665`*^9, 3.7166710770508313`*^9}, {
   3.716671134840579*^9, 3.716671135971878*^9}, {3.7198254746813087`*^9, 
   3.719825475217451*^9}, {3.7348015038859835`*^9, 3.7348015088925223`*^9}, {
   3.734801559159957*^9, 3.7348015650851173`*^9}, {3.748192754780232*^9, 
   3.748192838370331*^9}, {3.748197133874509*^9, 3.748197229470049*^9}, {
   3.748197378441827*^9, 3.748197390169345*^9}, {3.748265901484597*^9, 
   3.748265904740265*^9}, {3.748265956999032*^9, 3.7482659702719507`*^9}, 
   3.748774746791428*^9, 3.748782197231453*^9, {3.7487888048911004`*^9, 
   3.7487888059802055`*^9}, 3.7487889523492317`*^9, {3.7494584374285903`*^9, 
   3.749458479354499*^9}, {3.7494586247964354`*^9, 3.7494587526838503`*^9}, {
   3.749458803085863*^9, 3.7494588114405174`*^9}, 3.7494717011514206`*^9},
 ExpressionUUID -> "09bd2c59-97fe-406b-8035-c128314c4e86"]
}, Open  ]],

Cell[BoxData[
 RowBox[{
  RowBox[{"TestPoint", "=", " ", 
   RowBox[{"SetPrecision", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"g", "\[Rule]", "1.2094508496552547`"}], ",", 
       RowBox[{"k", "\[Rule]", "1.3369970510001397`"}], ",", 
       RowBox[{"\[Mu]2", "\[Rule]", "1"}], ",", 
       RowBox[{"\[Mu]1", "\[Rule]", "3"}], ",", 
       RowBox[{"K", "\[Rule]", "3"}], ",", 
       RowBox[{"\[Gamma]", "->", "0.7872741904711991`"}]}], "}"}], ",", 
     "50"}], "]"}]}], ";"}]], "Input",
 CellChangeTimes->{{3.7494604767939544`*^9, 3.749460543489599*^9}, 
   3.7494650044129553`*^9, {3.749465194039904*^9, 3.7494652040421505`*^9}, {
   3.749498028850665*^9, 3.749498060353456*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"fsolutionhex", "=", 
   RowBox[{"GenerateSolution", "[", 
    RowBox[{
     RowBox[{"g", "/.", "TestPoint"}], ",", 
     RowBox[{"k", "/.", "TestPoint"}], ",", 
     RowBox[{"\[Mu]1", "/.", "TestPoint"}], ",", 
     RowBox[{"K", "/.", "TestPoint"}]}], "]"}]}], ";"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.748244164508481*^9, 3.748244176032332*^9}, 
   3.748265242373149*^9, {3.748371437005432*^9, 3.74837143926686*^9}, 
   3.748782197235414*^9, 3.7487889523492317`*^9, {3.7487912558986964`*^9, 
   3.748791272273771*^9}, {3.748793049074587*^9, 3.748793050045586*^9}, {
   3.749458864936097*^9, 3.7494588680068793`*^9}, 3.7494604086083565`*^9, 
   3.74946054693436*^9, {3.74946521254042*^9, 3.7494652148133698`*^9}, {
   3.7494980749285436`*^9, 3.74949809780186*^9}, {3.749499343966523*^9, 
   3.749499349032048*^9}}],

Cell[BoxData[
 TemplateBox[{
  "Part","partd",
   "\"Part specification \\!\\(\\*RowBox[{\\\"a\\\", \\\"\[LeftDoubleBracket]\
\\\", \\\"1\\\", \\\"\[RightDoubleBracket]\\\"}]\\) is longer than depth of \
object.\"",2,72,2,31106859009397539697,"Local"},
  "MessageTemplate"]], "Message", "MSG",
 CellChangeTimes->{3.7537823370132275`*^9, 3.753783231980297*^9, 
  3.7537874591196423`*^9}],

Cell[BoxData[
 TemplateBox[{
  "Part","partd",
   "\"Part specification \\!\\(\\*RowBox[{\\\"a\\\", \\\"\[LeftDoubleBracket]\
\\\", \\\"1\\\", \\\"\[RightDoubleBracket]\\\"}]\\) is longer than depth of \
object.\"",2,72,3,31106859009397539697,"Local"},
  "MessageTemplate"]], "Message", "MSG",
 CellChangeTimes->{3.7537823370132275`*^9, 3.753783231980297*^9, 
  3.7537874591515594`*^9}],

Cell["For example:", "Text",
 CellChangeTimes->{{3.7494604256926355`*^9, 3.7494604295553026`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"fz", "[", 
    RowBox[{"1", ",", "1"}], "]"}], "[", "z", "]"}], "/.", 
  "fsolutionhex"}]], "Input",
 CellChangeTimes->{{3.7494604318481703`*^9, 3.7494604456403008`*^9}, {
  3.749471728181123*^9, 3.749471737791418*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{
    "1.03622638883446724310313248791730630780373160490824035383929054875392173\
943828`48.424122596298695", " ", "z"}]], " ", 
   RowBox[{"A", "[", 
    RowBox[{"1", ",", "1"}], "]"}]}], "+", 
  RowBox[{
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{
     RowBox[{
     "-", "1.03622638883446724310313248791730630780373160490824035383929054875\
392173943828`48.424122596298695"}], " ", "z"}]], " ", 
   RowBox[{"A", "[", 
    RowBox[{"1", ",", "2"}], "]"}]}], "+", 
  RowBox[{
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{
    "0.79819468789805755395516195796716030478843158410968946027179332167279295\
213853`48.227265231843205", " ", "z"}]], " ", 
   RowBox[{"A", "[", 
    RowBox[{"1", ",", "3"}], "]"}]}], "+", 
  RowBox[{
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{
     RowBox[{
     "-", "0.79819468789805755395516195796716030478843158410968946027179332167\
279295213853`48.227265231843205"}], " ", "z"}]], " ", 
   RowBox[{"A", "[", 
    RowBox[{"1", ",", "4"}], "]"}]}]}]], "Output",
 CellChangeTimes->{{3.748791269162113*^9, 3.7487912731479483`*^9}, {
   3.7487930506368847`*^9, 3.7487930668035126`*^9}, {3.7494716833032084`*^9, 
   3.7494717335966377`*^9}, 3.7494919931103086`*^9, 3.750444010047682*^9, 
   3.7504443953186646`*^9, 3.7537823732971525`*^9, 3.7537832628407393`*^9, 
   3.7537874925026493`*^9}]
}, Open  ]],

Cell["\<\
To verify these fields solve the bulk equations, we substitute them in, and \
then simplify the result \
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.7166710645876665`*^9, 3.7166710770508313`*^9}, {
   3.716671134840579*^9, 3.716671135971878*^9}, {3.7198254746813087`*^9, 
   3.719825475217451*^9}, {3.7348015038859835`*^9, 3.7348015088925223`*^9}, {
   3.734801559159957*^9, 3.7348015650851173`*^9}, {3.748192754780232*^9, 
   3.748192838370331*^9}, {3.748197133874509*^9, 3.748197229470049*^9}, {
   3.748197378441827*^9, 3.748197390169345*^9}, {3.748265901484597*^9, 
   3.748265904740265*^9}, {3.748265956999032*^9, 3.7482659702719507`*^9}, {
   3.74835877376761*^9, 3.748358840101597*^9}, 3.748782197237439*^9, 
   3.748788952350238*^9, 3.749460594149063*^9},
 ExpressionUUID -> "09bd2c59-97fe-406b-8035-c128314c4e86"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Chop", "[", 
  RowBox[{
   RowBox[{"Simplify", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"TrigSimplify", "[", "BulkEquationSeries", "]"}], "/.", 
      "TestPoint"}], "/.", "fsolutionhex"}], "]"}], ",", 
   RowBox[{"10", "^", 
    RowBox[{"-", "10"}]}]}], "]"}]], "Input",
 CellChangeTimes->{{3.750444427033129*^9, 3.750444429920459*^9}}],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "3"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 3, 3, 1],
    Editable->False], ",", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "3"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 3, 3, 1],
    Editable->False], ",", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "3"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 3, 3, 1],
    Editable->False], ",", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "3"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 3, 3, 1],
    Editable->False], ",", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "3"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 3, 3, 1],
    Editable->False], ",", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "3"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 3, 3, 1],
    Editable->False]}], "}"}]], "Output",
 CellChangeTimes->{3.7504447809137754`*^9, 3.7537823982493887`*^9, 
  3.7537832889483924`*^9, 3.753787515923641*^9}]
}, Open  ]],

Cell["", "Text",
 CellChangeTimes->{{3.749814249477998*^9, 3.749814252055662*^9}}],

Cell["", "Text",
 CellChangeTimes->{{3.7494924233672905`*^9, 3.749492423679004*^9}, {
  3.7494924966330338`*^9, 3.7494924969511814`*^9}}],

Cell[TextData[StyleBox["Validation the solution satisfies the boundary \
conditions", "Subsection"]], "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.748192982991539*^9, 3.748192988325735*^9}, 
   3.74878219724442*^9, 3.7487889523512273`*^9}],

Cell["\<\
The (numerical) constant of integrations can be found from SolveBC[g , k, \
\[Eta],  K, fsolution, boundarycondition]\
\>", "Text",
 CellChangeTimes->{{3.749464917656982*^9, 3.7494649523521833`*^9}, {
  3.749499664803006*^9, 3.7494996687881656`*^9}, {3.7494998008937044`*^9, 
  3.7494998047987394`*^9}, {3.749533861086837*^9, 3.749533864376462*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NumericalConstants", " ", "=", " ", 
  RowBox[{"SolveBC", "[", 
   RowBox[{
   "1.2094508496552547`", ",", "1.3369970510001397`", ",", "3.0", ",", "3.0", 
    ",", "fsolutionhex", ",", " ", "BoundaryConditions"}], "]"}]}]], "Input",
 CellChangeTimes->{{3.749464877041805*^9, 3.749464909994505*^9}, {
   3.749464978518196*^9, 3.7494649824696484`*^9}, 3.749465035451942*^9, {
   3.749465219125839*^9, 3.7494652239728713`*^9}, {3.749533608576769*^9, 
   3.7495336143754697`*^9}, {3.7495338493584123`*^9, 3.7495338569971223`*^9}, 
   3.7495348636338544`*^9}],

Cell[BoxData[
 TemplateBox[{
  "Solve","ratnz",
   "\"Solve was unable to solve the system with inexact coefficients. The \
answer was obtained by solving a corresponding exact system and numericizing \
the result.\"",2,75,4,31106859009397539697,"Local"},
  "MessageTemplate"]], "Message", "MSG",
 CellChangeTimes->{3.7495341557602663`*^9, 3.749535102423122*^9, 
  3.74953617677485*^9, 3.7495367190287247`*^9, 3.7495502029827805`*^9, 
  3.7504456812211294`*^9, 3.7537824875166173`*^9, 3.7537827811736526`*^9, 
  3.753783496049204*^9, 3.753787722725692*^9}],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"L0", "\[Rule]", 
    RowBox[{"-", "3.9968327791067006`*^-14"}]}], ",", 
   RowBox[{
    RowBox[{"A", "[", 
     RowBox[{"1", ",", "1"}], "]"}], "\[Rule]", 
    RowBox[{"-", "1.0041872177388824`"}]}], ",", 
   RowBox[{
    RowBox[{"A", "[", 
     RowBox[{"1", ",", "2"}], "]"}], "\[Rule]", 
    RowBox[{"-", "1.8930124853322294`"}]}], ",", 
   RowBox[{
    RowBox[{"A", "[", 
     RowBox[{"1", ",", "3"}], "]"}], "\[Rule]", "1.705524424958708`"}], ",", 
   RowBox[{
    RowBox[{"A", "[", 
     RowBox[{"1", ",", "4"}], "]"}], "\[Rule]", "2.5043980033701825`"}], ",", 
   
   RowBox[{
    RowBox[{"B", "[", 
     RowBox[{"1", ",", "3"}], "]"}], "\[Rule]", "1.3127227252577784`"}], ",", 
   
   RowBox[{
    RowBox[{"B", "[", 
     RowBox[{"1", ",", "4"}], "]"}], "\[Rule]", 
    RowBox[{"-", "0.7066285079416565`"}]}], ",", 
   RowBox[{"L1", "\[Rule]", 
    RowBox[{"-", "0.09968417506030337`"}]}], ",", 
   RowBox[{
    RowBox[{"A", "[", 
     RowBox[{"2", ",", 
      RowBox[{"-", "1"}]}], "]"}], "\[Rule]", 
    RowBox[{"-", "0.08277839253571262`"}]}], ",", 
   RowBox[{
    RowBox[{"A", "[", 
     RowBox[{"2", ",", "0"}], "]"}], "\[Rule]", "0.7330161794150175`"}], ",", 
   
   RowBox[{
    RowBox[{"A", "[", 
     RowBox[{"2", ",", "1"}], "]"}], "\[Rule]", 
    RowBox[{"-", "0.08252238937526937`"}]}], ",", 
   RowBox[{
    RowBox[{"A", "[", 
     RowBox[{"2", ",", "2"}], "]"}], "\[Rule]", 
    RowBox[{"-", "0.20148745972639534`"}]}], ",", 
   RowBox[{
    RowBox[{"A", "[", 
     RowBox[{"2", ",", "3"}], "]"}], "\[Rule]", "0.13800524543666046`"}], ",", 
   RowBox[{
    RowBox[{"A", "[", 
     RowBox[{"2", ",", "4"}], "]"}], "\[Rule]", 
    RowBox[{"-", "0.20468012065476324`"}]}], ",", 
   RowBox[{
    RowBox[{"A", "[", 
     RowBox[{"2", ",", "5"}], "]"}], "\[Rule]", 
    RowBox[{"-", "0.36981134013489464`"}]}], ",", 
   RowBox[{
    RowBox[{"A", "[", 
     RowBox[{"2", ",", "6"}], "]"}], "\[Rule]", "1.3514747102190223`"}], ",", 
   
   RowBox[{
    RowBox[{"A", "[", 
     RowBox[{"2", ",", "7"}], "]"}], "\[Rule]", "0.06250342016345271`"}], ",", 
   RowBox[{
    RowBox[{"A", "[", 
     RowBox[{"2", ",", "8"}], "]"}], "\[Rule]", 
    RowBox[{"-", "0.7090608546611205`"}]}], ",", 
   RowBox[{
    RowBox[{"A", "[", 
     RowBox[{"2", ",", "9"}], "]"}], "\[Rule]", 
    RowBox[{"-", "0.3861007036247837`"}]}], ",", 
   RowBox[{
    RowBox[{"A", "[", 
     RowBox[{"2", ",", "10"}], "]"}], "\[Rule]", 
    RowBox[{"-", "0.19840119781573143`"}]}], ",", 
   RowBox[{
    RowBox[{"A", "[", 
     RowBox[{"2", ",", "11"}], "]"}], "\[Rule]", "1.3257429051894472`"}], ",", 
   RowBox[{
    RowBox[{"A", "[", 
     RowBox[{"2", ",", "12"}], "]"}], "\[Rule]", 
    RowBox[{"-", "2.8587304453689164`"}]}], ",", 
   RowBox[{
    RowBox[{"B", "[", 
     RowBox[{"2", ",", "3"}], "]"}], "\[Rule]", 
    RowBox[{"-", "0.481710643017666`"}]}], ",", 
   RowBox[{
    RowBox[{"B", "[", 
     RowBox[{"2", ",", "4"}], "]"}], "\[Rule]", "0.3294266745764065`"}], ",", 
   
   RowBox[{
    RowBox[{"B", "[", 
     RowBox[{"2", ",", "7"}], "]"}], "\[Rule]", 
    RowBox[{"-", "0.9992196888575886`"}]}], ",", 
   RowBox[{
    RowBox[{"B", "[", 
     RowBox[{"2", ",", "8"}], "]"}], "\[Rule]", 
    RowBox[{"-", "0.1128287737917317`"}]}], ",", 
   RowBox[{
    RowBox[{"B", "[", 
     RowBox[{"2", ",", "9"}], "]"}], "\[Rule]", 
    RowBox[{"-", "6.624180832457668`"}]}], ",", 
   RowBox[{
    RowBox[{"B", "[", 
     RowBox[{"2", ",", "10"}], "]"}], "\[Rule]", "0.37502070958269834`"}]}], 
  "}"}]], "Output",
 CellChangeTimes->{3.7537834961269903`*^9, 3.753787722795615*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell[" For example:", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.7482608804202003`*^9, 3.748260931153734*^9}, {
   3.748260970268648*^9, 3.748260985377118*^9}, 3.748782197246414*^9, 
   3.7487889523532233`*^9, 3.7494649606200695`*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"A", "[", 
   RowBox[{"1", ",", "1"}], "]"}], "/.", "NumericalConstants"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.6657713232344956`*^9, 3.665771343381418*^9}, {
   3.665771418266056*^9, 3.665771443269935*^9}, {3.6657714854748755`*^9, 
   3.6657715331802435`*^9}, {3.665771716847458*^9, 3.665771720446126*^9}, {
   3.6657718139064054`*^9, 3.6657718741793876`*^9}, {3.665772091294901*^9, 
   3.665772215307926*^9}, {3.665772247964039*^9, 3.665772263610958*^9}, {
   3.665772485446594*^9, 3.6657725184863443`*^9}, 3.6657729166750526`*^9, {
   3.66577426017365*^9, 3.6657742603906803`*^9}, 3.6658296584374185`*^9, {
   3.6672460521798906`*^9, 3.667246110757153*^9}, 3.667301860838393*^9, {
   3.7058179970092716`*^9, 3.705818048247526*^9}, {3.705818285589779*^9, 
   3.7058183237035446`*^9}, 3.705818449364562*^9, {3.71612068835028*^9, 
   3.7161206885199823`*^9}, 3.7161210204739447`*^9, 3.7345160841540394`*^9, 
   3.748244002237558*^9, 3.74824513592642*^9, 3.748245199162204*^9, {
   3.748260944110375*^9, 3.7482609500335903`*^9}, 3.7487821972473836`*^9, 
   3.748788952354219*^9, {3.748798416883297*^9, 3.7487984172782364`*^9}, {
   3.7494649692280655`*^9, 3.749464985695998*^9}, 3.749465036687632*^9},
 ExpressionUUID -> "56930096-319d-4d45-b5d5-c082ff58afa2"]
}, Open  ]],

Cell[BoxData[
 RowBox[{"-", "1.0041872177388824`"}]], "Output",
 CellChangeTimes->{3.753783496184282*^9, 3.75378772284651*^9}],

Cell[CellGroupData[{

Cell["\<\
Finally, we expand the boundary conditions to third order in epsilon, \
substitute in the fields and the constants and evaluate at the test point, to \
show that they are satisfied up to machine precision. The Chop command rounds \
numbers smaller than 10^-30 to zero.\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.7482613237444077`*^9, 3.748261354007496*^9}, {
   3.748261408839826*^9, 3.748261423343892*^9}, {3.748367234219556*^9, 
   3.748367248653973*^9}, 3.7487821972643356`*^9, 3.7487889523552155`*^9}],

Cell[BoxData[
 RowBox[{"Chop", "[", 
  RowBox[{
   RowBox[{"FullSimplify", "[", 
    RowBox[{"TrigSimplify", "[", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"Series", "[", 
          RowBox[{"BoundaryConditions", ",", 
           RowBox[{"{", 
            RowBox[{"\[CurlyEpsilon]", ",", "0", ",", "2"}], "}"}]}], "]"}], "/.",
          "fsolutionhex"}], "/.", "NumericalConstants"}], "/.", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"z0", "\[Rule]", "0"}], ",", 
         RowBox[{"z1", "\[Rule]", "1"}]}], "}"}]}], "/.", "TestPoint"}], 
     "]"}], "]"}], ",", 
   RowBox[{"10", "^", 
    RowBox[{"-", "10"}]}]}], "]"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.6657713232344956`*^9, 3.665771343381418*^9}, {
   3.665771418266056*^9, 3.665771443269935*^9}, {3.6657714854748755`*^9, 
   3.6657715331802435`*^9}, {3.665771716847458*^9, 3.665771720446126*^9}, {
   3.6657718139064054`*^9, 3.6657718741793876`*^9}, {3.665772091294901*^9, 
   3.665772215307926*^9}, {3.665772247964039*^9, 3.665772263610958*^9}, {
   3.665772485446594*^9, 3.6657725184863443`*^9}, 3.6657729166750526`*^9, {
   3.66577426017365*^9, 3.6657742603906803`*^9}, 3.6658296584374185`*^9, {
   3.6672460521798906`*^9, 3.667246110757153*^9}, 3.667301860838393*^9, {
   3.7058179970092716`*^9, 3.705818048247526*^9}, {3.705818285589779*^9, 
   3.7058183237035446`*^9}, 3.705818449364562*^9, {3.71612068835028*^9, 
   3.7161206885199823`*^9}, 3.7161210204739447`*^9, 3.7345160841540394`*^9, 
   3.748244002237558*^9, 3.74824513592642*^9, 3.748245199162204*^9, {
   3.748260944110375*^9, 3.7482609500335903`*^9}, 3.7482611528669987`*^9, {
   3.7482612607979317`*^9, 3.748261267619701*^9}, {3.7482613798230553`*^9, 
   3.7482613948351583`*^9}, {3.7482615037390327`*^9, 3.7482615540063*^9}, 
   3.748363049102429*^9, {3.7483671689589977`*^9, 3.7483671715347033`*^9}, {
   3.748367220011492*^9, 3.7483672221377087`*^9}, {3.7483675709932613`*^9, 
   3.7483675787994757`*^9}, 3.748782197265361*^9, 3.7487889523552155`*^9, {
   3.748957483238025*^9, 3.748957485439419*^9}, {3.7489712037171593`*^9, 
   3.748971227725083*^9}, {3.748971709738352*^9, 3.7489717134753647`*^9}, {
   3.74897213241098*^9, 3.7489721437291574`*^9}, {3.749465061272871*^9, 
   3.7494650839880943`*^9}, 3.7494652295569277`*^9, {3.7495366912851496`*^9, 
   3.7495367149312677`*^9}, {3.7495397662899237`*^9, 
   3.7495397888145285`*^9}, {3.7495508757081647`*^9, 3.749550891786277*^9}, 
   3.7504453876730423`*^9},
 ExpressionUUID -> "56930096-319d-4d45-b5d5-c082ff58afa2"]
}, Open  ]],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "3"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 3, 3, 1],
    Editable->False], ",", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "3"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 3, 3, 1],
    Editable->False], ",", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "3"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 3, 3, 1],
    Editable->False], ",", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "3"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 3, 3, 1],
    Editable->False], ",", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "3"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 3, 3, 1],
    Editable->False], ",", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "3"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 3, 3, 1],
    Editable->False], ",", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "3"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 3, 3, 1],
    Editable->False], ",", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "3"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 3, 3, 1],
    Editable->False], ",", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "3"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 3, 3, 1],
    Editable->False], ",", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "\[CurlyEpsilon]", "]"}], "3"],
    SeriesData[$CellContext`\[CurlyEpsilon], 0, {}, 3, 3, 1],
    Editable->False]}], "}"}]], "Output",
 CellChangeTimes->{3.749539699154375*^9, 3.7495398870725*^9, 
  3.749550845441818*^9, 3.749550985488022*^9, 3.7504459004840508`*^9, 
  3.7537835722527275`*^9, 3.7537878030640683`*^9}],

Cell["", "Text",
 CellChangeTimes->{{3.7489724785133915`*^9, 3.7489724827179384`*^9}}],

Cell[CellGroupData[{

Cell[TextData[StyleBox["Energy Thickness integration", "Subsection"]], "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.748262074419416*^9, 3.74826208137881*^9}, 
   3.7487821972683554`*^9, 3.7487889523552155`*^9, {3.7494652486478434`*^9, 
   3.749465249497573*^9}}],

Cell["\<\
We finally integrate the averaged energies over thickness to find the energy \
coefficients as defined in the paper numerically\
\>", "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.7482620881141*^9, 3.748262101954483*^9}, {
   3.748363346970169*^9, 3.748363359624626*^9}, {3.748365214909356*^9, 
   3.748365240840098*^9}, {3.74836617852466*^9, 3.748366185813534*^9}, 
   3.7487821972683554`*^9, 3.7487889523552155`*^9, {3.7494652526222353`*^9, 
   3.7494652615543437`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"E2", "=", 
   RowBox[{
    RowBox[{"NIntegrate", "[", 
     RowBox[{
      RowBox[{"ReleaseHold", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"W2av", "/.", "fsolutionhex"}], "/.", "NumericalConstants"}],
         "/.", "TestPoint"}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"z", ",", "0", ",", "1"}], "}"}]}], "]"}], "+", 
    RowBox[{"NIntegrate", "[", 
     RowBox[{
      RowBox[{"ReleaseHold", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"W2Sav", "/.", "fsolutionhex"}], "/.", 
         "NumericalConstants"}], "/.", "TestPoint"}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"z", ",", 
        RowBox[{"-", "\[Infinity]"}], ",", "0"}], "}"}]}], "]"}]}]}], 
  "\[IndentingNewLine]"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.6657713232344956`*^9, 3.665771343381418*^9}, {
   3.665771418266056*^9, 3.665771443269935*^9}, {3.6657714854748755`*^9, 
   3.6657715331802435`*^9}, {3.665771716847458*^9, 3.665771720446126*^9}, {
   3.6657718139064054`*^9, 3.6657718741793876`*^9}, {3.665772091294901*^9, 
   3.665772215307926*^9}, {3.665772247964039*^9, 3.665772263610958*^9}, {
   3.665772485446594*^9, 3.6657725184863443`*^9}, 3.6657729166750526`*^9, {
   3.66577426017365*^9, 3.6657742603906803`*^9}, 3.6658296584374185`*^9, {
   3.6672460521798906`*^9, 3.667246110757153*^9}, 3.667301860838393*^9, {
   3.7058179970092716`*^9, 3.705818048247526*^9}, {3.705818285589779*^9, 
   3.7058183237035446`*^9}, 3.705818449364562*^9, {3.71612068835028*^9, 
   3.7161206885199823`*^9}, 3.7161210204739447`*^9, 3.7345160841540394`*^9, 
   3.748244002237558*^9, 3.74824513592642*^9, 3.748245199162204*^9, {
   3.748260944110375*^9, 3.7482609500335903`*^9}, 3.7482611528669987`*^9, {
   3.7482612607979317`*^9, 3.748261267619701*^9}, {3.7482613798230553`*^9, 
   3.7482613948351583`*^9}, {3.7482615037390327`*^9, 3.7482615540063*^9}, {
   3.748261810434952*^9, 3.748261852402668*^9}, {3.748261923192767*^9, 
   3.748261997419652*^9}, {3.748262036893588*^9, 3.7482620405479507`*^9}, {
   3.748262113533332*^9, 3.748262252817123*^9}, {3.748262305237101*^9, 
   3.748262310340988*^9}, {3.748262365269137*^9, 3.7482623722651787`*^9}, {
   3.7483633671292267`*^9, 3.748363383150691*^9}, {3.7483635959244337`*^9, 
   3.7483636870930433`*^9}, {3.748364333478301*^9, 3.748364334039871*^9}, {
   3.748365181454217*^9, 3.748365190982555*^9}, {3.748366111342016*^9, 
   3.7483661577719717`*^9}, 3.7487821972713184`*^9, 3.7487889523562126`*^9, {
   3.748955142243904*^9, 3.7489551504988284`*^9}, {3.749465105268177*^9, 
   3.7494651776846747`*^9}, 3.7494652666537066`*^9, {3.7495365575540504`*^9, 
   3.749536573912321*^9}, {3.7495395010037374`*^9, 3.7495395044206038`*^9}},
 ExpressionUUID -> "56930096-319d-4d45-b5d5-c082ff58afa2"]
}, Open  ]],

Cell[BoxData["1.056932319443149`*^-13"], "Output",
 CellChangeTimes->{3.7495511969949627`*^9, 3.7504459007280416`*^9, 
  3.7537835723734035`*^9, 3.753787803184747*^9}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"E3", "=", 
  RowBox[{
   RowBox[{"NIntegrate", "[", 
    RowBox[{
     RowBox[{"ReleaseHold", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"W3av", "/.", "fsolutionhex"}], "/.", "NumericalConstants"}], 
       "/.", "TestPoint"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"z", ",", "0", ",", "1"}], "}"}]}], "]"}], "+", 
   RowBox[{"NIntegrate", "[", 
    RowBox[{
     RowBox[{"ReleaseHold", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"W3Sav", "/.", "fsolutionhex"}], "/.", "NumericalConstants"}],
        "/.", "TestPoint"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"z", ",", 
       RowBox[{"-", "\[Infinity]"}], ",", "0"}], "}"}]}], "]"}]}]}]], "Input",\

 CellChangeTimes->{{3.7494652717979245`*^9, 3.74946528046175*^9}, {
   3.749536562207611*^9, 3.7495365750024056`*^9}, {3.7495395019841423`*^9, 
   3.749539506051243*^9}, 3.7495396423628473`*^9}],

Cell[BoxData["0.06645611670680018`"], "Output",
 CellChangeTimes->{3.7495511974363017`*^9, 3.7504459012541337`*^9, 
  3.7537835726666183`*^9, 3.7537878035359373`*^9}]
}, Open  ]],

Cell[BoxData[""], "Input"],

Cell["\<\
For Fig. 11, we repeat the process (CharacterizeInstabilityCompressible \
\[Rule] GenerateSolution \[Rule] SolveBC \[Rule] evaluate E3) using threshold \
parameters g=g* and k=k* for different values of K for TestPoint.\
\>", "Text",
 CellChangeTimes->{{3.749467238624465*^9, 3.749467271391466*^9}, {
  3.749536590969722*^9, 3.749536601480633*^9}, {3.7495395319759407`*^9, 
  3.7495396387884045`*^9}, {3.7495518907422442`*^9, 3.749551898545662*^9}, {
  3.7498162956341305`*^9, 3.7498163018995657`*^9}}],

Cell["", "Text",
 CellChangeTimes->{{3.7537851239383397`*^9, 3.7537851246843433`*^9}}],

Cell[TextData[{
 "Using ",
 StyleBox["CharacterizeInstabilityCompressible[30,5]", "Input"],
 " gives the threshold parameter of: (The function would take a long time to \
run but you can try to run the function to confirm the answer)"
}], "Text",
 CellChangeTimes->{{3.7537851179215374`*^9, 3.7537852570622*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{
    RowBox[{"g", "\[Rule]", "1.0407081123823319`"}], ",", 
    RowBox[{"k", "\[Rule]", "0.48513178949766383`"}], ",", 
    RowBox[{"\[Mu]2", "\[Rule]", "1"}], ",", 
    RowBox[{"\[Mu]1", "\[Rule]", "30"}], ",", 
    RowBox[{"K", "\[Rule]", "5"}], ",", 
    RowBox[{"\[Gamma]", "\[Rule]", "0.9736804467665771`"}]}], "}"}], ";"}]], \
"Input",
 CellChangeTimes->{{3.75378522233211*^9, 3.753785222338084*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"TestPoint", "=", " ", 
   RowBox[{"SetPrecision", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"g", "\[Rule]", "1.0407081123823319`"}], ",", 
       RowBox[{"k", "\[Rule]", "0.48513178949766383`"}], ",", 
       RowBox[{"\[Mu]2", "\[Rule]", "1"}], ",", 
       RowBox[{"\[Mu]1", "\[Rule]", " ", "30"}], ",", 
       RowBox[{"K", "\[Rule]", "5"}], ",", 
       RowBox[{"\[Gamma]", "\[Rule]", "0.9736804467665771`"}]}], "}"}], ",", 
     "50"}], "]"}]}], ";"}]], "Input",
 CellChangeTimes->{{3.7494604767939544`*^9, 3.749460543489599*^9}, 
   3.7494650044129553`*^9, {3.749465194039904*^9, 3.7494652040421505`*^9}, {
   3.749498028850665*^9, 3.749498060353456*^9}, {3.7504460245122566`*^9, 
   3.750446026349506*^9}, {3.750446490674841*^9, 3.750446496993677*^9}, {
   3.7504466614864016`*^9, 3.75044666492021*^9}, {3.7504504708827276`*^9, 
   3.7504504917578*^9}, {3.7504505283246126`*^9, 3.7504506465122075`*^9}, 
   3.750450722512328*^9, {3.7504509899524803`*^9, 3.750451007727662*^9}, {
   3.750451273049923*^9, 3.750451282311567*^9}, {3.7504515571077867`*^9, 
   3.750451567554369*^9}, {3.7504526997492304`*^9, 3.750452707519641*^9}, {
   3.7504530720821605`*^9, 3.7504530825606503`*^9}, {3.7504534520638*^9, 
   3.7504534597469516`*^9}, {3.7504538895188427`*^9, 
   3.7504538973830757`*^9}, {3.7504545047947493`*^9, 3.750454517251188*^9}, {
   3.750454594329418*^9, 3.750454609326579*^9}, {3.7504553561165023`*^9, 
   3.7504553666151576`*^9}, {3.7504557828360662`*^9, 
   3.7504557898156214`*^9}, {3.750456068275735*^9, 3.750456086604687*^9}, {
   3.750456341340436*^9, 3.7504563576837587`*^9}, {3.750488397532141*^9, 
   3.7504884075224266`*^9}, {3.750488655703679*^9, 3.750488664088256*^9}, {
   3.7504890655986032`*^9, 3.750489075231956*^9}, {3.7504895337020884`*^9, 
   3.7504895454130373`*^9}, {3.750489890149493*^9, 3.750489899632507*^9}, {
   3.750490308518818*^9, 3.7504903181888905`*^9}, {3.7504906014350734`*^9, 
   3.7504906104745255`*^9}, {3.750490937024991*^9, 3.7504909448547363`*^9}, {
   3.7504912024233665`*^9, 3.7504912138138714`*^9}, {3.7504915516753454`*^9, 
   3.750491559901348*^9}, {3.7504958129423957`*^9, 3.750495821955593*^9}, {
   3.750496108956829*^9, 3.750496117190778*^9}, {3.7504964613103666`*^9, 
   3.750496472561288*^9}, {3.750501928240241*^9, 3.7505019413571615`*^9}, {
   3.7505022272873597`*^9, 3.750502235155319*^9}, {3.750502495456785*^9, 
   3.7505025039649987`*^9}, {3.7505031145943336`*^9, 3.750503120129531*^9}, {
   3.75050347919663*^9, 3.7505034864991283`*^9}, {3.7505744000640473`*^9, 
   3.750574424997726*^9}, 3.750574603508723*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"fsolutionhex", "=", 
   RowBox[{"GenerateSolution", "[", 
    RowBox[{
     RowBox[{"g", "/.", "TestPoint"}], ",", 
     RowBox[{"k", "/.", "TestPoint"}], ",", 
     RowBox[{"\[Mu]1", "/.", "TestPoint"}], ",", 
     RowBox[{"K", "/.", "TestPoint"}]}], "]"}]}], ";"}]], "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.},
 CellChangeTimes->{{3.748244164508481*^9, 3.748244176032332*^9}, 
   3.748265242373149*^9, {3.748371437005432*^9, 3.74837143926686*^9}, 
   3.748782197235414*^9, 3.7487889523492317`*^9, {3.7487912558986964`*^9, 
   3.748791272273771*^9}, {3.748793049074587*^9, 3.748793050045586*^9}, {
   3.749458864936097*^9, 3.7494588680068793`*^9}, 3.7494604086083565`*^9, 
   3.74946054693436*^9, {3.74946521254042*^9, 3.7494652148133698`*^9}, {
   3.7494980749285436`*^9, 3.74949809780186*^9}, {3.749499343966523*^9, 
   3.749499349032048*^9}, 3.7504486100206523`*^9, 3.7504507212000103`*^9}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NumericalConstants", " ", "=", " ", 
   RowBox[{"SolveBC", "[", 
    RowBox[{
     RowBox[{"g", "/.", "TestPoint"}], ",", 
     RowBox[{"k", "/.", "TestPoint"}], ",", 
     RowBox[{"\[Mu]1", "/.", "TestPoint"}], ",", 
     RowBox[{"K", "/.", "TestPoint"}], ",", "fsolutionhex", ",", " ", 
     "BoundaryConditions"}], "]"}]}], ";"}]], "Input",
 CellChangeTimes->{{3.749464877041805*^9, 3.749464909994505*^9}, {
   3.749464978518196*^9, 3.7494649824696484`*^9}, 3.749465035451942*^9, {
   3.749465219125839*^9, 3.7494652239728713`*^9}, {3.749533608576769*^9, 
   3.7495336143754697`*^9}, {3.7495338493584123`*^9, 3.7495338569971223`*^9}, 
   3.7495348636338544`*^9, {3.7504436424409685`*^9, 3.750443673867797*^9}, 
   3.7504474564903717`*^9}],

Cell[BoxData[
 TemplateBox[{
  "Solve","ratnz",
   "\"Solve was unable to solve the system with inexact coefficients. The \
answer was obtained by solving a corresponding exact system and numericizing \
the result.\"",2,83,5,31106859009397539697,"Local"},
  "MessageTemplate"]], "Message", "MSG",
 CellChangeTimes->{3.750448408641927*^9, 3.750450362512712*^9, 
  3.7504508828239164`*^9, 3.75045124557016*^9, 3.750451526842945*^9, 
  3.7504518977494235`*^9, 3.750453031460256*^9, 3.7504534120106726`*^9, 
  3.750453795520279*^9, 3.7504542198522987`*^9, 3.7504548387156363`*^9, 
  3.7504553032639375`*^9, 3.750455642425175*^9, 3.750456031754525*^9, 
  3.7504562858289223`*^9, 3.7504565585625095`*^9, 3.750488611964878*^9, 
  3.750488890041381*^9, 3.750489313246872*^9, 3.7504897984181786`*^9, 
  3.7504901296937304`*^9, 3.750490563480418*^9, 3.7504908518499804`*^9, 
  3.750491157369735*^9, 3.7504914440980134`*^9, 3.750491776346467*^9, 
  3.7504960439172845`*^9, 3.7504963342737937`*^9, 3.750496677236884*^9, 
  3.7505021440351324`*^9, 3.7505024617392864`*^9, 3.750502744866621*^9, 
  3.7505033342911096`*^9, 3.7505037184081135`*^9, 3.753788073782222*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"E3", "=", 
  RowBox[{
   RowBox[{"NIntegrate", "[", 
    RowBox[{
     RowBox[{"ReleaseHold", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"W3av", "/.", "fsolutionhex"}], "/.", "NumericalConstants"}], 
       "/.", "TestPoint"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"z", ",", "0", ",", "1"}], "}"}]}], "]"}], "+", 
   RowBox[{"NIntegrate", "[", 
    RowBox[{
     RowBox[{"ReleaseHold", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"W3Sav", "/.", "fsolutionhex"}], "/.", "NumericalConstants"}],
        "/.", "TestPoint"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"z", ",", 
       RowBox[{"-", "\[Infinity]"}], ",", "0"}], "}"}]}], "]"}]}]}]], "Input",\

 CellChangeTimes->{{3.7494652717979245`*^9, 3.74946528046175*^9}, {
   3.749536562207611*^9, 3.7495365750024056`*^9}, {3.7495395019841423`*^9, 
   3.749539506051243*^9}, 3.7495396423628473`*^9}],

Cell[BoxData["0.014287760866800281`"], "Output",
 CellChangeTimes->{3.7495511974363017`*^9, 3.7504484094032583`*^9, 
  3.7504503628251677`*^9, 3.750450883195777*^9, 3.7504512459312506`*^9, 
  3.750451527291432*^9, 3.750451898369179*^9, 3.7504530319968295`*^9, 
  3.7504534126449757`*^9, 3.7504537961694126`*^9, 3.75045422049362*^9, 
  3.7504548393122263`*^9, 3.7504553038463964`*^9, 3.750455642756158*^9, 
  3.7504560321811547`*^9, 3.7504562861650267`*^9, 3.7504565588996363`*^9, 
  3.750488612305931*^9, 3.750488890393423*^9, 3.7504893137207747`*^9, 
  3.750489798854039*^9, 3.7504901300447903`*^9, 3.7504905638640485`*^9, 
  3.7504908522519894`*^9, 3.7504911577188005`*^9, 3.7504914444570837`*^9, 
  3.7504917767514315`*^9, 3.750496044274355*^9, 3.7504963346617837`*^9, 
  3.7504966776268864`*^9, 3.7505021444111357`*^9, 3.750502462149222*^9, 
  3.7505027452456055`*^9, 3.7505033346571293`*^9, 3.7505037188698792`*^9, 
  3.7537880741998806`*^9}]
}, Open  ]]
},
Evaluator->"Local",
WindowSize->{1520, 781},
WindowMargins->{{-8, Automatic}, {Automatic, 0}},
FrontEndVersion->"11.0 for Microsoft Windows (64-bit) (July 28, 2016)",
StyleDefinitions->FrontEnd`FileName[{"Article"}, "JournalArticle.nb", 
  CharacterEncoding -> "UTF-8"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[558, 20, 702, 13, 90, "Input"],
Cell[1263, 35, 184, 3, 43, "Text"],
Cell[1450, 40, 140, 1, 31, "Text"],
Cell[1593, 43, 534, 9, 41, "Text"],
Cell[2130, 54, 736, 11, 41, "Text"],
Cell[2869, 67, 304, 6, 23, "Text"],
Cell[3176, 75, 16, 0, 23, "Text"],
Cell[CellGroupData[{
Cell[3217, 79, 214, 3, 31, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.}],
Cell[3434, 84, 376, 7, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.}],
Cell[3813, 93, 277, 4, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.}],
Cell[4093, 99, 548, 9, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.}],
Cell[4644, 110, 627, 12, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.}],
Cell[5274, 124, 651, 11, 41, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.}],
Cell[5928, 137, 408, 7, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.}],
Cell[6339, 146, 833, 17, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[7209, 168, 378, 5, 31, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[7590, 175, 317, 6, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[7910, 183, 583, 15, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[8496, 200, 887, 13, 41, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[9386, 215, 867, 19, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[10256, 236, 220, 3, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[10479, 241, 527, 12, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[11009, 255, 383, 7, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[11395, 264, 2017, 33, 29, "Output",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[13415, 299, 222, 3, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[13640, 304, 494, 14, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[14137, 320, 234, 5, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[14374, 327, 208, 4, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[14585, 333, 4677, 119, 83, "Output",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[19265, 454, 253, 3, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[19521, 459, 400, 10, 56, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[19924, 471, 284, 5, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[20211, 478, 254, 4, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[20468, 484, 2441, 46, 87, "Output",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[22912, 532, 463, 8, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[23378, 542, 513, 12, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[23894, 556, 799, 21, 55, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[24696, 579, 193, 3, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[24892, 584, 378, 7, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[25273, 593, 830, 21, 29, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}]
}, Open  ]],
Cell[26118, 617, 286, 4, 29, "Input"],
Cell[CellGroupData[{
Cell[26429, 625, 217, 3, 31, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.}],
Cell[26649, 630, 211, 3, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.}],
Cell[26863, 635, 1424, 34, 76, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.}],
Cell[28290, 671, 779, 12, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.}],
Cell[29072, 685, 1451, 40, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.}],
Cell[30526, 727, 1501, 41, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.}],
Cell[32030, 770, 1349, 40, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.}],
Cell[33382, 812, 528, 12, 48, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.}],
Cell[33913, 826, 260, 6, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.}],
Cell[34176, 834, 388, 8, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.}],
Cell[34567, 844, 4437, 104, 106, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.}],
Cell[39007, 950, 736, 14, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10000.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[39780, 969, 214, 3, 31, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[39997, 974, 274, 5, 26, "Subsubsection",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[40274, 981, 515, 9, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[40792, 992, 1597, 30, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[42392, 1024, 1933, 32, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[44328, 1058, 1047, 17, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[45378, 1077, 248, 4, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[45629, 1083, 6415, 92, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[52047, 1177, 4891, 72, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[56941, 1251, 3840, 57, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[60784, 1310, 684, 12, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[61471, 1324, 535, 7, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}]
}, Open  ]],
Cell[62021, 1334, 1406, 36, 59, "Input"],
Cell[CellGroupData[{
Cell[63452, 1374, 213, 3, 27, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[63668, 1379, 256, 6, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[63927, 1387, 784, 15, 31, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[64714, 1404, 241, 4, 27, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[64958, 1410, 281, 6, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[65242, 1418, 2148, 42, 48, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[67393, 1462, 213, 3, 27, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[67609, 1467, 619, 10, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[68231, 1479, 434, 12, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[68668, 1493, 284, 5, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[68955, 1500, 2727, 59, 84, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[71685, 1561, 362, 5, 27, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[72050, 1568, 382, 7, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[72435, 1577, 1149, 20, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}]
}, Open  ]],
Cell[73599, 1600, 144, 1, 23, "Text"],
Cell[73746, 1603, 9671, 248, 228, "Input"],
Cell[83420, 1853, 188, 4, 23, "Text"],
Cell[CellGroupData[{
Cell[83633, 1861, 232, 4, 29, "Input"],
Cell[83868, 1867, 1128, 27, 35, "Output"]
}, Open  ]],
Cell[85011, 1897, 86, 1, 23, "Text"],
Cell[CellGroupData[{
Cell[85122, 1902, 256, 4, 27, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[85381, 1908, 567, 10, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[85951, 1920, 250, 4, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[86204, 1926, 1232, 20, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[87439, 1948, 43521, 932, 458, "Output",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[130963, 2882, 318, 6, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[131284, 2890, 390, 7, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[131677, 2899, 2057, 36, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[133737, 2937, 1255, 25, 66, "Output",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}]
}, Open  ]],
Cell[135007, 2965, 170, 3, 23, "Text"],
Cell[CellGroupData[{
Cell[135202, 2972, 126, 2, 29, "Input"],
Cell[135331, 2976, 1510, 42, 66, "Output"]
}, Open  ]],
Cell[CellGroupData[{
Cell[136878, 3023, 616, 10, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[137497, 3035, 2087, 36, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}]
}, Open  ]],
Cell[139599, 3074, 1627, 41, 35, "Output"],
Cell[141229, 3117, 357, 7, 23, "Text"],
Cell[141589, 3126, 190, 2, 27, "Text"],
Cell[141782, 3130, 382, 6, 23, "Text"],
Cell[CellGroupData[{
Cell[142189, 3140, 1879, 31, 29, "Input"],
Cell[144071, 3173, 41072, 865, 368, "Output"]
}, Open  ]],
Cell[185158, 4041, 834, 13, 41, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[185995, 4056, 528, 10, 29, "Input"],
Cell[186526, 4068, 131, 1, 23, "Text"],
Cell[CellGroupData[{
Cell[186682, 4073, 2212, 36, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[188897, 4111, 282, 5, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[189182, 4118, 1733, 26, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[190918, 4146, 500, 8, 31, "Output",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}],
Cell[191421, 4156, 558, 9, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10001.}]
}, Open  ]],
Cell[CellGroupData[{
Cell[192016, 4170, 1636, 34, 48, "Input"],
Cell[193655, 4206, 116, 1, 31, "Output"]
}, Open  ]],
Cell[CellGroupData[{
Cell[193808, 4212, 161, 2, 31, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[193972, 4216, 272, 5, 26, "Subsubsection",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[194247, 4223, 568, 10, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[194818, 4235, 4197, 106, 157, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[199018, 4343, 4416, 108, 152, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[203437, 4453, 3558, 93, 127, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[206998, 4548, 268, 4, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[207269, 4554, 6411, 92, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[213683, 4648, 4887, 72, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[218573, 4722, 3834, 57, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[222410, 4781, 233, 3, 27, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[222646, 4786, 276, 6, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[222925, 4794, 778, 15, 31, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[223706, 4811, 263, 4, 27, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[223972, 4817, 303, 6, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[224278, 4825, 2307, 48, 48, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[226588, 4875, 233, 3, 27, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[226824, 4880, 639, 10, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[227466, 4892, 543, 15, 43, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[228012, 4909, 439, 8, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[228454, 4919, 3287, 76, 102, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[231744, 4997, 323, 5, 27, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[232070, 5004, 485, 9, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[232558, 5015, 1227, 21, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[233788, 5038, 1271, 19, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[235062, 5059, 1167, 17, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}]
}, Open  ]],
Cell[236244, 5079, 709, 16, 29, "Input"],
Cell[236956, 5097, 901, 17, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[237860, 5116, 386, 8, 21, "Message"],
Cell[238249, 5126, 386, 8, 21, "Message"],
Cell[238638, 5136, 98, 1, 23, "Text"],
Cell[CellGroupData[{
Cell[238761, 5141, 266, 7, 29, "Input"],
Cell[239030, 5150, 1420, 36, 31, "Output"]
}, Open  ]],
Cell[240465, 5189, 878, 14, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[CellGroupData[{
Cell[241368, 5207, 367, 10, 29, "Input"],
Cell[241738, 5219, 1292, 34, 35, "Output"]
}, Open  ]],
Cell[243045, 5256, 82, 1, 23, "Text"],
Cell[243130, 5259, 137, 2, 23, "Text"],
Cell[243270, 5263, 279, 4, 27, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[243552, 5269, 359, 6, 23, "Text"],
Cell[CellGroupData[{
Cell[243936, 5279, 578, 10, 29, "Input"],
Cell[244517, 5291, 556, 10, 21, "Message"],
Cell[245076, 5303, 3622, 108, 75, "Output"]
}, Open  ]],
Cell[CellGroupData[{
Cell[248735, 5416, 276, 4, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[249014, 5422, 1344, 20, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}]
}, Open  ]],
Cell[250373, 5445, 126, 2, 29, "Output"],
Cell[CellGroupData[{
Cell[250524, 5451, 563, 9, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[251090, 5462, 2599, 47, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}]
}, Open  ]],
Cell[253704, 5512, 2107, 55, 35, "Output"],
Cell[255814, 5569, 86, 1, 23, "Text"],
Cell[CellGroupData[{
Cell[255925, 5574, 302, 4, 27, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[256230, 5580, 521, 9, 23, "Text",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[256754, 5591, 2826, 51, 48, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}]
}, Open  ]],
Cell[259595, 5645, 167, 2, 31, "Output"],
Cell[CellGroupData[{
Cell[259787, 5651, 906, 25, 29, "Input"],
Cell[260696, 5678, 166, 2, 29, "Output"]
}, Open  ]],
Cell[260877, 5683, 26, 0, 29, "Input"],
Cell[260906, 5685, 512, 8, 23, "Text"],
Cell[261421, 5695, 86, 1, 23, "Text"],
Cell[261510, 5698, 312, 6, 23, "Text"],
Cell[261825, 5706, 457, 11, 29, "Input"],
Cell[262285, 5719, 2641, 42, 29, "Input"],
Cell[264929, 5763, 949, 17, 29, "Input",
 CellGroupingRules->{"GroupTogetherGrouping", 10002.}],
Cell[CellGroupData[{
Cell[265903, 5784, 776, 15, 29, "Input"],
Cell[266682, 5801, 1154, 18, 21, "Message"]
}, Open  ]],
Cell[CellGroupData[{
Cell[267873, 5824, 906, 25, 29, "Input"],
Cell[268782, 5851, 947, 13, 29, "Output"]
}, Open  ]]
}
]
*)

