import numpy as np
import matplotlib.pyplot as plt
import re
import os
import sympy
import math
import sys

def read_energies(filename):
    """
    Read the energies of the gaussian output - returns a list with all excitation energies in J
    """
    with open(filename,"r") as f:
        content = f.read()

    split_content = content.split('\n')
    Excited = re.compile(" Excited State",re.MULTILINE)
    
    energies = []
    eV_to_J = 1.60217662*10**(-19) #J/eV
    for i,line in enumerate(split_content):
        if re.match(" SCF Done:",line):
            scf_energy = float(line.split()[4])*27.212
            #print(scf_energy)
            energies.append(scf_energy)
        
        if re.match(Excited,line):
            ex_energy = float(line.split()[4])

            energies.append(ex_energy)
            
    zeropoint = energies[0]
    energies[1:] = [zeropoint+ex_energy for ex_energy in energies[1:]]

    return np.array(energies)    

def read_normal_modes(filename):
    """
    Read the normal mode diplacement matrices from the optimized + freqeuency calculation
    """
    with open(filename,"r") as f:
        content = f.read()
    
    split_content = content.split('\n')
    
    atom_num = re.compile(" NAtoms= ",re.MULTILINE)

    for i,line in enumerate(split_content):
        if re.match(" NAtoms= ",line):
            numatoms = line.split()[1]
            break
    
    patt = re.compile("Frequencies", re.MULTILINE)
    
    #read all lines with the word "Frequencies" in the beginning
    freq_lines = []
    for i, line in enumerate(split_content):
        if re.search(patt,line):
            freq_lines.append(i)
            
    normal_modes = []

    for i in freq_lines:
        mode1 = []
        mode2 = []
        mode3 = []
        frequencies = split_content[i].split()[2:] #split the lines with the word "Frequencies" in it
        reduces_massees = split_content[i+1].split()[3:]
        #Identification of the mode eg. mode1, mode2 ..
        mode1.append(float(frequencies[0]))
        mode1.append(float(reduces_massees[0]))
        mode2.append(float(frequencies[1]))
        mode2.append(float(reduces_massees[1]))
        mode3.append(float(frequencies[2]))
        mode3.append(float(reduces_massees[2]))
        #read the lines with the displacement matrix in them
        for j in np.arange(int(i)+5,int(i)+int(numatoms)+5,1):
            line = split_content[j].split()
            mode1.append([float(line[2]),float(line[3]),float(line[4])])
            mode2.append([float(line[5]),float(line[6]),float(line[7])])
            mode3.append([float(line[8]),float(line[9]),float(line[10])])

        #each mode is a separate list in the normal_modes array
        #the modes are flatten for the projection of the forces
        mode1[2:] = np.array(mode1[2:]).flatten()
        mode2[2:] = np.array(mode2[2:]).flatten()
        mode3[2:] = np.array(mode3[2:]).flatten()
        

        normal_modes.append(mode1)
        normal_modes.append(mode2)
        normal_modes.append(mode3)
    
    return np.array(normal_modes)

def read_forces(filename):
    """
    Read the force matrix from the excited state calculation
    """
    with open(filename,"r") as f:
        content = f.read()
    
    split_content = content.split('\n')

    for i,line in enumerate(split_content):
        if re.match(" NAtoms= ",line):
            numatoms = line.split()[1]
            break
            
    #\s represents a undefined number of whitespaces
    patt = re.compile("Forces\s\(Hartrees\/Bohr\)",re.MULTILINE)
    
    for i, line in enumerate(split_content):
        if re.search(patt,line):
            force_line = i
    #the force matrix is written with two header lines 
    force_matrix = np.zeros((int(numatoms),3))
   

    for k,j in enumerate(np.arange(int(force_line)+3,int(force_line)+int(numatoms)+3)):
        #first two character of string is center num and atom num
        forces = split_content[j].split()[2:]
        #print(forces)
        force_matrix[k][0] = forces[0]
        force_matrix[k][1] = forces[1]
        force_matrix[k][2] = forces[2]

    #force matrix is flatten to project into normal modes
    return force_matrix.flatten()

def vector_projection(vec_a,vec_b):
    return np.dot(vec_a,vec_b)/np.linalg.norm(vec_b)


# # Read the normal modes of the optimized ground state
normal_modes = read_normal_modes(sys.argv[1])

# # Read the files with forces, these are named with the word "root" in the name
files= [filename for filename in os.listdir(".") if "root" in filename]
files.sort()

# # calculate the gradient $grad = \frac{\vec{force} \cdot \vec{Q_\xi} }{|\vec{Q_\xi}|}$ and the coupling constants

# The derivatives then needs to be multipled with 1/$\omega_\xi$ - the frequency of the investigated normal mode. The normal mode frequencies should be saved from the orginal frequency calculation. 

frequencies = normal_modes[:,0]
masses = normal_modes[:,1]

coupling_constants = []
huang_rhys = []

hbar = 1.054571628*10**(-34) #J*s
c = 299792458*10**2 #cm/s
hartree_to_joule = 4.359744650*10**(-18)#J/hartree
bohr_to_m = 5.2918*10**(-11) #bohr/m
A_to_m = 10**-10

for j in range(len(files)):
    fm = read_forces(files[j])
    for i in range(len(normal_modes)):
        grad = vector_projection(fm,normal_modes[i][2:]) *hartree_to_joule/bohr_to_m

        freq = float(frequencies[i])
        mu = float(masses[i]) * 1.66053904*10**-27 #amu to kg
        g = (1/np.sqrt((2*hbar*mu*(2*np.pi*freq*c)**3)))*grad
        coupling_constants.append(("mode"+str(i+1),"state"+str(j+1),g))
        huang_rhys.append(("mode"+str(i+1),"state"+str(j+1),float(freq),float(0.5*g**2)))

# # Collect the coupling pr. state into spectral densities pr. state (adiabatic)


num_states = len(files) #number of excited states the gradient is calculated for

huang_rhys = np.array(huang_rhys)
hr = huang_rhys[:,3]
hr = np.array(hr)

modes = huang_rhys[:,0]
modes = [int(mode.strip("mode"))-1 for mode in modes]

freq = huang_rhys[:,2]
freq = [float(fr) for fr in freq]
freq = np.array(freq)

coupling_constant = np.array(coupling_constants)
g = np.array(coupling_constant[:,2],dtype=float)

num_normal = len(normal_modes)

#save .csv files with the frequencies in cm^-1 and dimensionless coupling constants
k = 0
for i in range(num_states):
    k += num_normal
    freq_list = freq[i*num_normal:k+1]
    #hr_list = hr[i*num_normal:k+1]
    g_list = g[i*num_normal:k+1]
    
    #use only modes with freqeuncies higher than 200 cm-1
    #hr_plot=[]
    g_plot = []
    freq_plot = []  
    for j in range(len(freq_list)):
        if freq_list[j] > 200:
            freq_plot.append(float(freq_list[j]))
            #hr_plot.append(float(hr_list[j]))
            g_plot.append(g_list[j])

    root = files[i].split("root")[-1].split(".")[0]
    
    with open("coupling_constant" + str(root) + ".csv","w") as f:
        f.write("{} \n".format(",".join(map(str,freq_plot))))
        f.write("{} \n".format(",".join(map(str,g_plot))))
    



