module fitutil_extended
!***************************************************************
!* set of subroutines/functions to be used with vfitdriver.f90 *
!***************************************************************

implicit none
! max_b: maximum number of blocks
! max_p
! max_r
! max_rr
! max_par : maximum number of parameters which can be fitted
!max_theta
integer(4),parameter :: max_b=100,max_p=10,max_r=10,max_rr=70,max_par=50,max_theta=40

type pot
integer(4) :: npa
real(8)    :: a(max_par,max_b)
integer(4) :: ntotal
integer(4) :: nblk
integer(4) :: nangle
integer(4) :: nr
integer(4) :: isym
integer(4) :: npblk(max_b)
real(8)    :: angle(max_b)
real(8)    :: r(max_b)
real(8)    :: rr(max_rr,max_b)
real(8)    :: v(max_rr,max_b)
real(8)    :: pinv(max_theta,max_theta)
integer(4) :: maxpws
integer(4) :: minmps
integer(4) :: maxmps 
integer(4) :: mpsstp
real(8)    :: updmax
real(8)    :: wfac
real(8)    :: ang(max_theta)
real(8)    :: rmol(max_r)
integer(4) :: mld
real(8)    :: re
logical    :: fitflg(max_b)
real(8)    :: rinv(max_r,max_r)
integer(4) :: maxit
real(8)    :: eps
real(8)    :: xr(max_b)
end type pot

type(pot) :: pots(max_p) 

contains

subroutine gues(e1,e2,r1,r2,a0,r0)
implicit none
real(8),intent(in) :: e1,e2,r1,r2
real(8),intent(out) :: a0,r0
real(8) :: a1,a2  
!.....fit exponential exp(-a0*(r-r0)) to two points
a1=log(e1)
a2=log(e2)
a0=(a2-a1)/(r1-r2)
r0=r1+a1/a0
return
end subroutine gues

subroutine matinv(a,det,n,nc)
implicit none
integer(4),intent(in) :: n,nc
real(8),intent(inout) :: a(n,n)
real(8),intent(out) :: det
real(8) :: amax,save
integer(4) :: ik(100),jk(100)
integer(4) :: i,j,k,l

10    det=1.
11    do 100 k=1,nc
amax=0.
21    do 30 i=k,nc
      do 30 j=k,nc
23       if (dabs(amax)-dabs(a(i,j))) 24,24,30
24       amax=a(i,j)
     ik(k)=i
     jk(k)=j
30    continue
31    if (amax) 41,32,41
32    det=0.
    goto 140
41    i=ik(k)
if (i-k) 21,51,43
43    do 50 j=1,nc
save=a(k,j)
a(k,j)=a(i,j)
50       a(i,j)=-save
51    j=jk(k)
if (j-k) 21,61,53
53    do 60 i=1,nc
  save=a(i,k)
  a(i,k)=a(i,j)
60       a(i,j)=-save
61    do 70 i=1,nc
if (i-k) 63,70,63
63       a(i,k)=-a(i,k)/amax
70    continue
71    do 80 i=1,nc
  do 80 j=1,nc
  if (i-k) 74,80,74
74       if (j-k) 75,80,75
75       a(i,j)=a(i,j)+a(i,k)*a(k,j)
80    continue
81    do 90 j=1,nc
if (j-k) 83,90,83
83       a(k,j)=a(k,j)/amax
90    continue
  a(k,k)=1./amax
100   det=det*amax
101   do 130 l=1,nc
   k=nc-l+1
   j=ik(k)
   if (j-k) 111,111,105
105      do 110 i=1,nc
 save=a(i,k)
 a(i,k)=-a(i,j)
110        a(i,j)=save
111      i=jk(k)
  if (i-k) 130,130,113
113      do 120 j=1,nc
   save=a(k,j)
   a(k,j)=-a(i,j)
120         a(i,j)=save
130   continue
140   return
end subroutine matinv

function dlm0(l,m,theta)
implicit none
integer(4),intent(in) :: l,m
real(8),intent(in) :: theta
real(8) :: dlm0
integer(4) :: i,l_
real(8) :: x,y,thedeg,pm1,pm2,pp,rat,ai,al,al2
real(8),parameter :: pi=3.1415926535897932d0
!
!...function to calculate dlm0(cos(theta)) as defined in "brink and satchler"
!
thedeg=(theta*pi)/180d0

!  if m>l pm1=0 !

if (m.gt.l) then
  pm1=0d0
  return
endif
x=cos(thedeg)
if (m.lt.0) then 
  write (6,*) '	NEGATIVE M IN LEGENDRE ROUTINE:	 ABORT'
  stop
endif
if (m==0) then 
!  here for regular legendre polynomials
  pm1=1d0
  pm2=0d0
  do l_=1,l
    pp=((2d0*l_-1d0)*x*pm1-(l_-1d0)*pm2)/float(l_)
    pm2=pm1
    pm1=pp
  enddo
  else
!  here for alexander-legendre polynomials
  rat=1.
  do i=2,2*m,2
    ai=i
    rat=rat*((ai-1d0)/ai)
  enddo
  y=sin(thedeg)
  pm1=sqrt(rat)*(y**m)
  pm2=0d0
  do l_=m+1,l
    al=(l_+m)*(l_-m)
    al=1d0/al
    al2=((l_+m-1)*(l_-m-1))*al
    al=sqrt(al)
    al2=sqrt(al2)
    pp=(2d0*l_-1d0)*x*pm1*al-pm2*al2
    pm2=pm1
    pm1=pp
  enddo
  pm1=((-1d0)**m)*pm1  !correct phase
endif
dlm0=pm1
return
end function dlm0 

subroutine choles(a,x,idim)
implicit none
integer(4),intent(in) :: idim
real(8),intent(inout) :: a(idim,idim+1)
real(8),intent(inout) ::x(idim)
integer(4) :: nrow,ncol,k,il,l,ll,j,i,m
real(8) :: pivot,temp,sum

!...cholesky decomposition

       nrow=idim
       ncol=idim+1

!...pivotisierung

       do 25 k=1,nrow
       pivot=a(k,k)
       il=k
       do 10 l=k+1,nrow
          if (abs(a(l,k)).lt.abs(pivot)) goto 10
          pivot=a(l,k)
          il=l
 10    continue
       if (il.eq.k) goto 25

       do 20 ll=1,ncol
          temp=a(k,ll)
          a(k,ll)=a(il,ll)
          a(il,ll)=temp
 20     continue
 25   continue

      do 30 j=2,ncol
 30       a(1,j)=a(1,j)/a(1,1)

      do 80 l=2,nrow
        do 50 i=l,nrow
           sum=0.
           do 40 k=1,l-1
 40           sum=sum+a(i,k)*a(k,l)
           a(i,l)=a(i,l)-sum
 50     continue
        do 70 j=l+1,ncol
           sum=0.
           do 60 k=1,l-1
 60           sum=sum+a(l,k)*a(k,j)
           a(l,j)=(a(l,j)-sum)/a(l,l)
 70     continue
 80   continue

      x(nrow)=a(nrow,ncol)
      do 100 m=1,nrow-1
        i=nrow-m
        sum=0.
        do 90 j=i+1,nrow
 90        sum=sum+a(i,j)*x(j)
        x(i)=a(i,ncol)-sum
 100  continue
      return
end subroutine choles

subroutine readv(filnam,readfl,nterms)
implicit none
character(len=80),intent(in) :: filnam
logical,intent(out) :: readfl
integer(4),intent(out) :: nterms
real(8) :: rbig,theta,rsmall,energy,rsmall_old,theta_old
integer(4) :: i,ibl,i_nr,i_nangle,ipot,activity
!...subroutine to read in potential
!   potential must be given in the following form
!
!   r  angle  r   energy

!   where r refers to the distance between the center of mass
!   of the diatomic and the atom. r is the bond distance of the
!   diatomic and angle is the angle between the figure axis of
!   the molecule and r.

!   on entry: filnam    filename of datafile
!             verify    flag, if true control output to screen

!   variables in common blocks

!   copot: ntotal           total number of points
!          nblk       number of blocks in i-th potential
!          nangle     number of different angles in i-th potential
!          nr               number of different r
!          npblk(iblk) number of points in i-th block and in i-th potential
!          angle(iblk) angle in i-th block and in i-th potential
!          r(iblk)     bond distance in i-th block and in i-th potential
!          rr(n,iblk)  array of rr-values 
!          v(n,iblk)   array of energies

readfl=.false.
                      
open(unit=1,file=filnam,status='OLD')
read(1,*)
read(1,*) nterms
write(*,*) nterms
main_loop: do ipot=1,nterms
             pots(ipot)%fitflg=.false.
             pots(ipot)%npblk=0
             pots(ipot)%angle=0.0
             pots(ipot)%r=0.0
             pots(ipot)%rr=0.0
             pots(ipot)%v=0.0
             pots(ipot)%ang=0.0
             pots(ipot)%rmol=0.0
             pots(ipot)%nblk=0
	     pots(ipot)%ntotal=0
	     pots(ipot)%isym=2
             read(1,*)
             read(1,*) pots(ipot)%nangle,pots(ipot)%nr
             read(1,*) rbig,theta,rsmall,energy 
             theta_old=theta
             rsmall_old=rsmall
	     do i_nr=1,pots(ipot)%nr
               pots(ipot)%rmol(i_nr)=rsmall
               do i_nangle=1,pots(ipot)%nangle
                 if(theta.gt.91.0) pots(ipot)%isym=1
		 pots(ipot)%ang(i_nangle)=theta
       	 	 pots(ipot)%nblk=pots(ipot)%nblk+1
                 pots(ipot)%angle(pots(ipot)%nblk)=theta
		 pots(ipot)%r(pots(ipot)%nblk)=rsmall
                 pots(ipot)%npblk(pots(ipot)%nblk)=0 
                 do while (theta==theta_old .and. rsmall==rsmall_old)
                   pots(ipot)%npblk(pots(ipot)%nblk)=pots(ipot)%npblk(pots(ipot)%nblk)+1	
	           pots(ipot)%rr(pots(ipot)%npblk(pots(ipot)%nblk),pots(ipot)%nblk)=rbig
                   pots(ipot)%v(pots(ipot)%npblk(pots(ipot)%nblk),pots(ipot)%nblk)=energy
                   pots(ipot)%ntotal=pots(ipot)%ntotal+1
		   read(1,*,iostat=activity) rbig,theta,rsmall,energy
!!                   write(*,*) i_nr,i_nangle,rbig,theta,rsmall,energy
                   if (activity<0) exit main_loop
		 enddo
		 theta_old=theta
		 rsmall_old=rsmall
               enddo
             enddo

           enddo main_loop
do ipot=1,nterms
!  write(*,*) pots(ipot)%nr,pots(ipot)%nangle
!  read(*,*)
  ibl=0  
  do i_nr=1,pots(ipot)%nr
    do i_nangle=1,pots(ipot)%nangle
      ibl=ibl+1
!      write(*,*) pots(ipot)%npblk(ibl)
!      read(*,*)
      do i=1,pots(ipot)%npblk(ibl)
	write(6,*) ipot,i_nr,i_nangle,i,pots(ipot)%r(ibl),pots(ipot)%angle(ibl),pots(ipot)%rr(i,ibl),pots(ipot)%v(i,ibl)
      enddo
    enddo
  enddo
enddo

do ipot=1,nterms
  ! initialize matrices
  call inimat(0,ipot)
  ! guess short-range damping term r1
  do ibl=1,pots(ipot)%nblk
    if (pots(ipot)%npblk(ibl).gt.1) then
      call gues(pots(ipot)%v(1,ibl),pots(ipot)%v(2,ibl),pots(ipot)%rr(1,ibl),pots(ipot)%rr(2,ibl), &
                pots(ipot)%a(1,ibl),pots(ipot)%a(pots(ipot)%npa+3,ibl))
    else
      pots(ipot)%a(1,ibl)=0
      pots(ipot)%a(pots(ipot)%npa+3,ibl)=0
    endif
  enddo
enddo

ipot=1 
write(6,'(/,a,/,a,i3,t39,a,i3,/,a,i3,t39,a,i4,/,a,i3,t39,a,i3,/,a,/)') & 
        '---------------------------------------------------------------------------', &
	'  NUMBER OF POTENTIAL (NTERMS): ',nterms,                                     &
	'  CURRENT POTENTIAL:            ',ipot,                                       &
	'  NUMBER OF BLOCKS:             ',pots(ipot)%nblk,                            &
	'  TOTAL NUMBER OF POINTS:       ',pots(ipot)%ntotal,                          &
	'  NUMBER OF ANGLES:             ',pots(ipot)%nangle,                          &
	'  NUMBER OF BOND DISTANCES:     ',pots(ipot)%nr,                              &
	'---------------------------------------------------------------------------'
readfl=.true.
return
end subroutine readv
      
function fct(r,iblk,ipot)
implicit none
real(8),intent(in) :: r
integer(4),intent(in) :: iblk,ipot
real(8) :: fct,tanhyb,fex,pfac,rpowi,pexfac,fmfac,jst,powmj
integer(4) :: i,j

!...this function is defined as follows:
!                                          2                    maxpws
!     b(r)=exp(-a *(r-r1))*{ a  + a *r + a *r + ... + a         *r       }
!                1          2    3      4            maxpws+2
!
!                                       minmps                minmps+mpsstp
!       -(1/2)*(1+tanh((r-r0)/rref)){a         /r       + a          /r
!                            maxpws+3             maxpws+4
!
!                        maxmps
!       + . . . + a    /r       }
!                  npa


!r0 is set to 0
!     a(npa+2,iblk,ipot)=0.0
tanhyb=(tanh((r-pots(ipot)%a(pots(ipot)%npa+2,iblk))/pots(ipot)%a(pots(ipot)%npa+1,iblk))+1.d0)*0.5d0
fex=exp(-pots(ipot)%a(1,iblk)*(r-pots(ipot)%a(pots(ipot)%npa+3,iblk)))
if(fex.gt.1.d10) fex=1.d10
pfac=0.
rpowi=1.
do i=2,pots(ipot)%maxpws
  pfac=pfac+pots(ipot)%a(i,iblk)*rpowi
  rpowi=rpowi*r
enddo
pexfac=fex*pfac
if(pexfac.gt.1.d20) pexfac=1.d20
fmfac=0.
jst=pots(ipot)%maxpws+1
if (.not. (jst.gt.pots(ipot)%npa)) then
  powmj=r**pots(ipot)%minmps
  do j=jst,pots(ipot)%npa
    fmfac=fmfac+pots(ipot)%a(j,iblk)/powmj
   powmj=powmj*r**pots(ipot)%mpsstp
  enddo
endif
fct=pexfac-tanhyb*fmfac
return
end function fct

subroutine inimat(ijob,ipot)
implicit none
integer(4),intent(in) :: ijob,ipot
integer(4) :: lmax,i,j,l,n
real(8) :: theta,det,rd
!...subroutine to initialize transformation matrices
!   on entry: ijob    0: do job 1 and 2
!                         -1
!                     1: p  is computed (stored in pinv)
!                         -1
!                     2: r  is computed (stored in rinv)

!                   -1
!...job 1: compute p   maxtrix
!
if (ijob.eq.1.or.ijob.eq.0) then
  lmax=(pots(ipot)%nangle-1)*pots(ipot)%isym+pots(ipot)%mld
!
!..built up p matrix for calculated angles
!
  do i=1,pots(ipot)%nangle
    j=0
    theta=pots(ipot)%ang(i)
    do l=pots(ipot)%mld,lmax,pots(ipot)%isym
      j=j+1
      pots(ipot)%pinv(i,j)=dlm0(l,pots(ipot)%mld,theta)
    enddo
  enddo
!...now invert matrix
  call matinv(pots(ipot)%pinv,det,max_theta,pots(ipot)%nangle)
!Begin Guillaume
!write(*,*) pots(ipot)%pinv(1:pots(ipot)%nangle,1:pots(ipot)%nangle)
!read(*,*)
!End Guillaume

endif
!                   -1
!...job 2: compute r
!
if (ijob.eq.2.or.ijob.eq.0) then
!
!..built up r matrix
!
  do i=1,pots(ipot)%nr
    j=1
    rd=pots(ipot)%rmol(i)-pots(ipot)%re
    pots(ipot)%rinv(1,i)=1d0
    do n=1,pots(ipot)%nr
      j=j+1
      pots(ipot)%rinv(j,i)=rd
!Begin Old line
!      rd=rd*rd
!End Old line
!Begin New line
      rd=rd*(pots(ipot)%rmol(i)-pots(ipot)%re)
!End Old line
    enddo
  enddo
!...matrix inversion
!write(*,*) "Avant",pots(ipot)%rinv(1:pots(ipot)%nr,1:pots(ipot)%nr)
!read(*,*)
call matinv(pots(ipot)%rinv,det,max_r,pots(ipot)%nr)
!Begin Guillaume
!write(*,*) pots(ipot)%rinv(1:pots(ipot)%nr,1:pots(ipot)%nr) 
!read(*,*)
!End Guillaume
endif
return
end subroutine inimat

subroutine fitpot(autom,autmin,autmax,autstp,iblk,ipot,readfl)
implicit none
logical,intent(in) :: autom,readfl
integer(4),intent(in) ::iblk,ipot
real(8),intent(in) :: autmin,autmax,autstp
real(8) :: fehler,fehold,fehmin,abest,da1,update,a 
integer(4) :: i,iter

!...routine to fit one block in one potential
!   on entry:   iblk -> block number

da1=pots(ipot)%updmax

!...check if there are any data points

if (.not.readfl) then
  write(6,*) ' %VFIT-ERROR: no data points read in'
  return
end if

!...check if there are more coefficients than data points

if (pots(ipot)%npa.gt.pots(ipot)%npblk(iblk)) then
         write(6,*) ' %VFIT-WRNG: more coefficients than data points'
end if

!...automatic, find best start value for a1

if(autom) then
  write(6,'(/a,/,1x,a,/,a,f10.4,a,f10.4,a,f10.4)') ' AUTOMATIC PROCESSING','----------------------',&
  ' AUTMIN: ',autmin,' AUTMAX: ',autmax,' AUTSTP: ',autstp
  fehold=1.d10
  pots(ipot)%a(1,iblk)=autmin
  do while (pots(ipot)%a(1,iblk).le.autmax)
    fehold=fehler
    call linequ(pots(ipot)%rr(1:pots(ipot)%npblk(iblk),iblk),pots(ipot)%v(1:pots(ipot)%npblk(iblk),iblk),&
                pots(ipot)%npblk(iblk),iblk,ipot)
    fehler=0.   
    do i=1,pots(ipot)%npblk(iblk)    
      a=fct(pots(ipot)%rr(i,iblk),iblk,ipot)
      if (abs(a-pots(ipot)%v(i,iblk)).gt.1.d15) then
	fehler=1.d15
      else if (fehler.lt.1.d15) then
	fehler=fehler+(a-pots(ipot)%v(i,iblk))**2
      endif
    enddo
    fehler=sqrt(fehler/pots(ipot)%npblk(iblk))
    if (fehler.lt.fehold) then
      abest=pots(ipot)%a(1,iblk)
      fehold=fehler
    endif
    pots(ipot)%a(1,iblk)=pots(ipot)%a(1,iblk)+autstp
  enddo
  write(6,'(a,i1,a,1pe15.6)') ' START VALUE FOR A(1,',iblk,') SET TO ',abest
  pots(ipot)%a(1,iblk)=abest
  da1=autstp*0.1
  pots(ipot)%eps=pots(ipot)%eps*0.01
end if
!
!...go ahead
!
fehler=0.
fehmin=1.0d10
iter=0
!
!...write out some information
!
write(6,'(a,/,a,/)') ' CURRENT FIT PARAMETERS:',' -----------------------'
      write(6,'(a,12x,i3,t35,a,1pe15.5,/,a,1pe15.5,t35,a,1pe15.5,/,a,1pe15.5,t35,a,1pe15.5,/,a,&
       &12x,i3,t35,a,12x,i3,/,a,i2,t26,a,i2,t55,a,i2)') ' MAXIT:        ',pots(ipot)%maxit,&
       ' AGUESS        ',pots(ipot)%a(1,iblk),' MAX. UPDATE:  ',da1,' ACCURACY:     ',pots(ipot)%eps,&
       ' REFERENCE:    ',pots(ipot)%a(pots(ipot)%npa+1,iblk),' WEIGHT(EXPO.):',pots(ipot)%xr(iblk),&
       ' COEFFICIENTS: ',pots(ipot)%npa,' MAXPWS:       ',pots(ipot)%maxpws-2,&
       ' MINMPS: ',pots(ipot)%minmps,' MAXMPS:  ',pots(ipot)%maxmps,' MPSSTP: ',pots(ipot)%mpsstp
      write(6,'(/a)') '  ITER       A(1)          UPDATE           SQRSUM'
!
!...iteration loop starts here
!
      do while (iter<=pots(ipot)%maxit .and. abs(da1)>=pots(ipot)%eps)
        pots(ipot)%a(1,iblk)=pots(ipot)%a(1,iblk)+da1
        if(iter.eq.0) then
          pots(ipot)%a(1,iblk)=pots(ipot)%a(1,iblk)-da1
          update=0.
        else
          update=da1
        end if
        fehold=fehler
        call linequ(pots(ipot)%rr(1:pots(ipot)%npblk(iblk),iblk),pots(ipot)%v(1:pots(ipot)%npblk(iblk),iblk),&
                    pots(ipot)%npblk(iblk),iblk,ipot)
        fehler=0.
        do i=1,pots(ipot)%npblk(iblk)
          a=fct(pots(ipot)%rr(i,iblk),iblk,ipot)
          if (abs(a-pots(ipot)%v(i,iblk)).gt.1.d15) then
            fehler=1.d15
          else if (fehler.lt.1.d15) then
            fehler=fehler+(a-pots(ipot)%v(i,iblk))**2
          endif
        enddo
        fehler=sqrt(fehler/pots(ipot)%npblk(iblk))
        write(6,'(3x,i2,1x,1pe15.6,1x,e15.6,1x,1pe15.6)') iter,pots(ipot)%a(1,iblk),update,fehler
        iter=iter+1
!
!...save best value for a(1)
!
        if(fehler.lt.fehmin) then
          abest=pots(ipot)%a(1,iblk)
	  fehmin=fehler
        endif
        if ((fehler-fehold)>=0) da1=-da1/2. 
      enddo
!
!...here if fit is completed
!
350   pots(ipot)%fitflg(iblk)=.true.
      write(6,'(/a,i2,a,/,a,/)') ' FIT FOR BLOCK ',iblk,' COMPLETED',' --------------------------'
      write(6,'(a,3x,i7,t40,a,6x,i7,/,a,3x,f7.2,t40,a,4x,f9.4,/,a,1x,f9.4,t40,a,1pe15.6,/)') &
       ' NUMBER OF COEFFICIENTS: ',pots(ipot)%npa,' NUMBER OF POINTS:       ',pots(ipot)%npblk(iblk),&
       ' ANGLE:                  ',pots(ipot)%angle(iblk),' BOND DISTANCE:          ',pots(ipot)%r(iblk),&
       ' REFERENCE:              ',pots(ipot)%a(pots(ipot)%npa+1,iblk),' FINAL SQUARE SUM:     ',fehler
      call report(iblk,ipot)
      return

end subroutine fitpot
      
subroutine linequ(rr,v,np,iblk,ipot)
implicit none
integer(4),intent(in) :: np,iblk,ipot
real(8),intent(in) :: rr(np),v(np)
real(8) :: g(pots(ipot)%npa-1),h(pots(ipot)%npa-1,pots(ipot)%npa)
integer(4) :: i,j,ip,ii
real(8) :: x,tanhyb,const,rpowj,rpowi,powmj,powmi,weight

!...subroutine to set up a system of linear equations for
!   least square fit
!   on entry:  rr    -> array containing r - values
!              v     -> array containing energy values
!              g,h   -> scratch arrays
!              np    -> number of points in rr() and v()
!              iblk  -> block number
!  on return:  a(2...npa,iblk) contains new coefficients for block iblk

pots(ipot)%maxpws=pots(ipot)%maxpws-1
do i=1,pots(ipot)%npa-1
  g(i)=0.
  do j=1,pots(ipot)%npa
    h(i,j)=0.
  enddo
enddo

!...loop over all points starts here
write(*,*) pots(ipot)%maxpws,pots(ipot)%minmps,pots(ipot)%maxmps,pots(ipot)%mpsstp


do ip=1,np
  write(*,*) 'rr,e',rr(ip),v(ip)
  x=rr(ip)
  tanhyb=(tanh((x-pots(ipot)%a(pots(ipot)%npa+2,iblk))/pots(ipot)%a(pots(ipot)%npa+1,iblk))+1.d0)*0.5d0
  const=exp(-pots(ipot)%a(1,iblk)*(x-pots(ipot)%a(pots(ipot)%npa+3,iblk)))

  write(*,*) '#pots(ipot)%wfac,tanhyb,const',pots(ipot)%wfac,tanhyb,const
  if (pots(ipot)%wfac.ne.0) then
    if(abs(pots(ipot)%v(ip,iblk)).le.1.d-5) then
      weight = 1.d0/1.d-5**pots(ipot)%xr(iblk)
    else
      weight = 1.d0/abs(pots(ipot)%v(ip,iblk))**pots(ipot)%xr(iblk)
    endif
  endif
  !   call pesi(ip,pots(ipot)%v,np,weight,pots(ipot)%xr(iblk))

  ! ......        call weisub(x,weight,iblk)                              

  write(6,*) 'pots(ipot)%wfac',pots(ipot)%wfac
  if (pots(ipot)%wfac.eq.0.0) then
    weight = 1.
    write(6,*) 'weight',weight
  endif
!  weight=1.
  write(*,*) weight
  rpowi=1.
  do i=1,pots(ipot)%maxpws
    h(i,pots(ipot)%npa)=h(i,pots(ipot)%npa)+weight*v(ip)*const*rpowi
    write(*,*) h(i,pots(ipot)%npa)
 !  read(*,*) 
    rpowj=1.
    do j=1,pots(ipot)%maxpws
      h(i,j)=h(i,j)+weight*const*rpowi*const*rpowj
      write(*,*) h(i,j)
!      read(*,*)
      rpowj=rpowj*x
    enddo
    powmj=x**pots(ipot)%minmps
    do j=pots(ipot)%maxpws+1,pots(ipot)%npa-1
      h(i,j)=h(i,j)-weight*const*rpowi*tanhyb/powmj
      powmj=powmj*x**pots(ipot)%mpsstp	
    enddo
    rpowi=rpowi*x
  enddo
  powmi=x**pots(ipot)%minmps
  do i=pots(ipot)%maxpws+1,pots(ipot)%npa-1
    h(i,pots(ipot)%npa)=h(i,pots(ipot)%npa)-weight*v(ip)*tanhyb/powmi
    rpowj=1.
    do j=1,pots(ipot)%maxpws
      h(i,j)=h(i,j)-weight*(tanhyb/powmi)*const*rpowj
      rpowj=rpowj*x
      !????????????
    enddo
    powmj=x**pots(ipot)%minmps
    do j=pots(ipot)%maxpws+1,pots(ipot)%npa-1
      h(i,j)=h(i,j)+weight*(tanhyb/powmi)*(tanhyb/powmj)
      powmj=powmj*x**pots(ipot)%mpsstp
    enddo
    powmi=powmi*x**pots(ipot)%mpsstp
  enddo
enddo  
!...now perform cholesky decomposition

do i=1,pots(ipot)%npa-1
  g(i)=h(i,pots(ipot)%npa)
  write(6,*) g(i)
enddo
call choles(h,g,pots(ipot)%npa-1)
ii=0
do i=2,pots(ipot)%npa
  ii=ii+1
  pots(ipot)%a(i,iblk)=g(ii)
  write(6,*) pots(ipot)%a(i,iblk) 
enddo
pots(ipot)%maxpws=pots(ipot)%maxpws+1
return
end subroutine linequ

subroutine report(iblk,ipot)
implicit none
integer(4),intent(in) :: iblk,ipot
real(8) :: sproz,proz,fehler,qproz
integer(4) :: i,j
!...subroutine to write out some information

!     on entry:    noout, if 0 print only coefficients
!                         if 1 print data points, fitted points,
!                              differences and coefficients
!                  iblk -> block number

!...check if fit for all blocks has been done

if (.not.pots(ipot)%fitflg(iblk)) then
  write(6,'(a,i2,a)') ' %VFIT-ERROR: block ',iblk,' not fitted'
  return
end if
write(6,'(a,i3,a,i2,/)') ' SURFACE (',pots(ipot)%npblk(iblk),' POINTS) BLOCK NUMBER: ',iblk
write(6,'(5x,a,6x,a,7x,a,10x,a,11x,a,5x,a,7x,a)') 'RR','ANGLE','R ','INPUT','FIT','DIFFERENCE','%'
sproz = 0.0
do i=1,pots(ipot)%npblk(iblk)
  fehler = pots(ipot)%v(i,iblk) - fct(pots(ipot)%rr(i,iblk),iblk,ipot)
  if (abs(pots(ipot)%v(i,iblk)).lt.1.0d-16) then
    proz = 0.0
  else 
    proz = abs(fehler/(pots(ipot)%v(i,iblk))*100.)
  end if
  sproz = sproz + proz 
  write(6,'(f8.2,f10.4,f10.5,2f15.8,1x,e15.8,1x,f6.1)') pots(ipot)%rr(i,iblk),pots(ipot)%angle(iblk),pots(ipot)%r(iblk),&
  pots(ipot)%v(i,iblk),fct(pots(ipot)%rr(i,iblk),iblk,ipot),fehler,proz
enddo
qproz = sproz/pots(ipot)%npblk(iblk)
write(6,'(1x,t15,a,f10.2)') 'Fehlerdurchschnitt = ',qproz
write(6,'(a,i3,/,a)') ' COEFFICIENTS: ',pots(ipot)%npa,' -----------------'
do i=1,pots(ipot)%npa,4
  write(6,'(1x,4((a),i2,(a),g14.6,1x))') ('A',j,':',pots(ipot)%a(j,iblk),j=i,min0(i+3,pots(ipot)%npa))
enddo
return
end subroutine report

subroutine pesi(ip,v,np,peso,we)
implicit none
integer(4),intent(in) :: ip,np
real(8),intent(in) :: v(np),we
real(8),intent(out) :: peso
!...subroutine to calculate weight factor

if(abs(v(ip)).le.1.d-5) then
  peso = 1.d0/1.d-5**we
else
  peso = 1.d0/abs(v(ip))**we
endif
return
end subroutine pesi  	

subroutine savfit(filnam,nterms)
implicit none
character(len=80),intent(in) :: filnam
integer(4),intent(in) :: nterms
character(len=80) :: label
logical :: exstfl,opnfl
integer(4) :: ierr,ipot
!
!...subroutine to save fitted surface on unformatted files (filnam.srf).
!
!   on entry:   filnam  filename
!
!   author: b. follmeg
!   current revision date: 20-june-1988
! ------------------------------------------------------------------------

!...now go ahead
inquire(file=filnam,exist=exstfl,opened=opnfl)
open(unit=1,file=filnam,status='UNKNOWN',form='UNFORMATTED',access='SEQUENTIAL')
if (exstfl) rewind(1)

write(6,'(a)') 'Label ?>'
read(5,'(a)') label
write(1,iostat=ierr) label
write(1,iostat=ierr) nterms
do ipot=1,nterms
  write(1,iostat=ierr) pots(ipot)%npa,pots(ipot)%a,pots(ipot)%ntotal,pots(ipot)%nblk,pots(ipot)%nangle,pots(ipot)%nr,&
                       pots(ipot)%isym,pots(ipot)%npblk,pots(ipot)%angle,pots(ipot)%r,pots(ipot)%rr,pots(ipot)%v,&
                       pots(ipot)%maxpws,pots(ipot)%minmps,pots(ipot)%maxmps,pots(ipot)%mpsstp,pots(ipot)%updmax,&
                       pots(ipot)%wfac,pots(ipot)%ang,pots(ipot)%rmol,pots(ipot)%re,pots(ipot)%mld,pots(ipot)%fitflg,&
                       pots(ipot)%pinv,pots(ipot)%xr
  write(6,*) 'pots(ipot)%nblk',pots(ipot)%nblk
enddo
write(6,'(a,(a))') ' SURFACE SAVED ON FILE ',filnam
close(1)
return
end subroutine savfit

subroutine loafit(filnam,nterms,readfl,input)
implicit none
character(len=80),intent(in) :: filnam
integer(4),intent(in) :: input
integer(4),intent(out) :: nterms
logical,intent(out) :: readfl
character(len=80) :: label
logical :: exstfl,openfl
real(8) :: re
integer(4) :: ierr,i,ntblk,ntot,ntpa,ipot
!
!...subroutine to load fitted surface on unformatted files (filnam.srf).
!
!   on entry:   filnam  filename
!

ntblk=0
ntot=0
ntpa=0
ierr=0
open(unit=1,file=filnam,status='UNKNOWN',form='UNFORMATTED',access='SEQUENTIAL')

read(1,iostat=ierr) label
write(input,*) 'label : ',label
read(1,iostat=ierr) nterms
write(input,*) 'nterms : ',nterms
do ipot=1,nterms
  read(1,iostat=ierr) pots(ipot)%npa,pots(ipot)%a,pots(ipot)%ntotal,pots(ipot)%nblk,pots(ipot)%nangle,pots(ipot)%nr,&
                       pots(ipot)%isym,pots(ipot)%npblk,pots(ipot)%angle,pots(ipot)%r,pots(ipot)%rr,pots(ipot)%v,&
                       pots(ipot)%maxpws,pots(ipot)%minmps,pots(ipot)%maxmps,pots(ipot)%mpsstp,pots(ipot)%updmax,&
                       pots(ipot)%wfac,pots(ipot)%ang,pots(ipot)%rmol,pots(ipot)%re,pots(ipot)%mld,pots(ipot)%fitflg,&
                       pots(ipot)%pinv,pots(ipot)%xr
enddo
do ipot=1,nterms
  ntblk=ntblk+pots(ipot)%nblk
  write(input,*) 'ntblk',ntblk
  ntot=ntot+pots(ipot)%ntotal
  ntpa=ntpa+pots(ipot)%npa
enddo

!	initialize matrices
do i=1,nterms
  call inimat(0,i)
enddo

ipot=1
write(input,'(a,a,/)') ' SURFACE LOADED FROM FILE ',filnam
write(input,'(a,(a),/,a,/,a,i3,t40,a,i3,/,a,i3,t40,a,i3,/,a,/)') ' LABEL:',label(:72), &
     '---------------------------------------------------------------------------',&
     '  NUMBER OF POTENTIALS (NTERMS): ',nterms,'  TOTAL NUMBER OF BLOCKS:        ',ntblk,&
     '  TOTAL NUMBER OF POINTS:        ',ntot,'  TOTAL NUMBER OF COEFFICIENTS:  ',ntpa,&
     '---------------------------------------------------------------------------'
readfl=.true.
close(1)
return

end subroutine loafit
     
function vcalc(rbig,theta,rd,ipot)
implicit none
real(8),intent(in) :: rbig,theta,rd
integer(4),intent(in) :: ipot
real(8) :: vcalc,vvlsum,vvl(max_theta),vvlp(max_theta),pvec(max_theta),rvecp(max_theta), &
           rvecpp(max_theta),b(max_theta,max_r),dr
integer(4) :: i,l,lmax,n,j,iblk
!
!...function to calculate potential for any given geometry
!
lmax=(pots(ipot)%nangle-1)*pots(ipot)%isym+pots(ipot)%mld
vcalc=0.
!                            mld,0                  mld,0
!...calculate pvec: pvec(1)=d     (theta), pvec(2)=d       (theta), ...
!                            l                      l+isym
!
!   where isym = 2 if homonuclear molecule and isym =1 if heteronuclear
!   molecule

i=0
do l=pots(ipot)%mld,lmax,pots(ipot)%isym
  i=i+1
  pvec(i)=dlm0(l,pots(ipot)%mld,theta)
enddo

!   now obtain vector of vvl(nangle) coefficients and calculate energy.
!   note that vvl(1) = vv0 is the isotropic term if mld (mproj) is 0.
!
!                      -1       -1
!   v(r,theta,r)=pvec*p  *b(r)*r  *rvec = pvec * vvl * rvec
!
!   first built up b(r) matrix

iblk=0
do l=1,pots(ipot)%nr
  do n=1,pots(ipot)%nangle
    iblk=iblk+1
    b(n,l)=fct(rbig,iblk,ipot)
  enddo
enddo

!...fast loop if nr = 1 .and. novib
if (pots(ipot)%nr.gt.1) then
  !...here if normal treatment of r - dependent potential
  !                                             2
  !...calculate rvec: rvec = { 1., (r-re), (r-re) , ... }
  !                       -1     -1
  !    and multiply with r  :   r  * rvec = rvec'
  do i=1,pots(ipot)%nr
    rvecp(i)=0.
    dr=1.
    do j=1,pots(ipot)%nr
      rvecp(i)=rvecp(i)+pots(ipot)%rinv(i,j)*dr
      dr=dr*rd
    enddo
  enddo
  !  calculate  b(r) * rvec' = rvec''
  do i=1,pots(ipot)%nangle
    rvecpp(i)=0.
    do j=1,pots(ipot)%nr
      rvecpp(i)=rvecpp(i)+b(i,j)*rvecp(j)
    enddo
  enddo
  !              -1
  !  calculate  p  * rvec'' = vvl  ( note that vvl(1) = vv0 if mld (mproj) is 0 )
  do i=1,pots(ipot)%nangle
    vvl(i)=0.
    do j=1,pots(ipot)%nangle
      vvl(i)=vvl(i)+pots(ipot)%pinv(i,j)*rvecpp(j)
    enddo
  enddo
else
!                                                  -1
!...here if number of bond distances is 1:  vvl = p   * b(r) where
!   b(r) is a now column vector rather than a matrix
  do i=1,pots(ipot)%nangle
    vvl(i)=0.
    do j=1,pots(ipot)%nangle
      vvl(i)=vvl(i)+pots(ipot)%pinv(i,j)*b(j,1)
    enddo
  enddo
end if
!
!  sum over all terms: v(r,theta,r) = pvec(1)*vvl(1) + pvec(2)*vvl(2) +...
!
vvlsum=0.
do i=1,pots(ipot)%nangle
  vvlsum=vvlsum+vvl(i)
  vvlp(i)=pvec(i)*vvl(i)
  vcalc=vcalc+vvlp(i)
enddo
return
end function vcalc

subroutine anapot(iflag,turnflg,energ,sc1,sc2,ipot)
implicit none
integer(4),intent(in) :: iflag,ipot
logical,intent(in) :: turnflg
real(8),intent(in) :: energ
real(8),intent(out) :: sc1(max_b),sc2(max_b)
real(8) :: tol,shift,elast,energy,dx,x1,x2,y1,y2,x,xn,yn
integer(4) :: mmaxit,iblk,iter,i,miter,iterm

!...subroutine to analyse fitted potential,x1,x2,y1,y2
!
!   a) find rr-values at which v(rr)=0
!   b) find rr-values at which v(rr)=minimum

data tol,mmaxit/1.d-10,30/
pots(ipot)%maxit=1000
shift=0.0

!..calculate shift for turning points

if (turnflg) shift=energ/219474.817d0

!...check if all blocks have been fitted

do i=1,pots(ipot)%nblk
  if(.not.pots(ipot)%fitflg(i)) then
    write(6,'(a,i2,a)') ' %VFIT-ERROR: fit for block ',i,' has not be done'
    return
  end if
enddo
 
!...find minima

do iblk=1,pots(ipot)%nblk
  iter=0
  x=pots(ipot)%rr(1,iblk)
  dx=1.
  energy=fct(x,iblk,ipot)
  do
    x=x+dx
    iter=iter+1
    if (iter.gt.pots(ipot)%maxit) then
      sc1(iblk)=-1.
      exit 
    endif
    elast=energy
    energy=fct(x,iblk,ipot)
    if ((elast-energy)==0) then
      sc1(iblk)=x
      exit
    elseif ((elast-energy)<=0) then
      dx=-dx*0.5
      if (abs(dx).le.0.001) then
        sc1(iblk)=x	
	exit
      endif
    endif
  enddo
enddo
if (iflag.eq.0) return

!..find point at which v(rr) is 0 or equal to given energy (energ)

loop: do iblk=1,pots(ipot)%nblk
      dx=1.
      miter=0
      do
        x1=sc1(iblk)
        x2=sc1(iblk)-dx

        !..for turning points of v1 and v2 potentials

        if (turnflg.and.(ipot.eq.3.or.ipot.eq.4).and.energ.ne.0) then
          x1=10.0
          x2=x1-dx
        end if

        y1=fct(x1,iblk,ipot)-shift
        y2=fct(x2,iblk,ipot)-shift
        if (y1*y2.lt.0) exit 
        miter=miter+1
        dx=dx+1.
        if(miter.gt.mmaxit) then
          sc2(iblk)=-1.
          cycle loop 
        end if
      enddo
      iter=0
      do
        xn=(x1-x2)/2.+x2
        iter=iter+1
        yn=fct(xn,iblk,ipot)-shift
        if (yn*y1.gt.0) then
          x1=xn
          y1=yn
        else
          x2=xn
          y2=yn
        end if
        if (iter.gt.pots(ipot)%maxit) then
          sc2(iblk)=-1.0
          cycle loop  
        end if
        if (abs(yn).le.tol) exit
      enddo
      sc2(iblk)=xn
      enddo loop

!...now write out minima and point at which potential is 0 or
!...turning points, respectively.

      if (turnflg) then
        write(6,'(/,a,f6.1,a,/,a,/)') ' TURNING POINTS FOR E = ',energ,' CM-1',' ----------------------------------'
        write(6,'(a)') ' BLOCK  ANGLE       R      RTURN'
        do i=1,pots(ipot)%nblk
          write(6,'(t2,i3,t7,f8.4,t16,f8.4,t25,f9.5)') i,pots(ipot)%angle(i),pots(ipot)%r(i),sc2(i)
        enddo
      else
        write(6,'(/,a,/,a,/)') ' POTENTIAL ANALYZED',' ------------------'
        write(6,'(a,a)') ' BLOCK  ANGLE	R	RRMIN	 V(RRMIN)','	  RRZERO     V(RRZERO)'
        do i=1,pots(ipot)%nblk
          write(6,'(t2,i3,t7,f8.4,t16,f8.4,t25,f9.5,t35,1pe14.6,t51,0pf9.5,t61,1pe14.6)') &
                    i,pots(ipot)%angle(i),pots(ipot)%r(i),sc1(i),fct(sc1(i),i,ipot),sc2(i),fct(sc2(i),i,ipot)
        enddo
      end if
      return
end subroutine anapot

subroutine check(ipot)
implicit none
integer(4),intent(in) :: ipot
real(8) :: enew,eold,x,dx
real(8) :: sc1(max_b),sc2(max_b)
integer(4) :: ierr,i,j,iblk

!...subroutine to analyze fitted potential

!real(8) :: sc1(2500),sc2(2500)


!...check if all block have been fitted

do i=1,pots(ipot)%nblk
  if(.not.pots(ipot)%fitflg(i)) then
  write(6,'(a,i2,a)') ' %VFIT-ERROR: fit for block ',i,' has not be done'
  return
  endif
enddo

!...first find minima

call anapot(0,.false.,0d0,sc1,sc2,ipot)

ierr=0
write(6,'(/,a,/,a)') ' POTENTIAL CHECK',' ---------------'
do iblk=1,pots(ipot)%nblk
  x=sc1(iblk)
!...check if there is a minimum
  if(x.lt.0.) then
    write(6,'(a, i2,a)') ' %VFIT-MSG: no minimum found in block ',iblk,' assuming repulsive potential'
  else 
!...here if there is one. test short range potential first.
    eold=fct(x,iblk,ipot)
    dx=-abs((x-2.)/50.)
!   loop1: do j=1,50
    loop1: do j=1,200
             x=x+dx
             enew=fct(x,iblk,ipot)
             if (enew.lt.eold) then
               ierr=ierr+1
               write(6,'(a,a,i2,a,f9.4,a)') ' %VFIT-MSG: short range potential not monotonic',' for block ',iblk,' at ',x,' (a.u)'
               exit loop1 
             end if
             eold=enew
           enddo loop1
!...here for long range potential check
    x=sc1(iblk)
    eold=fct(x,iblk,ipot)
    dx=0.5
    loop2: do j=1,100
             x=x+dx
             enew=fct(x,iblk,ipot)
             if (enew.lt.eold) then
               ierr=ierr+1
               write(6,'(a,a,i2,a,f9.4,a)') ' %VFIT-MSG: long  range potential not monotonic',' for block ',iblk,' at ',x,' (a.u)'
               exit loop2
             end if
             eold=enew
           enddo loop2
  endif
enddo
if(ierr.eq.0) write(6,*) ' POTENTIAL CHECKED, NO ERRORS DETECTED'
return
end subroutine check

subroutine contour(nterms) 

!     Needed information to draw the plot :
!     assuming there is Nvar different variables
!     name(i) contains the name of the variable i
!     fixed(i) is a flag : t/f  if the variable is/is not fixed
!     the number of the first moving variable is stored in move(1)
!     the number of the second moving variable is stored in move(2)
!     range(i,1) contains :
!       . for a non-fixed variable
!        => the starting value for variable(moving(i)), i=1 or 2
!       .  for a fixed variable
!        => the value of the fixed variable
!     range(i,2) contains the final value for variable(moving(i)), i=1 or 2
!     n_step(i) contains the number of steps for variable(moving(i)), i=1 or 2
!     file_ contains the filename where the data is stored
!      the data contains three parts :
!       > the first part consists in a commentar typed by the user
!       > the second part consists in the description of the variables :
!         % name
!         % fixed variable or not
!         % either the fixed value or start value,end value and number of steps
!       > the third part consists in the potential surface :
!         % the two number of the non-fixed variables
!         % a matrix which contains the cut of the potential surface
!     value stores the value of the variables just before calling the potential function

implicit none     
integer(4),intent(in) :: nterms 
character(len=80) file_
integer*4 i,i_theta,i_rr,poten,number_of_rr_points,number_of_theta_points
real(8) :: r_value,rr_value,theta_value,rr_min,rr_max,theta,theta_min,theta_max
real(8),allocatable :: grid(:,:)

write(6,'("Plot arguments :",/)')
write(6,*) 'Please give r'
read(5,*) r_value
write(6,*) 'Please give rr_min , rr_max , number_of_rr_points :'
read(5,*) rr_min,rr_max,number_of_rr_points
write(6,*) 'Please give theta_min , theta_max , number_of_theta_points :'
read(5,*) theta_min,theta_max,number_of_theta_points  
allocate(grid(number_of_theta_points,number_of_rr_points))
grid=0d0

write(6,'(/,"Name of the file ? >",1X,$)')
read(5,'(A80)') file_
open(unit=99,file=file_,status='UNKNOWN',access='SEQUENTIAL')
write(99,'(a25)') '# Begin global parameters'
write(99,'(a12)') 'set tics out'
write(99,'(a10)') 'set nogrid'
write(99,'(a10)') 'set output'
write(99,'(a16)') 'set terminal x11'
write(99,'(a25)') 'set key top right outside'
write(99,'(a13)') 'set nosurface'
write(99,'(a16)') 'set contour base'
write(99,'(a19)') 'set clabel "%10.3e"'
write(99,'(a17)') 'set size 0.9,1.09'
write(99,'(a19)') 'set origin 0.,-0.05'
write(99,'(a20)') 'set view 180,180,1,1'
write(99,'(a15)') 'set autoscale z'
write(99,'(a18)') 'set xlabel "    r"'
write(99,'(a22)') 'set ylabel "    theta"'
write(99,'(a29)') 'set zlabel "    wavefunction"'
write(99,'(a13,E11.4,a2,E11.4,a10)') 'set xrange [ ',rr_min,': ',rr_max,' ] reverse'
write(99,'(a13,E11.4,a2,E11.4,a2)') 'set yrange [ ',theta_min,': ',theta_max,' ]'
write(99,'(a25)') 'set cntrparam cubicspline'
write(99,'(a22)') 'set cntrparam points 5'
write(99,'(a23)') '# End global parameters'
write(99,'(a1)') ' '

do poten=1,nterms
  write(6,'(/,80(1h*))')
  write(6,'("*",78X,"*")')
  write(6,'("* Display of potential surface ",i2,49X,"*")') poten
  write(6,'("*",78X,"*")')
  !      write(*,'("* The potential surface number",1X,I2,1X,"will be displayed",28X,"*")') 10
  !      write(*,'("*",78X,"*")')
  write(6,'(80(1h*),/)')

  ! We can now recorder all the information
  do i_theta=1,number_of_theta_points
    do i_rr=1,number_of_rr_points
      rr_value= ((rr_max-rr_min)*(i_rr-1))/(number_of_rr_points-1)+rr_min
      theta_value=((theta_max-theta_min)*(i_theta-1))/(number_of_theta_points-1)+theta_min
      grid(i_theta,i_rr)=vcalc(rr_value,theta_value,r_value-pots(poten)%re,poten)*219474.63d0/1d6  !conversion from microhartrees to wavenumbers
    enddo
  enddo
  
  if (poten==1) then
    write(99,'(a23)') 'set title "Potential 1"'
  else
    write(99,'(a23)') 'set title "Potential 2"' 
  endif
  write(6,*) minval(grid),(/ (minval(grid)*(20d0-i)/20d0,i=0,20) /)
  write(99,'((a30),21(f14.5,","),(f14.5,",",f14.5))') 'set cntrparam levels discrete ',(/ (minval(grid)*(20d0-i)/20d0,i=0,20) /),100.0,200.0
  write(99,'(a38)') 'splot "-" notitle with lines; pause -1'
  do i_theta=1,number_of_theta_points
    do i_rr=1,number_of_rr_points
      rr_value= ((rr_max-rr_min)*(i_rr-1))/(number_of_rr_points-1)+rr_min
      theta_value=((theta_max-theta_min)*(i_theta-1))/(number_of_theta_points-1)+theta_min
      write(99,'(3d14.5)') rr_value,theta_value,grid(i_theta,i_rr) 
    enddo
    write(99,'(3d14.5)')
  enddo
  write(99,'(a1)') 'e'
enddo

close(99)
deallocate(grid)
             
end subroutine contour
 	
subroutine initialize(autom,autmin,autmax,autstp,guess,weight,azfl,iblk,potential,label,nterms,readfl)
implicit none
logical(4),intent(out) :: autom,azfl,readfl
real(8),intent(out) :: autmin,autmax,autstp,guess,weight
integer(4),intent(out) :: iblk,potential,nterms
character(len=80) :: label

integer(4) :: i,ipot

!...default settings
guess=0.0d0
do ipot=1,max_p
  pots(ipot)%npa=8
  pots(ipot)%maxpws=5
  pots(ipot)%minmps=6
  pots(ipot)%maxmps=10
  pots(ipot)%mpsstp=2
  pots(ipot)%isym=2
  pots(ipot)%mld=0
  pots(ipot)%maxit=10
  pots(ipot)%eps=0.01d0
  pots(ipot)%updmax=0.1d0
  pots(ipot)%xr=2.0
  do i=1,max_b
    pots(ipot)%a(1,i)=guess
!  if (pots(ipot)%wfac.ne.0) call pesi(ip,pots(ipot)%v,np,weight,pots(ipot)%xr(iblk))
    pots(ipot)%a(pots(ipot)%npa+3,i)=0.
    pots(ipot)%a(pots(ipot)%npa+2,i)=0.
    pots(ipot)%a(pots(ipot)%npa+1,i)=0.
  enddo
  pots(ipot)%fitflg=.false.
  pots(ipot)%nblk=0
  pots(ipot)%wfac=0.0
enddo

weight=0.d0
label=' VFIT POTENTIAL'
azfl=.true.
iblk=1
nterms=0
readfl=.false.
autom=.false.
autmin=0.5
autmax=5.0
autstp=0.2
!guess=0.0
potential=1

end subroutine initialize

subroutine convert(filnam,nterms)
implicit none
character(len=80),intent(in) :: filnam
integer(4),intent(in) :: nterms
logical :: exstfl,opnfl
integer(4) :: ierr,ipot,lambda,i,j,k,block
real(8) :: sign_

! convert the datafile corresponding to the A' and A'' surfaces
! write a file corresponding :
!                                  A'+A''
! the first surface corresponds to ------
!                                    2
!                                          lambda
!                                   A'+(-1)      A''
! the second surface corresponds to ----------------
!                                          2

write(*,*) 'pots(1)%nr= ',pots(1)%nr
write(*,*) 'pots(1)%nangle= ',pots(1)%nangle
write(*,*) 'pots(2)%nr= ',pots(2)%nr
write(*,*) 'pots(2)%nangle= ',pots(2)%nangle
write(*,*) 'Value of lambda (lambda>0)'
read(*,*) lambda
sign_=(-1)**lambda
!Now go ahead
inquire(file=filnam,exist=exstfl,opened=opnfl)
open(unit=1,file=filnam,status='UNKNOWN',form='FORMATTED')
if (exstfl) rewind(1)

write(1,*) 'File created by convert, #Number of potentials:'
write(1,*) nterms
write(1,*) '#Size of first potential'
write(1,*) pots(1)%nangle,pots(1)%nr
block=0
do i=1,pots(1)%nr
  do j=1,pots(1)%nangle
    block=(i-1)*pots(1)%nangle+j
    do k=1,pots(1)%npblk(block)
      write(1,*) pots(1)%rr(k,block),pots(1)%angle(block),pots(1)%r(block),(pots(1)%v(k,block)+pots(2)%v(k,block))/2d0
    enddo
  enddo
enddo
write(1,*) '-1.0 0.0 0.0 0.0'
if (nterms==2) then
  write(1,*) '#Size of second potential'
  write(1,*) pots(2)%nangle-2,pots(2)%nr
  block=0
  do i=1,pots(2)%nr
    do j=2,pots(2)%nangle-1
      block=(i-1)*pots(2)%nangle+j
      do k=1,pots(2)%npblk(block)
        write(1,*) pots(2)%rr(k,block),pots(2)%angle(block),pots(2)%r(block),sign_*(pots(1)%v(k,block)-pots(2)%v(k,block))/2d0
      enddo
    enddo
  enddo
write(1,*) '-1.0 0.0 0.0 0.0'
write(1,*) '                '
endif

write(6,'(a,(a))') ' SURFACE SAVED ON FILE ',filnam
close(1)
end subroutine convert

subroutine savfit_formatted(filnam,nterms)
implicit none
character(len=80),intent(in) :: filnam
integer(4),intent(in) :: nterms
character(len=80) :: label
logical :: exstfl,opnfl
integer(4) :: ierr,ipot
!
!...subroutine to save fitted surface on unformatted files (filnam.srf).
!
!   on entry:   filnam  filename
!
!   author: b. follmeg
!   current revision date: 20-june-1988
! ------------------------------------------------------------------------

!...now go ahead
inquire(file=filnam,exist=exstfl,opened=opnfl)
!open(unit=1,file=filnam,status='UNKNOWN',form='FORMATTED',access='SEQUENTIAL')
open(unit=1,file=filnam,status='UNKNOWN',form='FORMATTED')
if (exstfl) rewind(1)

write(6,'(a)') 'Label ?>'
read(5,'(a)') label
write(1,"(a80)",iostat=ierr) label
write(1,"(i10)",iostat=ierr) nterms
do ipot=1,nterms
  write(1,"(i10)",iostat=ierr) pots(ipot)%npa
  write(1,"(e30.20)",iostat=ierr) pots(ipot)%a
  write(1,"(i10)",iostat=ierr) pots(ipot)%ntotal
  write(1,"(i10)",iostat=ierr) pots(ipot)%nblk
  write(1,"(i10)",iostat=ierr) pots(ipot)%nangle
  write(1,"(i10)",iostat=ierr) pots(ipot)%nr
  write(1,"(i10)",iostat=ierr) pots(ipot)%isym
  write(1,"(i10)",iostat=ierr) pots(ipot)%npblk
  write(1,"(e30.20)",iostat=ierr) pots(ipot)%angle
  write(1,"(e30.20)",iostat=ierr) pots(ipot)%r
  write(1,"(e30.20)",iostat=ierr) pots(ipot)%rr
  write(1,"(e30.20)",iostat=ierr) pots(ipot)%v
  write(1,"(i10)",iostat=ierr) pots(ipot)%maxpws
  write(1,"(i10)",iostat=ierr) pots(ipot)%minmps
  write(1,"(i10)",iostat=ierr) pots(ipot)%maxmps
  write(1,"(i10)",iostat=ierr) pots(ipot)%mpsstp
  write(1,"(e30.20)",iostat=ierr) pots(ipot)%updmax
  write(1,"(e30.20)",iostat=ierr) pots(ipot)%wfac
  write(1,"(e30.20)",iostat=ierr) pots(ipot)%ang
  write(1,"(e30.20)",iostat=ierr) pots(ipot)%rmol
  write(1,"(e30.20)",iostat=ierr) pots(ipot)%re
  write(1,"(i10)",iostat=ierr) pots(ipot)%mld
  write(1,"(l1)",iostat=ierr) pots(ipot)%fitflg
  write(1,"(e30.20)",iostat=ierr) pots(ipot)%pinv
  write(1,"(e30.20)",iostat=ierr) pots(ipot)%xr
  write(6,*) 'pots(ipot)%nblk',pots(ipot)%nblk
enddo
write(6,'(a,(a))') ' SURFACE SAVED ON FILE ',filnam
close(1)
return
end subroutine savfit_formatted

subroutine loafit_formatted(filnam,nterms,readfl,input)
implicit none
character(len=80),intent(in) :: filnam
integer(4),intent(in) :: input
integer(4),intent(out) :: nterms
logical,intent(out) :: readfl
character(len=80) :: label
logical :: exstfl,openfl
real(8) :: re
integer(4) :: ierr,i,ntblk,ntot,ntpa,ipot
!
!...subroutine to load fitted surface on unformatted files (filnam.srf).
!
!   on entry:   filnam  filename
!

ntblk=0
ntot=0
ntpa=0
ierr=0
open(unit=1,file=filnam,status='UNKNOWN',form='FORMATTED')

read(1,"(a80)",iostat=ierr) label
write(input,*) 'label : ',label
read(1,"(i10)",iostat=ierr) nterms
write(input,*) 'nterms : ',nterms
do ipot=1,nterms
  read(1,"(i10)",iostat=ierr) pots(ipot)%npa
  read(1,"(e30.20)",iostat=ierr) pots(ipot)%a
  read(1,"(i10)",iostat=ierr) pots(ipot)%ntotal
  read(1,"(i10)",iostat=ierr) pots(ipot)%nblk
  read(1,"(i10)",iostat=ierr) pots(ipot)%nangle
  read(1,"(i10)",iostat=ierr) pots(ipot)%nr
  read(1,"(i10)",iostat=ierr) pots(ipot)%isym
  read(1,"(i10)",iostat=ierr) pots(ipot)%npblk
  read(1,"(e30.20)",iostat=ierr) pots(ipot)%angle
  read(1,"(e30.20)",iostat=ierr) pots(ipot)%r
  read(1,"(e30.20)",iostat=ierr) pots(ipot)%rr
  read(1,"(e30.20)",iostat=ierr) pots(ipot)%v
  read(1,"(i10)",iostat=ierr) pots(ipot)%maxpws
  read(1,"(i10)",iostat=ierr) pots(ipot)%minmps
  read(1,"(i10)",iostat=ierr) pots(ipot)%maxmps
  read(1,"(i10)",iostat=ierr) pots(ipot)%mpsstp
  read(1,"(e30.20)",iostat=ierr) pots(ipot)%updmax
  read(1,"(e30.20)",iostat=ierr) pots(ipot)%wfac
  read(1,"(e30.20)",iostat=ierr) pots(ipot)%ang
  read(1,"(e30.20)",iostat=ierr) pots(ipot)%rmol
  read(1,"(e30.20)",iostat=ierr) pots(ipot)%re
  read(1,"(i10)",iostat=ierr) pots(ipot)%mld
  read(1,"(l1)",iostat=ierr) pots(ipot)%fitflg
  read(1,"(e30.20)",iostat=ierr) pots(ipot)%pinv
  read(1,"(e30.20)",iostat=ierr) pots(ipot)%xr
enddo
do ipot=1,nterms
  ntblk=ntblk+pots(ipot)%nblk
  write(input,*) 'ntblk',ntblk
  ntot=ntot+pots(ipot)%ntotal
  ntpa=ntpa+pots(ipot)%npa
enddo

!       initialize matrices
do i=1,nterms
  call inimat(0,i)
enddo

ipot=1
write(input,'(a,a,/)') ' SURFACE LOADED FROM FILE ',filnam
write(input,'(a,(a),/,a,/,a,i3,t40,a,i3,/,a,i4,t40,a,i3,/,a,/)') ' LABEL:',label(:72), &
     '---------------------------------------------------------------------------',&
     '  NUMBER OF POTENTIALS (NTERMS): ',nterms,'  TOTAL NUMBER OF BLOCKS:        ',ntblk,&
     '  TOTAL NUMBER OF POINTS:       ',ntot,'  TOTAL NUMBER OF COEFFICIENTS:  ',ntpa,&
     '---------------------------------------------------------------------------'
readfl=.true.
close(1)
return

end subroutine loafit_formatted

subroutine deepneg(nterms) 

!  Look for points on the surface
!  where V < -10000 cm-1
!     
implicit none     
integer(4),intent(in) :: nterms
integer(4) poten,i_r,i_rr,i_theta,number_of_r_points,number_of_rr_points,number_of_theta_points
real(8), parameter :: critical_value=-10000d0
real(8) :: r_value,r_min,r_max,rr_value,rr_min,rr_max,theta_value,theta_min,theta_max,tampon

write(6,'("Loops arguments :",/)')
write(6,*) 'Please give r_min,r_max, number_of_r_points'
read(5,*) r_min,r_max,number_of_r_points
write(6,*) 'Please give rr_min , rr_max , number_of_rr_points :'
read(5,*) rr_min,rr_max,number_of_rr_points
write(6,*) 'Please give theta_min , theta_max , number_of_theta_points :'
read(5,*) theta_min,theta_max,number_of_theta_points  

do poten=1,nterms
  write(6,'(/,36(1h*))')
  write(6,'("*",34X,"*")')
  write(6,'("* Check for very deep points in ",i2,1X,"*")') poten
  write(6,'("*",34X,"*")')
  write(6,'(36(1h*),/)')

  ! We can now recorder all the informationdo the loops
  do i_r=1,number_of_r_points
    do i_theta=1,number_of_theta_points
      do i_rr=1,number_of_rr_points
        r_value=((r_max-r_min)*(i_r-1))/(number_of_r_points-1)+r_min
        rr_value=((rr_max-rr_min)*(i_rr-1))/(number_of_rr_points-1)+rr_min
        theta_value=((theta_max-theta_min)*(i_theta-1))/(number_of_theta_points-1)+theta_min
        tampon=vcalc(rr_value,theta_value,r_value-pots(poten)%re,poten)*219474.63d0  !conversion from hartrees to wavenumbers
        if (tampon<=critical_value) then
          write(6,'(/,"R= ",f30.20,/,"theta= ",f30.20,/,"r= ",f30.20,/,"V(cm-1)= ",f30.20,/)') rr_value,theta_value,r_value,tampon
        endif
      enddo
    enddo
  enddo
enddo        
end subroutine deepneg

subroutine loafit_formatted_without_output(filnam,nterms,readfl,input)
implicit none
character(len=80),intent(in) :: filnam
integer(4),intent(in) :: input
integer(4),intent(out) :: nterms
logical,intent(out) :: readfl
character(len=80) :: label
logical :: exstfl,openfl
real(8) :: re
integer(4) :: ierr,i,ntblk,ntot,ntpa,ipot
!
!...subroutine to load fitted surface on unformatted files (filnam.srf).
!
!   on entry: filnam    filename
!

ntblk=0
ntot=0
ntpa=0
ierr=0
open(unit=1,file=filnam,status='UNKNOWN',form='FORMATTED')

read(1,"(a80)",iostat=ierr) label
!write(input,*) 'label : ',label
read(1,"(i10)",iostat=ierr) nterms
!write(input,*) 'nterms : ',nterms
do ipot=1,nterms
  read(1,"(i10)",iostat=ierr) pots(ipot)%npa
  read(1,"(e30.20)",iostat=ierr) pots(ipot)%a
  read(1,"(i10)",iostat=ierr) pots(ipot)%ntotal
  read(1,"(i10)",iostat=ierr) pots(ipot)%nblk
  read(1,"(i10)",iostat=ierr) pots(ipot)%nangle
  read(1,"(i10)",iostat=ierr) pots(ipot)%nr
  read(1,"(i10)",iostat=ierr) pots(ipot)%isym
  read(1,"(i10)",iostat=ierr) pots(ipot)%npblk
  read(1,"(e30.20)",iostat=ierr) pots(ipot)%angle
  read(1,"(e30.20)",iostat=ierr) pots(ipot)%r
  read(1,"(e30.20)",iostat=ierr) pots(ipot)%rr
  read(1,"(e30.20)",iostat=ierr) pots(ipot)%v
  read(1,"(i10)",iostat=ierr) pots(ipot)%maxpws
  read(1,"(i10)",iostat=ierr) pots(ipot)%minmps
  read(1,"(i10)",iostat=ierr) pots(ipot)%maxmps
  read(1,"(i10)",iostat=ierr) pots(ipot)%mpsstp
  read(1,"(e30.20)",iostat=ierr) pots(ipot)%updmax
  read(1,"(e30.20)",iostat=ierr) pots(ipot)%wfac
  read(1,"(e30.20)",iostat=ierr) pots(ipot)%ang
  read(1,"(e30.20)",iostat=ierr) pots(ipot)%rmol
  read(1,"(e30.20)",iostat=ierr) pots(ipot)%re
  read(1,"(i10)",iostat=ierr) pots(ipot)%mld
  read(1,"(l1)",iostat=ierr) pots(ipot)%fitflg
  read(1,"(e30.20)",iostat=ierr) pots(ipot)%pinv
  read(1,"(e30.20)",iostat=ierr) pots(ipot)%xr
enddo
do ipot=1,nterms
  ntblk=ntblk+pots(ipot)%nblk
  !write(input,*) 'ntblk',ntblk
  ntot=ntot+pots(ipot)%ntotal
  ntpa=ntpa+pots(ipot)%npa
enddo

!       initialize matrices
do i=1,nterms
  call inimat(0,i)
enddo

ipot=1
write(input,'(a,a,/)') ' SURFACE LOADED FROM FILE ',filnam
write(input,'(a,(a),/,a,/,a,i3,t40,a,i3,/,a,i4,t40,a,i3,/,a,/)') ' LABEL:',label(:72), &
     '---------------------------------------------------------------------------',&
     '  NUMBER OF POTENTIALS (NTERMS): ',nterms,'  TOTAL NUMBER OF BLOCKS:        ',ntblk,&
     '  TOTAL NUMBER OF POINTS:       ',ntot,'  TOTAL NUMBER OF COEFFICIENTS:  ',ntpa,&
     '---------------------------------------------------------------------------'
readfl=.true.
close(1)
return

end subroutine loafit_formatted_without_output

subroutine potentiel(ideriv, rbig, theta, rd, ipot, vcalc)
implicit none
real(8),intent(inout) :: rbig,theta,rd
integer(4),intent(in) :: ipot, ideriv
real(8) :: vcalc,vvlsum,vvl(max_theta),vvlp(max_theta),pvec(max_theta),rvecp(max_theta), &
           rvecpp(max_theta),b(max_theta,max_r),dr
integer(4) :: i,l,lmax,n,j,iblk
character*80 :: filnam
integer(4) :: nterms
logical :: readfl

! Loads the data contained in the input file
! Definition of subroutine loafit_formatted_without_output
!subroutine loafit_formatted_without_output(filnam,nterms,readfl,input)
!implicit none 
!character(len=80),intent(in) :: filnam
!integer(4),intent(in) :: input
!integer(4),intent(out) :: nterms
!logical,intent(out) :: readfl
!character(len=80) :: label
!logical :: exstfl,openfl
!real(8) :: re 
!integer(4) :: ierr,i,ntblk,ntot,ntpa,ipot
!
!...subroutine to load fitted surface on unformatted files (filnam.srf).
!
!   on entry:   filnam  filename
!

if (ideriv .eq. -1) then
  !write(*,*) 'Point 1'
  ! Name of the input file
  filnam = 'fit_IO_Ar.dat'
  !write(*,*) 'Point 2'
  call loafit_formatted_without_output(filnam, nterms, readfl, 6)
  !write(*,*) 'Point 3'
  ! Initialisation de epsilon et RM
  ! rbig = 0.529d0
  ! vcalc = 0.2..d0 
endif

if (ideriv .eq. 0) then
  !write(*,*) 'Point 4'
  !...function to calculate potential for any given geometry
  !
  lmax=(pots(ipot)%nangle-1)*pots(ipot)%isym+pots(ipot)%mld
  vcalc=0d0
  !                            mld,0                  mld,0
  !...calculate pvec: pvec(1)=d     (theta), pvec(2)=d       (theta), ...
  !                            l                      l+isym
  !
  !   where isym = 2 if homonuclear molecule and isym =1 if heteronuclear
  !   molecule

  !Now, the potential is in the memory
  !We can shift the bond length of the diatom, as required
  !by the lines of code used below
  rd = rd-pots(ipot)%re 

  i=0
  do l=pots(ipot)%mld,lmax,pots(ipot)%isym
    i=i+1
    pvec(i)=dlm0(l,pots(ipot)%mld,theta)
  enddo

  !   now obtain vector of vvl(nangle) coefficients and calculate energy.
  !   note that vvl(1) = vv0 is the isotropic term if mld (mproj) is 0.
  !
  !                      -1       -1
  !   v(r,theta,r)=pvec*p  *b(r)*r  *rvec = pvec * vvl * rvec
  !
  !   first built up b(r) matrix

  iblk=0
  do l=1,pots(ipot)%nr
    do n=1,pots(ipot)%nangle
      iblk=iblk+1
      b(n,l)=fct(rbig,iblk,ipot)
    enddo
  enddo

  !...fast loop if nr = 1 .and. novib
  if (pots(ipot)%nr.gt.1) then
    !...here if normal treatment of r - dependent potential
    !                                             2
    !...calculate rvec: rvec = { 1., (r-re), (r-re) , ... }
    !                       -1     -1
    !    and multiply with r  :   r  * rvec = rvec'
    do i=1,pots(ipot)%nr
      rvecp(i)=0d0
      dr=1d0
      do j=1,pots(ipot)%nr
        rvecp(i)=rvecp(i)+pots(ipot)%rinv(i,j)*dr
        dr=dr*rd
      enddo
    enddo
    !  calculate  b(r) * rvec' = rvec''
    do i=1,pots(ipot)%nangle
      rvecpp(i)=0d0
      do j=1,pots(ipot)%nr
        rvecpp(i)=rvecpp(i)+b(i,j)*rvecp(j)
      enddo
    enddo
    !              -1
    !  calculate  p  * rvec'' = vvl  ( note that vvl(1) = vv0 if mld (mproj) is 0 )
    do i=1,pots(ipot)%nangle
      vvl(i)=0d0
      do j=1,pots(ipot)%nangle
        vvl(i)=vvl(i)+pots(ipot)%pinv(i,j)*rvecpp(j)
      enddo
    enddo
  else
  !                                                  -1
  !...here if number of bond distances is 1:  vvl = p   * b(r) where
  !   b(r) is a now column vector rather than a matrix
    do i=1,pots(ipot)%nangle
      vvl(i)=0.
      do j=1,pots(ipot)%nangle
        vvl(i)=vvl(i)+pots(ipot)%pinv(i,j)*b(j,1)
      enddo
    enddo
  end if
  !
  !  sum over all terms: v(r,theta,r) = pvec(1)*vvl(1) + pvec(2)*vvl(2) +...
  !
  vvlsum=0.
  do i=1,pots(ipot)%nangle
    vvlsum=vvlsum+vvl(i)
    vvlp(i)=pvec(i)*vvl(i)
    vcalc=vcalc+vvlp(i)
  enddo
  return
endif
end subroutine potentiel 

end module fitutil_extended
