within ;
package sfbVFc_VFcHomogenous

  model sfbVFc_VFCHomogenous

    import Modelica.SIunits.*;

    // outputs
    output Real conversion_NIPAM;
    output Real conversion_VFc;

    parameter Temperature T = 65 + 273.15;

    Real ferroFraction;

    // input
    Real runFeed;

    // some variables
    Real realTime( start = 0);
    Real feedTime( start = 0);
    parameter Real time_feed1 = 3000;
    parameter Real time_feed2 = 6000;
    parameter Real time_feed3 = 9500;
    parameter Real ferroFraction_feed = 0.15;

    // model componments
    moleBalance         moleBalance0;
    microgelProperties  microgelProperties0;

  equation

    // get outputs
    conversion_VFc = moleBalance0.conversion_VFc;
    conversion_NIPAM = moleBalance0.conversion_NIPAM;

    ferroFraction = microgelProperties0.ferroFraction;

    // schedule
    der(realTime) = 1;

    if feedTime>moleBalance0.feedTime then
      runFeed = 0;
      der(feedTime) = 0;
    elseif realTime>time_feed3+2/8*moleBalance0.feedTime then
      runFeed = 0;
      der(feedTime) = 0;
    elseif realTime>time_feed3 then
      runFeed = 1;
      der(feedTime) = 1;
    elseif realTime>time_feed2+3/8*moleBalance0.feedTime then
      runFeed = 0;
      der(feedTime) = 0;
    elseif realTime>time_feed2 then
      runFeed = 1;
      der(feedTime) = 1;
    elseif realTime>time_feed1+3/8*moleBalance0.feedTime then
      runFeed = 0;
      der(feedTime) = 0;
    elseif realTime>time_feed1 then
      runFeed = 1;
      der(feedTime) = 1;
    else
      runFeed = 0;
      der(feedTime) = 0;
    end if;

    // set flow
    moleBalance0.runFeed = runFeed;

    moleBalance0.T = T;

    // connection of microgel properties
    connect(moleBalance0.microgelExchange,
            microgelProperties0.microgelExchange);

  end sfbVFc_VFCHomogenous;

  model moleBalance

    import Modelica.SIunits.*;

    // import parameter
    kineticParameter pKinetic;
    reactantParameter pReactant;

    // add microgel properties
    connectorMicrogel microgelExchange;

    // set temperature
    Temperature T;

    // moles and concentrations of reactants
    MolarFlowRate nF_H2O                          "molar flow [mol/s]";
    AmountOfSubstance n_H2O( start = 4.675+0.83325*4/8)  "number of moles [mol] start value equals 85ml";
    Concentration c_H2O                           "concentration [mol/m^3]";
    Mass m_H2O                                    "mass [kg]";

    AmountOfSubstance n_I(   start = 0.2E-3)     "number of moles [mol]start value equals 0.05g";
    Concentration c_I                             "concentration [mol/m^3]";
    Mass m_I                                      "mass [kg]";

    AmountOfSubstance n_R                 "number of moles [mol]";
    Concentration c_R                             "concentration [mol/m^3]";

    AmountOfSubstance n_NIPAM( start = 12E-3)    "number of moles [mol]";
    Concentration c_NIPAM                           "concentration [mol/m^3]";
    Mass m_NIPAM                                    "mass [kg]";

    AmountOfSubstance n_BIS(start = 0.7E-3)      "number of moles [mol]";
    Concentration c_BIS                           "concentration [mol/m^3]";
    Mass m_BIS                                    "mass [kg]";

    MolarFlowRate nF_VFc                          "molar flow [mol/s]";
    AmountOfSubstance n_VFc(start = 0.5E-3*4/8)   "number of moles [mol]";
    Concentration c_VFc                           "concentration [mol/m^3]";
    Mass m_VFc                                    "mass [kg]";

    AmountOfSubstance n_PDB(   start = 0)         "number of moles [mol]";
    Concentration c_PDB                           "concentration [mol/m^3]";

    AmountOfSubstance n_R_NIPAM( start = 0)         "number of moles [mol]";
    Concentration c_R_NIPAM                         "concentration [mol/m^3]";

    AmountOfSubstance n_R_BIS( start = 0)         "number of moles [mol]";
    Concentration c_R_BIS                         "concentration [mol/m^3]";

    AmountOfSubstance n_R_VFc( start = 0)         "number of moles [mol]";
    Concentration c_R_VFc                         "concentration [mol/m^3]";

    AmountOfSubstance n_pNIPAM(  start = 0)         "number of moles [mol]";
    Mass m_pNIPAM                                   "mass [kg]";

    AmountOfSubstance n_pBIS(  start = 0)         "number of moles [mol]";
    Mass m_pBIS                                   "mass [kg]";

    AmountOfSubstance n_pVFc(  start = 0)         "number of moles [mol]";
    Mass m_pVFc                                   "mass [kg]";

    AmountOfSubstance n_X(   start = 0)           "number of moles [mol]";
    Concentration c_X                             "concentration [mol/m^3]";

    // reaction rates
    Real Rd           "reaction rate [mol/m^3/s]";
    Real Ri_NIPAM       "reaction rate [mol/m^3/s]";
    Real Ri_BIS       "reaction rate [mol/m^3/s]";
    Real Ri_VFc       "reaction rate [mol/m^3/s]";

    Real Rp_NIPAM_NIPAM   "reaction rate [mol/m^3/s]";
    Real Rp_BIS_NIPAM   "reaction rate [mol/m^3/s]";
    Real Rp_NIPAM_BIS   "reaction rate [mol/m^3/s]";
    Real Rp_BIS_BIS   "reaction rate [mol/m^3/s]";

    Real Rp_VFc_VFc   "reaction rate [mol/m^3/s]";
    Real Rp_NIPAM_VFc   "reaction rate [mol/m^3/s]";
    Real Rp_BIS_VFc   "reaction rate [mol/m^3/s]";
    Real Rp_VFc_NIPAM   "reaction rate [mol/m^3/s]";
    Real Rp_VFc_BIS   "reaction rate [mol/m^3/s]";

    Real Rp_NIPAM_PDB   "reaction rate [mol/m^3/s]";
    Real Rp_BIS_PDB   "reaction rate [mol/m^3/s]";
    Real Rp_VFc_PDB   "reaction rate [mol/m^3/s]";

    Real Rt_NIPAM       "reaction rate [mol/m^3/s]";
    Real Rt_BIS       "reaction rate [mol/m^3/s]";
    Real Rt_VFc       "reaction rate [mol/m^3/s]";

    Volume Vol        "volume [m^3]";

    // conversion calculation
    Real conversion_NIPAM;
    Real conversion_VFc;

    // flow realted stuff
    parameter AmountOfSubstance n_H2O_fed = 0.83325*4/8 "number of moles [mol]";
    parameter MoleFraction x_VFc_fed = 0.5E-3 / 0.83325 "molar fraction [mol/mol]";
    parameter MolarFlowRate nF_fed = 0.061105 "molar flow [mol/s]";
    Real feedTime;
    Real runFeed;

  equation

    // determine duration of feeding
    feedTime = n_H2O_fed / nF_fed;

    // mole balances, concentrations and masses of reactants
    nF_H2O        = nF_fed*runFeed;
    der(n_H2O)    = nF_H2O;
    n_H2O         = c_H2O*Vol;
    m_H2O         = pReactant.Mw_H2O*n_H2O;

    der(n_I)      = - Rd*Vol;
    n_I           = c_I*Vol;
    m_I           = pReactant.Mw_I*n_I;

    0             = (2*Rd - Ri_NIPAM - Ri_BIS - Ri_VFc)*Vol;
    n_R           = c_R*Vol;

    der(n_NIPAM)    = -(Rp_NIPAM_NIPAM+Rp_BIS_NIPAM+Rp_VFc_NIPAM)*Vol;
    n_NIPAM         = c_NIPAM*Vol;
    m_NIPAM         = pReactant.Mw_NIPAM*n_NIPAM;

    der(n_BIS)    = -(Rp_NIPAM_BIS+Rp_BIS_BIS+Rp_VFc_BIS)*Vol;
    n_BIS         = c_BIS*Vol;
    m_BIS         = pReactant.Mw_BIS*n_BIS;

    nF_VFc        = nF_fed*x_VFc_fed*runFeed;
    der(n_VFc)    = nF_VFc-(Rp_VFc_VFc+Rp_NIPAM_VFc+Rp_BIS_VFc)*Vol;
    n_VFc         = c_VFc*Vol;
    m_VFc         = pReactant.Mw_VFc*n_VFc;

    der(n_PDB)    = (Rp_NIPAM_BIS+Rp_BIS_BIS+Rp_VFc_BIS - Rp_NIPAM_PDB-Rp_BIS_PDB-Rp_VFc_PDB)*Vol;
    n_PDB         = c_PDB*Vol;

    der(n_pNIPAM)   = (Rp_NIPAM_NIPAM + Rp_BIS_NIPAM + Rp_VFc_NIPAM)*Vol;
    m_pNIPAM        = pReactant.Mw_NIPAM*n_pNIPAM;

    der(n_pBIS)   = (+ Rp_NIPAM_BIS + Rp_BIS_BIS + Rp_VFc_BIS)*Vol;
    m_pBIS        = pReactant.Mw_BIS*n_pBIS;

    der(n_pVFc)   = (+ Rp_NIPAM_VFc + Rp_BIS_VFc + Rp_VFc_VFc)*Vol;
    m_pVFc        = pReactant.Mw_VFc*n_pVFc;

    der(n_R_NIPAM)  = (Ri_NIPAM - Rp_NIPAM_BIS + Rp_BIS_NIPAM - Rp_NIPAM_VFc + Rp_VFc_NIPAM - Rp_NIPAM_PDB - Rt_NIPAM)*Vol;
    n_R_NIPAM       = c_R_NIPAM*Vol;

    der(n_R_BIS)  = (Ri_BIS + Rp_NIPAM_BIS - Rp_BIS_NIPAM - Rp_BIS_VFc + Rp_VFc_BIS + Rp_NIPAM_PDB + Rp_VFc_PDB - Rt_BIS)*Vol;
    n_R_BIS       = c_R_BIS*Vol;

    der(n_R_VFc)  = (Ri_VFc + Rp_NIPAM_VFc - Rp_VFc_NIPAM + Rp_BIS_VFc - Rp_VFc_BIS - Rp_VFc_PDB  - Rt_VFc)*Vol;
    n_R_VFc       = c_R_VFc*Vol;

    der(n_X)      = (Rp_NIPAM_PDB + Rp_BIS_PDB)*Vol;
    n_X           = c_X * Vol;

    // volume of the solution assuming a lot of solvent
    pReactant.rho_H2O*Vol = m_H2O;

    // set temperature of reaction rates
    pKinetic.T  = T;

    // reaction rates
    Rd           = pKinetic.kd*c_I;
    Ri_NIPAM      = pKinetic.kp_NIPAM_NIPAM*c_R*c_NIPAM;
    Ri_BIS        = pKinetic.kp_NIPAM_BIS*c_R*c_BIS;
    Ri_VFc        = pKinetic.kp_NIPAM_VFc*c_R*c_VFc;

    Rp_NIPAM_NIPAM  = pKinetic.kp_NIPAM_NIPAM*c_NIPAM*c_R_NIPAM;
    Rp_BIS_NIPAM  = pKinetic.kp_BIS_NIPAM*c_NIPAM*c_R_BIS;
    Rp_NIPAM_BIS  = pKinetic.kp_NIPAM_BIS*c_BIS*c_R_NIPAM;
    Rp_BIS_BIS  = pKinetic.kp_BIS_BIS*c_BIS*c_R_BIS;

    Rp_VFc_VFc  = pKinetic.kp_VFc_VFc*c_R_VFc*c_VFc;
    Rp_NIPAM_VFc  = pKinetic.kp_NIPAM_VFc*c_VFc*c_R_NIPAM;
    Rp_BIS_VFc  = pKinetic.kp_BIS_VFc*c_VFc*c_R_BIS;
    Rp_VFc_NIPAM  = pKinetic.kp_VFc_NIPAM*c_R_VFc*c_NIPAM;
    Rp_VFc_BIS  = pKinetic.kp_VFc_BIS*c_R_VFc*c_BIS;

    Rp_NIPAM_PDB  = pKinetic.kp_NIPAM_BIS*c_R_NIPAM*c_PDB;
    Rp_BIS_PDB  = pKinetic.kp_BIS_BIS*c_R_BIS*c_PDB;
    Rp_VFc_PDB  = pKinetic.kp_VFc_BIS*c_R_VFc*c_PDB;

    Rt_NIPAM      = 2*pKinetic.kt*c_R_NIPAM*c_R_NIPAM + pKinetic.kt*c_R_NIPAM*(c_R_BIS+c_R_VFc);
    Rt_BIS      = 2*pKinetic.kt*c_R_BIS*c_R_BIS + pKinetic.kt*(c_R_NIPAM+c_R_VFc)*c_R_BIS;
    Rt_VFc      = 2*pKinetic.kt*c_R_VFc*c_R_VFc + pKinetic.kt*c_R_VFc*(c_R_NIPAM+c_R_BIS);

    // conversion calculation
    conversion_NIPAM = (n_pNIPAM)/(n_NIPAM+n_pNIPAM);
    conversion_VFc = (n_pVFc)/(n_VFc+n_pVFc+1E-9);

    // set variables in microgel properties
    microgelExchange.R_I = Ri_NIPAM + Ri_BIS + Ri_VFc;
    microgelExchange.R_NIPAM = Rp_NIPAM_NIPAM+Rp_BIS_NIPAM+Rp_VFc_NIPAM;
    microgelExchange.R_BIS = Rp_NIPAM_BIS+Rp_BIS_BIS+Rp_VFc_BIS;
    microgelExchange.R_VFc = Rp_NIPAM_VFc+Rp_BIS_VFc+Rp_VFc_VFc;
    microgelExchange.R_PDB = Rp_NIPAM_PDB+Rp_BIS_PDB;

    microgelExchange.kp_NIPAM_NIPAM = pKinetic.kp_NIPAM_NIPAM;
    microgelExchange.kp_NIPAM_BIS = pKinetic.kp_NIPAM_BIS;
    microgelExchange.kp_BIS_NIPAM = pKinetic.kp_BIS_NIPAM;
    microgelExchange.kp_BIS_BIS = pKinetic.kp_BIS_BIS;

    microgelExchange.kp_VFc_VFc = pKinetic.kp_VFc_VFc;
    microgelExchange.kp_NIPAM_VFc = pKinetic.kp_NIPAM_VFc;
    microgelExchange.kp_VFc_NIPAM = pKinetic.kp_VFc_NIPAM;
    microgelExchange.kp_BIS_VFc = pKinetic.kp_BIS_VFc;
    microgelExchange.kp_VFc_BIS = pKinetic.kp_VFc_BIS;

    microgelExchange.n_pNIPAM = n_pNIPAM;
    microgelExchange.n_pBIS = n_pBIS;
    microgelExchange.n_pVFc = n_pVFc;

    microgelExchange.c_NIPAM = c_NIPAM;
    microgelExchange.c_BIS = c_BIS;
    microgelExchange.c_VFc = c_VFc;
    microgelExchange.c_PDB = c_PDB;

    microgelExchange.c_R_NIPAM = c_R_NIPAM;
    microgelExchange.c_R_BIS = c_R_BIS;
    microgelExchange.c_R_VFc = c_R_VFc;

    microgelExchange.Mw_NIPAM = pReactant.Mw_NIPAM;
    microgelExchange.Mw_BIS = pReactant.Mw_BIS;
    microgelExchange.Mw_VFc = pReactant.Mw_VFc;
    microgelExchange.rho_Poly = pReactant.rho_Poly;

    microgelExchange.Vol = Vol;

  end moleBalance;

  model microgelProperties

    import Modelica.SIunits.*;

    // connector
    connectorMicrogel microgelExchange;

    parameter Real pi = 3.14159265359;
    parameter Real N_A = 6.023E23;

    // cross linker fraction
    Real monomerFraction;
    Real crossLinkerFraction;
    Real ferroFraction;

    // size related variables
    Concentration c_particle_R_NIPAM;
    Concentration c_particle_R_BIS;
    Concentration c_particle_R_VFc;

    Real n_particle_NIPAM( start = 1) "number of moles [mol]";
    Real n_particle_BIS( start = 1) "number of moles [mol]";
    Real n_particle_VFc( start = 1) "number of moles [mol]";

    Real gr_n_particle_NIPAM "number of moles [mol]";
    Real gr_n_particle_BIS "number of moles [mol]";
    Real gr_n_particle_VFc "number of moles [mol]";

    Real gr_R_coag_NIPAM "number of moles [mol]";
    Real gr_R_coag_BIS "number of moles [mol]";
    Real gr_R_coag_VFc "number of moles [mol]";

    Real N( start = 1E-10);
    Real c_N;
    Real nBar2;
    Real R_coag;

    Real n_particle_NIPAM_ave;
    Real n_particle_BIS_ave;
    Real n_particle_VFc_ave;

    parameter Real c_surf = 0.1104 "concentration of surfactant mol/m^3";
    parameter Real c_CMC = 0.92 "critical micelle concentration mol/m^3";

    parameter Real k_coag = 2*1.7;

    Diameter   d_particle         "particle diameter [nm]";
    Radius     r_particle         "particle diameter [nm]";
    Volume     V_particle         "particle diameter [nm^3]";
    Diameter   d_particle_log;

  equation

    // crosslinker fraction in the droplets
    monomerFraction = +microgelExchange.R_NIPAM/(microgelExchange.R_NIPAM+microgelExchange.R_BIS+microgelExchange.R_VFc+1E-6);
    crossLinkerFraction = +microgelExchange.R_BIS/(microgelExchange.R_NIPAM+microgelExchange.R_BIS+microgelExchange.R_VFc+1E-6);
    ferroFraction = +microgelExchange.R_VFc/(microgelExchange.R_NIPAM+microgelExchange.R_BIS+microgelExchange.R_VFc+1E-9);

    // number Partickes
    R_coag = k_coag*exp(-2*c_surf)*( microgelExchange.R_BIS+microgelExchange.R_NIPAM+microgelExchange.R_PDB+microgelExchange.R_VFc)/(microgelExchange.R_I+1E-9)*c_N^2;

    der(N)= (microgelExchange.R_I - R_coag)*microgelExchange.Vol;
    c_N = N/microgelExchange.Vol;

    nBar2 = (microgelExchange.c_R_NIPAM+microgelExchange.c_R_BIS+microgelExchange.c_R_VFc)/c_N;

    n_particle_NIPAM_ave = microgelExchange.n_pNIPAM / N;
    n_particle_BIS_ave = microgelExchange.n_pBIS / N;
    n_particle_VFc_ave = microgelExchange.n_pVFc / N;

    // droplet diameter
    c_particle_R_NIPAM = nBar2*microgelExchange.c_R_NIPAM/(microgelExchange.c_R_NIPAM+microgelExchange.c_R_BIS+microgelExchange.c_R_VFc+1E-9);
    c_particle_R_BIS = nBar2*microgelExchange.c_R_BIS/(microgelExchange.c_R_NIPAM+microgelExchange.c_R_BIS+microgelExchange.c_R_VFc+1E-9);
    c_particle_R_VFc = nBar2*microgelExchange.c_R_VFc/(microgelExchange.c_R_NIPAM+microgelExchange.c_R_BIS+microgelExchange.c_R_VFc+1E-9);

    gr_n_particle_NIPAM = (microgelExchange.kp_NIPAM_NIPAM*c_particle_R_NIPAM + microgelExchange.kp_BIS_NIPAM*c_particle_R_BIS + microgelExchange.kp_VFc_NIPAM*c_particle_R_VFc)*microgelExchange.c_NIPAM;
    gr_n_particle_BIS = (microgelExchange.kp_NIPAM_BIS*c_particle_R_NIPAM + microgelExchange.kp_BIS_BIS*c_particle_R_BIS + microgelExchange.kp_VFc_BIS*c_particle_R_VFc)*microgelExchange.c_BIS;
    gr_n_particle_VFc = (microgelExchange.kp_NIPAM_VFc*c_particle_R_NIPAM + microgelExchange.kp_BIS_VFc*c_particle_R_BIS + microgelExchange.kp_VFc_VFc*c_particle_R_VFc)*microgelExchange.c_VFc;

    gr_R_coag_NIPAM = + R_coag/N*microgelExchange.Vol * n_particle_NIPAM_ave;
    gr_R_coag_BIS = + R_coag/N*microgelExchange.Vol * n_particle_BIS_ave;
    gr_R_coag_VFc = + R_coag/N*microgelExchange.Vol * n_particle_VFc_ave;

    der(n_particle_NIPAM) = gr_n_particle_NIPAM+ gr_R_coag_NIPAM;
    der(n_particle_BIS) = gr_n_particle_BIS + gr_R_coag_BIS;
    der(n_particle_VFc) = gr_n_particle_VFc+ gr_R_coag_VFc;

    V_particle = 1/6*pi*d_particle^3;
    V_particle*microgelExchange.rho_Poly = (n_particle_NIPAM*microgelExchange.Mw_NIPAM+n_particle_BIS*microgelExchange.Mw_BIS+n_particle_VFc*microgelExchange.Mw_VFc)/N_A*1E27;

    r_particle = d_particle/2;
    d_particle_log = log10(d_particle+1E0);

  end microgelProperties;

  model kineticParameter

     import Modelica.SIunits.*;

     // initiator decomposition
     parameter Real kd_0 =             34.45       "reaction constant [1/s]";
     parameter Real Ead_0 =           123400       "activation energy*gas constant [J/mol/K]";
     Real kd                                       "reaction constant [1/s]";

     // propagation reaction
     parameter Real kp_NIPAM_NIPAM_0 =     6.1900        "reaction constant [m^3/mol/s]";
     parameter Real Eap_NIPAM_NIPAM =      16160        "activation energy*gas constant [K]";
     Real kp_NIPAM_NIPAM                                "reaction constant [m^3/mol/s]";

     parameter Real kp_NIPAM_BIS_0 =     6.9899        "reaction constant [m^3/mol/s]";
     parameter Real Eap_NIPAM_BIS =      18170        "activation energy*gas constant [K]";
     Real kp_NIPAM_BIS                                "reaction constant [m^3/mol/s]";

     parameter Real kp_BIS_NIPAM_0 =     14.2776        "reaction constant [m^3/mol/s]";
     parameter Real Eap_BIS_NIPAM =      30050        "activation energy*gas constant [K]";
     Real kp_BIS_NIPAM                                "reaction constant [m^3/mol/s]";

     parameter Real kp_BIS_BIS_0 =     15.1572        "reaction constant [m^3/mol/s]";
     parameter Real Eap_BIS_BIS =      26627        "activation energy*gas constant [K]";
     Real kp_BIS_BIS                                "reaction constant [m^3/mol/s]";

     parameter Real r_VFc_VFc = 0.0244;
     parameter Real r_x_VFc = 2.72178;
     parameter Real r_VFc_x = 0.00010799;

     Real kp_VFc_VFc                   "reaction constant [m^3/mol/s]";
     Real kp_NIPAM_VFc                 "reaction constant [m^3/mol/s]";
     Real kp_BIS_VFc                    "reaction constant [m^3/mol/s]";
     Real kp_VFc_NIPAM              "reaction constant [m^3/mol/s]";
     Real kp_VFc_BIS                   "reaction constant [m^3/mol/s]";

     // termination reaction (combined diffusion and reaction)
     parameter Real kt =               8.7967          "reaction constant [m^3/mol/s]";
     // temperature
     Temperature T                                  "reactor temperature [K]";

     // ideal gas constant
     parameter Real Rgas =             8.314        "ideal gas constant [J/mol/K]";

  equation

     // initiator decomposition
     kd         = exp(kd_0         - Ead_0/(Rgas*T));

     // propagation reaction
     kp_NIPAM_NIPAM = exp(kp_NIPAM_NIPAM_0 - Eap_NIPAM_NIPAM/(Rgas*T));
     kp_NIPAM_BIS = exp(kp_NIPAM_BIS_0 - Eap_NIPAM_BIS/(Rgas*T));
     kp_BIS_NIPAM = exp(kp_BIS_NIPAM_0 - Eap_BIS_NIPAM/(Rgas*T));
     kp_BIS_BIS = exp(kp_BIS_BIS_0 - Eap_BIS_BIS/(Rgas*T));

     kp_VFc_VFc = r_VFc_VFc * kp_NIPAM_NIPAM;
     kp_NIPAM_VFc = r_x_VFc * kp_NIPAM_NIPAM;
     kp_BIS_VFc = r_x_VFc * kp_BIS_NIPAM;
     kp_VFc_NIPAM = r_VFc_x * kp_NIPAM_NIPAM;
     kp_VFc_BIS = r_VFc_x * kp_NIPAM_BIS;

  end kineticParameter;

  model reactantParameter

    import Modelica.SIunits.*;

    parameter MolarMass Mw_I = 0.27119       "molecular weight [kg/mol]";
    parameter MolarMass Mw_NIPAM = 0.11316     "molecular weight [kg/mol]";
    parameter MolarMass Mw_BIS = 0.15417     "molecular weight [kg/mol]";
    parameter MolarMass Mw_VFc = 0.21207      "molecular weight [kg/mol]";
    parameter MolarMass Mw_H2O = 0.018015     "molecular weight [kg/mol]";
    parameter MolarMass Mw_O2 = 0.0320       "molecular weight [kg/mol]";

    parameter Density rho_H2O = 977.63       "density [kg/m^3]";
    parameter Density rho_NIPAM = 1100         "density [kg/m^3]";
    parameter Density rho_BIS = 1100         "density [kg/m^3]";
    parameter Density rho_Poly = 1370        "density [kg/m^3]";

    parameter SpecificHeatCapacity cp_H2O = 4190   "specific heat capacity [J/kg/K]";

    parameter Concentration c_NIPAM_sat = 265 "critical concentration [mol/m^3]";
    //Quelle:https://pubchem.ncbi.nlm.nih.gov/compound/n-isopropylacrylamide#section=Color 265 mol/m^3
    parameter Concentration c_BIS_sat =  129 "critical mass concentration [mol/m^3]";
    //Quelle:https://www.sigmaaldrich.com/catalog/product/sial/146072?lang=de&region=DE

  end reactantParameter;

  connector connectorMicrogel

    import Modelica.SIunits.*;

    Real R_I;
    Real R_NIPAM;
    Real R_BIS;
    Real R_VFc;
    Real R_PDB;

    Real kp_NIPAM_NIPAM;
    Real kp_NIPAM_BIS;
    Real kp_BIS_NIPAM;
    Real kp_BIS_BIS;

    Real kp_VFc_VFc;
    Real kp_NIPAM_VFc;
    Real kp_VFc_NIPAM;
    Real kp_BIS_VFc;
    Real kp_VFc_BIS;

    Concentration n_pNIPAM;
    Concentration n_pBIS;
    Concentration n_pVFc;

    Concentration c_NIPAM;
    Concentration c_BIS;
    Concentration c_PDB;
    Concentration c_VFc;

    Concentration c_R_NIPAM;
    Concentration c_R_BIS;
    Concentration c_R_VFc;

    MolarMass Mw_NIPAM;
    MolarMass Mw_BIS;
    MolarMass Mw_VFc;
    Density rho_Poly;

    Volume Vol;

  end connectorMicrogel;

end sfbVFc_VFcHomogenous;
