%%
% Function to perform maximum likelihood factor analysis (MLFA) and produce
% results consistent with a singular value decomposition (SVD) format such
% that Xhat=USV'.  Results are provided in the original space of the data
% so that any pre-processing (centering, scaling should be done external to 
% the function if desired.
%
% Note: This function uses the function factoran.m in the MatLab Statistics
%       and Machine Learning Toolbox.
%
% Usage:
% [U,S,V] = mlfa_usv(X,nfac) - returns the nfac dimensional MLFA
%          decomposition of the data in SVD format (U is the left
%          singular matrix, V is the right singular matrix, and S is the
%          diagonal matrix of singular values.
% [U,S,V,Psi] = mlfa_usv(X,nfac) - also returns the estimate of measurement
%          error variances for each column of data in Psi. Note that 
%          these are scaled to the original space of the data and not the
%          correlation space.
% [U,S,V,Psi,MLFAout] = mlfa_usv(X,nfac) - also returns the structured 
%          variable MLFAout indicating exit conditions of the algorithm.
%          The fields are:
%            MLFAout.hwflg = set to true if a Heywood case is encountered,
%                         false otherwise.
%            MLFAout.hwindx = a vector containing indices of variables 
%                         that gave a Heywood condition.
%            MLFAout.retindx = a vector containing the indices of retained
%                         variables if those giving rise to a Heywood  
%                         condition were removed (see "heyopt" option).
%            MLFAout.stata = a structured variable containing the 
%                         statistical information returned by factoran 
%                         (log-likelihood, degrees of freedom, chi-
%                         squared value, p-value). (Note: the statistics
%                         will be slightly different if the data are not
%                         mean-centered.)
% [...] = mlfa_usv(X,nfac,options) - Allows specification of option pairs 
%           as described below.
% Options:
%   ['delta',deltaval] - sets the limiting limiting value for the 
%        detection of Heywood cases. deltval must be a number greater 
%        than zero.
%        Default: 1e-6
%        Comment: Ideally set to less than the minimum ratio of 
%                 measurement noise variance to column variance (about 
%                 zero) for all variables. Too large values can lead to
%                 false detection of Heywood cases. Too small values may
%                 affect scores when heyopt=0 (see below).
%   ['heyopt',heyval] - determines action to be taken in the event of 
%        Heywood case.  Allowed values [0 1 2].
%           heyval=0 indicates that flagged elements of Psi(and the scores)
%                    should be estimated using the converged Psi values 
%                    (i.e. with some approaching delta).
%           heyval=1 indicates that the variance of flagged variables 
%                    should be set to the smallest element of Psi that did
%                    not give a Heywood case.
%           heyval=2 indicates the variables giving rise to a Heywood case
%                    should be removed. In this case, the dimensions of
%                    S,V and Psi will be reduced. Removed and retained 
%                    variables are indicated in PAFout.hwindx and
%                    PAFout.retindx.
%         Default: 1
%
%%
%
function [U,S,V,Psi,MLFAout]=mlfa_usv(X,nfac,varargin);
%
%%
% Parse and check the inputs and initialize variables
%
validateattributes(X,{'double'},{'ndims',2},1);
[nrow,ncol]=size(X);
if nrow<2 | ncol<2
    error('Expected input number 1 to be a matrix.')
end
imin=min([nrow ncol]);
validateattributes(nfac,{'double'},{'scalar','integer','>',0,'<',imin},2);
p=inputParser;
testfcn=@(x)validateattributes(x,{'double'},{'scalar','>',0,'<',0.1},'delta');
addParameter(p,'delta',1e-6,testfcn);
testfcn=@(x)validateattributes(x,{'double'},{'integer','scalar','>=',0,'<',3},'heyopt');
addParameter(p,'heyopt',1,testfcn);
parse(p,varargin{:})
deltaval=p.Results.delta;
heyopt=p.Results.heyopt;
%
MLFAout.hwflg=false;         % 0=normal, 1=Heywood case
MLFAout.hwindx=[];           % Indices of Heywood cases
MLFAout.retindx=[1:ncol];    % Indices of retained variables
%%
% Calculate correlation matrix about zero and perform MLFA
%
CovX=X'*X/nrow;             % Covariance about zero
Xsd=sqrt(diag(CovX))';      % Variances on diagonal of X
Xsc=X./(ones(nrow,1)*Xsd);  % Scaled X values for scores calculation
R=CovX./(Xsd'*Xsd);    % Correlation about zero
%
[Lam,Psi,T,stats]=factoran(R,nfac,'xtype','covariance','rotate','none',...
    'nobs',nrow,'delta',deltaval);
MLFAout.stats=stats;                 % Saves statistics
Rtemp=Lam*Lam';                      % Orthogonalizes loadings
[U,S,V]=svds(Rtemp,nfac);
Lam=U*sqrt(S);
%%
% Check for Heywood cases and adjust results accordingly
%
PsiOrig=Psi;
indx=[];
if ~isfield(stats,'p')             % This is way to check for Heywood
    MLFAout.hwflg=true;            % Set flag
    indx=find(Psi<deltaval*1.001); % Note: This is a way to locate Heywood
end                                % variables since they may not reach delta
MLFAout.hwindx=indx;
if ~isempty(indx)               % Heywood cases - need adjustments
    if length(indx)>(ncol-nfac) & heyopt>0  % Check unlikely case of mostly Heywood
        warning('Too many Heywood instances - heyopt set to 0.')
        heyopt=0;
    end
%
% Process Heywood cases.  Note: No adjustmennt needed for heyopt=0.
%
    if heyopt==1              % Replace elements with smallest delta
        Psi(indx)=-100;       % To distinguish removed elements
        Psimin=min(abs(Psi)); % Minimum non-Heywood value
        Psi(indx)=Psimin;     % Replace Heywood cases
    elseif heyopt==2          % Eliminate Heywood variables
        retindx=[1:ncol];
        retindx(indx)=[];     % Indices of variables retained
        MLFAout.retindx=retindx;
        Xsc(:,indx)=[];       % Remove variables from X
        Lam(indx,:)=[];       % Remove variables from loadings
        Psi(indx)=[];         % Remove variables from unique variances
        CovX(indx,:)=[]; CovX(:,indx)=[];    % Fix covariance
        Xsd(indx)=[];         % Fix SD values for scaling
        R(indx,:)=[]; R(:,indx)=[];          % Fix correlation matrix
        Rtemp=Lam*Lam';       % This part re-orthogonalizes loadings
        [U,S,V]=svds(Rtemp,nfac);
        Lam=U*sqrt(S);
    end
%
end
%
% Now generate results scaled back to the original space
%
Siginv=diag(1./Psi);                   % Weight matrix
Xhat=Xsc*Siginv*Lam*inv(Lam'*Siginv*Lam)*Lam'; % ML projection
Xhatsc=Xhat.*(ones(nrow,1)*Xsd);       % Rescale projected data
[U,S,V]=svds(Xhatsc,nfac);             % PCA format
Psi=Psi.*(Xsd'.^2);                    % Scaled variance estimates
%%
