%%
% Function to perform principal axis factorization (PAF) and produce
% results consistent with a singular value decomposition (SVD) format such
% that Xhat=USV'.  Results are provided in the original space of the data
% so that any pre-processing (centering, scaling should be done external to 
% the function if desired.
%
% Usage:
% [U,S,V] = paf_usv(X,nfac) - returns the nfac dimensional PAF 
%          decomposition of the data in SVD format (U is the left
%          singular matrix, V is the right singular matrix, and S is the
%          diagonal matrix of singular values.
% [U,S,V,Psi] = paf_usv(X,nfac) - also returns the estimate of measurement
%          error variances for each column of data in Psi. Note that 
%          these are scaled to the original space of the data and not the
%          correlation space.
% [U,S,V,Psi,PAFout] = paf_usv(X,nfac) - also returns the structured 
%          variable PAFout indicating exit conditions of the algorithm.
%          The fields are:
%             PAFout.maxflg - set to true if the maximum number of 
%                         iterations (1000) occurred before convergence, 
%                         false otherwise.
%             PAFout.hwflg = set to true if a Heywood case is encountered,
%                         false otherwise.
%             PAFout.hwindx = a vector containing indices of variables 
%                         that gave a Heywood condition.
%             PAFout.retindx = a vector containing the indices of retained
%                         variables if those giving rise to a Heywood  
%                         condition were removed (see "heyopt" option).
% [...] = paf_usv(X,nfac,options) - Allows specification of option pairs 
%           as described below.
% Options:
%   ['delta',deltaval] - sets the limiting limiting value for the 
%        detection of Heywood cases. deltval must be a number greater 
%        than zero.
%        Default: 1e-6
%        Comment: Ideally set to less than the minimum ratio of 
%                 measurement noise variance to column variance (about 
%                 zero) for all variables. Too large values can lead to
%                 false detection of Heywood cases. Too small values may
%                 affect scores when heyopt=0 (see below).
%   ['heyopt',heyval] - determines action to be taken in the event of 
%        Heywood case.  Allowed values [0 1 2].
%           heyval=0 indicates that flagged elements of Psi(and the scores)
%                    should be estimated using 'deltaval'.
%           heyval=1 indicates that the variance of flagged variables 
%                    should be set to the smallest element of Psi that did
%                    not give a Heywood case.
%           heyval=2 indicates the variables giving rise to a Heywood case
%                    should be removed. In this case, the dimensions of
%                    S,V and Psi will be reduced. Removed and retained 
%                    variables are indicated in PAFout.hwindx and
%                    PAFout.retindx.
%         Default: 1
%
%%
%
function [U,S,V,Psi,PAFout]=paf_usv(X,nfac,varargin);
%
%%
% Parse and check the inputs and initialize variables
%
validateattributes(X,{'double'},{'ndims',2},1);
[nrow,ncol]=size(X);
if nrow<2 | ncol<2
    error('Expected input number 1 to be a matrix.')
end
imin=min([nrow ncol]);
validateattributes(nfac,{'double'},{'scalar','integer','>',0,'<',imin},2);
p=inputParser;
testfcn=@(x)validateattributes(x,{'double'},{'scalar','>',0,'<',0.1},'delta');
addParameter(p,'delta',1e-6,testfcn);
testfcn=@(x)validateattributes(x,{'double'},{'integer','scalar','>=',0,'<',3},'heyopt');
addParameter(p,'heyopt',1,testfcn);
parse(p,varargin{:})
deltaval=p.Results.delta;
heyopt=p.Results.heyopt;
%
PAFout.maxflg=false;        % 0=normal, 1=maxiterations
PAFout.hwflg=false;         % 0=normal, 1=Heywood case
PAFout.hwindx=[];           % Indices of Heywood cases
PAFout.retindx=[1:ncol];    % Indices of retained variables
tol=1e-10;                  % Tolerance for convergence
maxiter=1000;               % Maximum number of iteration
%%
% Calculate correlation matrix about zero and perform PAF
%
CovX=X'*X/nrow;             % Covariance about zero
Xsd=sqrt(diag(CovX))';      % Variances on diagonal of X
Xsc=X./(ones(nrow,1)*Xsd);  % Scaled X values for scores calculation
R=CovX./(Xsd'*Xsd);    % Correlation about zero
%
count=0;                    % Loop counter
convflg=false;              % Convergence flag
Psi=zeros(ncol,1);          % Unique variances
%
while ~convflg & count<maxiter    % Loop for PAF
    Z=R-diag(Psi);
    [U,S,V]=svds(Z,nfac);
    L=U*sqrt(S);
    Psiold=Psi;
    for j=1:ncol
        Psi(j)=1-L(j,:)*L(j,:)';
    end
    convtst=max(abs(Psi-Psiold));
    if convtst<tol
        convflg=true;
    else
        count=count+1;
    end
end
%
if count>=maxiter        % Sets flag for non-convergence
    PAFout.maxflg=true;
end
Lam=L;                   % Lambda values
%%
% Check for Heywood cases and adjust results accordingly
%
PsiOrig=Psi;
indx=find(Psi<deltaval)';
PAFout.hwindx=indx;
if ~isempty(indx)               % Heywood cases - need adjustments
    warning('Heywood case detected in PAF decomposition.')
    PAFout.hwflg=true;          % Set Heywood indicator
    if length(indx)>(ncol-nfac) & heyopt>0  % Check unlikely case of mostly Heywood
        warning('Too many Heywood instances - heyopt set to 0.')
        heyopt=0;
    end
%
    if heyopt==0            % Replace elements with delta value
        Psi(indx)=deltaval;
%
    elseif heyopt==2        % Eliminate Heywood variables
        retindx=[1:ncol];
        retindx(indx)=[];   % Indices of variables retained
        PAFout.retindx=retindx;
        Xsc(:,indx)=[];     % Remove variables from X
        Lam(indx,:)=[];     % Remove variables from loadings
        Psi(indx)=[];       % Remove variables from unique variances
        CovX(indx,:)=[]; CovX(:,indx)=[];    % Fix covariance
        Xsd(indx)=[];       % Fix SD values for scaling
        R(indx,:)=[]; R(:,indx)=[];          % Fix correlation matrix
        Rtemp=Lam*Lam';     % This part re-orthogonalizes loadings
        [U,S,V]=svds(Rtemp,nfac);
        Lam=U*sqrt(S);
%
    else                      % Replace elements with smallest delta
        Psi(indx)=-100;       % To distinguish removed elements
        Psimin=min(abs(Psi)); % Minimum non-Heywood value
        Psi(indx)=Psimin;     % Replace Heywood cases
    end
end
%
% Now generate results scaled back to the original space
%
Siginv=diag(1./Psi);                   % Weight matrix
Xhat=Xsc*Siginv*Lam*inv(Lam'*Siginv*Lam)*Lam'; % ML projection
Xhatsc=Xhat.*(ones(nrow,1)*Xsd);       % Rescale projected data
[U,S,V]=svds(Xhatsc,nfac);             % PCA format
Psi=Psi.*(Xsd'.^2);                 % Scaled variance estimates
%%
