# -*- coding: utf-8 -*-
"""
This program is part of the supplementary information for
Spinodal decomposition of chemically fueled polymer solutions
by Jonas Heckel, Fabio Batti, Robert T. Mathers and Andreas Walther

If you use this script for your scientific work, please give proper credit to the publication.

*** Description
This tool is for fitting experimental data to a chemical reaction network (CRN): 
A + B -> C + D  with rate constant k1
A -> C          with rate constant k2, which is known already
D -> B          with rate constant k3
which is defined in the crn_diff function. 

The implementation takes experimental concentrations only for species A and fits these to the CRN to obtain k1, k3.
Fitting is done using the lmfit package and leastsquare optimization. Multiple datasets are optimized simulataneously.

*** License
***********
MIT License

Copyright (c) 2021 Jonas Heckel

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
***********
"""


### imports
import numpy as np
import matplotlib.pyplot as plt
from lmfit import minimize, Parameters, report_fit
from scipy.integrate import odeint


### settings
# reaction pH, this will choose the correct rate constants
pH = 7.0

# map of rate constants vs pH. These will be used as initial estimates for the parameter or fixed values if fixed below
k_pH_map = {
        5.3: [1000, 0.0033832, 0.01],
        6.1: [500, 4.5213496E-4, 0.05],
        7.0: [100, 1.4893208E-4, 0.1]}
k1_guess, k2, k3_guess = k_pH_map.get(pH)

# differential representation of the chemical reaction network. Try if rate constanst have been initialized, if not use parameters passed to the function.
def crn_diff(C0, t, ps):
    try:
        k1 = ps['k1'].value
        k2 = ps['k2'].value
        k3 = ps['k3'].value
    except:
        k1, k2, k3 = ps
    r1 = k1*C0[0]*C0[1]
    r2 = k2*C0[0]
    r3 = k3*C0[3]
    
    dAdt = - r1 - r2
    dBdt = - r1  + r3
    dCdt = r1 + r2
    dDdt = r1 - r3
    return [dAdt, dBdt, dCdt, dDdt]

# solver for the differential form of the CRN using ODE int
def crn_solve(t, C0, ps):
    Ct = odeint(crn_diff, C0, t, args=(ps,))
    return Ct
    
# Calculate residuals for each individual dataset and flatten into a 1D array
def residual(ps, t, data, C0):
    ndata, _ = data.shape
    resid = 0.0*data[:]
   
    for i in range(ndata):
        model = crn_solve(t, C0[i, :], ps)[:,0]
        resid[i, :] = model -  data[i, :]
    return resid.flatten()

### Input data ###
# t is a time array
# datN are data points with the initial  concentration (mol/L) of A determined by the first element
t = [0, 2, 17.6666667, 33.3333333]

dat1 = [0.004, 0.0030034, 3.5466576E-4, -4.7116479E-5]
dat2 = [0.004, 0.0026363, 3.1456881E-4, 1.2061298E-6]
dat3 = [0.004, 0.0030821, 4.1965472E-4, -1.3823532E-5]
dat4 = [0.004, 0.0031401, 5.6909596E-4, 4.3132245E-5]
dat5 = [0.004, 0.0033205, 8.6164575E-4, 1.234866E-4]

dat6 = [0.002, 0.0012783, 6.3758589E-5, np.NaN]
dat7 = [0.002, 0.0013217, 7.2825116E-5, np.NaN]
dat8 = [0.002, 0.0013205, 5.798839E-5, np.NaN]
dat9 = [0.002, 0.0012672, 1.0690763E-4, np.NaN]
dat10 = [0.002, 0.0011341, 3.9915216E-5, np.NaN]

# create a 2D array comprised of all datasets
data = []
data.append(dat1)
data.append(dat2)
data.append(dat3)
data.append(dat4)
data.append(dat5)
data.append(dat6)
data.append(dat7)
data.append(dat8)
data.append(dat9)
data.append(dat10)
data = np.array(data)

# create initial concentrations of all 4 species in mol/L (A = EDC, B = acid monomer units, C = EDU, D = anhydride monomer units)
C0 = []
for row in data:
    C0.append([row[0], 0.002, 0,0])
C0 = np.array(C0)


# set parameters incluing bounds. k2 is determined in a different experiment and is thus fixed
params = Parameters()
params.add('k1', value=k1_guess, min=0, max=5000)
params.add('k2', value=k2, vary=False)
params.add('k3', value=k3_guess, min=0, max=1)

print(data)
# fit model and find predicted values
result = minimize(residual, params, args=(t, data, C0), method='leastsq', nan_policy='omit')
final_params = [result.params['k1'], k2, result.params['k3']]

# create separate t axis with higher resolution for plotting the results
t_solution = np.linspace(0, 60, 240)


# plot data and fitted curves
for row in data:
    plt.plot(t, row, 'o')
for row in C0:
    plt.plot(t_solution, crn_solve(t_solution, row, final_params)[:,0], '-', linewidth=2)

# display fitted statistics
report_fit(result)



### raw kinetic data ###

### pH = 5.2
#t = [0, 2, 17.6666667, 33.3333333, 49.0666667]
#
#dat1 = [0.004, 0.0022038, 0.0013217, 6.5084885E-4, 2.2339331E-4]
#dat2 = [0.004, 0.0019324, 0.0010199, 3.55751E-4, 1.1245562E-4]
#dat3 = [0.004, 0.001764, 8.6087557E-4, 2.6921218E-4, 7.4462741E-5]
#dat4 = [0.004, 0.0022474, 0.0013866, 7.5418571E-4, 2.8670518E-4]
#dat5 = [0.004, 0.002119, 0.001216, 5.452251E-4, 1.5559138E-4]
#
#dat6 = [0.002, 3.8029305E-4, 4.5967425E-5, np.NaN, np.NaN]
#dat7 = [0.002, 4.3630224E-4, 6.6726637E-5, np.NaN, np.NaN]
#dat8 = [0.002, 2.8400379E-4, 6.027824E-5, np.NaN, np.NaN]
#dat9 = [0.002, 3.9336993E-4, 5.2541349E-5, np.NaN, np.NaN]
#dat10 = [0.002, 3.2242813E-4, 4.3689262E-5, np.NaN, np.NaN]


### pH = 6.1
#t = [0, 2, 17.6666667, 33.3333333]

#dat1 = [0.004, 0.0019142, 5.5577422E-4, 5.5558146E-5]
#dat2 = [0.004, 0.0018167, 4.4945537E-4, 3.0076431E-5]
#dat3 = [0.004, 0.0013784, 1.2534222E-4, 5.6392561E-5]
#dat4 = [0.004, 0.0020152, 6.8616166E-4, 7.882031E-5]
#dat5 = [0.004, 0.0021681, 7.3249839E-4, 9.1413789E-5]
#
#
#dat6 = [0.002, 5.9714558E-4, 6.8620349E-5, 3.8413157E-5]
#dat7 = [0.002, 0.0010235, 8.5700479E-5, np.NaN]
#dat8 = [0.002, 4.8142323E-4, 6.0109037E-5, 6.0207692E-5]
#dat9 = [0.002, 5.9997422E-4, 4.011784E-5, 2.7698843E-6]
#dat10 = [0.002, 5.9919292E-4, 5.2637923E-5, 2.9638423E-5]


### pH = 7.0
#t = [0, 2, 17.6666667, 33.3333333]
#
#dat1 = [0.004, 0.0030034, 3.5466576E-4, -4.7116479E-5]
#dat2 = [0.004, 0.0026363, 3.1456881E-4, 1.2061298E-6]
#dat3 = [0.004, 0.0030821, 4.1965472E-4, -1.3823532E-5]
#dat4 = [0.004, 0.0031401, 5.6909596E-4, 4.3132245E-5]
#dat5 = [0.004, 0.0033205, 8.6164575E-4, 1.234866E-4]
#
#dat6 = [0.002, 0.0012783, 6.3758589E-5, np.NaN]
#dat7 = [0.002, 0.0013217, 7.2825116E-5, np.NaN]
#dat8 = [0.002, 0.0013205, 5.798839E-5, np.NaN]
#dat9 = [0.002, 0.0012672, 1.0690763E-4, np.NaN]
#dat10 = [0.002, 0.0011341, 3.9915216E-5, np.NaN]

