# -*- coding: utf-8 -*-
"""
This program is part of the supplementary information for
Spinodal decomposition of chemically fueled polymer solutions
by Jonas Heckel, Fabio Batti, Robert T. Mathers and Andreas Walther

If you use this script for your scientific work, please give proper credit to the publication.

*** Description
This tool performs an uncertainty analysis for a chemical reaction network (CRN) using the uncertanpy package:
https://github.com/simetenn/uncertainpy

The analysis needs initial concentrations (assumed to be accurately known) and rate constants and their uncertainties.
This allows calculating time-evolution of any species in the system. 

*** License
***********
MIT License

Copyright (c) 2021 Jonas Heckel

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
***********
"""

# import
import numpy as np
from scipy.integrate import odeint
import uncertainpy as un
import chaospy as cp
import pandas as pd


# name of the experiment. Only for the output filename
experimentname = "~experimentname"

# pH of the experiment for correct rate constants
pH = 6.1

# rate constants for the model, k1 and k3 have uncertainties, for k2 we assume sigma = 0, as it is not used in the fitting process and is very small anyways
k_pH_map = {
        5.3: [938.384382, 158.720185, 0.0033832, 0, 0.0213004, 9.5899E-4],
        6.1: [603.239945, 85.1658186, 4.5213496E-4, 0, 0.054178,	0.0042834],
        7.0: [106.146026, 8.7868039, 1.4893208E-4, 0 , 0.1807041, 0.0364393 ]}
k1, k1_sigma, k2, k2_sigma, k3, k3_sigma = k_pH_map.get(pH)


# this line seems to fix some kind of bug in Windows with multithreading used by uncertainpy. Clear cache after each run of the program when using Windows10
if __name__ == '__main__':    
    
          
    # define the chemical reaction network and solver
    def crn(k1, k2, k3):
                  
        # initial concentrations, A = EDC, B = fuelable monomer units, C = EDU, D = anhydride units, concentrations in mol/L.        
        cA0 = 0.016
        cB0 = 0.02
        cC0 = 0.0
        cD0 = 0.0
        C0 = [cA0, cB0, cC0, cD0]       
        
        # boundaries in which the CRN should be solved (min and max values for t) and density of datapoints (dt).
        tmin= 0
        tmax = 100
        dt  = 0.05
        time = np.arange(tmin, tmax + dt, dt)
        
        # definition of differential equation
        def dCdt(C0, time):                     
            
            r1 = k1*C0[0]*C0[1] # activation reaction
            r2 = k2*C0[0]       # EDC hydrolysis
            r3 = k3*C0[3]       # deactivation reaction
        
            dAdt = - r1 - r2
            dBdt = - r1  + r3
            dCdt = r1 + r2 
            dDdt = r1 - r3
            return [dAdt, dBdt, dCdt, dDdt]
        
        # solve ordinary differential equation
        Ct = odeint(dCdt, C0, time)
        
        # output EDC concentration (cA) for all times defined above. This can be changed to any desired compound in the system
        cA = Ct[:,0]
        
        return (time, cA)  
    
   
    # Initialize the uncertainty analysis model
    model = un.Model(run=crn,
                     labels=["time / min", "c / mol/L"])
    
    # assume normal distributions for activation and deactivation reaction rate constants
    k1 = cp.Normal(mu=k1, sigma=k1_sigma)
    k3 = cp.Normal(mu=k3, sigma=k3_sigma)
    
    # Define the parameter dictionary
    parameters = {"k1": k1, "k2": k2, "k3": k3}
    
    # Create the parameters
    parameters = un.Parameters(parameters)
    
    # perform uncertainty analysis
    UQ = un.UncertaintyQuantification(model=model, parameters=parameters)
        
    data = UQ.quantify(seed=1, method = "mc")
    
    # save output as csv including time, mean concentration, as well as p5 and p95 for the concentrations
    output = pd.DataFrame({"time":data["crn"].time, "mean":data["crn"].mean, "p5":data["crn"].percentile_5, "p95":data["crn"].percentile_95})
    output.to_csv(path_or_buf=experimentname+"_EDC_model.txt", sep='\t' ,index=False)






