# -*- coding: utf-8 -*-
"""
This program is part of the supplementary information for
Spinodal decomposition of chemically fueled polymer solutions
by Jonas Heckel, Fabio Batti, Robert T. Mathers and Andreas Walther

If you use this script for your scientific work, please give proper credit to the publication.

*** Description
Tool for the calculation of radially averaged Fourier Transforms of images (for example patterns formed in spinodal decomposition etc.)


The method for radial averageing is inspired by Naveen Venkatesan, X-Ray Scattering Tools, 2020
https://github.com/n-venkatesan/xray-scattering-tools



*** License
***********
MIT License

Copyright (c) 2021 Jonas Heckel

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
***********
"""


#imports
import cv2
import numpy as np
from matplotlib import pyplot as plt
import pandas as pd
import os


#################################################################################################################################################################################################################################
#setup# 

# path of the experiment containing the file(s). Square images only.
path = 'C:/Users/jonas/Documents/PhD/experiments/JH-265/265-5/'

# function analyzing the images
def fft_analysis (path, file):
    "This function performs FFT analysis on one image"
    
    #import file and calculate square of 2D FFT
    img = cv2.imread(path+file, 0)
    f = np.fft.fft2(img)
    fshift = np.fft.fftshift(f)
    fft_squared = np.abs(fshift)**2
    
    # if magnitude spectrum is desired
    #magnitude_spectrum = 20*np.log(np.abs(fshift))
    
    # normalize fourier image. calculate contrast range. this is relevant for creating a Fourier image to show
    #magnitude_spectrum = 255 * magnitude_spectrum / np.max(magnitude_spectrum)
    #magnitude_spectrum = magnitude_spectrum.astype(np.uint8)
    
    
    # plotting function for quick preview, enable as needed
    #plt.subplot(121),plt.imshow(img, cmap = 'gray')
    #plt.title('Input Image'), plt.xticks([]), plt.yticks([])
    #plt.subplot(122),plt.imshow(magnitude_spectrum, cmap = 'gray')
    #plt.title('Magnitude Spectrum'), plt.xticks([]), plt.yticks([])
    #plt.show()
    
    # Get image parameters
    x_size = fft_squared.shape[0]
    y_size = fft_squared.shape[1]
    
    # calculate center of FFT image
    cen_x = int(x_size/2)
    cen_y = int(y_size/2)
    
    # find radial distances. this creates a mesh grid for coordinates centered on the image
    [X, Y] = np.meshgrid(np.arange(y_size) - cen_x, np.arange(x_size) - cen_y)
    R = np.sqrt(np.square(X) + np.square(Y))
    
    # initialize variables for radial averaging.
    rad = np.arange(1, np.max(R), 1)
    intensity = np.zeros(len(rad))
    index = 0
    
    # calculate radial average
    bin_size = 1
    for i in rad:
        mask = (np.greater(R, i - bin_size) & np.less(R, i + bin_size))
        values = fft_squared[mask]
        intensity[index] = np.mean(values)
        index += 1
    
    # save output data to specified path
    output_data = pd.DataFrame({'x':rad, 'y':intensity})
    output_data.to_csv(path_or_buf=path+file+'.fft.txt', sep='\t' ,index=False)
    return

# perform analysis function for all tif-images in the specified path
# counter is nice to have since the calculation takes a few seconds per image
i = 0

for filename in os.listdir(path):    
    if filename.endswith(".tif"):
        fft_analysis(path, filename)
        i = i+1
        print('{} images processed.'.format(i))