#=
Scripts for MT in CO2BOLs
Thomas Moore et al. 2021
In this file, we include scripts used to generate images.
=#

#############
#DEFINE TYPES
#############

#= 
We begin by defining abstract CO2BOL type, with a struct for 
each CO2BOL (e.g. EEMPA, IPADM, etc.). This will allow us
to overload functions with multiple methods, providing the 
particular CO2BOL of interest as a function parameter
=#

abstract type CO2BOL end
struct EEMPA <: CO2BOL end
struct IPADM <: CO2BOL end
struct DBUHex <: CO2BOL end
struct GAP1TEG <: CO2BOL end
struct MEA <: CO2BOL end
struct K2CO3 <: CO2BOL end

#=
We now define abstract type for whether a function is 
applied to an ideal solution, or whether nonidealities 
are accounted for.
=#

abstract type ThermoMethod end
struct Ideal <: ThermoMethod end
struct NonIdeal <: ThermoMethod end

#=
We now define abstract type for different apparatus types.
=#

abstract type Apparatus end
struct WettedWall <: Apparatus end

##############
#CALL PACKAGES
##############

using Images
using Optim 
using Plots
MColor = ["#0072BD", "#D95319", "#EDB120", "#7E2F8E", "#77AC30", "#4DBEEE"];
using LaTeXStrings
using DifferentialEquations
using QuadGK
using LinearAlgebra
using SpecialFunctions
using BenchmarkTools
using Sundials

################################################################
#GENERAL PURPOSE SOLVER FOR SURFACE RENEWAL THEORY OF DANCKWERTS
################################################################

#Inputs:
# - n, number of species, integer
# - R, function for reaction rate. Function input is local concentrations (mol/m3). Function output is rate of change of local concentrations (mol/m3.s). Input and ouput are both vectors of length n.
# - D, vector of diffusivities (m2/s). Vector of length n
# - c0, Initial concentrations of all species, mol/m3. Should satisfy R(c0) = 0; if not will return error. Vector of length n.
# - csurf, surface concentration of species 1 (dissolving gas, most often CO2), mol/m3
# - s, surface renewal rate, 1/second.
# - N, number of cells to divide domain into (non-uniform mesh is used)
# - L, length of the domain. This *should* be infinite, but in practice must extend far enough that concentrations are unchanged by surface effects at times of order 1/s. 
# - tfinal, the endpoint of the time integration. Again this should be infinite, but in practice must be much larger than 1/s.
# - a, exponential growth factor for cell width.
# - OM, number of orders of magnitude of a over which to grow cells

#Outputs:
# - Flux as predicted by surface renewal theory.
# - Solution to diffusion-reaction problem in stationary fluid.

function SurfaceRenewalModel(n, R, D::Array{Float64}, c0, csurf, s; N = 50, L = 300e-6, 
        tfinal = 100.0, a = 2.0, OM = 7.0)
    
    #Check that the initial conditions are at equilibrium
    if norm(R(c0)) > 1e-6
        error("Initial conditions are not at equilibrium.")
    end
    
    #Preliminary Calculations
    #Boundary Points
    b = a .^(LinRange(0.0, OM, N+1))
    b = b .- b[1]
    b = b .* (L/b[end])

    #Cell Center Points & Cell Widths
    x = (b[2:N+1] .+ b[1:N]) ./ 2
    Δb = b[2:N+1] .- b[1:N]
    Δx = x[2:N] .- x[1:N-1]

    #Set Initial Conditions (average values within cells)
    c0_array = zeros(N,n)
    for i = 1:N
        for j = 1:n
            c0_array[i, j] = c0[j]
        end
    end

    #ODE Function
    function odefun!(dcdt, c, (JLeft, JRight, n, N, csurf, D, R, Δb, Δx), t)

        #Calculate dcdt in cell adjacent to the wall
        #Start with CO2 at gas-liquid interface
        JLeft = -D[1]*(c[1,1] - csurf)/(Δb[1]/2)
        JRight = -D[1]*(c[2, 1] - c[1, 1])/Δx[1]
        dcdt[1, 1] = ((JLeft) - (JRight))/Δb[1] + R(@view c[1, :])[1]

        #Then do all other species at the gas-liquid interface
        if n > 1
            for j = 2:n
                JLeft = 0.0
                JRight = -D[j]*(c[2, j] - c[1, j])/Δx[1]
                dcdt[1, j] = ((JLeft) - (JRight))/Δb[1] + R(@view c[1, :])[j]
            end
        end

        #Calculate dcdt in cell adjacent to end of domain
        for j = 1:n
            JLeft = -D[j]*(c[N, j] - c[N-1, j])/Δx[end]
            JRight = 0.0
            dcdt[N, j] = ((JLeft) - (JRight))/Δb[end] + R(@view c[N, :])[j]
        end

        #Calculate rate of change of concentration in interior cells
        for i = 2:N-1
            for j = 1:n
                JLeft = -D[j]*(c[i, j] - c[i-1, j])/Δx[i-1]
                JRight = -D[j]*(c[i+1, j] - c[i, j])/Δx[i]
                dcdt[i, j] = ((JLeft) - (JRight))/Δb[i] + R(@view c[i, :])[j]
            end
        end

    end
    
    #Solve ODE
    tspan = (0.0, tfinal)
    prob = ODEProblem(odefun!, c0_array, tspan, (0.0, 0.0, n, N, csurf, D, R, Δb, Δx))
    sol = solve(prob, Sundials.CVODE_BDF())
    
    #Calculate Flux as Predicted by Surface Renewal Theory
    J(t) = -D[1]*(sol(t)[1] - csurf)/(Δb[1]/2)
    integrand(θ) = J(θ)*s*exp(-θ*s)
    Jav = quadgk(integrand, 0, tfinal, rtol = 1e-4)
    return (Jav[1], sol)
end

##########################
#ESTIMATE HENRY'S CONSTANT 
##########################

#=
Here, we estimate the Henry's constant for CO2 in IPADM-2-BOL based 
on the CO2 solubility in similar organic liquids. 
=#

function H(T, CO2BOL::IPADM; Hwater0 = 3.4e-4, n = 3.5, ΔH = -12500)
    R = 8.314
    H25 = Hwater0*n         #mol/m3.Pa
    return H25 * exp(-ΔH/R * (1/T - 1/298.15))
end

function H(T, CO2BOL::EEMPA; Hwater0 = 3.4e-4, n = 3.5, ΔH = -12500)
    R = 8.314
    H25 = Hwater0*n         #mol/m3.Pa
    return H25 * exp(-ΔH/R * (1/T - 1/298.15))
end

function H(T, CO2BOL::DBUHex; Hwater0 = 3.4e-4, n = 3.5, ΔH = -12500)
    R = 8.314
    H25 = Hwater0*n         #mol/m3.Pa
    return H25 * exp(-ΔH/R * (1/T - 1/298.15))
end

function H(T, CO2BOL::GAP1TEG; Hwater0 = 3.4e-4, n = 3.5, ΔH = -12500)
    R = 8.314
    H25 = Hwater0*n         #mol/m3.Pa
    return H25 * exp(-ΔH/R * (1/T - 1/298.15))
end

#########################################
#ESTIMATE INTRINSIC EQUILIBRIUM CONSTANTS
#########################################

#Calculate K(T) based on functions for KH(T) and H(T)

function K(T, method::Ideal, CO2BOL::IPADM; Hwater0 = 3.4e-4, n = 3.5, ΔH = -12500)
    return KH(T, Ideal(), IPADM())/H(T, IPADM(), Hwater0 = Hwater0, n = n, ΔH = ΔH)
end

function K(T, method::Ideal, CO2BOL::EEMPA; Hwater0 = 3.4e-4, n = 3.5, ΔH = -12500)
    return KH(T, Ideal(), EEMPA())/H(T, EEMPA(), Hwater0 = Hwater0, n = n, ΔH = ΔH)
end

function K(T, method::Ideal, CO2BOL::DBUHex; Hwater0 = 3.4e-4, n = 3.5, ΔH = -12500)
    return KH(T, Ideal(), DBUHex())/H(T, DBUHex(), Hwater0 = Hwater0, n = n, ΔH = ΔH)
end

function K(T, method::Ideal, CO2BOL::GAP1TEG; Hwater0 = 3.4e-4, n = 3.5, ΔH = -12500)
    return KH(T, Ideal(), GAP1TEG())/H(T, GAP1TEG(), Hwater0 = Hwater0, n = n, ΔH = ΔH)
end

function K(α, T, method::NonIdeal, CO2BOL::EEMPA; Hwater0 = 3.4e-4, n = 3.5, ΔH = -12500)
    return KH(α, T, NonIdeal(), EEMPA())/H(T, EEMPA(), Hwater0 = Hwater0, n = n, ΔH = ΔH)
end

###################################
#INITIAL EQUILIBRIUM CONCENTRATIONS
###################################

#Calculate equilibrium concentrations as a function of loading and temperature
#T in Kelvin

function c_eq(α, T, method::Ideal, CO2BOL::IPADM; ρ = 1100, MW = 0.1714, n = 3.5)
    
    Ctot = ρ/MW            #Total CO2BOL Concentration, mol/m3
    
    CA0 = α/K(T, Ideal(), IPADM(), n = n)
    CC0 = Ctot / (1 + 1/α)
    CB0 = Ctot - CC0
    
    return [CA0, CB0, CC0]
end
        
function c_eq(α, T, method::Ideal, CO2BOL::EEMPA; ρ = 950.0, MW = 0.2163, n = 3.5)
    
    Ctot = ρ/MW            #Total CO2BOL Concentration, mol/m3
    
    CA0 = α/K(T, Ideal(), EEMPA(), n = n)
    CC0 = Ctot / (1 + 1/α)
    CB0 = Ctot - CC0
    
    return [CA0, CB0, CC0]
end

function c_eq(α, T, method::Ideal, CO2BOL::DBUHex; ρ = 916.0, MW = 0.1272, n = 3.5)
    
    Ctot = ρ/2/MW            #Total Reactive CO2BOL Concentration, mol/m3
    
    CA0 = α/K(T, Ideal(), DBUHex(), n = n)
    CC0 = Ctot / (1 + 1/α)
    CB0 = Ctot - CC0
    
    return [CA0, CB0, CC0]
end

function c_eq(α, T, method::Ideal, CO2BOL::GAP1TEG; ρ = 1000.0, MW = 0.25, n = 3.5)
    
    Ctot = ρ/MW*0.6            #Total Reactive CO2BOL Concentration, mol/m3
    
    CA0 = α/K(T, Ideal(), GAP1TEG(), n = n)
    CC0 = Ctot / (1 + 1/α)
    CB0 = Ctot - CC0
    
    return [CA0, CB0, CC0]
end

function c_eq(α, T, method::NonIdeal, CO2BOL::EEMPA; ρ = 950.0, MW = 0.2163, n = 3.5)
    
    Ctot = ρ/MW            #Total CO2BOL Concentration, mol/m3
    
    CC0 = α*Ctot
    CB0 = Ctot - 2CC0
    CA0 = 1/K(α,T,NonIdeal(),EEMPA(), n = n) * α/(1-2α)^2

    return [CA0, CB0, CC0]
end

#########################################
#NET REACTION RATE EXPRESSIONS (mol/m3.s)
#########################################

#Calculate reaction rate expressions in mol/m3.s
#T in Kelvin
#Return reaction rate for all species in mol/m3.s

function RateExpression(c, T, CO2BOL::IPADM; k = 1e3, n = 3.5)
    rate = k*(c[1]*c[2] - c[3]/(K(T,Ideal(),IPADM(), n = n))) 
    return [-rate, -rate, rate]
end

function RateExpression(c, T, CO2BOL::EEMPA; k = 1e3, n = 3.5)
    rate = k*(c[1]*c[2] - c[3]/(K(T,Ideal(),EEMPA(), n = n))) 
    return [-rate, -rate, rate]
end

function RateExpression(c, T, CO2BOL::DBUHex; k = 1e3, n = 3.5)
    rate = k*(c[1]*c[2] - c[3]/(K(T,Ideal(),DBUHex(), n = n))) 
    return [-rate, -rate, rate]
end

function RateExpression(c, T, CO2BOL::GAP1TEG; k = 1e3, n = 3.5)
    rate = k*(c[1]*c[2] - c[3]/(K(T,Ideal(),GAP1TEG(), n = n))) 
    return [-rate, -rate, rate]
end

function RateExpression(c, T, CO2BOL::EEMPA, method::NonIdeal; k = 1e3, n = 3.5)
    α = c[3]/(c[2]+2c[3])
    c1eq = (1/K(α,T,NonIdeal(),EEMPA(), n = n))α/(1-2α)^2
    rate = k*c[2]*(c[1] - c1eq) 
    return [-rate, -2*rate, rate]
end


##########################################
#LIQUID FILM MASS TRANFER COEFFICIENT, kL0
##########################################

#Calculate kL0 in wetted wall column
#using correlation of Vivian and Peaceman (1956)
#T in Kelvin
#Return kL0 in m/s
#h, column height, m
#D, gas diffusivity, m2/s
#μ, liquid viscosity
#ρ, liquid density
#Γ, mass flow rate of liquid per unit perimeter
#g, gravitational constant
#Choose units so bracketted terms are dimensionless 

function kL0Correlation(h, D, μ, ρ, Γ, Apparatus::WettedWall, g = 9.81)
    return D/h * 0.433 * (μ/(ρ*D))^(1/2) * (ρ^2*g*h^3/μ^2)^(1/6) * (4Γ/μ)^0.4
end

##########################
#SURFACE RENEWAL RATE, 1/s
##########################

#Calculate surface renewal rate in wetted wall column
#Use Vivan and Peacemen correlation of kL0
#Input terms as for kL0 correlation above.
#Return surface renewal rate, 1/s

function SurfaceRenewalRate(h, D, μ, ρ, Γ, Apparatus::WettedWall)
    kL0 = kL0Correlation(h, D, μ, ρ, Γ, WettedWall())
    return kL0^2/D
end

#########################################
#DIFFUSIVITY FROM WILKE CHANG CORRELATION
#########################################

#Return Diffusivity of solute in m2/s
#Input T in K, Viscosity in Pa.s,
#MW of solvent in kg/mol, 
#Volume of solute at liquid boiling point in m3/mol
#ϕsolvent is 2.6 of water, 1.0 for unassociated solvents.

function D_Wilke(T, μ; ϕsolvent = 2.6, MWsolvent = 0.018, Vliqsolute = 18e-6)
    return 7.4e-8/1e4 * (ϕsolvent*MWsolvent*1e3)^(0.5)*T/(μ*1e3*(Vliqsolute*1e6)^0.6)
end


####################
#VLE FOR IPADM-2-BOL
####################

α_40C_IPADM = [0.007, 0.025, 0.042, 0.06, 0.076, 0.11, 0.143, 0.176, 0.194, 0.208]
p_40C_IPADM = [75.0, 110.0, 140.0, 200.0, 265, 400, 630, 950,1170, 1500]
α_60C_IPADM = [0.007, 0.025, 0.042, 0.06, 0.076, 0.11, 0.143, 0.176, 0.194, 0.201]
p_60C_IPADM = [180.0, 500.0, 720.0, 1080.0, 1450, 2400, 3400, 5300,7800, 7400]
α_80C_IPADM = [0.007,0.011, 0.025, 0.039, 0.042, 0.06, 0.076, 0.11, 0.143, 0.176, 0.194, 0.207]
p_80C_IPADM = [550.0, 1500.0, 2000.0,5200, 3200.0, 5000.0, 6800, 11000, 14500, 18500,55000, 25000]
α_100C_IPADM = [0.011, 0.025, 0.039,  0.06, 0.076, 0.11, 0.143, 0.176, 0.194]
p_100C_IPADM = [8500.0, 10000.0, 21000, 31000.0, 46000, 90000, 210000, 350000,450000];

#Calculate pCO2_eq as a function of α, T, and parameters A, B and C.
pCO2_model_IPADM(α, T, params) = α/(1-α)*exp(params[1]+params[2]/(8.314*T))

#Calculate Sum of Squares Error as Function of Parameter Choice
function objective_IPADM(params)
    SSE = 0
    for i in 1:length(α_40C_IPADM)
        SSE += (log(pCO2_model_IPADM(α_40C_IPADM[i], 313.15, params)) - log(p_40C_IPADM[i]))^2
    end
    for i in 1:length(α_60C_IPADM)
        SSE += (log(pCO2_model_IPADM(α_60C_IPADM[i], 333.15, params)) - log(p_60C_IPADM[i]))^2
    end
    for i in 1:length(α_80C_IPADM)
        SSE += (log(pCO2_model_IPADM(α_80C_IPADM[i], 353.15, params)) - log(p_80C_IPADM[i]))^2
    end
    for i in 1:length(α_100C_IPADM)
        SSE += (log(pCO2_model_IPADM(α_100C_IPADM[i], 373.15, params)) - log(p_100C_IPADM[i]))^2
    end
    return SSE
end

#Initial Guess
params0_IPADM = [30.0, -100.0]

#Optimize via Nelder Mead Algorithm
res_IPADM = Optim.optimize(objective_IPADM, params0_IPADM, iterations = 10000)
param_vals_IPADM = res_IPADM.minimizer

function KH(T, method::Ideal, CO2BOL::IPADM; params = param_vals_IPADM)
    κ = exp(params[1]+params[2]/(8.314*T))
    return 1/κ
end

function pCO2(α, T, method::Ideal, CO2BOL::IPADM)
    return α/(1-α)/KH(T, Ideal(), IPADM())
end

#######################################
#VLE FOR EEMPA, IDEAL 1-1 STOICHIOMETRY
#######################################

α_30C_EEMPA = [0.0055, 0.01, 0.025, 0.05, 0.1, 0.2, 0.3, 0.4, 0.43]
p_30C_EEMPA = [160, 210, 400, 680,1230, 3000, 6300, 14000, 18600]
α_40C_EEMPA = [0.0055, 0.01, 0.025, 0.05, 0.1, 0.2, 0.3, 0.4]
p_40C_EEMPA = [200, 340, 820, 1500,3100, 7900, 17000, 35000]
α_75C_EEMPA = [0.0055, 0.01, 0.025, 0.05, 0.1]
p_75C_EEMPA = [2350, 4150, 10500,20500, 41000]

#Calcualte pCO2_eq as a function of α, T, and parameters A, B and C.
#pCO2_model(α, T, params) = exp(params[1]+params[2]/(8.314*T)+params[3]*α+params[4]*sqrt(α))*(α/(1-α))
pCO2_model(α, T, params) = exp(params[1]+params[2]/(8.314*T))*(α/(1-α))

#Calculate Sum of Squares Error as Function of Parameter Choice
function objective(params)
    SSE = 0
    for i in 1:length(α_30C_EEMPA)
        SSE += (log(pCO2_model(α_30C_EEMPA[i], 303.15, params)) - log(p_30C_EEMPA[i]))^2
    end
    for i in 1:length(α_40C_EEMPA)
        SSE += (log(pCO2_model(α_40C_EEMPA[i], 313.15, params)) - log(p_40C_EEMPA[i]))^2
    end
    for i in 1:length(α_75C_EEMPA)
        SSE += (log(pCO2_model(α_75C_EEMPA[i], 348.15, params)) - log(p_75C_EEMPA[i]))^2
    end
    return SSE
end

#Initial Guess
#params0 = [30.0, -2000.0, 10.0, 10.0]
params0_EEMPA = [30.0, -2000.0]

#Optimize via Nelder Mead Algorithm
res_EEMPA = Optim.optimize(objective, params0_EEMPA, iterations = 10000)
param_vals_EEMPA = res_EEMPA.minimizer;

function KH(T, method::Ideal, CO2BOL::EEMPA; params = param_vals_EEMPA)
    κ = exp(params[1]+params[2]/(8.314*T))
    return 1/κ
end

function pCO2(α, T, method::Ideal, CO2BOL::EEMPA)
    return α/(1-α)/KH(T, Ideal(), EEMPA())
end

##########################################
#VLE FOR EEMPA, NONIDEAL 1-2 STOICHIOMETRY
##########################################

#Calcualte pCO2_eq as a function of α, T, and parameters A, B and C.
#pCO2_model(α, T, params) = exp(params[1]+params[2]/(8.314*T)+params[3]*α+params[4]*sqrt(α))*(α/(1-α))
pCO2_model(α, T, params) = exp(params[1]+params[2]/(8.314*T) +params[3]*α+params[4]*sqrt(α))*(α/(1-2α)^2)

#Calculate Sum of Squares Error as Function of Parameter Choice
function objective(params)
    SSE = 0
    for i in 1:length(α_30C_EEMPA)
        SSE += (log(pCO2_model(α_30C_EEMPA[i], 303.15, params)) - log(p_30C_EEMPA[i]))^2
    end
    for i in 1:length(α_40C_EEMPA)
        SSE += (log(pCO2_model(α_40C_EEMPA[i], 313.15, params)) - log(p_40C_EEMPA[i]))^2
    end
    for i in 1:length(α_75C_EEMPA)
        SSE += (log(pCO2_model(α_75C_EEMPA[i], 348.15, params)) - log(p_75C_EEMPA[i]))^2
    end
    return SSE
end

#Initial Guess
#params0 = [30.0, -2000.0, 10.0, 10.0]
params0_EEMPA_NonIdeal = [30.0, -2000.0, 10.0, 10.0]

#Optimize via Nelder Mead Algorithm
res_EEMPA_NonIdeal = Optim.optimize(objective, params0_EEMPA_NonIdeal, iterations = 10000)
param_vals_EEMPA_NonIdeal = res_EEMPA_NonIdeal.minimizer;

function KH(α, T, method::NonIdeal, CO2BOL::EEMPA; params = param_vals_EEMPA_NonIdeal)
    κ = exp(params[1]+params[2]/(8.314*T)+params[3]*α+params[4]*sqrt(α))
    return 1/κ
end

function pCO2(α, T, method::NonIdeal, CO2BOL::EEMPA)
    return α/(1-2α)^2/KH(α, T, NonIdeal(), EEMPA())
end

###############
#VLE FOR DBUHex
###############

α_35C_DBUHex = [0.0146, 0.0291, 0.0582, 0.2574, 0.4744, 0.7111]
p_35C_DBUHex = [337.1, 431.2, 505.3, 3130.2, 8936.0, 27917.9]
α_45C_DBUHex = [0.0146, 0.0582, 0.2574, 0.4744, 0.7111]
p_45C_DBUHex = [101.8, 1282.8, 14031.9, 21458.5, 26196.3]
α_55C_DBUHex = [0.0146, 0.0582, 0.2574, 0.4744, 0.7111]
p_55C_DBUHex = [237.4, 2003.0, 25338.3, 38827.6, 30775.0]

#Calcualte pCO2_eq as a function of α, T, and parameters A, B and C.
#pCO2_model(α, T, params) = exp(params[1]+params[2]/(8.314*T)+params[3]*α+params[4]*sqrt(α))*(α/(1-α))
pCO2_model(α, T, params) = exp(params[1]+params[2]/(8.314*T))*(α/(1-α))

#Calculate Sum of Squares Error as Function of Parameter Choice
function objective(params)
    SSE = 0
    for i in 1:length(α_35C_DBUHex)
        SSE += (log(pCO2_model(α_35C_DBUHex[i], 308.15, params)) - log(p_35C_DBUHex[i]))^2
    end
    for i in 1:length(α_45C_DBUHex)
        SSE += (log(pCO2_model(α_45C_DBUHex[i], 318.15, params)) - log(p_45C_DBUHex[i]))^2
    end
    for i in 1:length(α_55C_DBUHex)
        SSE += (log(pCO2_model(α_55C_DBUHex[i], 328.15, params)) - log(p_55C_DBUHex[i]))^2
    end
    return SSE
end

#Initial Guess
params0_DBUHex = [30.0, -2000.0]

#Optimize via Nelder Mead Algorithm
res_DBUHex = Optim.optimize(objective, params0_DBUHex, iterations = 10000)
param_vals_DBUHex = res_DBUHex.minimizer;

function KH(T, method::Ideal, CO2BOL::DBUHex; params = param_vals_DBUHex)
    κ = exp(params[1]+params[2]/(8.314*T))
    return 1/κ
end

function pCO2(α, T, method::Ideal, CO2BOL::DBUHex)
    return α/(1-α)/KH(T, Ideal(), DBUHex())
end

####################################
#IMPLEMENT MEA SURFACE RENEWAL MODEL
####################################

#------------------
#VLE for CO2 in MEA
#------------------
param_vals_MEA = [30.96, -10584, -7.187]
function KH(α, T, method::NonIdeal, CO2BOL::MEA; params = param_vals_MEA, x_am = 0.11)
    κ = 1e3*exp(params[1] + params[2]/T + params[3]*x_am*α)
    return 1/κ
end

function pCO2(α, T, method::NonIdeal, CO2BOL::MEA; x_am = 0.11, params = param_vals_MEA)
    return α^2 / (1-2α)^2 / KH(α, T, NonIdeal(), MEA(), params = param_vals_MEA, x_am = x_am)
end

#------------------------------------------
#Henry's constant for CO2 in MEA, mol/Pa.m3
#------------------------------------------
function H(T, CO2BOL::MEA)
    C1 = -4052.21; C2 = 110950.0; C3 = 704.824; C4 = -1.097938; 
    C5 = -118.7848; C6 = 2580.65; C7 = 21.6287; C8 = -0.044901;
    return 1.0/(exp(C1 + C2/T + C3*log(T) + C4*T) * exp(C5 + C6/T + C7*log(T) + C8*T))
end

#------------------------
#Equilibrium Constant MEA
#------------------------
function K(α, T, method::NonIdeal, CO2BOL::MEA)
    return KH(α, T, NonIdeal(), MEA())/H(T, MEA())
end

#----------------------------------
#Initial Equilibrium Concentrations
#----------------------------------
function c_eq(α, T, method::NonIdeal, CO2BOL::MEA; Ctot = 5000.0)
        
    CMEACOO_0 = α*Ctot
    CMEA_0 = Ctot - 2CMEACOO_0
    CCO2_0 = 1/K(α,T,NonIdeal(),MEA())*α^2/(1-2α)^2

    return [CCO2_0, CMEA_0, CMEACOO_0]
end

#--------------------------
#Rate Expression (mol/m3.s)
#-------------------------
function RateExpression(c, T, method::NonIdeal, CO2BOL::MEA; k = 4.4e8*exp(-5400/T))
    α = c[3]/(c[2]+2c[3])
    c1eq = (1/K(α,T,NonIdeal(),MEA()))*α^2/(1-2α)^2
    rate = k*c[2]*(c[1] - c1eq) 
    return [-rate, -2*rate, rate]
end

#####################################
#IMPLEMENT K2CO3 SURFACE RENEWAL MODEL
#####################################

#--------------------
#VLE for CO2 in K2CO3
#--------------------
param_vals_K2CO3 = [26.3, -4311.5, 34.58, -10.78]
function KH(α, T, CO2BOL::K2CO3; params = param_vals_K2CO3, x_K2CO30 = 0.053)
    κ = x_K2CO30*exp(params[1] + params[2]/T + params[3]*x_K2CO30*α + params[4]*sqrt(x_K2CO30*α))
    return 1/κ
end

function pCO2(α, T, CO2BOL::K2CO3; x_K2CO30 = 0.053, params = param_vals_K2CO3)
    pCO2 = α^2 / (1-α) / KH(α, T, K2CO3(), params = param_vals_K2CO3, x_K2CO30 = x_K2CO30)
end

#--------------------------------------------
#Henry's constant for CO2 in K2CO3, mol/Pa.m3
#--------------------------------------------
function H(T, CO2BOL::K2CO3; cK2CO3 = 3000)
    H_CO2_Water = 3.54e-7*exp(2044/T)
    hg = -0.172 + (T - 298.15)*(-0.338e-3)
    hK = 0.0922; hCO32 = 0.1423
    cK = 2*cK2CO3/1000; cCO32 = cK2CO3/1000
    H_CO2_Water*10^((hK+hg)*cK+(hCO32+hg)*cCO32)
    return H_CO2_Water*10^((hK+hg)*cK+(hCO32+hg)*cCO32)
end

#--------------------------
#Equilibrium Constant K2CO3
#--------------------------
function K(α, T, CO2BOL::K2CO3)
    return KH(α, T, K2CO3())/H(T, K2CO3())
end

#----------------------------------
#Initial Equilibrium Concentrations
#----------------------------------
function c_eq(α, T, CO2BOL::K2CO3; CK = 6000.0)
        
    CKHCO3_0 = α*CK
    CK2CO3_0 = (CK - CKHCO3_0)/2
    CCO2_0 = 1/K(α,T,K2CO3())*α^2/(1-α)

    return [CCO2_0, CK2CO3_0, CKHCO3_0]
end

#--------------------------
#Rate Expression (mol/m3.s)
#-------------------------
function RateExpression(c, T,  CO2BOL::K2CO3; k = 10^(13.635 - 2895/T + 0.08*(0.5*(6*1+3*4)))/1000)
    α = c[3]/(2c[2]+c[3])
    c1eq = (1/K(α,T,K2CO3()))*α^2/(1-α)
    Keqinf_OH = 10^(-1568.9/T + 2.5866 + 6.737e-3*T)
    Keq_OH = Keqinf_OH / (10^(1.01*sqrt(6)/(1 + 1.49*sqrt(6)) + 6.1e-2*6))
    cOH = Keq_OH * c[2]/c[3]
    rate = k*cOH*(c[1] - c1eq) 
    return [-rate, -rate, 2*rate]
end


######################################################
#TEST SURFACE RENEWAL CODE AGAINST ANALYTICAL SOLUTION
######################################################

#=
The code below checks that the ODE solver above does
indeed correctly solve a simple diffusion-reaction PDE:

$$
\frac{\partial c}{\partial t} = D\frac{\partial^2 c}{\partial x^2} - kc
$$

$$
c|_{t=0} = 0; \quad c|_{x=0} = c_{\text{surf}}.
$$

This has analytical solution (see Danckwerts' Gas Liquid Reactions),

$$
c/c_{\text{surf}} = \frac{1}{2} e^{-x/\sqrt{D/k}} \text{erfc} \left[\frac{x}{2\sqrt{Dt}}-\sqrt{kt}\right] + \frac{1}{2} e^{x/\sqrt{D/k}} \text{erfc} \left[\frac{x}{2\sqrt{Dt}}+\sqrt{kt}\right] 
$$

We solve this numerical for the following values: $k = 10 s^{-1}; D = 10^{-9} m^2/s$. This results in a characteristic length scale of $\sqrt{D/k} = 10\mu m$ and a characteristic timescale of $1/k = 0.1s$. We therefore solve over $L = 100 \mu m$, and look at time points $\mathcal{O}(0.1s)$.
=#

#=
#CODE START:
#Parameters
D = [1e-9]; csurf = 1.0; c0 = 0.0; R(c) = -10*c[1]
s = 1.0; n = 1; N = 50; L = 300e-6; tfinal = 10.0; 
a = 2; OM = 7

#Analytical Solution
function AnalyticalSolution(x, t; D = D, k = k, csurf = csurf)
    return 0.5*exp(-x/sqrt(D[1]/k))*erfc(x/(2*sqrt(D[1]*t))-sqrt(k*t)) +
           0.5*exp( x/sqrt(D[1]/k))*erfc(x/(2*sqrt(D[1]*t))+sqrt(k*t))
end

#Computational Solution
sol = SurfaceRenewalModel(n, R, D, c0, csurf, s, N = N, L = L, tfinal = tfinal, a = a, OM = OM);
@time sol = SurfaceRenewalModel(n, R, D, c0, csurf, s, N = N, L = L, tfinal = tfinal, a = a, OM = OM);

#Plots
t1 = 0.01; t2 = 0.1; t3 = tfinal

#xvals for plot & error calculation
b = a .^(LinRange(0.0, OM, N+1))
b = b .- b[1]
b = b .* (L/b[end])
x = (b[2:N+1] .+ b[1:N]) ./ 2

#Average Error at t = 0.1s
av_error = sum(abs(sol[2](0.1)[i] - AnalyticalSolution(x[i], 0.1)) for i in 1:N)/N
println("At t = 0.1s, the average of the absolute error across all points is ", av_error)

scatter(x, sol[2](t1), marker = :circle, color = :blue)
plot!(x, x -> AnalyticalSolution(x, t1), color = :blue)
scatter!(x, sol[2](t2), marker = :circle, color = :red)
plot!(x, x -> AnalyticalSolution(x, t2), color = :red)
scatter!(x, sol[2](t3), marker = :circle, color = :black)
plot!(x, x -> AnalyticalSolution(x, t3), color = :black, legend = false)
#:CODE END
=#
