﻿/*
 * Title: Flow rate control for offline high-pressure nanoESI using spray current as the feedback signal
 * 
 * Author: CHEN, Lee Chuin (leechuin@yamanashi.ac.jp) [2023]
 * 
 * - Tested using Microsoft Visual C# 2010 Express under Windows 7
 * - Agilent 34461A digital multimeter (DMM) is used for the acquisition of spray current
 * - Agilent EDU36311A programmable power supply is used to generate the control signal to the HV module
 * - Their IVI drivers (downloaded from Agilent's website) are required
 * - Resource Description needs to be edited accordingly.
 * - serialPort2 is used to send START signal to an Arduino microcontroller which is used to turn ON the MS acquisition
 * 
 * THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 * */

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;

using System.IO;
using System.Threading;
using System.Threading.Tasks;


//About IVI driver
//Install the IVI driver (downloaded from Agilent website)
//Add the COM object by: Project -> add reference -> find IVI AgXXX
using Agilent.Ag3446x.Interop; // Multimeter
using Agilent.AgilentE36xx.Interop; //DC power supply

//remark: The fastest sampling time ~ 20ms

namespace lockspray
{
    public partial class Form1 : Form
    {

        StreamWriter outputfile = null;
        long TimeOriginForSaving = 0;
        BackgroundWorker xworker = null; // Lock single spray current
        BackgroundWorker yworker = null; // Continuous monitoring spray current
        BackgroundWorker zworker = null; // Lock Multiple spray current
        BackgroundWorker wworker = null; // HV sequence without the lockin of spray current. (For performance comparison only)
        AutoResetEvent yreset = new AutoResetEvent(false);
        AutoResetEvent xreset = new AutoResetEvent(false);
        AutoResetEvent zreset = new AutoResetEvent(false);
        AutoResetEvent wreset = new AutoResetEvent(false);


        Ag3446x dmm = new Ag3446x();
        AgilentE36xx DCS = new AgilentE36xx();
        public Form1()
        {
            InitializeComponent();
            xworker = new BackgroundWorker();
            xworker.WorkerSupportsCancellation = true;
            xworker.WorkerReportsProgress = true;
            xworker.DoWork += new DoWorkEventHandler(xworker_DoWork);
            xworker.ProgressChanged += new ProgressChangedEventHandler(xworker_ProgressChanged);
            xworker.RunWorkerCompleted += new RunWorkerCompletedEventHandler(xworker_RunWorkerCompleted);

            //yworker is for the constant monitoring of spray current
            yworker = new BackgroundWorker();
            yworker.WorkerSupportsCancellation = true;
            yworker.WorkerReportsProgress = true;
            yworker.DoWork += new DoWorkEventHandler(yworker_DoWork);
            yworker.ProgressChanged += new ProgressChangedEventHandler(yworker_ProgressChanged);
            yworker.RunWorkerCompleted += new RunWorkerCompletedEventHandler(yworker_RunWorkerCompleted);

            //zworker is for the constant monitoring of spray current
            zworker = new BackgroundWorker();
            zworker.WorkerSupportsCancellation = true;
            zworker.WorkerReportsProgress = true;
            zworker.DoWork += new DoWorkEventHandler(zworker_DoWork);
            zworker.ProgressChanged += new ProgressChangedEventHandler(zworker_ProgressChanged);
            zworker.RunWorkerCompleted += new RunWorkerCompletedEventHandler(zworker_RunWorkerCompleted);

            wworker = new BackgroundWorker();
            wworker.WorkerSupportsCancellation = true;
            wworker.WorkerReportsProgress = true;
            wworker.DoWork += new DoWorkEventHandler(wworker_DoWork);
            wworker.ProgressChanged += new ProgressChangedEventHandler(wworker_ProgressChanged);
            wworker.RunWorkerCompleted += new RunWorkerCompletedEventHandler(wworker_RunWorkerCompleted);

            this.FormClosing += new FormClosingEventHandler(Form1_FormClosing);




        }
        void wHoldHVForFiniteTime(double V_max, double HV, int Delay, long DurationTick)
        {
            if (DurationTick == 0)
            {
                return;//bye bye
            }
            

            IAgilentE36xxOutput pOutput1 = DCS.Outputs.get_Item(DCS.Outputs.get_Name(1));
            double dmm_data;

            long TimeOrigin = DateTime.Now.Ticks;
            long ElapsedTicks = DateTime.Now.Ticks - TimeOrigin;
            while (wworker.CancellationPending != true & HV < V_max & ElapsedTicks < DurationTick)
            {
                pOutput1.VoltageLevel = HV;
                
                dmm.Measurement.Initiate(); // Agilent dmm
                //dmm.System.WaitForOperationComplete(500);
                dmm_data = dmm.Measurement.Fetch(500);
                //Display the value on the screen
                List<object> Status = new List<object>();
                Status.Add(dmm_data);
                wworker.ReportProgress(0, Status);
                Thread.Sleep(Delay);
                ElapsedTicks = DateTime.Now.Ticks - TimeOrigin;

            }
            return;
        }
        void wworker_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            serialPort2.Write("a"); //Arduino Turn OFF Relay
            serialPort2.Write("b"); //Arduino Turn OFF LED
            
            textBox_nolock_HV_1.Enabled = true;
            textBox_nolock_HV_2.Enabled = true;
            textBox_nolock_HV_3.Enabled = true;
            textBox_nolock_HV_4.Enabled = true;
            textBox_nolock_HV_5.Enabled = true;
            textBox_nolock_HV_6.Enabled = true;
            textBox_nolock_HV_7.Enabled = true;
            textBox_nolock_HV_8.Enabled = true;
            textBox_nolock_duration_1.Enabled = true;
            textBox_nolock_duration_2.Enabled = true;
            textBox_nolock_duration_3.Enabled = true;
            textBox_nolock_duration_4.Enabled = true;
            textBox_nolock_duration_5.Enabled = true;
            textBox_nolock_duration_6.Enabled = true;
            textBox_nolock_duration_7.Enabled = true;
            textBox_nolock_duration_8.Enabled = true;

            button_Sequence.Enabled = true;
            button_Stop.Enabled = true;

            button_dmm_NULL.Enabled = true;
            button_dmm_unNull.Enabled = true;
            button_go.Enabled = true;
            checkBox_SaveSprayCurrrent.Enabled = true;
            if (checkBox_SaveSprayCurrrent.Checked == true)
            {
                outputfile.Close();
            }
            button_HV_StartSequence.Text = "Start HV";
            myPrint("<< Exit w worker loop <<");
        }

        void wworker_ProgressChanged(object sender, ProgressChangedEventArgs e)
        {
            if (e.ProgressPercentage == 0)
            {
                List<object> argumentlist = e.UserState as List<object>;
                double Readout_AgilentDMM = (double)argumentlist[0];
                textBox_AgDMMDisplay.Text = Math.Round(Readout_AgilentDMM, 3).ToString();
                if (checkBox_SaveSprayCurrrent.Checked == true)
                {
                    outputfile.WriteLine(((DateTime.Now.Ticks - TimeOriginForSaving) * 1E-7).ToString() + ", " + Readout_AgilentDMM.ToString() + ", " + DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel.ToString());
                }
            }
            else
            {
                myPrint("HV state -> " + e.ProgressPercentage.ToString());
                textBox_nolock_HV_1.BackColor = Color.White;
                textBox_nolock_HV_2.BackColor = Color.White;
                textBox_nolock_HV_3.BackColor = Color.White;
                textBox_nolock_HV_4.BackColor = Color.White;
                textBox_nolock_HV_5.BackColor = Color.White;
                textBox_nolock_HV_6.BackColor = Color.White;
                textBox_nolock_HV_7.BackColor = Color.White;
                textBox_nolock_HV_8.BackColor = Color.White;
                textBox_nolock_duration_1.BackColor = Color.White;
                textBox_nolock_duration_2.BackColor = Color.White;
                textBox_nolock_duration_3.BackColor = Color.White;
                textBox_nolock_duration_4.BackColor = Color.White;
                textBox_nolock_duration_5.BackColor = Color.White;
                textBox_nolock_duration_6.BackColor = Color.White;
                textBox_nolock_duration_7.BackColor = Color.White;
                textBox_nolock_duration_8.BackColor = Color.White;

                switch (e.ProgressPercentage)
                {
                    case 1: textBox_nolock_HV_1.BackColor = Color.Pink;
                        textBox_nolock_duration_1.BackColor = Color.Pink;
                        break;
                    case 2: textBox_nolock_HV_2.BackColor = Color.Pink;
                        textBox_nolock_duration_2.BackColor = Color.Pink;
                        break;
                    case 3: textBox_nolock_HV_3.BackColor = Color.Pink;
                        textBox_nolock_duration_3.BackColor = Color.Pink;
                        break;
                    case 4: textBox_nolock_HV_4.BackColor = Color.Pink;
                        textBox_nolock_duration_4.BackColor = Color.Pink;
                        break;
                    case 5: textBox_nolock_HV_5.BackColor = Color.Pink;
                        textBox_nolock_duration_5.BackColor = Color.Pink;
                        break;
                    case 6: textBox_nolock_HV_6.BackColor = Color.Pink;
                        textBox_nolock_duration_6.BackColor = Color.Pink;
                        break;
                    case 7: textBox_nolock_HV_7.BackColor = Color.Pink;
                        textBox_nolock_duration_7.BackColor = Color.Pink;
                        break;
                    case 8: textBox_nolock_HV_8.BackColor = Color.Pink;
                        textBox_nolock_duration_8.BackColor = Color.Pink;
                        break;
                    default: break;

                }
            }
        }

        void wworker_DoWork(object sender, DoWorkEventArgs e)
        {
            List<object> argumentlist = e.Argument as List<object>;
            double HV_1 = (double)argumentlist[0];
            double Duration_1 = (double)argumentlist[1];
            double HV_2 = (double)argumentlist[2];
            double Duration_2 = (double)argumentlist[3];
            double HV_3 = (double)argumentlist[4];
            double Duration_3 = (double)argumentlist[5];
            double HV_4 = (double)argumentlist[6];
            double Duration_4 = (double)argumentlist[7];
            double HV_5 = (double)argumentlist[8];
            double Duration_5 = (double)argumentlist[9];
            double HV_6 = (double)argumentlist[10];
            double Duration_6 = (double)argumentlist[11];
            double HV_7 = (double)argumentlist[12];
            double Duration_7 = (double)argumentlist[13];
            double HV_8 = (double)argumentlist[14];
            double Duration_8 = (double)argumentlist[15];
            
            IAgilentE36xxOutput pOutput1 = DCS.Outputs.get_Item(DCS.Outputs.get_Name(1));

           

            if (checkBox_SaveSprayCurrrent.Checked == true)
            {
                outputfile = new StreamWriter(textBox_FileName.Text);
                outputfile.WriteLine("# Data acquired from Agilent 34461A DMM and Agilent DC supply");
                outputfile.WriteLine("# Date: " + DateTime.Now.ToShortDateString() + " Time: " + DateTime.Now.ToLongTimeString());
                TimeOriginForSaving = DateTime.Now.Ticks;
            }
            int Delay = int.Parse(textBox_I_Lock_Delay.Text);



            //MS synchronization
            serialPort2.Write("A"); //Arduino Turn On Relay
            serialPort2.Write("B"); //Arduino Turn On LED

            while (wworker.CancellationPending != true)
            {
                wworker.ReportProgress(1);
                wHoldHVForFiniteTime(5, HV_1, Delay, (long)Math.Round(Duration_1 * 1E7, 7));
                wworker.ReportProgress(2);
                wHoldHVForFiniteTime(5, HV_2, Delay, (long)Math.Round(Duration_2 * 1E7, 7));
                wworker.ReportProgress(3);
                wHoldHVForFiniteTime(5, HV_3, Delay, (long)Math.Round(Duration_3 * 1E7, 7));
                wworker.ReportProgress(4);
                wHoldHVForFiniteTime(5, HV_4, Delay, (long)Math.Round(Duration_4 * 1E7, 7));
                wworker.ReportProgress(5);
                wHoldHVForFiniteTime(5, HV_5, Delay, (long)Math.Round(Duration_5 * 1E7, 7));
                wworker.ReportProgress(6);
                wHoldHVForFiniteTime(5, HV_6, Delay, (long)Math.Round(Duration_6 * 1E7, 7));
                wworker.ReportProgress(7);
                wHoldHVForFiniteTime(5, HV_7, Delay, (long)Math.Round(Duration_7 * 1E7, 7));
                wworker.ReportProgress(8);
                wHoldHVForFiniteTime(5, HV_8, Delay, (long)Math.Round(Duration_8 * 1E7, 7));
                wworker.ReportProgress(9);

            }
            wreset.Set();
        }

        void zworker_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            //MS synchronization
            serialPort2.Write("a"); //Arduino Turn OFF Relay
            serialPort2.Write("b"); //Arduino Turn OFF LED
            
            textBox_Target_1.BackColor = Color.White;
            textBox_Target_2.BackColor = Color.White;
            textBox_Target_3.BackColor = Color.White;
            textBox_Target_4.BackColor = Color.White;
            textBox_Target_5.BackColor = Color.White;
            textBox_Target_6.BackColor = Color.White;
            textBox_Target_7.BackColor = Color.White;
            textBox_Target_8.BackColor = Color.White;
            textBox_Duration_1.BackColor = Color.White;
            textBox_Duration_2.BackColor = Color.White;
            textBox_Duration_3.BackColor = Color.White;
            textBox_Duration_4.BackColor = Color.White;
            textBox_Duration_5.BackColor = Color.White;
            textBox_Duration_6.BackColor = Color.White;
            textBox_Duration_7.BackColor = Color.White;
            textBox_Duration_8.BackColor = Color.White;
            textBox_Target_1.Enabled = true;
            textBox_Target_2.Enabled = true;
            textBox_Target_3.Enabled = true;
            textBox_Target_4.Enabled = true;
            textBox_Target_5.Enabled = true;
            textBox_Target_6.Enabled = true;
            textBox_Target_7.Enabled = true;
            textBox_Target_8.Enabled = true;
            textBox_Duration_1.Enabled = true;
            textBox_Duration_2.Enabled = true;
            textBox_Duration_3.Enabled = true;
            textBox_Duration_4.Enabled = true;
            textBox_Duration_5.Enabled = true;
            textBox_Duration_6.Enabled = true;
            textBox_Duration_7.Enabled = true;
            textBox_Duration_8.Enabled = true;

            button_dmm_NULL.Enabled = true;
            button_dmm_unNull.Enabled = true;
            button_go.Enabled = true;
            checkBox_SaveSprayCurrrent.Enabled = true;
            if (checkBox_SaveSprayCurrrent.Checked == true)
            {
                outputfile.Close();
            }
            myPrint("<< Exit zworker loop <<");
        }

        void zworker_ProgressChanged(object sender, ProgressChangedEventArgs e)
        {
            if (e.ProgressPercentage == 0)
            {
                List<object> argumentlist = e.UserState as List<object>;
                double Readout_AgilentDMM = (double)argumentlist[0];
                textBox_AgDMMDisplay.Text = Math.Round(Readout_AgilentDMM, 3).ToString();
                if (checkBox_SaveSprayCurrrent.Checked == true)
                {
                    outputfile.WriteLine(((DateTime.Now.Ticks - TimeOriginForSaving) * 1E-7).ToString() + ", " + Readout_AgilentDMM.ToString() + ", " + DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel.ToString());
                }
            }
            else
            {
                myPrint("Current state -> " + e.ProgressPercentage.ToString());
                textBox_Target_1.BackColor = Color.White;
                textBox_Target_2.BackColor = Color.White;
                textBox_Target_3.BackColor = Color.White;
                textBox_Target_4.BackColor = Color.White;
                textBox_Target_5.BackColor = Color.White;
                textBox_Target_6.BackColor = Color.White;
                textBox_Target_7.BackColor = Color.White;
                textBox_Target_8.BackColor = Color.White;
                textBox_Duration_1.BackColor = Color.White;
                textBox_Duration_2.BackColor = Color.White;
                textBox_Duration_3.BackColor = Color.White;
                textBox_Duration_4.BackColor = Color.White;
                textBox_Duration_5.BackColor = Color.White;
                textBox_Duration_6.BackColor = Color.White;
                textBox_Duration_7.BackColor = Color.White;
                textBox_Duration_8.BackColor = Color.White;
                switch (e.ProgressPercentage)
                {
                    case 1: textBox_Target_1.BackColor = Color.Yellow;
                            textBox_Duration_1.BackColor = Color.Yellow; 
                            break;
                    case 2: textBox_Target_2.BackColor = Color.Yellow;
                            textBox_Duration_2.BackColor = Color.Yellow;
                            break;
                    case 3: textBox_Target_3.BackColor = Color.Yellow;
                            textBox_Duration_3.BackColor = Color.Yellow;
                            break;
                    case 4: textBox_Target_4.BackColor = Color.Yellow;
                            textBox_Duration_4.BackColor = Color.Yellow;
                            break;
                    case 5: textBox_Target_5.BackColor = Color.Yellow;
                            textBox_Duration_5.BackColor = Color.Yellow;
                            break;
                    case 6: textBox_Target_6.BackColor = Color.Yellow;
                            textBox_Duration_6.BackColor = Color.Yellow;
                            break;
                    case 7: textBox_Target_7.BackColor = Color.Yellow;
                            textBox_Duration_7.BackColor = Color.Yellow;
                            break;
                    case 8: textBox_Target_8.BackColor = Color.Yellow;
                            textBox_Duration_8.BackColor = Color.Yellow;
                            break;
                    default: break;

                }
                
            }
                       

        }

        void zworker_DoWork(object sender, DoWorkEventArgs e)
        {
            List<object> argumentlist = e.Argument as List<object>;
            double Vmax = (double)argumentlist[0];
            double Vinit_step = (double)argumentlist[1];
            double Iinit_threshold_1 = (double)argumentlist[2];
            double Iinit_threshold_2 = (double)argumentlist[3];
            int Vinit_delay = (int)argumentlist[4];
            double Gain = (double)argumentlist[5];
            int Ilock_delay = (int)argumentlist[6];
            double I_Target_1 = (double)argumentlist[7];
            double I_Duration_1 = (double)argumentlist[8];
            double I_Target_2 = (double)argumentlist[9];
            double I_Duration_2 = (double)argumentlist[10];
            double I_Target_3 = (double)argumentlist[11];
            double I_Duration_3 = (double)argumentlist[12];
            double I_Target_4 = (double)argumentlist[13];
            double I_Duration_4 = (double)argumentlist[14];
            double I_Target_5 = (double)argumentlist[15];
            double I_Duration_5 = (double)argumentlist[16];
            double I_Target_6 = (double)argumentlist[17];
            double I_Duration_6 = (double)argumentlist[18];
            double I_Target_7 = (double)argumentlist[19];
            double I_Duration_7 = (double)argumentlist[20];
            double I_Target_8 = (double)argumentlist[21];
            double I_Duration_8 = (double)argumentlist[22];
            double PositiveOrNegative = (double)argumentlist[23];

            IAgilentE36xxOutput pOutput1 = DCS.Outputs.get_Item(DCS.Outputs.get_Name(1));
            

            if (checkBox_SaveSprayCurrrent.Checked == true)
            {
                outputfile = new StreamWriter(textBox_FileName.Text);
                outputfile.WriteLine("# Data acquired from Agilent 34461A DMM and Agilent DC supply");
                outputfile.WriteLine("# Date: " + DateTime.Now.ToShortDateString() + " Time: " + DateTime.Now.ToLongTimeString());
                TimeOriginForSaving = DateTime.Now.Ticks;
            }

            //MS synchronization
            serialPort2.Write("A"); //Arduino Turn On Relay
            serialPort2.Write("B"); //Arduino Turn On LED

            while (zworker.CancellationPending != true)
            {
                zInitiateFirstSpray(Vmax, Vinit_step, Iinit_threshold_1, Iinit_threshold_2, Vinit_delay, PositiveOrNegative);
                zworker.ReportProgress(1);
                zHoldSprayForFiniteTime(Vmax, I_Target_1, Gain, Ilock_delay, (long)Math.Round(I_Duration_1 * 1E7, 7), PositiveOrNegative);
                zworker.ReportProgress(2);
                zHoldSprayForFiniteTime(Vmax, I_Target_2, Gain, Ilock_delay, (long)Math.Round(I_Duration_2 * 1E7, 7), PositiveOrNegative);
                zworker.ReportProgress(3);
                zHoldSprayForFiniteTime(Vmax, I_Target_3, Gain, Ilock_delay, (long)Math.Round(I_Duration_3 * 1E7, 7), PositiveOrNegative);
                zworker.ReportProgress(4);
                zHoldSprayForFiniteTime(Vmax, I_Target_4, Gain, Ilock_delay, (long)Math.Round(I_Duration_4 * 1E7, 7), PositiveOrNegative);
                zworker.ReportProgress(5);
                zHoldSprayForFiniteTime(Vmax, I_Target_5, Gain, Ilock_delay, (long)Math.Round(I_Duration_5 * 1E7, 7), PositiveOrNegative);
                zworker.ReportProgress(6);
                zHoldSprayForFiniteTime(Vmax, I_Target_6, Gain, Ilock_delay, (long)Math.Round(I_Duration_6 * 1E7, 7), PositiveOrNegative);
                zworker.ReportProgress(7);
                zHoldSprayForFiniteTime(Vmax, I_Target_7, Gain, Ilock_delay, (long)Math.Round(I_Duration_7 * 1E7, 7), PositiveOrNegative);
                zworker.ReportProgress(8);
                zHoldSprayForFiniteTime(Vmax, I_Target_8, Gain, Ilock_delay, (long)Math.Round(I_Duration_8 * 1E7, 7), PositiveOrNegative);
                

            }

            zreset.Set();
        }

        void yworker_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            //myPrint("<< yworker stop <<");
            //throw new NotImplementedException();
        }

        void yworker_ProgressChanged(object sender, ProgressChangedEventArgs e)
        {
            List<object> argumentlist = e.UserState as List<object>;
            double DC_Value = (double)argumentlist[0];
            textBox_AgDMMDisplay.Text = Math.Round(DC_Value, 3).ToString();
            
            
        }

        void yworker_DoWork(object sender, DoWorkEventArgs e)
        {
            List<object> argumentlist = e.Argument as List<object>;
            int Delay = (int)argumentlist[0];
            long DurationTicks = (long)argumentlist[1]; //if == 0, then no duration. Run forever until cancel
            
            double dmm_data;

            if (DurationTicks == 0)
            {
                while (yworker.CancellationPending != true)
                {
                    dmm.Measurement.Initiate(); // Agilent dmm
                    //dmm.System.WaitForOperationComplete(500);
                    dmm_data = dmm.Measurement.Fetch(500);

                    
                    List<object> Status = new List<object>();
                    Status.Add(dmm_data);
                    yworker.ReportProgress(1, Status);
                    Thread.Sleep(Delay);
                }
            }
            else
            {
                long TimeOrigin = DateTime.Now.Ticks;
                long ElapsedTicks = DateTime.Now.Ticks - TimeOrigin;
                while (yworker.CancellationPending != true & ElapsedTicks<DurationTicks)
                {
                    dmm.Measurement.Initiate(); // Agilent dmm
                    //dmm.System.WaitForOperationComplete(500);
                    dmm_data = dmm.Measurement.Fetch(500);
                    List<object> Status = new List<object>();
                    Status.Add(dmm_data);
                    yworker.ReportProgress(1, Status);
                    Thread.Sleep(Delay);
                    ElapsedTicks = DateTime.Now.Ticks - TimeOrigin;
                }
            }

            yreset.Set();
            //throw new NotImplementedException();
        }

        void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            //myPrint("Shuting down ...");
            DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel = 0;
            DCS.Outputs.Enabled = true;
            if (checkBox_SaveSprayCurrrent.Checked == true)
            {
                outputfile.Close();
            }
            if (yworker.IsBusy == true)
            {
                yworker.CancelAsync();
                yreset.WaitOne();
                
            }
            Thread.Sleep(600);
        }

        void xworker_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            //MS synchronization
            serialPort2.Write("a"); //Arduino Turn OFF Relay
            serialPort2.Write("b"); //Arduino Turn OFF LED
            
            button_Sequence.Enabled = true;
            button_dmm_NULL.Enabled = true;
            button_dmm_unNull.Enabled = true;
            checkBox_SaveSprayCurrrent.Enabled = true;
            if (checkBox_SaveSprayCurrrent.Checked == true)
            {
                outputfile.Close();
            }
            myPrint("<< Exit xworker loop <<");
        }

        void xworker_ProgressChanged(object sender, ProgressChangedEventArgs e)
        {
            List<object> argumentlist = e.UserState as List<object>;
            double Readout_AgilentDMM = (double)argumentlist[0];
            //String Readout_FlukeDMM = (String)argumentlist[1];

            textBox_AgDMMDisplay.Text = Math.Round(Readout_AgilentDMM, 3).ToString();
            if (checkBox_SaveSprayCurrrent.Checked == true)
            {
                //outputfile.WriteLine(((DateTime.Now.Ticks - TimeOriginForSaving) * 1E-7).ToString() + ", " + Readout_AgilentDMM.ToString()+ ", " + );
                outputfile.WriteLine(((DateTime.Now.Ticks - TimeOriginForSaving) * 1E-7).ToString() + ", " + Readout_AgilentDMM.ToString() + ", " + DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel.ToString());
            }
            
        }
        double xmyReadDMM(double NegativeOrPositive)
        {
            double dmm_data;
            dmm.Measurement.Initiate(); // Agilent dmm
            //dmm.System.WaitForOperationComplete(500);
            dmm_data = dmm.Measurement.Fetch(500);


            dmm_data = dmm_data * NegativeOrPositive;//Positve = 1, Negative = -1

            //Display the value on the screen
            List<object> Status = new List<object>();
            Status.Add(dmm_data);
            xworker.ReportProgress(1, Status);
        return dmm_data;
        }
        double zmyReadDMM(double PositveOrNegative)
        {
            double dmm_data;
            dmm.Measurement.Initiate(); // Agilent dmm
            //dmm.System.WaitForOperationComplete(500);
            dmm_data = dmm.Measurement.Fetch(500);

            
            dmm_data = dmm_data * PositveOrNegative;

            //Display the value on the screen
            List<object> Status = new List<object>();
            Status.Add(dmm_data);
            zworker.ReportProgress(0, Status); //Progress 0 means it is measuring spray current.
            return dmm_data;
        }

        void xInitiateFirstSpray(double V_max, double V_step, double I_threshold_1, double I_threshold_2, int Delay, double NegativeOrPositive)
        {
            //Fire the first spray
            //Method: Increase the voltage gradually until the spray current > I_Threshold
            double Ivalue;
            double vset = 0;
            
            IAgilentE36xxOutput pOutput1 = DCS.Outputs.get_Item(DCS.Outputs.get_Name(1));

            Ivalue = xmyReadDMM(NegativeOrPositive);
            if (Ivalue > I_threshold_1)
            {
                return;
            }

            //If spray current < I_threshold_1, then perform initialization sequence
            while (Ivalue < I_threshold_2 & vset < V_max & xworker.CancellationPending != true)
            {
                pOutput1.VoltageLevel = vset;
                Thread.Sleep(Delay);
                Ivalue = xmyReadDMM(NegativeOrPositive);
                vset += V_step;
            }

        }
        void zInitiateFirstSpray(double V_max, double V_step, double I_threshold_1, double I_threshold_2, int Delay, double PositiveOrNegative)
        {
            //Fire the first spray
            //Method: Increase the voltage gradually until the spray current > I_Threshold
            double Ivalue;
            double vset = 0;

            IAgilentE36xxOutput pOutput1 = DCS.Outputs.get_Item(DCS.Outputs.get_Name(1));

            Ivalue = zmyReadDMM(PositiveOrNegative);
            if (Ivalue > I_threshold_1)
            {
                return;
            }

            //If spray current < I_threshold_1, then perform initialization sequence
            while (Ivalue < I_threshold_2 & vset < V_max & zworker.CancellationPending != true)
            {
                pOutput1.VoltageLevel = vset;
                Thread.Sleep(Delay);
                Ivalue = zmyReadDMM(PositiveOrNegative);
                vset += V_step;
            }

        }
        void xHoldSpray(double V_max, double I_target, double Gain, int Delay, double NegativeOrPositive)
        {
            
            double Ivalue;
            double Vpre;
            double vset = 0;
            
            IAgilentE36xxOutput pOutput1 = DCS.Outputs.get_Item(DCS.Outputs.get_Name(1));

            Vpre = pOutput1.VoltageLevel;
            Ivalue = xmyReadDMM(NegativeOrPositive);
            Thread.Sleep(10);
            vset = Vpre + (I_target - Ivalue) * Gain;

            while (xworker.CancellationPending != true & vset < V_max)
            {
                pOutput1.VoltageLevel = vset;
                Vpre = pOutput1.VoltageLevel;
                Ivalue = xmyReadDMM(NegativeOrPositive);
                Thread.Sleep(Delay);
                vset = Vpre + (I_target - Ivalue) * Gain;
                
            }
        }
        void zHoldSprayForFiniteTime(double V_max, double I_target, double Gain, int Delay, long DurationTick, double PositiveOrNegative)
        {
            if (DurationTick == 0)
            {
                return;//bye bye
            }
            double Ivalue;
            double Vpre;
            double vset = 0;

            IAgilentE36xxOutput pOutput1 = DCS.Outputs.get_Item(DCS.Outputs.get_Name(1));

            Vpre = pOutput1.VoltageLevel;
            Ivalue = zmyReadDMM(PositiveOrNegative);
            Thread.Sleep(10);
            vset = Vpre + (I_target - Ivalue) * Gain;

            long TimeOrigin = DateTime.Now.Ticks;
            long ElapsedTicks = DateTime.Now.Ticks - TimeOrigin;
            while (zworker.CancellationPending != true & vset < V_max & ElapsedTicks<DurationTick)
            {
                pOutput1.VoltageLevel = vset;
                Vpre = pOutput1.VoltageLevel;
                Ivalue = zmyReadDMM(PositiveOrNegative);
                Thread.Sleep(Delay);
                vset = Vpre + (I_target - Ivalue) * Gain;
                ElapsedTicks = DateTime.Now.Ticks - TimeOrigin;

            }
            return;
        }
        
        //****************************************************************
        //This is the Main Loop to Lock Single Spray Current
        //****************************************************************
        void xworker_DoWork(object sender, DoWorkEventArgs e)
        {
            List<object> argumentlist = e.Argument as List<object>;
            double Vmax = (double) argumentlist[0];
            double Vinit_step = (double) argumentlist[1];
            double Iinit_threshold_1 = (double)argumentlist[2];
            double Iinit_threshold_2 = (double)argumentlist[3];
            int Vinit_delay = (int)argumentlist[4];
            double Itarget = (double)argumentlist[5];
            double Gain = (double) argumentlist[6];
            int Ilock_delay = (int)argumentlist[7];
            double NegativeOrPositive = (double)argumentlist[8];

           

            IAgilentE36xxOutput pOutput1 = DCS.Outputs.get_Item(DCS.Outputs.get_Name(1));

            
            if (checkBox_SaveSprayCurrrent.Checked == true)
            {
                outputfile = new StreamWriter(textBox_FileName.Text);
                outputfile.WriteLine("# Data acquired from Agilent 34461A DMM and Agilent DC supply");
                outputfile.WriteLine("# Date: " + DateTime.Now.ToShortDateString() + " Time: " + DateTime.Now.ToLongTimeString());
                TimeOriginForSaving = DateTime.Now.Ticks;
            }

            
            
            //MS synchronization
            serialPort2.Write("A"); //Arduino Turn On Relay
            serialPort2.Write("B"); //Arduino Turn On LED

            while (xworker.CancellationPending != true)
            {

                xInitiateFirstSpray(Vmax, Vinit_step, Iinit_threshold_1, Iinit_threshold_2, Vinit_delay, NegativeOrPositive);
                xHoldSpray(Vmax, Itarget, Gain, Ilock_delay, NegativeOrPositive);

            }
            
            xreset.Set();
        }
        void myPrint(String x){
            richTextBox_Console.AppendText(x + "\n");
            richTextBox_Console.ScrollToCaret();
        }

        void myInitHardWare()
        {

            //Agilent 34461A
            try
            {
                dmm.Initialize("USB0::0x0957::0x1A07::MY53203329::0::INSTR", false, true);
                myPrint("-- Init Agilent 3446x DMM --");
                myPrint("Identifier: " + dmm.Identity.Identifier.ToString());
                myPrint("Revision: " + dmm.Identity.Revision.ToString());
                myPrint("Vendor: " + dmm.Identity.Vendor.ToString());
                myPrint("Description: " + dmm.Identity.Description.ToString());
                myPrint("Model: " + dmm.Identity.InstrumentModel.ToString());
                myPrint("FirmwareRev: " + dmm.Identity.InstrumentFirmwareRevision.ToString());
                myPrint("Serial #: " + dmm.System.SerialNumber.ToString());
                myPrint("----\n");
                
            }
            catch (Exception ex)
            {
                MessageBox.Show("No communication with Agilent 34461A !\nClick OK to close.");

            }

            // Agilent EDU36311
            try
            {

                // Edit resource and options as needed.  Resource is ignored if option Simulate=true
                string resourceDesc = "USB0::0x2A8D::0x8F01::CN61190020::0::INSTR";
                
                // Initialize the driver.  See driver help topic "Initializing the IVI-COM Driver" for additional information
                //DCS.Initialize(resourceDesc, idquery, reset, initOptions);
                DCS.Initialize(resourceDesc, false, true);


                // Print a few IIviDriverIdentity properties
                myPrint("-- Init Agilent EDU36311 DC power supply --");
                myPrint("Identifier: " + DCS.Identity.Identifier);
                myPrint("Revision: " + DCS.Identity.Revision);
                myPrint("Vendor: " + DCS.Identity.Vendor);
                myPrint("Model: " + DCS.Identity.InstrumentModel);
                myPrint("FirmwareRev: " + DCS.Identity.InstrumentFirmwareRevision);
                myPrint("----\n");

            }

            catch (Exception ex)
            {
                MessageBox.Show("No communication with Agilent EDU36311 !\nClick OK to close.");

            }


            myPrint("Open COM 20 for Arduino");
            try
            {
                serialPort2.PortName = "COM20";
                serialPort2.Open();

            }
            catch (Exception ex)
            {
                MessageBox.Show("No communication with Arduino!");

            }
        }
        void myInitHardwareConfig()
        {
            //Agilent DMM
            // Configure for a single reading, 10 Volt range
            dmm.DCVoltage.Configure(10, 1e-3);
            // Set 10 Power Line Cycles of integration time on the measurement
            dmm.DCVoltage.NPLC = 0.2;
            dmm.DCVoltage.AutoZero = Ag3446xAutoZeroEnum.Ag3446xAutoZeroOff;

        }

        private void Form1_Load(object sender, EventArgs e)
        {
            myPrint("Welcome to Lock Spray");
            myInitHardWare();
            myInitHardwareConfig();
            myPrint("Turn ON DC power supply");
            DCS.Outputs.Enabled = true;
            Summon_yworker_MonitorCurrent(0);
            button_Stop.Enabled = false;

            comboBox_ElectrosprayMode.Items.Add("Positive");
            comboBox_ElectrosprayMode.Items.Add("Negative");
            comboBox_ElectrosprayMode.Text = "Positive";
            
        }
        

        void Summon_xworker_LockCurrent()
        {
            double NegativeOrPositive = 1;
            if (comboBox_ElectrosprayMode.Text == "Negative")
            {
                NegativeOrPositive = -1;
            }
            
            double Vmax = double.Parse(textBox_V_Max.Text);
            double Vinit_step = double.Parse(textBox_V_InitStep.Text);
            double Iinit_threshold_1 = double.Parse(textBox_I_InitThreshold_1.Text);
            double Iinit_threshold_2 = double.Parse(textBox_I_InitThreshold_2.Text);
            int Vinit_delay = int.Parse(textBox_V_InitDelay.Text);
            double Itarget = double.Parse(textBox_I_Target.Text);
            double Gain = double.Parse(textBox_FeedBackGain.Text);
            int Ilock_delay = int.Parse(textBox_I_Lock_Delay.Text);

            //Passing the argument to background xworker            
            List<object> arguments = new List<object>();
            arguments.Add(Vmax);
            arguments.Add(Vinit_step);
            arguments.Add(Iinit_threshold_1);
            arguments.Add(Iinit_threshold_2);
            arguments.Add(Vinit_delay);
            arguments.Add(Itarget);
            arguments.Add(Gain);
            arguments.Add(Ilock_delay);
            arguments.Add(NegativeOrPositive);

            button_Sequence.Enabled = false;
            button_dmm_NULL.Enabled = false;
            button_dmm_unNull.Enabled = false;              

            xworker.RunWorkerAsync(arguments);
        }
        void Summon_zworker_SpraySequence()
        {

            double PositiveOrNegative = 1;
            if (comboBox_ElectrosprayMode.Text == "Negative")
            {
                PositiveOrNegative = -1;
            }
            
            double Vmax = double.Parse(textBox_V_Max.Text);
            double Vinit_step = double.Parse(textBox_V_InitStep.Text);
            double Iinit_threshold_1 = double.Parse(textBox_I_InitThreshold_1.Text);
            double Iinit_threshold_2 = double.Parse(textBox_I_InitThreshold_2.Text);
            int Vinit_delay = int.Parse(textBox_V_InitDelay.Text);
            double Gain = double.Parse(textBox_FeedBackGain.Text);
            int Ilock_delay = int.Parse(textBox_I_Lock_Delay.Text);
            double I_Target_1 = double.Parse(textBox_Target_1.Text);
            double I_Duration_1 = double.Parse(textBox_Duration_1.Text);
            double I_Target_2 = double.Parse(textBox_Target_2.Text);
            double I_Duration_2 = double.Parse(textBox_Duration_2.Text);
            double I_Target_3 = double.Parse(textBox_Target_3.Text);
            double I_Duration_3 = double.Parse(textBox_Duration_3.Text);
            double I_Target_4 = double.Parse(textBox_Target_4.Text);
            double I_Duration_4 = double.Parse(textBox_Duration_4.Text);
            double I_Target_5 = double.Parse(textBox_Target_5.Text);
            double I_Duration_5 = double.Parse(textBox_Duration_5.Text);
            double I_Target_6 = double.Parse(textBox_Target_6.Text);
            double I_Duration_6 = double.Parse(textBox_Duration_6.Text);
            double I_Target_7 = double.Parse(textBox_Target_7.Text);
            double I_Duration_7 = double.Parse(textBox_Duration_7.Text);
            double I_Target_8 = double.Parse(textBox_Target_8.Text);
            double I_Duration_8 = double.Parse(textBox_Duration_8.Text);
            

            //Passing the argument to background xworker            
            List<object> arguments = new List<object>();
            arguments.Add(Vmax);
            arguments.Add(Vinit_step);
            arguments.Add(Iinit_threshold_1);
            arguments.Add(Iinit_threshold_2);
            arguments.Add(Vinit_delay);
            arguments.Add(Gain);
            arguments.Add(Ilock_delay);
            arguments.Add(I_Target_1);
            arguments.Add(I_Duration_1);
            arguments.Add(I_Target_2);
            arguments.Add(I_Duration_2);
            arguments.Add(I_Target_3);
            arguments.Add(I_Duration_3);
            arguments.Add(I_Target_4);
            arguments.Add(I_Duration_4);
            arguments.Add(I_Target_5);
            arguments.Add(I_Duration_5);
            arguments.Add(I_Target_6);
            arguments.Add(I_Duration_6);
            arguments.Add(I_Target_7);
            arguments.Add(I_Duration_7);
            arguments.Add(I_Target_8);
            arguments.Add(I_Duration_8);
            arguments.Add(PositiveOrNegative);

            textBox_Target_1.Enabled = false;
            textBox_Target_2.Enabled = false;
            textBox_Target_3.Enabled = false;
            textBox_Target_4.Enabled = false;
            textBox_Target_5.Enabled = false;
            textBox_Target_6.Enabled = false;
            textBox_Target_7.Enabled = false;
            textBox_Target_8.Enabled = false;
            textBox_Duration_1.Enabled = false;
            textBox_Duration_2.Enabled = false;
            textBox_Duration_3.Enabled = false;
            textBox_Duration_4.Enabled = false;
            textBox_Duration_5.Enabled = false;
            textBox_Duration_6.Enabled = false;
            textBox_Duration_7.Enabled = false;
            textBox_Duration_8.Enabled = false;
            button_dmm_NULL.Enabled = false;
            button_dmm_unNull.Enabled = false;
            button_go.Enabled = false;
            zworker.RunWorkerAsync(arguments);
        }
        void Summon_wworker_HV_SequenceWithoutLocking()
        {
            double HV_1 = double.Parse(textBox_nolock_HV_1.Text);
            double HV_Duration_1 = double.Parse(textBox_nolock_duration_1.Text);
            double HV_2 = double.Parse(textBox_nolock_HV_2.Text);
            double HV_Duration_2 = double.Parse(textBox_nolock_duration_2.Text);
            double HV_3 = double.Parse(textBox_nolock_HV_3.Text);
            double HV_Duration_3 = double.Parse(textBox_nolock_duration_3.Text);
            double HV_4 = double.Parse(textBox_nolock_HV_4.Text);
            double HV_Duration_4 = double.Parse(textBox_nolock_duration_4.Text);
            double HV_5 = double.Parse(textBox_nolock_HV_5.Text);
            double HV_Duration_5 = double.Parse(textBox_nolock_duration_5.Text);
            double HV_6 = double.Parse(textBox_nolock_HV_6.Text);
            double HV_Duration_6 = double.Parse(textBox_nolock_duration_6.Text);
            double HV_7 = double.Parse(textBox_nolock_HV_7.Text);
            double HV_Duration_7 = double.Parse(textBox_nolock_duration_7.Text);
            double HV_8 = double.Parse(textBox_nolock_HV_8.Text);
            double HV_Duration_8 = double.Parse(textBox_nolock_duration_8.Text);

            //Passing the argument to background wworker            
            List<object> arguments = new List<object>();
            arguments.Add(HV_1);
            arguments.Add(HV_Duration_1);
            arguments.Add(HV_2);
            arguments.Add(HV_Duration_2);
            arguments.Add(HV_3);
            arguments.Add(HV_Duration_3);
            arguments.Add(HV_4);
            arguments.Add(HV_Duration_4);
            arguments.Add(HV_5);
            arguments.Add(HV_Duration_5);
            arguments.Add(HV_6);
            arguments.Add(HV_Duration_6);
            arguments.Add(HV_7);
            arguments.Add(HV_Duration_7);
            arguments.Add(HV_8);
            arguments.Add(HV_Duration_8);
            textBox_nolock_HV_1.Enabled = false;
            textBox_nolock_HV_2.Enabled = false;
            textBox_nolock_HV_3.Enabled = false;
            textBox_nolock_HV_4.Enabled = false;
            textBox_nolock_HV_5.Enabled = false;
            textBox_nolock_HV_6.Enabled = false;
            textBox_nolock_HV_7.Enabled = false;
            textBox_nolock_HV_8.Enabled = false;
            textBox_nolock_duration_1.Enabled = false;
            textBox_nolock_duration_2.Enabled = false;
            textBox_nolock_duration_3.Enabled = false;
            textBox_nolock_duration_4.Enabled = false;
            textBox_nolock_duration_5.Enabled = false;
            textBox_nolock_duration_6.Enabled = false;
            textBox_nolock_duration_7.Enabled = false;
            textBox_nolock_duration_8.Enabled = false;
            wworker.RunWorkerAsync(arguments);
           

        }
        void Summon_yworker_MonitorCurrent(double Duration)
        {
            int Ilock_delay = int.Parse(textBox_I_Lock_Delay.Text);

            //Convert to the Date Time Tick unit: 1s = 1E7 ticks
            long DurationTick = (long)Math.Round(Duration * 1E7, 7);

            List<object> arguments = new List<object>();
            arguments.Add(Ilock_delay);
            arguments.Add(DurationTick);

            yworker.RunWorkerAsync(arguments);

        }

        private void button_go_Click(object sender, EventArgs e)
        {
            //Stop the Agilent DMM from measuring the value
            if (button_go.Text == "Lock" && xworker.IsBusy == false)
            {
                if (yworker.IsBusy == true)
                {
                    yworker.CancelAsync();
                    yreset.WaitOne();
                }
                
                button_go.Text = "Cancel";
                Summon_xworker_LockCurrent();
                myPrint(">> Enter main loop >>");
                checkBox_SaveSprayCurrrent.Enabled = false;   

            }
            else
            {
                xworker.CancelAsync();
                xreset.WaitOne();
                button_go.Text = "Lock";
                Summon_yworker_MonitorCurrent(0);
                
            }
        }

        private void button_ResetHV_Click(object sender, EventArgs e)
        {
            DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel = 0;
            textBox_HV_SetValue.Text = DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel.ToString();
        }

        private void button_dmm_NULL_Click(object sender, EventArgs e)
        {
            
            if (button_dmm_NULL.Text == "Null")
            {
                yworker.CancelAsync();
                yreset.WaitOne();
                button_dmm_NULL.Text = "Restart";
                button_dmm_unNull.Enabled = false;
                
            }
            else
            {

                yreset.Reset();
                while (yworker.IsBusy == true)
                {
                    
                    myPrint("Wait");
                }
                dmm.DCVoltage.NullEnabled = true;
                myPrint("PresentNull value :" + dmm.DCVoltage.NullValue.ToString());
                myPrint("PresentNull value :" + dmm.DCVoltage.NullValue.ToString());
                Summon_yworker_MonitorCurrent(0);
                button_dmm_NULL.Text = "Null";
                button_dmm_unNull.Enabled = true;

            }

                

        }

        private void button_dmm_unNull_Click(object sender, EventArgs e)
        {
            if (button_dmm_unNull.Text == "unNull")
            {
                yworker.CancelAsync();
                yreset.WaitOne();
                button_dmm_unNull.Text = "Restart";
                button_dmm_NULL.Enabled = false;
             }
            else
            {
                dmm.DCVoltage.NullEnabled = false;
                Summon_yworker_MonitorCurrent(0);
                button_dmm_unNull.Text = "unNull";
                button_dmm_NULL.Enabled = true;
                

            }
        }

        private void button_Sequence_Click(object sender, EventArgs e)
        {
            if (button_Sequence.Enabled == true)
            {
                checkBox_SaveSprayCurrrent.Enabled = false;            
                button_Sequence.Enabled = false;
                button_Stop.Enabled = true;
                if (yworker.IsBusy == true)
                {
                    myPrint("Cancel yworker");
                    yworker.CancelAsync();
                    yreset.WaitOne();
                }
                if (xworker.IsBusy == true)
                {
                    myPrint("Cancel xworker");
                    xworker.CancelAsync();
                    xreset.WaitOne();
                }
                myPrint("summon z");
                Summon_zworker_SpraySequence();
                myPrint("Done..Sequence");
            }
           
        }

        private void button_Stop_Click(object sender, EventArgs e)
        {
            if (button_Sequence.Enabled == false)
            {
                button_Stop.Enabled = false;
                button_Sequence.Enabled = true;

                myPrint("Stop");
                zworker.CancelAsync();
                zreset.WaitOne();
                myPrint("Done");
                Summon_yworker_MonitorCurrent(0);
            }
            

        }

        private void button_Test_Click(object sender, EventArgs e)
        {
            System.Diagnostics.Process pEditor; pEditor = new System.Diagnostics.Process();
            pEditor.StartInfo.FileName = @"C:\Users\t12ee069Yuki\AppData\Local\Programs\Python\Python38-32\python.exe";
            pEditor.StartInfo.Arguments = "\"plot_lockspray.py\"" + " " + textBox_FileName.Text;
            pEditor.Start();
            pEditor.WaitForExit();   
        }

        private void button2_Click(object sender, EventArgs e)
        {
            DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel = DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel -= 1;
            textBox_HV_SetValue.Text = (DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel * 1000).ToString();
        }

        private void button_HV_Digit4_Up_Click(object sender, EventArgs e)
        {
            DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel = DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel += 1;
            textBox_HV_SetValue.Text = (DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel * 1000).ToString();
        }

        private void button_HV_Digit3_Up_Click(object sender, EventArgs e)
        {
            DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel = DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel += 0.1;
            textBox_HV_SetValue.Text = (DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel * 1000).ToString();
        }

        private void button_HV_Digit3_Down_Click(object sender, EventArgs e)
        {
            DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel = DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel -= 0.1;
            textBox_HV_SetValue.Text = (DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel * 1000).ToString();
        }

        private void button_HV_Digit2_Up_Click(object sender, EventArgs e)
        {
            DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel = DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel += 0.01;
            textBox_HV_SetValue.Text = (DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel * 1000).ToString();
        }

        private void button_HV_Digit2_Down_Click(object sender, EventArgs e)
        {
            DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel = DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel -= 0.01;
            textBox_HV_SetValue.Text = (DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel * 1000).ToString();
        }

        private void button_HV_Digit1_Up_Click(object sender, EventArgs e)
        {
            DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel = DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel += 0.001;
            textBox_HV_SetValue.Text = (DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel * 1000).ToString();
        }

        private void button_HV_Digit1_Down_Click(object sender, EventArgs e)
        {
            DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel = DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel -= 0.001;
            textBox_HV_SetValue.Text = (DCS.Outputs.get_Item(DCS.Outputs.get_Name(1)).VoltageLevel * 1000).ToString();
        }

        private void button_HV_StartSequence_Click(object sender, EventArgs e)
        {
            if (button_HV_StartSequence.Text == "Start HV")
            {
                checkBox_SaveSprayCurrrent.Enabled = false;
                button_Sequence.Enabled = false;
                button_Stop.Enabled = false;
                button_dmm_NULL.Enabled = false;
                button_dmm_unNull.Enabled = false;
                button_go.Enabled = false;
                if (yworker.IsBusy == true)
                {
                    myPrint("Cancel yworker");
                    yworker.CancelAsync();
                    yreset.WaitOne();
                }
                if (xworker.IsBusy == true)
                {
                    myPrint("Cancel xworker");
                    xworker.CancelAsync();
                    xreset.WaitOne();
                }
                myPrint("summon w worker");
                Summon_wworker_HV_SequenceWithoutLocking();
                button_HV_StartSequence.Text = "Stop";
            }
            else
            {
                wworker.CancelAsync();
                wreset.WaitOne();
                Summon_yworker_MonitorCurrent(0);

            }
        }

        private void button_OpenFile_Click(object sender, EventArgs e)
        {
            using (OpenFileDialog openFileDialog = new OpenFileDialog())
            {
                //openFileDialog.InitialDirectory = "c:\\";
                openFileDialog.InitialDirectory = Directory.GetCurrentDirectory();

                openFileDialog.Filter = "txt files (*.txt)|*.txt|All files (*.*)|*.*";
                openFileDialog.FilterIndex = 2;
                openFileDialog.RestoreDirectory = true;

                if (openFileDialog.ShowDialog() == DialogResult.OK)
                {
                    //Get the path of specified file
                    String f = Path.GetFileName(openFileDialog.FileName);
                    System.Diagnostics.Process pEditor; pEditor = new System.Diagnostics.Process();
                    pEditor.StartInfo.FileName = @"C:\Users\t12ee069Yuki\AppData\Local\Programs\Python\Python38-32\python.exe";
                    pEditor.StartInfo.Arguments = "\"plot_lockspray.py\"" + " " + f;
                    pEditor.Start();
                    pEditor.WaitForExit();


                }

 
            }
        }
    }
}
