MODULE CRP_GasPhase
  USE CRP_Constants
  USE CRP_General
  USE Spline

  IMPLICIT NONE
CONTAINS

!*******************************************************************************
! Switch_Function
!*******************************************************************************
!> This routine computes the switch function used to switch from the 6D
!> full potential to the 2D gas phase potential.
!*******************************************************************************
  SUBROUTINE Switch_Function( Zcc, Fso, dFso, PES )
    IMPLICIT NONE

    TYPE (CRP_GasPhase_2D) :: PES
    REAL       :: Zcc, Fso, dFso

    IF (Zcc .LE. PES%ZsoMin) THEN
      Fso = 1.00
      dFso = 0.00
      RETURN
    END IF

    IF (zcc .GE. PES%ZsoMax) THEN
      Fso = 0.00
      dFso = 0.00
      RETURN
    END IF

    Fso = 0.500 + 0.500 * COS((Zcc - PES%ZsoMin) * CRP_Constants_PI / (PES%ZsoMax - PES%ZsoMin))
    dFso = -0.500 * CRP_Constants_PI / (PES%ZsoMax - PES%ZsoMin) * SIN((Zcc - PES%ZsoMin) * &
             CRP_Constants_PI / (PES%ZsoMax - PES%ZsoMin))
  END SUBROUTINE

!*******************************************************************************
! Calculate_CRP_GasPhase_Potential
!*******************************************************************************
!>    In this routine we define a 2D potential dependent on r and Z,
!>    which is used far from the surface to switch the full 6D PES
!>    to the H2 potential in the vacuum.
!>
!>    This 2D potential is a sum of 2 1D potentials depending just on r
!>    and Z. Then, in the region where the PES id equal to this 2D
!>    potential the dynamics is "trivial". The r dependence doen't
!>    change with Z. Then ZPE is constant and the molecules are
!>    accelerated toward the surface due to the attraction with Z.
!>    Then, the results of the dynamics should not depend on
!>    the shape of the Z dependence of this 2D potential.
!.    Just the difference of the 1D Z dependent term between
!>    ZSOMAX and infinitty is important.
!*********************************************************************
  SUBROUTINE Calculate_CRP_GasPhase_Potential( Zcc, ro, V2D, dV2DinfZ, dV2DinfR, PES )
    IMPLICIT NONE

    TYPE (CRP_GasPhase_2D) :: PES
    REAL       :: Zcc, ro, V2D, dV2DinfZ, dV2DinfR, V1D, VH2

    ! Compute the contribution from r
    CALL SPLINT( PES%Rdep%X, PES%Rdep%F, PES%Rdep%Coefs, PES%Rdep%NX, ro, VH2, dV2DinfR )

    ! Compute the contribution from Z
    IF (Zcc .GE. PES%ZsoMin .AND. Zcc .LE. PES%Zdep%X(PES%Zdep%NX)) THEN
      ! The molecule is within the area spanned by the interpolated Z dependency
      CALL SPLINT( PES%Zdep%X, PES%Zdep%F, PES%Zdep%Coefs, PES%Zdep%NX, Zcc, V1D, dV2DinfZ )
    ELSE
      IF (Zcc .GT. PES%Zdep%X(PES%Zdep%NX)) THEN
        ! The molecule is above the area spanned by the interpolated Z dependency
        V1D = PES%Zdep%F(PES%Zdep%NX)
        dV2DinfZ = 0.00
      ELSE
        PRINT *, "ERROR IN V2DINF"
        STOP
      END IF
    END IF

    V2D = VH2 + V1D
  END SUBROUTINE
END MODULE
