import logging
from opcua import Client
from opcua import ua

# Set up logging for interactions and errors
logging.basicConfig(level=logging.DEBUG, filename='log.txt', filemode='a',
                    format='%(asctime)s - %(levelname)s - %(message)s')

error_logging = logging.getLogger('errorLogger')
error_logging.setLevel(logging.ERROR)
handler = logging.FileHandler('error_log.txt', 'a')
handler.setFormatter(logging.Formatter('%(asctime)s - %(levelname)s - %(message)s'))
error_logging.addHandler(handler)

class Combi9SputteringChamber:
    def __init__(self, address):
        self.client = None
        self.address = address
        self.node_ids = {}
        self.load_node_ids()

    def load_node_ids(self):
        try:
            with open('opc_nodeIDs.txt', 'r') as file:
                for line in file:
                    tag, ns_i = line.strip().split(' ', 1)
                    ns, i_value = ns_i.split(';')
                    node_id = i_value.split('=')[1]
                    self.node_ids[tag] = int(node_id)
        except Exception as e:
            error_logging.error(f"Failed to load node IDs: {e}")

    def connect(self):
        try:
            self.client = Client(self.address)
            self.client.connect()
            logging.info("Connected to the PLC")
        except Exception as e:
            error_logging.error(f"Failed to connect: {e}")

    def disconnect(self):
        if self.client:
            self.client.disconnect()
            logging.info("Disconnected from the PLC")

    def set_shutter_state(self, shutter, state):
        if not self.client:
            logging.error("Attempt to set shutter state without connection")
            return
        try:
            node = self.client.get_node(f"ns=2;i={self.node_ids[f'Autonomous.Shutters.{shutter}']}")
            node.set_value(ua.Variant(state, ua.VariantType.Int32))
            logging.info(f"Set {shutter} shutter to {state}")
        except Exception as e:
            error_logging.error(f"Error setting {shutter} shutter state: {e}")

    def get_shutter_state(self, shutter):
        if not self.client:
            logging.error("Attempt to get shutter state without connection")
            return
        try:
            node = self.client.get_node(f"ns=2;i={self.node_ids[f'Autonomous.Shutters.{shutter}']}")
            state = node.get_value()
            logging.info(f"{shutter} shutter state: {state}")
            return state
        except Exception as e:
            error_logging.error(f"Error getting {shutter} shutter state: {e}")

# Unit tests
import unittest

class TestCombi9SputteringChamber(unittest.TestCase):
    def test_node_id_loading(self):
        chamber = Combi9SputteringChamber('opc.tcp://192.168.2.20:4840')
        self.assertIn('Autonomous.NoFlow', chamber.node_ids)
        self.assertIsInstance(chamber.node_ids['Autonomous.NoFlow'], int)

    def test_shutter_operations(self):
        chamber = Combi9SputteringChamber('opc.tcp://192.168.2.20:4840')
        chamber.connect()
        chamber.set_shutter_state('G1', 1)  # Open G1
        state = chamber.get_shutter_state('G1')
        chamber.disconnect()
        self.assertEqual(state, 1)

if __name__ == "__main__":
    unittest.main()
