import tkinter as tk
from tkinter import ttk, messagebox
from opcua import Client
from opcua import ua
import logging

# Assuming the Combi9SputteringChamber class code has been properly imported and available as part of the module Combi9
from Combi9 import Combi9SputteringChamber

class GUI:
    def __init__(self, master):
        self.master = master
        master.title("Combi9 Sputtering Chamber Control")

        # Initialization of the class
        self.address_var = tk.StringVar()
        self.shutter_var = tk.StringVar()
        self.state_var = tk.IntVar()
        
        # Connection Status
        self.connection_status = tk.StringVar(value="Disconnected")
        
        # Setup frames for a cleaner layout
        self.frame_connection = ttk.Frame(master)
        self.frame_connection.grid(row=0, column=0, padx=10, pady=10, sticky="ew")
        
        self.frame_controls = ttk.Frame(master)
        self.frame_controls.grid(row=1, column=0, padx=10, pady=10, sticky="ew")

        self.frame_status = ttk.Frame(master)
        self.frame_status.grid(row=2, column=0, padx=10, pady=10, sticky="ew")
        
        # Connection Frame Widgets
        ttk.Label(self.frame_connection, text="PLC Address:").grid(row=0, column=0, sticky="w")
        ttk.Entry(self.frame_connection, textvariable=self.address_var).grid(row=0, column=1, sticky="ew")
        ttk.Button(self.frame_connection, text="Connect", command=self.connect).grid(row=0, column=2)
        ttk.Button(self.frame_connection, text="Disconnect", command=self.disconnect).grid(row=0, column=3)
        ttk.Label(self.frame_connection, text="Connection Status:").grid(row=1, column=0, sticky="w")
        ttk.Label(self.frame_connection, textvariable=self.connection_status).grid(row=1, column=1, sticky="w")
        
        # Controls Frame Widgets
        ttk.Label(self.frame_controls, text="Shutter:").grid(row=0, column=0, sticky="w")
        shutter_options = ['G1', 'G2', 'G3', 'G4']
        shutter_combobox = ttk.Combobox(self.frame_controls, textvariable=self.shutter_var, values=shutter_options, state="readonly")
        shutter_combobox.grid(row=0, column=1, sticky="ew")
        shutter_combobox.current(0)  # Optional: set a default value
        ttk.Label(self.frame_controls, text="State (0 or 1):").grid(row=1, column=0, sticky="w")
        ttk.Entry(self.frame_controls, textvariable=self.state_var).grid(row=1, column=1, sticky="ew")
        ttk.Button(self.frame_controls, text="Set Shutter State", command=self.set_shutter_state).grid(row=1, column=2)
        
        # Chamber Status and Error Handling
        ttk.Label(self.frame_status, text="Chamber Status").grid(row=0, column=0, sticky="w")
        self.status_message = tk.StringVar(value="Ready")
        ttk.Label(self.frame_status, textvariable=self.status_message).grid(row=0, column=1, sticky="w")

    def connect(self):
        address = self.address_var.get()
        if not address:
            messagebox.showerror("Error", "Please provide a valid address.")
            return
        # Assuming chamber initialization and connection methods are correctly defined
        self.chamber = Combi9SputteringChamber(address)
        try:
            self.chamber.connect()
            self.connection_status.set("Connected")
        except Exception as e:
            messagebox.showerror("Connection Error", str(e))

    def disconnect(self):
        if hasattr(self, 'chamber'):
            self.chamber.disconnect()
            self.connection_status.set("Disconnected")
        else:
            messagebox.showerror("Error", "No active connection to disconnect.")

    def set_shutter_state(self):
        if not hasattr(self, 'chamber'):
            messagebox.showerror("Error", "No active connection. Please connect first.")
            return
        shutter = self.shutter_var.get()
        state = self.state_var.get()
        try:
            self.chamber.set_shutter_state(shutter, state)
            self.status_message.set(f"Shutter {shutter} set to {state}")
        except Exception as e:
            messagebox.showerror("Operation Error", str(e))

root = tk.Tk()
app = GUI(root)
root.mainloop()
