import sys
from PyQt5 import QtWidgets, QtCore
from PyQt5.QtWidgets import QApplication, QMainWindow, QPushButton, QVBoxLayout, QWidget, QLabel, QLineEdit, QCheckBox, QComboBox, QTextEdit, QGroupBox, QHBoxLayout

# Make sure to import your Keithley2400 class
from Keithley2400 import Keithley2400

class KeithleyGUI(QMainWindow):
    def __init__(self, instrument):
        super().__init__()
        self.instrument = instrument
        self.instrument_connected = False
        self.initUI()

    def initUI(self):
        self.setWindowTitle('Keithley 2400 Control Panel')
        self.setGeometry(100, 100, 900, 700)

        widget = QWidget(self)
        self.setCentralWidget(widget)
        main_layout = QVBoxLayout(widget)

        # Connection Section
        connection_group = QGroupBox('Connection Management')
        connection_layout = QVBoxLayout(connection_group)
        self.connect_button = QPushButton('Connect', self)
        self.connect_button.clicked.connect(self.connect_instrument)
        self.disconnect_button = QPushButton('Disconnect', self)
        self.disconnect_button.clicked.connect(self.disconnect_instrument)
        self.disconnect_button.setEnabled(False)  # Initially disabled
        connection_layout.addWidget(self.connect_button)
        connection_layout.addWidget(self.disconnect_button)
        main_layout.addWidget(connection_group)

        # Output Section
        output_group = QGroupBox('Output Control')
        output_layout = QVBoxLayout(output_group)
        self.enable_output_checkbox = QCheckBox('Enable Output', self)
        self.enable_output_checkbox.stateChanged.connect(self.toggle_output)
        self.enable_output_checkbox.setEnabled(False)
        output_layout.addWidget(self.enable_output_checkbox)
        main_layout.addWidget(output_group)

        # Compliance Section
        compliance_group = QGroupBox('Compliance Settings')
        compliance_layout = QVBoxLayout(compliance_group)
        self.compliance_mode_combo = QComboBox(self)
        self.compliance_mode_combo.addItems(['current', 'voltage'])
        self.compliance_value_line = QLineEdit(self)
        self.set_compliance_button = QPushButton('Set Compliance', self)
        self.set_compliance_button.clicked.connect(self.set_compliance)
        self.set_compliance_button.setEnabled(False)
        compliance_layout.addWidget(self.compliance_mode_combo)
        compliance_layout.addWidget(self.compliance_value_line)
        compliance_layout.addWidget(self.set_compliance_button)
        main_layout.addWidget(compliance_group)

        # Function Control Section
        function_group = QGroupBox('Function Control')
        function_layout = QVBoxLayout(function_group)
        self.source_function_combo = QComboBox(self)
        self.source_function_combo.addItems(['VOLT', 'CURR'])
        self.measure_function_combo = QComboBox(self)
        self.measure_function_combo.addItems(['VOLT', 'CURR', 'RES'])  # Assuming Resistance can be measured
        self.set_source_function_button = QPushButton('Set Source Function', self)
        self.set_source_function_button.clicked.connect(self.set_source_function)
        self.set_measure_function_button = QPushButton('Set Measure Function', self)
        self.set_measure_function_button.clicked.connect(self.set_measure_function)
        self.set_source_function_button.setEnabled(False)
        self.set_measure_function_button.setEnabled(False)
        function_layout.addWidget(self.source_function_combo)
        function_layout.addWidget(self.measure_function_combo)
        function_layout.addWidget(self.set_source_function_button)
        function_layout.addWidget(self.set_measure_function_button)
        main_layout.addWidget(function_group)

        # Source Value Section
        source_value_group = QGroupBox('Source Value Settings')
        source_value_layout = QVBoxLayout(source_value_group)
        self.source_value_mode_combo = QComboBox(self)
        self.source_value_mode_combo.addItems(['VOLT', 'CURR'])
        self.source_value_line = QLineEdit(self)
        self.set_source_value_button = QPushButton('Set Source Value', self)
        self.set_source_value_button.clicked.connect(self.set_source_value)
        self.set_source_value_button.setEnabled(False)
        source_value_layout.addWidget(self.source_value_mode_combo)
        source_value_layout.addWidget(self.source_value_line)
        source_value_layout.addWidget(self.set_source_value_button)
        main_layout.addWidget(source_value_group)

        # Data Management
        data_group = QGroupBox('Data Management')
        data_layout = QVBoxLayout(data_group)
        self.read_data_button = QPushButton('Read Data', self)
        self.read_data_button.clicked.connect(self.read_data)
        self.data_display = QTextEdit(self)
        self.trigger_count_line = QLineEdit(self)
        self.set_trigger_button = QPushButton('Set Trigger Count', self)
        self.set_trigger_button.clicked.connect(self.set_trigger_count)
        self.read_data_button.setEnabled(False)
        self.set_trigger_button.setEnabled(False)
        data_layout.addWidget(self.read_data_button)
        data_layout.addWidget(self.data_display)
        data_layout.addWidget(self.trigger_count_line)
        data_layout.addWidget(self.set_trigger_button)
        main_layout.addWidget(data_group)

        widget.setLayout(main_layout)

    def connect_instrument(self):
        try:
            self.instrument.connect()
            self.instrument_connected = True
            self.statusBar().showMessage('Connected')
            self.disconnect_button.setEnabled(True)
            self.enable_output_checkbox.setEnabled(True)
            self.set_compliance_button.setEnabled(True)
            self.set_source_function_button.setEnabled(True)
            self.set_measure_function_button.setEnabled(True)
            self.set_source_value_button.setEnabled(True)
            self.read_data_button.setEnabled(True)
            self.set_trigger_button.setEnabled(True)
        except Exception as e:
            self.statusBar().showMessage(f'Error: {str(e)}')
            self.instrument_connected = False

    def toggle_output(self, state):
        if self.instrument_connected:
            enable = state == QtCore.Qt.Checked
            try:
                self.instrument.enable_output(enable)
                self.statusBar().showMessage('Output toggled')
            except Exception as e:
                self.statusBar().showMessage(f'Error: {str(e)}')
        else:
            self.statusBar().showMessage('Instrument not connected')

    def set_compliance(self):
        if self.instrument_connected:
            mode = self.compliance_mode_combo.currentText()
            value = float(self.compliance_value_line.text())
            try:
                self.instrument.set_compliance(mode, value)
                self.statusBar().showMessage('Compliance Set')
            except Exception as e:
                self.statusBar().showMessage(f'Error: {str(e)}')
        else:
            self.statusBar().showMessage('Instrument not connected')

    def set_source_function(self):
        if self.instrument_connected:
            function = self.source_function_combo.currentText()
            try:
                self.instrument.set_source_function(function)
                self.statusBar().showMessage('Source Function Set')
            except Exception as e:
                self.statusBar().showMessage(f'Error: {str(e)}')
        else:
            self.statusBar().showMessage('Instrument not connected')

    def set_measure_function(self):
        if self.instrument_connected:
            function = self.measure_function_combo.currentText()
            try:
                self.instrument.set_measure_function(function)
                self.statusBar().showMessage('Measure Function Set')
            except Exception as e:
                self.statusBar().showMessage(f'Error: {str(e)}')
        else:
            self.statusBar().showMessage('Instrument not connected')

    def set_source_value(self):
        if self.instrument_connected:
            mode = self.source_value_mode_combo.currentText()
            value = float(self.source_value_line.text())
            try:
                self.instrument.source_value(mode, value)
                self.instrument.enable_output(True)  # Enable output when setting source value
                self.enable_output_checkbox.setChecked(True)
                self.statusBar().showMessage(f'Source {mode} set to {value}, Output Enabled')
            except Exception as e:
                self.statusBar().showMessage(f'Error: {str(e)}')
        else:
            self.statusBar().showMessage('Instrument not connected')

    def read_data(self):
        if self.instrument_connected:
            try:
                data = self.instrument.read_data()
                self.data_display.setText(data)
                self.instrument.log_data(data.split(','))
                self.instrument.enable_output(False)  # Turn off output after reading data
                self.enable_output_checkbox.setChecked(False)
                self.statusBar().showMessage('Data Read and Logged, Output Disabled')
            except Exception as e:
                self.statusBar().showMessage(f'Error: {str(e)}')
        else:
            self.statusBar().showMessage('Instrument not connected')

    def set_trigger_count(self):
        if self.instrument_connected:
            count = int(self.trigger_count_line.text())
            try:
                self.instrument.set_trigger_count(count)
                self.statusBar().showMessage('Trigger Count Set')
            except Exception as e:
                self.statusBar().showMessage(f'Error: {str(e)}')
        else:
            self.statusBar().showMessage('Instrument not connected')

    def disconnect_instrument(self):
        if self.instrument_connected:
            try:
                self.instrument.disconnect()
                self.instrument_connected = False
                self.disconnect_button.setEnabled(False)
                self.enable_output_checkbox.setEnabled(False)
                self.set_compliance_button.setEnabled(False)
                self.set_source_function_button.setEnabled(False)
                self.set_measure_function_button.setEnabled(False)
                self.set_source_value_button.setEnabled(False)
                self.read_data_button.setEnabled(False)
                self.set_trigger_button.setEnabled(False)
                self.statusBar().showMessage('Disconnected')
            except Exception as e:
                self.statusBar().showMessage(f'Error: {str(e)}')
        else:
            self.statusBar().showMessage('Instrument not connected')

def main():
    app = QApplication(sys.argv)
    keithley = Keithley2400('ASRL5::INSTR')  # Replace 'ASRL5::INSTR' with your instrument's resource name
    ex = KeithleyGUI(keithley)
    ex.show()
    sys.exit(app.exec_())

if __name__ == '__main__':
    main()
