import pyvisa as visa # Expert review
import csv
import logging
from datetime import datetime

# Configure logging
logging.basicConfig(filename='error_log.txt', level=logging.ERROR, format='%(asctime)s - %(message)s')

class Keithley2400:
    def __init__(self, resource_name):
        self.resource_manager = visa.ResourceManager()
        self.instrument = None
        self.resource_name = resource_name

    def connect(self):
        try:
            self.instrument = self.resource_manager.open_resource(self.resource_name)
            self.instrument.write('*RST')  # Reset the instrument to its default settings
            logging.info("Instrument connected and reset")
        except Exception as e:
            logging.error(f"Failed to connect to the instrument: {e}")
            raise

    def enable_output(self, enable=True):
        try:
            self.instrument.write(f'OUTP:STAT {1 if enable else 0}')
        except Exception as e:
            logging.error(f"Failed to {'enable' if enable else 'disable'} output: {e}")
            raise

    def set_compliance(self, mode, value):
        try:
            if mode.lower() in ['curr', 'current']:
                self.instrument.write(f':SENS:CURR:PROT {value}')
            elif mode.lower() in ['volt', 'voltage']:
                self.instrument.write(f':SENS:VOLT:PROT {value}')
            else:
                raise ValueError("Invalid mode. Choose 'current' or 'voltage'.")
        except Exception as e:
            logging.error(f"Failed to set {mode} compliance to {value}: {e}")
            raise

    def set_source_function(self, function):
        try:
            self.instrument.write(f':SOUR:FUNC {function}')
        except Exception as e:
            logging.error(f"Failed to set source function to {function}: {e}")
            raise

    def set_measure_function(self, function):
        try:
            self.instrument.write(f':SENS:FUNC "{function}"')
        except Exception as e:
            logging.error(f"Failed to set measure function to {function}: {e}")
            raise

    def source_value(self, mode, value):
        try:
            self.instrument.write(f':SOUR:{mode} {value}')
        except Exception as e:
            logging.error(f"Failed to source {mode} with value {value}: {e}")
            raise

    def set_trigger_count(self, count):
        try:
            self.instrument.write(f':TRIG:COUN {count}')
        except Exception as e:
            logging.error(f"Failed to set trigger count to {count}: {e}")
            raise

    def read_data(self):
        try:
            return self.instrument.query('READ?')
        except Exception as e:
            logging.error("Failed to read data from the instrument: {e}")
            raise

    def log_data(self, data):
        with open('measurement_log.csv', 'a', newline='') as file:
            writer = csv.writer(file)
            writer.writerow([datetime.now().strftime("%Y-%m-%d %H:%M:%S"), *data])

    def disconnect(self):
        if self.instrument:
            self.instrument.close()
            logging.info("Instrument disconnected")

# Usage example
if __name__ == "__main__":
    keithley = Keithley2400('ASRL5::INSTR')
    try:
        keithley.connect()
        keithley.enable_output(True)
        keithley.set_compliance(mode="CURR",value=0.1)
        keithley.set_source_function('VOLT')
        keithley.set_measure_function('CURR')
        keithley.source_value('VOLT', 5.0)  # Example: source 5 V
        keithley.enable_output(True) # Expert review.
        data = keithley.read_data()
        keithley.log_data(data.split(','))
        print(data)
    finally:
        keithley.enable_output(False) # Expert review
        keithley.disconnect()