(* ::Package:: *)

AverageOverMolecules::usage = 
"AverageOverMolecules[dataset, property, {startMol, endMol}, {startFrame, endFrame}, totalMolecules] computes the average molecular property vector for a selected subset of molecules and frames from a dataset.

Arguments:
  dataset         \[LongDash] Path to the dataset file (e.g., \"traj_lf.dat\")
  property        \[LongDash] Property to average (e.g., \"dipole\", \"J1\", \"s2\")
  {startMol, endMol} \[LongDash] Range of molecule indices to include in the average
  {startFrame, endFrame} \[LongDash] Range of frame indices to process
  totalMolecules  \[LongDash] Total number of molecules per frame in the dataset

The function uses interpolated quantum chemistry data (via InterpolateQuantumChemistryData) to transform local property vectors to the lab frame and averages them across molecules. The output is saved to a file named based on the input dataset and property.

Progress is displayed during processing.";


AverageOverMolecules[dataset_, property_, moleculeRange_, frameRange_, totalMolecules_] :=
 Module[
   {
     inputFile, fx, fy, fz, baseName, outputMX, outputDAT,
     startMol, endMol, startFrame, endFrame,
     inputStream, results, frameData, moleculeBlocks,
     selected, thetas, fxvals, fyvals, fzvals,
     localVectors, rotated, rotationMatrix, frameIndex, averageVector
   },
   
   (* Input ranges *)
   {startMol, endMol} = moleculeRange;
   {startFrame, endFrame} = frameRange;
   
   (* Property selector *)
   inputFile = Switch[property,
     "dipole", "me.dat",
     "J1", "JFH1.dat",
     "J2", "JFH2.dat",
     "J3", "JFH3.dat",
     "s1", "sF1.dat",
     "s2", "sF2.dat",
     "s3", "sF3.dat",
     _, Message[AverageOverMolecules::badprop, property]; Return[$Failed];
   ];
   
   {fx, fy, fz} = InterpolateQuantumChemistryData[inputFile, "Periodic" -> True];
   
   (* Output paths *)
   baseName = StringDelete[dataset, ".dat"];
   outputMX = baseName <> "_" <> property <> ".mx";
   outputDAT = baseName <> "_" <> property <> ".dat";
   If[FileExistsQ[outputDAT], DeleteFile[outputDAT]];
   If[FileExistsQ[outputMX], DeleteFile[outputMX]];
   
   inputStream = OpenRead[dataset];
   results = {};
   
   Monitor[
     Do[
       frameData = Quiet@ReadList[inputStream, Number, 10 * totalMolecules];
       If[Length[frameData] != 10 * totalMolecules,
         Message[AverageOverMolecules::shortframe, frameIndex, Length[frameData]];
         Break[];
       ];
       
       moleculeBlocks = Partition[frameData, 10];
       selected = moleculeBlocks[[startMol ;; endMol]];
       thetas = selected[[All, 10]];
       
       fxvals = fx /@ thetas;
       fyvals = fy /@ thetas;
       fzvals = fz /@ thetas;
       localVectors = Transpose[{fxvals, fyvals, fzvals}];
       
       rotated = MapThread[
         Function[{block, vec},
           rotationMatrix = Partition[block[[1 ;; 9]], 3];
           rotationMatrix . vec
         ],
         {selected, localVectors}
       ];
       
       averageVector = Mean[rotated];
       AppendTo[results, averageVector];
       ,
       {frameIndex, startFrame, endFrame}
     ],
     Row[{"Processing frame ", Dynamic[frameIndex], " of ", endFrame}]
   ];
   
   Close[inputStream];
   
   (* Export outputs *)
   Export[outputMX, results];
   Export[outputDAT, results, "Table"];
 ];

(* Helpful messages *)
AverageOverMolecules::shortframe = "Stopped at frame `1`: incomplete frame (read `2` numbers instead of expected full frame).";
AverageOverMolecules::badprop = "Unrecognized property: `1`. Use one of: \"dipole\", \"J1\", \"J2\", etc.";

