(* ::Package:: *)

Clear[GenerateMolecularPropertiesFunctions]
GenerateMolecularPropertiesFunctions::usage =
"GenerateMolecularPropertiesFunctions[\"JFile\" -> \"J_iso_anti.dat\", \"SigmaFile\" -> \"s_iso_anti.dat\"] \
returns an association containing isotropic and antisymmetric interpolation functions \
for both couplings and shieldings.";

Options[GenerateMolecularPropertiesFunctions] = {
  "JFile" -> "Data//J_iso_anti.dat",
  "SigmaFile" -> "Data//s_iso_anti.dat"
};

GenerateMolecularPropertiesFunctions[opts : OptionsPattern[]] := Module[
  {
    jRaw, sRaw,
    anglesJ, anglesS,
    j1IsoFunc, j2IsoFunc, j3IsoFunc,
    j1Func, j2Func, j3Func,
    \[Sigma]1IsoFunc, \[Sigma]2IsoFunc, \[Sigma]3IsoFunc,
    \[Sigma]1Func, \[Sigma]2Func, \[Sigma]3Func
  },

  (* --- Import files --- *)
  jRaw = Import[OptionValue["JFile"]];
  sRaw = Import[OptionValue["SigmaFile"]];

  (* --- Convert angles --- *)
  anglesJ = jRaw[[All, 1]] Degree;
  anglesS = sRaw[[All, 1]] Degree;

  (* --- Coupling interpolation --- *)
  j1IsoFunc = Interpolation[Transpose[{anglesJ, Abs[jRaw[[All, 2]]]}], InterpolationOrder -> 2];
  j2IsoFunc = Interpolation[Transpose[{anglesJ, Abs[jRaw[[All, 4]]]}], InterpolationOrder -> 2];
  j3IsoFunc = Interpolation[Transpose[{anglesJ, Abs[jRaw[[All, 6]]]}], InterpolationOrder -> 2];

  j1Func = Interpolation[Transpose[{anglesJ, Abs[jRaw[[All, 3]]]}], InterpolationOrder -> 2];
  j2Func = Interpolation[Transpose[{anglesJ, Abs[jRaw[[All, 5]]]}], InterpolationOrder -> 2];
  j3Func = Interpolation[Transpose[{anglesJ, Abs[jRaw[[All, 7]]]}], InterpolationOrder -> 2];

  (* --- Shielding interpolation --- *)
  \[Sigma]1IsoFunc = Interpolation[Transpose[{anglesS, Abs[sRaw[[All, 2]]]}], InterpolationOrder -> 2];
  \[Sigma]2IsoFunc = Interpolation[Transpose[{anglesS, Abs[sRaw[[All, 4]]]}], InterpolationOrder -> 2];
  \[Sigma]3IsoFunc = Interpolation[Transpose[{anglesS, Abs[sRaw[[All, 6]]]}], InterpolationOrder -> 2];

  \[Sigma]1Func = Interpolation[Transpose[{anglesS, Abs[sRaw[[All, 3]]]}], InterpolationOrder -> 2];
  \[Sigma]2Func = Interpolation[Transpose[{anglesS, Abs[sRaw[[All, 5]]]}], InterpolationOrder -> 2];
  \[Sigma]3Func = Interpolation[Transpose[{anglesS, Abs[sRaw[[All, 7]]]}], InterpolationOrder -> 2];

  (* --- Return association --- *)
  <|
    "Couplings" -> <|
      "Isotropic" -> {j1IsoFunc, j2IsoFunc, j3IsoFunc},
      "Antisymmetric" -> {j1Func, j2Func, j3Func}
    |>,
    "Shieldings" -> <|
      "Isotropic" -> {\[Sigma]1IsoFunc, \[Sigma]2IsoFunc, \[Sigma]3IsoFunc},
      "Antisymmetric" -> {\[Sigma]1Func, \[Sigma]2Func, \[Sigma]3Func}
    |>
  |>
];
