(* ::Package:: *)

BeginPackage["NMER`"];


GromacsToXYZ::usage =
"GromacsToXYZ[selectionIndex, totalFrames, inputFileName, atomNamesList, outputFileName, mode] \
extracts atomic coordinates from a GROMACS .gro file and exports them to a flat XYZ-style file. \
The function supports two modes:

- mode = \"molecule\" (default): extracts a single molecule (given by selectionIndex) across multiple frames.
- mode = \"frame\": extracts all molecules from a single frame (given by selectionIndex).

The output file contains atomic coordinates (without XYZ headers or atom counts). \
Coordinates are scaled from nanometers to angstroms (\[Times]10).";


GromacsToXYZ[
  selectionIndex_,            (* moleculeIndex or frameIndex *)
  totalFrames_,               (* how many frames to read *)
  inputFileName_,             (* input GROMACS .gro file *)
  atomNamesList_,             (* list of atom names *)
  outputFileName_,            (* output XYZ file *)
  mode_: "molecule"           (* "molecule" or "frame" *)
] :=
Module[{inputStream, outputStream, numAtomsPerMolecule, currentMoleculeCoords, selectedCoords, 
        frameCounter = 1, moleculeCounter, currentFrameLines, line, lineCount = 0},

  If[!FileExistsQ[inputFileName],
    Return["Error: Input file not found."]
  ];

  numAtomsPerMolecule = Length[atomNamesList];

  inputStream = OpenRead[inputFileName];
  outputStream = OpenWrite[outputFileName];

  Monitor[
    While[frameCounter <= totalFrames,
      If[Find[inputStream, "t="] === EndOfFile, Break[]];
      If[Skip[inputStream, Number] === EndOfFile, Break[]];

      selectedCoords = {};

      If[mode === "molecule",
        (* Loop over all molecules, extract the selected one *)
        Do[
          currentMoleculeCoords = Table[
            line = Quiet[Read[inputStream, {Word, Word, Number, Number, Number, Number}]];
            If[line === EndOfFile || line === $Failed, Return[$Failed]];
            10.0 {line[[4]], line[[5]], line[[6]]}
            ,
            {numAtomsPerMolecule}
          ];

          If[moleculeCounter == selectionIndex,
            selectedCoords = currentMoleculeCoords;
          ];
          ,
          {moleculeCounter, 1, 828}
        ];

        (* Write selected molecule to file *)
        If[selectedCoords =!= {},
          MapThread[
            WriteString[
              outputStream,
              #1, " ",
              NumberForm[#2[[1]], {8, 4}], " ",
              NumberForm[#2[[2]], {8, 4}], " ",
              NumberForm[#2[[3]], {8, 4}], "\n"
            ] &,
            {atomNamesList, selectedCoords}
          ];
        ];
        ,
        (* mode === "frame": extract all molecules in this frameIndex *)
        If[frameCounter == selectionIndex,
          Do[
            currentMoleculeCoords = Table[
              line = Quiet[Read[inputStream, {Word, Word, Number, Number, Number, Number}]];
              If[line === EndOfFile || line === $Failed, Return[$Failed]];
              10.0 {line[[4]], line[[5]], line[[6]]}
              ,
              {numAtomsPerMolecule}
            ];

            (* Write molecule to output *)
            MapThread[
              WriteString[
                outputStream,
                #1, " ",
                NumberForm[#2[[1]], {8, 4}], " ",
                NumberForm[#2[[2]], {8, 4}], " ",
                NumberForm[#2[[3]], {8, 4}], "\n"
              ] &,
              {atomNamesList, currentMoleculeCoords}
            ];
            ,
            {moleculeCounter, 1, 828}
          ];
        ,
        (* Skip remaining lines for other frames *)
          Skip[inputStream, {Record, 828 * numAtomsPerMolecule}]
        ];
      ];

      frameCounter++;
    ],
    ProgressIndicator[frameCounter, {1, totalFrames}]
  ];

  Close[inputStream];
  Close[outputStream];

  Print[
    "\:2705 Exported ", 
    If[mode === "molecule", "molecule ", "frame "], selectionIndex, 
    If[mode === "molecule", " across " <> ToString[totalFrames] <> " frames", " (all molecules)"], 
    " to ", outputFileName
  ];

  outputFileName
]


EndPackage[];
