(* ::Package:: *)

InterpolateQuantumChemistryData::usage = 
"InterpolateQuantumChemistryData[filename, OptionsPattern[]] reads quantum chemistry data from the specified file, applies a fixed rotation matrix to the vectors, and returns three interpolation functions for the x, y, and z components of the rotated property vector as functions of angle \[Theta] (in degrees).

Options:
  \"Periodic\" -> True | False (default: False)
    When set to True, the function treats the angle domain as periodic and appends a point at \[Theta] = 180\[Degree] to ensure smooth interpolation around the wrap-around point.";



Options[InterpolateQuantumChemistryData] = {"Periodic" -> False};

InterpolateQuantumChemistryData[dataset_, OptionsPattern[]] :=
 Module[
   {periodicQ, rotationMatrix, raw, thetas, vectors, rotatedVectors, dataX, dataY, dataZ},
   
   periodicQ = OptionValue["Periodic"];
   
   rotationMatrix = {
     {0.45760482701384925, -0.8890042984534848, 0.016406694513291426},
     {0.8891504688820242, 0.4575874075160142, -0.005020773809860704},
     {-0.0030440073097153892, 0.016885550450034176, 0.9998527952681324}
   };
   
   (* Read the full dataset: {\[Theta], px, py, pz} *)
   raw = Quiet@ReadList[dataset, {Number, Number, Number, Number}];
   If[raw === $Failed || raw === {}, 
     Message[InterpolateQuantumChemistryData::readfail, dataset]; Return[$Failed]
   ];
   
   thetas = raw[[All, 1]];
   vectors = raw[[All, {2, 3, 4}]];  (* px, py, pz *)
   
   (* Apply rotation to each vector *)
   rotatedVectors = (Transpose[rotationMatrix] . #) & /@ vectors;
   rotatedVectors = Transpose[rotatedVectors]; (* Separate x, y, z components *)

   (* Combine into {\[Theta], value} format for interpolation *)
   dataX = Transpose[{thetas, rotatedVectors[[1]]}];
   dataY = Transpose[{thetas, rotatedVectors[[2]]}];
   dataZ = Transpose[{thetas, rotatedVectors[[3]]}];
   
   {
     Interpolation[dataX, PeriodicInterpolation -> periodicQ],
     Interpolation[dataY, PeriodicInterpolation -> periodicQ],
     Interpolation[dataZ, PeriodicInterpolation -> periodicQ]
   }
];
