(* ::Package:: *)

PlotAveragedVectors::usage =
"PlotAveragedVectors[file, OptionsPattern[]] reads a file containing per-frame averaged molecular property vectors (e.g. dipoles) and plots the selected components over time (frame index).

Arguments:
  file \[LongDash] Path to the file containing a table of vectors (one per frame, each a 3D list)

Options:
  \"Components\" -> {\"x\", \"y\", \"z\", \"norm\"} (default: {\"x\", \"y\", \"z\"})
    Select which components of the vectors to plot: x, y, z, and/or the vector norm (magnitude).
    
  \"FrameRange\" -> {start, end} or Automatic (default: Automatic)
    Restrict the plot to a specific range of frames.
    
  \"PlotJoined\" -> True | False (default: True)
    Whether to connect points with lines (True for line plot, False for scatter).
    
  \"PlotRange\" -> Automatic | All | {{xmin, xmax}, {ymin, ymax}} (default: Automatic)
    Specify plot range settings as in ListLinePlot.
    
  \"Legend\" -> True | False (default: True)
    Toggle legend for plotted components.
    
  \"Label\" -> string (default: \"Frame\")
    Label for the x-axis (e.g., \"Time\" or \"Frame\").

Returns:
  A ListLinePlot showing the evolution of the selected vector components over frames.";


Options[PlotAveragedVectors] = {
  "Components" -> {"x", "y", "z"},
  "FrameRange" -> Automatic,
  "FrameStep" -> 0.1,  (* time per frame in picoseconds *)
  "PlotJoined" -> True,
  "PlotRange" -> Automatic,
  "Legend" -> True,
  "Label" -> "Time (ps)",
  "VariableSymbol" -> "\[Mu]"
};

PlotAveragedVectors[file_, OptionsPattern[]] :=
 Module[
   {
     components, frameRange, frameStep, joinedQ, plotRange, legendQ, label, symbol,
     data, nFrames, times, series, labels, ext, start, end
   },
   
   (* Read options *)
   components = OptionValue["Components"];
   frameRange = OptionValue["FrameRange"];
   frameStep = OptionValue["FrameStep"];
   joinedQ = OptionValue["PlotJoined"];
   plotRange = OptionValue["PlotRange"];
   legendQ = OptionValue["Legend"];
   label = OptionValue["Label"];
   symbol = ToString[OptionValue["VariableSymbol"]];
   
   (* Import data *)
   ext = FileExtension[file];
   data = Switch[ext,
     "mx", Import[file],
     "dat", Import[file],
     _, Message[PlotAveragedVectors::badext, ext]; Return[$Failed];
   ];
   
   (* Validate shape *)
   If[!MatrixQ[data] || Dimensions[data][[2]] =!= 3,
     Message[PlotAveragedVectors::baddata, file];
     Return[$Failed];
   ];
   
   nFrames = Length[data];
   times = N[Range[0, nFrames - 1] * frameStep];  (* Time axis in ps *)

   (* Apply frame slicing if requested *)
   If[MatchQ[frameRange, {start_, end_}] && start <= end,
     start = Max[start, 1];
     end = Min[end, nFrames];
     times = times[[start ;; end]];
     data = data[[start ;; end]];
   ];
   
   (* Build plot data *)
   series = {};
   labels = {};
   
   If[MemberQ[components, "x"],
     AppendTo[series, data[[All, 1]]];
     AppendTo[labels, symbol <> "\:2093"];
   ];
   If[MemberQ[components, "y"],
     AppendTo[series, data[[All, 2]]];
     AppendTo[labels, symbol <> "\:1d67"];
   ];
   If[MemberQ[components, "z"],
     AppendTo[series, data[[All, 3]]];
     AppendTo[labels, symbol <> "\:d835\:dc67"];
   ];
   If[MemberQ[components, "norm"],
     AppendTo[series, Norm /@ data];
     AppendTo[labels, "\:2016" <> symbol <> "\:2016"];
   ];
   
   ListLinePlot[
     Transpose[{times, #}] & /@ series,
     PlotRange -> plotRange,
     Joined -> joinedQ,
     PlotLegends -> If[legendQ, labels, None],
     AxesLabel -> {label, symbol <> " component"},
     ImageSize -> Large
   ]
];

(* Error messages *)
PlotAveragedVectors::baddata = "The file `1` must contain a numeric matrix with 3 columns.";
PlotAveragedVectors::badext = "Unsupported file extension: `1`. Use a .mx or .dat file.";

