(* ::Package:: *)

ClearAll[VisualizeBoltzmannDistribution];
Options[VisualizeBoltzmannDistribution] = {
  "GridRadii" -> {0.2, 0.4, 0.6, 0.8, 1.0},
  "Annotations" -> {-56.4, -176.0}
};

VisualizeBoltzmannDistribution[filename_, opts : OptionsPattern[]] := Module[
  {
    rawData, weightedData, boltzmann, angleRange, fInterp, integral, normalizedData,
    polarCurve, polarPoints, gridDecorations, radii, annotations, annotationGraphics
  },

  (* Import and Boltzmann weight *)
  rawData = Import[filename];
  boltzmann = Exp[-#[[2]]/2.479] & /@ rawData;
  weightedData = MapThread[{#1[[1]], #2} &, {rawData, boltzmann}];
  
  (* Interpolate and normalize *)
  angleRange = {Min[weightedData[[All, 1]]], Max[weightedData[[All, 1]]]};
  fInterp = Interpolation[weightedData];
  integral = NIntegrate[fInterp[t], {t, angleRange[[1]], angleRange[[2]]}];
  normalizedData = weightedData;
  normalizedData[[All, 2]] = normalizedData[[All, 2]] / integral;
  fInterp = Interpolation[normalizedData];

  (* Plot core visuals *)
  polarCurve = PolarPlot[fInterp[t], {t, angleRange[[1]], angleRange[[2]]},
    PlotStyle -> {Thickness[0.01], Red}, Axes -> None, PlotRange -> All];
  polarPoints = ListPolarPlot[normalizedData,
    PlotStyle -> {PointSize[0.02], Red}, Axes -> None, PlotRange -> All];

  (* Polar grid *)
  radii = OptionValue["GridRadii"];
  gridDecorations = Graphics[
    {
      Opacity[0.9],
      Table[Line[{{0, 0}, 1.02 {Cos[i], Sin[i]}}], {i, 0, 2 Pi, Pi/6}],
      Table[Text[Style[Superscript[ToString[180 If[i > Pi, i - 2 Pi, i]/Pi], "\[SmallCircle]"], 30],
        1.15 {Cos[i], Sin[i]}], {i, 0, 2 Pi - Pi/6, Pi/6}],
      Table[BezierCurve[Table[r {Sin[i], Cos[i]}, {i, 0, 2 Pi, Pi/60}]], {r, radii}]
    }
  ];

  (* Angle Annotations *)
  annotations = OptionValue["Annotations"];
annotationGraphics = Graphics[
  {
    Thickness[Large],
    Table[
      Module[{\[Theta] = a Degree, labelAngle},
        labelAngle = If[Abs[a] > 90, \[Theta] + Pi, \[Theta]]; (* flip text if too rotated *)
        {
          Line[{{0, 0}, 1.25 {Cos[\[Theta]], Sin[\[Theta]]}}],
          Rotate[
            Text[
              Style[Superscript[ToString[a], "\[SmallCircle]"], 35, Bold],
              1.4 {Cos[\[Theta]], Sin[\[Theta]]}],
            labelAngle
          ]
        }
      ],
      {a, annotations}
    ]
  }
];

  (* Combine all layers *)
  Show[
    gridDecorations, polarCurve, polarPoints, annotationGraphics,
    ImageSize -> 600
  ]
];
