(* ::Package:: *)

Clear[VisualizeIsotropicAndAntisymmetricPart]
Options[VisualizeIsotropicAndAntisymmetricPart] = {
  "Type" -> "Couplings",
  "ThetaRange" -> {-180, 180},
  "StepSize" -> 10,
  "Functions" -> {},
  "GridScale" -> Automatic
};

VisualizeIsotropicAndAntisymmetricPart[opts : OptionsPattern[]] := Module[
  {
    \[Theta]Range, d\[Theta], \[Theta]List, \[Theta]Rad,
    isoFuncs, antiFuncs,
    isoData, antiData,
    maxValue, gridScale, tag,
    toPolar, polarGrid, radialBeziers, radialTicks,
    labelStyle, plot1, plot2
  },

  \[Theta]Range = OptionValue["ThetaRange"];
  d\[Theta] = OptionValue["StepSize"];
  \[Theta]List = Range @@ Append[\[Theta]Range, d\[Theta]];
  \[Theta]Rad = \[Theta]List Degree;
  {isoFuncs, antiFuncs} = OptionValue["Functions"];
  tag = OptionValue["Type"];
  labelStyle = 50;

  (* === Convert to polar data === *)
  toPolar[f_] := Transpose[{\[Theta]Rad, Abs[f /@ \[Theta]Rad]}];
  isoData = toPolar /@ isoFuncs;
  antiData = toPolar /@ antiFuncs;

  (* === Dynamic or manual grid scaling === *)
  maxValue = Max[Flatten[Last /@ Join[isoData, antiData]]];
  gridScale = Replace[OptionValue["GridScale"], Automatic -> maxValue];

  (* === Polar grid drawing === *)
  radialBeziers[r_] := BezierCurve[Table[r {Sin[i], Cos[i]}, {i, 0, 2 Pi, Pi/60}]];
  radialTicks[r_] := Table[
    Text[
      Style[Superscript[ToString[180 Mod[i, 2 Pi]/Pi], "\[Degree]"], 30],
      1.1 r {Cos[i], Sin[i]}
    ],
    {i, 0, 2 Pi - Pi/6, Pi/6}
  ];

  polarGrid[rmax_] := Graphics[{
    Opacity[0.9],
    Table[Line[{{0, 0}, rmax {Sin[i], Cos[i]}}], {i, 0, 2 Pi, Pi/6}],
    radialTicks[rmax],
    radialBeziers /@ {rmax, rmax * 2/3, rmax * 1/3}
  }];

  (* === Plots === *)
  plot1 = Show[
    polarGrid[gridScale],
    ListPolarPlot[isoData,
      Joined -> True,
      PlotStyle -> {Thickness[0.01]},
      Axes -> None
    ],
    ImageSize -> 600
  ];

  plot2 = Show[
    polarGrid[gridScale/10],
    ListPolarPlot[antiData,
      Joined -> True,
      PlotStyle -> Thickness[0.01],
      PlotLegends -> {
        Style["\!\(\*SubscriptBox[\"F\", \"a\"]\)", labelStyle],
        Style["\!\(\*SubscriptBox[\"F\", \"b\"]\)", labelStyle],
        Style["\!\(\*SubscriptBox[\"F\", \"c\"]\)", labelStyle]
      },
      Axes -> None
    ],
    ImageSize -> 600
  ];

  Grid[
    {
      {
        Text[Style[
          Switch[tag,
            "Couplings",
            "\!\(\*SuperscriptBox[\(\[InvisiblePrefixScriptBase]\), \(3\)]\)\!\(\*SubscriptBox[\(J\), \(iso\)]\)(\!\(\*SuperscriptBox[\(\[InvisiblePrefixScriptBase]\), \(19\)]\)F, \!\(\*SuperscriptBox[\(\[InvisiblePrefixScriptBase]\), \(1\)]\)H)",
            "Shieldings",
            "\!\(\*SubscriptBox[\(\[Sigma]\), \(iso\)]\)(\!\(\*SuperscriptBox[\(\[InvisiblePrefixScriptBase]\), \(19\)]\)F)"
          ],
          labelStyle]
        ],
        Text[Style[
          Switch[tag,
            "Couplings",
            "\!\(\*SuperscriptBox[\(J\), \(\[Star]\)]\)(\!\(\*SuperscriptBox[\(\[InvisiblePrefixScriptBase]\), \(19\)]\)F, \!\(\*SuperscriptBox[\(\[InvisiblePrefixScriptBase]\), \(1\)]\)H)",
            "Shieldings",
            "\!\(\*SuperscriptBox[\(\[Sigma]\), \(\[Star]\)]\)(\!\(\*SuperscriptBox[\(\[InvisiblePrefixScriptBase]\), \(19\)]\)F)"
          ],
          labelStyle]
        ]
      },
      {plot1, plot2}
    }
  ]
];

