(* ::Package:: *)

Clear[VisualizeMolecularProperties2D]
VisualizeMolecularProperties2D::usage =
"VisualizeMolecularProperties2D[
  fieldType, projection, dipoleFunc, fieldFuncs, opts
] projects dipole and vector field (coupling or shielding) interpolation functions into 2D.

Arguments:
- fieldType: \"Couplings\" or \"Shieldings\"
- projection: \"YZ\" or \"XZ\"
- dipoleFunc: interpolation function \[Mu](\[Theta]) (input in radians)
- fieldFuncs: list of interpolated vector field functions (J(\[Theta]) or \[Sigma](\[Theta]))
Options:
  HighlightAngle -> \[Theta] (in degrees)
  ThetaRange -> {minDeg, maxDeg}
  StepSize -> angle increment in degrees
  ScaleDipole -> \[Mu] scale (default 200)
  ScaleField -> J/\[Sigma] scale (default 100/50)
  ImageSize -> size of plot";


Options[VisualizeMolecularProperties2D] = {
  HighlightAngle -> -60,
  DipoleThetaRange -> {-180, 180},
  FieldThetaRange -> {-180, 180},
  StepSize -> 10,
  ScaleDipole -> 200,
  ScaleField -> Automatic,
  ImageSize -> 400
};

VisualizeMolecularProperties2D[
  fieldType_?StringQ,
  projection_?StringQ,
  dipoleFunc_InterpolatingFunction,
  fieldFuncs_List,
  OptionsPattern[]
] := Module[
  {
    projIndex,
    \[Mu]RangeDeg, \[Sigma]RangeDeg, d\[Theta], \[Theta]\[Mu]List, \[Theta]\[Sigma]List,
    highlight, \[Mu]list, fieldLists,
    sDip, sField, size, project2D,
    dipGraphics, fieldGraphics, baseGraphics,
    defaultScale
  },

  (* Option values *)
  {\[Mu]RangeDeg, \[Sigma]RangeDeg, d\[Theta], highlight, sDip, sField, size} = 
    OptionValue[{
      DipoleThetaRange, FieldThetaRange, StepSize,
      HighlightAngle, ScaleDipole, ScaleField, ImageSize
    }];

  \[Theta]\[Mu]List = Degree Range @@ Append[\[Mu]RangeDeg, d\[Theta]];
  \[Theta]\[Sigma]List = Degree Range @@ Append[\[Sigma]RangeDeg, d\[Theta]];

  (* Projection index *)
  projIndex = Switch[projection,
    "YZ", {2, 3},
    "XZ", {1, 3},
    _, Message[VisualizeMolecularProperties2D::badproj, projection]; Return[$Failed]
  ];

  project2D[v_] := v[[projIndex]];

  defaultScale = Switch[fieldType, "Couplings", 100, "Shieldings", 50, _, 100];
  If[sField === Automatic, sField = defaultScale];

  (* Sample dipole values *)
  \[Mu]list = Table[
    {ToString[Round[\[Theta] Degree^-1]], dipoleFunc[\[Theta]]},
    {\[Theta], \[Theta]\[Mu]List}
  ];

  (* Sample field values *)
  fieldLists = Table[
    Table[
      {ToString[Round[\[Theta] Degree^-1]], f[\[Theta]]},
      {\[Theta], \[Theta]\[Sigma]List}
    ],
    {f, fieldFuncs}
  ];

  (* Dipole arrows *)
  dipGraphics = Table[
    {
      If[\[Mu]list[[i, 1]] == ToString[highlight], Opacity[1.0], Opacity[0.3]],
      Darker[Green],
      Arrow[{{0, 0}, sDip project2D[\[Mu]list[[i, 2]]]}],
      Opacity[0.8],
      Text[Style[\[Mu]list[[i, 1]], 16], sDip project2D[\[Mu]list[[i, 2]]] 1.1]
    },
    {i, Length[\[Mu]list]}
  ];

  (* Field vectors *)
  fieldGraphics = Flatten[
    Table[
      Table[
        {
          If[fieldLists[[fi, i, 1]] == ToString[highlight], Opacity[1.0], Opacity[0.1]],
          Darker[Orange],
          Arrow[{{0, 0}, sField project2D[fieldLists[[fi, i, 2]]]}],
          If[fieldLists[[fi, i, 1]] == ToString[highlight], Opacity[1.0], Opacity[0]],
          Text[Style[fieldLists[[fi, i, 1]], 16], sField project2D[fieldLists[[fi, i, 2]]] 1.1]
        },
        {i, Length[fieldLists[[fi]]]}
      ],
      {fi, Length[fieldFuncs]}
    ]
  ];

  baseGraphics = Graphics[{
    White, Line[{{-150, 0}, {100, 0}}],
    Black,
    Arrow[{{0, 0}, {200, 0}}],
    Text[Style[Switch[projection, "YZ", "y", "XZ", "x"], Large], {220, 0}],
    Arrow[{{0, 0}, {0, 200}}],
    Text[Style["z", Large], {0, 220}],
    Thickness[0.006],
    Circle[{0, 0}, 10], Disk[{0, 0}, 3]
  }];

  Show[
    Graphics[{dipGraphics, fieldGraphics}],
    baseGraphics,
    ImageSize -> size
  ]
];
