(* ::Package:: *)

Clear[VisualizeMolecularProperties3D]
VisualizeMolecularProperties3D::usage =
"VisualizeMolecularProperties[
  tfp, ellipsoid, dipoleFunc, dipoleParams,
  fieldFuncs, fieldCenters, fieldParams,
  fieldType
] generates a 3D molecular visualization including dipole vectors and either \
antisymmetric couplings (\"Couplings\") or shieldings (\"Shieldings\").

Parameters:
- tfp: Transformed molecular structure (e.g. from TransformToMolecularFrame)
- ellipsoid: Probability ellipsoid (Graphics3D)
- dipoleFunc: Interpolated dipole vector function
- dipoleParams: {thetaMin, thetaMax, step, highlightAngle}
- fieldFuncs: List of interpolated functions (either J(\[Theta]) or \[Sigma](\[Theta]))
- fieldCenters: Corresponding list of 3D centers
- fieldParams: {thetaMin, thetaMax, step, highlightAngle}
- fieldType: \"Couplings\" or \"Shieldings\"";


VisualizeMolecularProperties3D[
  tfp_, ellipsoid_, 
  dipoleFunc_, {dipMin_, dipMax_, dipStep_, dipHighlight_},
  fieldFuncs_List, fieldCenters_List, {fMin_, fMax_, fStep_, fHighlight_},
  fieldType_?StringQ
] := Module[
  {axes, dipoles, fieldVectors},

  (* Coordinate axes *)
  axes = Graphics3D[{
    Opacity[1],
    Black,
    Arrow[Tube[{{0, 0, 0}, {400, 0, 0}}]],
    Text[Style["x", Large], {420, 0, 0}],
    Arrow[Tube[{{0, 0, 0}, {0, 400, 0}}]],
    Text[Style["y", Large], {0, 420, 0}],
    Arrow[Tube[{{0, 0, 0}, {0, 0, 400}}]],
    Text[Style["z", Large], {0, 0, 420}]
  }];

  (* Dipole vectors \[Mu](\[Theta]) *)
  dipoles = PlotInterpolatedDipoles[dipoleFunc, {dipMin, dipMax}, dipStep, dipHighlight];

  (* Switch field type: either J(\[Theta]) or \[Sigma](\[Theta]) *)
  fieldVectors = Switch[fieldType,
    "Couplings",
      Show @@ Table[
        PlotInterpolatedCouplings[
          fieldFuncs[[i]], fieldCenters[[i]], {fMin, fMax}, fStep, fHighlight
        ],
        {i, Length[fieldFuncs]}
      ],
    "Shieldings",
      Show @@ Table[
        PlotInterpolatedShieldings[
          fieldFuncs[[i]], fieldCenters[[i]], {fMin, fMax}, fStep, fHighlight
        ],
        {i, Length[fieldFuncs]}
      ],
    _,
      Message[VisualizeMolecularProperties3D::badtype, fieldType]; 
      Return[$Failed]
  ];

  (* Final combined plot *)
  Show[tfp, ellipsoid, axes, dipoles, fieldVectors, ImageSize -> 600]
];
