from openbabel import pybel 
import math
from rdkit import Chem
from rdkit.Chem import Draw
from rdkit.Chem.Draw import IPythonConsole
from rdkit.Chem.MolStandardize import rdMolStandardize
from rdkit.Chem.EnumerateStereoisomers import EnumerateStereoisomers,StereoEnumerationOptions
IPythonConsole.drawOptions.comicMode=True
from rdkit import RDLogger
RDLogger.DisableLog('rdApp.info')
import rdkit
# print(rdkit.__version__)

def stereoisomers(smiles) :
    mol = Chem.MolFromSmiles(smiles)
    chiral_centers = Chem.FindMolChiralCenters(mol, includeUnassigned=True)
    print("Chiral Centers:", chiral_centers)
    options = StereoEnumerationOptions(onlyUnassigned=False)
        
    isomers = list(EnumerateStereoisomers(mol,options=options))
    Lsmiles =[]    
    for iso in isomers:
        Lsmiles.append(Chem.MolToSmiles(iso, isomericSmiles=True))
    return Lsmiles
    
def tautomers(smiles):
    '''based on https://bitsilla.com/blog/2021/06/standardizing-a-molecule-using-rdkit/  '''
    # follows the steps in
    # https://github.com/greglandrum/RSC_OpenScience_Standardization_202104/blob/main/MolStandardize%20pieces.ipynb
    # as described **excellently** (by Greg) in
    # https://www.youtube.com/watch?v=eWTApNX8dJQ
    mol = Chem.MolFromSmiles(smiles)
    # removeHs, disconnect metal atoms, normalize the molecule, reionize the molecule
    clean_mol = rdMolStandardize.Cleanup(mol) 
     
    # if many fragments, get the "parent" (the actual mol we are interested in) 
    parent_clean_mol = rdMolStandardize.FragmentParent(clean_mol)
         
    # try to neutralize molecule
    uncharger = rdMolStandardize.Uncharger() # annoying, but necessary as no convenience method exists
    uncharged_parent_clean_mol = uncharger.uncharge(parent_clean_mol)
     
    # note that no attempt is made at reionization at this step
    # nor at ionization at some pH (rdkit has no pKa caculator)
    # the main aim to to represent all molecules from different sources
    # in a (single) standard way, for use in ML, catalogue, etc.
     
    te = rdMolStandardize.TautomerEnumerator() # idem
    taut_uncharged_parent_clean_mol = te.Canonicalize(uncharged_parent_clean_mol)
    tauts =te.Enumerate(uncharged_parent_clean_mol)
    Lsmiles =[]
    for iRmol in tauts:
        Lsmiles.append(Chem.MolToSmiles(iRmol))
    Lsmiles=list(set(Lsmiles))
    return Lsmiles   #,tauts, taut_uncharged_parent_clean_mol

def sample_pH(smiles,pH=7):
    # Create an Open Babel molecule from the SMILES string
    pybmol = pybel.readstring("smi", smiles)

    # Create a copy of the molecule to manipulate
    Lmol=[]
    Lmol.append(pybmol.clone)
    # Lmol.append(pybmol.clone)
    if pH==7 :
        Lmol.append(pybmol.clone)
    else :
        for iph in pH :
            Lmol.append(pybmol.clone)

    # Remove existing hydrogens
    Lmol[0].OBMol.DeleteHydrogens()
    Lmol[1].OBMol.DeleteHydrogens()
    # Add hydrogens without considering explicit charges
    Lmol[0].OBMol.AddHydrogens(False)        
    # Add hydrogens with explicit charge consideration
    # Lmol[1].OBMol.AddHydrogens(False, True, 7)
    if pH!=7 :
        for iph in range(len(pH)) :
            Lmol[1+iph].OBMol.DeleteHydrogens()
        # Add hydrogens without considering explicit charges
            Lmol[1+iph].OBMol.AddHydrogens(False, True, pH[iph])
    # Get SMILES representations
    Lsmiles = []
    for imol in Lmol:
        Lsmiles.append(imol.write("smi").strip())
    Lsmiles=list(set(Lsmiles))
    return Lsmiles

def stereo_then_pH(smiles = "Oc1c(cccc3)c3nc2ccncc12",pH=[0,7.,14.]):
    # Example usage
# input_smiles = "NCC(=O)O"  # Acetic acid
    # input_smiles = "Oc1c(cccc3)c3nc2ccncc12"  # Acetic acid    
    Lsmiles1 = stereoisomers(smiles)
    unique_union = set(Lsmiles1) | set([smiles])
    # print("I0 SMILES with AddHydrogens(True):", Chem.MolToSmiles(newmol))
    for jsmiles in Lsmiles1:
        # print("I1 SMILES with AddHydrogens(True):", jsmiles)         
        Lsmiles2= sample_pH(jsmiles,pH=pH)
        unique_union = set(unique_union) | set(Lsmiles2)
    unique_union=list(unique_union)
    return unique_union


def stan_then_pH(smiles = "Oc1c(cccc3)c3nc2ccncc12",pH=[0,7.,14.]):
    # Example usage
# input_smiles = "NCC(=O)O"  # Acetic acid
    # input_smiles = "Oc1c(cccc3)c3nc2ccncc12"  # Acetic acid
    Lsmiles1 = tautomers(smiles)
    unique_union = set(Lsmiles1) | set([smiles])
    # print("I0 SMILES with AddHydrogens(True):", Chem.MolToSmiles(newmol))
    for jsmiles in Lsmiles1:
        # print("I1 SMILES with AddHydrogens(True):", jsmiles)         
        Lsmiles2= sample_pH(jsmiles,pH=pH)
        unique_union = set(unique_union) | set(Lsmiles2)
    unique_union=list(unique_union)
    return unique_union
        # print("\n")        
        # for ismiles in unique_union:
        #     print("I 2 SMILES with AddHydrogens(True):",ismiles)
        # Draw.MolsToGridImage(allmol)

def pH_then_st(smiles = "Oc1c(cccc3)c3nc2ccncc12",pH=[0,7,14]):
    # Example usage
# input_smiles = "NCC(=O)O"  # Acetic acid
    # input_smiles = "Oc1c(cccc3)c3nc2ccncc12"  # Acetic acid
    Lsmiles1= sample_pH(smiles,pH=pH)
    unique_union = set(Lsmiles1)
    # for ismiles in Lsmiles1:
        # print("SMILES with AddHydrogens(True):", ismiles)
    for ismiles in Lsmiles1:
        Lsmiles2= tautomers(ismiles)
        unique_union = set(unique_union) | set(Lsmiles2)
        # print("2 SMILES with AddHydrogens(True):", Chem.MolToSmiles(newmol))
        # Draw.MolsToGridImage(allmol)

    unique_union=list(unique_union)
    return unique_union

if __name__ == "__main__" :
    # smiles = "Oc1c(cccc3)c3nc2ccncc12"
    # smiles ="O=c1c2ccccc2[nH]c2ccncc12"
    # smiles ="NCC(=O)O"
    ismiles ='C1CCC(CC1)C[C@@H](C(=O)N[C@@H](C[C@@H]2CCNC2=O)C=O)NC(=O)C3=CC4=CC=CC=C4N3'
    smiles ='C1CCC(CC1)CC(C(=O)NC(CC2CCNC2=O)C=O)NC(=O)C3=CC4=CC=CC=C4N3'

    # Lsmiles = sample_pH(smiles =smiles,pH=[0,7,14])
    # print("\n")        
    # for ismiles in Lsmiles:
        # print("I 2 SMILES with AddHydrogens(True):",ismiles)

    # Lsmiles = stan_then_pH(smiles =smiles,pH=[0.,7.,14.])
    # Lsmiles =pH_then_st(smiles =smiles)
    # Lsmiles= sample_pH(smiles =smiles,pH=[0.,7.,14.])
    # Lsmiles=tautomers(smiles =smiles)
    Lsmiles=stereoisomers(smiles =smiles)

    for ismiles in Lsmiles:
        print(" ALL SMILES :",ismiles)
        # Draw.MolsToGridImage(allmol)
    if ismiles in Lsmiles : 
        print ("OK")
    else :
        print ("not found")
        
def is_isomeric_smiles(smiles: str) -> bool:
    mol = Chem.MolFromSmiles(smiles)
    if not mol:
        raise ValueError("Invalid SMILES string.")
    
    # Check for chiral tags (stereocenters)
    has_chiral_centers = any(atom.HasProp('_ChiralityPossible') for atom in mol.GetAtoms())
    
    # Check for double bond stereochemistry
    has_double_bond_stereo = any(
        bond.GetStereo() != Chem.BondStereo.STEREONONE for bond in mol.GetBonds()
    )
    
    return has_chiral_centers or has_double_bond_stereo