import re
import argparse

def extract_spin_multiplicity(logfile):
    """
    Reads a Gaussian log file and returns the estimated spin multiplicity
    found in the file (if any).

    Parameters:
        logfile (str): Path to the Gaussian log file.

    Returns:
        int or None: The estimated spin multiplicity, or None if not found.
    """
    multiplicity_patterns = [
        r"Multiplicity\s*=\s*(\d+)",
        r"Spin multiplicity\s*=\s*(\d+)"
    ]
    
    with open(logfile, 'r') as f:
        for line in f:
            for pattern in multiplicity_patterns:
                match = re.search(pattern, line, re.IGNORECASE)
                if match:
                    return int(match.group(1))
    return None

def main():
    parser = argparse.ArgumentParser(
        description="Extract the estimated spin multiplicity from a Gaussian log file."
    )
    parser.add_argument("logfile", help="Path to the Gaussian log file")
    args = parser.parse_args()

    multiplicity = extract_spin_multiplicity(args.logfile)
    
    if multiplicity is not None:
        print(f"Estimated spin multiplicity: {multiplicity}")
        # To convert multiplicity to spin state (S = (Multiplicity-1)/2):
        spin_state = (multiplicity - 1) / 2
        print(f"Estimated spin state (S): {spin_state}")
    else:
        print("No spin multiplicity information found in the log file.")

if __name__ == "__main__":
    main()
