import os
import warnings
from pathlib import Path
from openbabel import openbabel as ob
from vina import Vina
import subprocess
import warnings
from numpy import array, dot
import numpy as np
import argparse
from rdkit import Chem
from rdkit.Chem import AllChem
from openbabel import openbabel as ob
import sys
import os
from openbabel import openbabel as ob
from rdkit import Chem
from rdkit.Chem import AllChem

class Vina_dock:
    """
    A class for automating rigid molecular docking using the Vina algorithm. This class initializes with protein 
    and ligand XYZ files, configures the docking grid, performs necessary file format conversions, executes 
    the docking process with specified parameters, and generates XYZ files for the best docking poses 
    for further analysis or visualization.
    
    Attributes:
        protein (str): Path to the protein file in XYZ format.
        ligand (str): Path to the ligand file in XYZ format.
        grid_center (list): Coordinates for the center of the docking grid.
        grid_size (list): Dimensions of the docking grid.
        grid_spacing (str): Spacing of the grid for docking.
        seed (int): Seed for reproducibility.
        n_poses (int): Number of docking poses to generate.
        energy_range: Energy range for pose output
        exhaustiveness: Exhaustiveness of the search. 
        py2 (str): Path to the Python 2 interpreter for running external scripts.
        utils_path (str): Path to utility scripts for file preparation.
    """

    def __init__(self, protein="", ligand="", smiles = "", grid_center=None, grid_size=None, grid_spacing=None, seed=None, n_poses=9, energy_range=7.0, exhaustiveness=8, py2=None, utils_path=None, output_all_poses= False):
        """
        Initializes the Vina_dock class with protein and ligand files, grid parameters, and paths 
        for utility scripts. Validates that necessary paths are set, and prepares files for docking.

        Parameters:
            protein (str): Path to the protein file in XYZ format.
            ligand (str): Path to the ligand file in XYZ format.
            grid_center (list, optional): Coordinates (x, y, z) for the center of the docking grid. If None, the ligand's center of mass is used.
            grid_size (list, optional): Size of the docking grid, in 1 or 3 dimensions.
            grid_spacing (float, optional): Grid spacing for docking calculations. Defaults to 0.375.
            seed (int, optional): Seed for reproducibility of docking results. Defaults to None.
            n_poses (int): Number of docking poses to generate. Defaults to 9.
            energy_range (float, optional): Energy range for pose output. Defaults to 7.0.  
            exhaustiveness (int, optional): Exhaustiveness of the search. Defaults to 8.
            py2 (str, optional): Path to the Python 2 interpreter for external scripts.
            utils_path (str, optional): The path to the directory containing the utility scripts required for file preparation. 
            This should be the path to the MGLTools installation, specifically the folder where 
            the AutoDockTools/Utilities24 directory is located. For example, 
            /path/to/MGLToolsPckgs/AutoDockTools/Utilities24. 
            MGLTools must be installed to run the necessary file conversion scripts for docking.
        """
        
        self.protein = protein
        self.ligand = ligand
        self.smiles = smiles
        self.seed = seed
        self.n_poses = n_poses
        self.energy_range = energy_range  
        self.exhaustiveness = exhaustiveness  
        self.output_all_poses = output_all_poses
         # Set py2 and utils_path either from parameters or environment variables
        self.py2 = py2 if py2 is not None else os.getenv('AUTODOCKTOOLS_PY2_PATH')
        self.utils_path = utils_path if utils_path is not None else os.getenv('AUTODOCKTOOLS_UTILS_PATH')
        
        # If grid_center is not provided, calculate it using the ligand's center of mass
        if grid_center is None:
            self.grid_center = self.calculate_mass_center(self.ligand)
        else:
            self.grid_center = grid_center
        
        # If grid_spacing is not provided, default to 0.375
        self.grid_spacing = grid_spacing if grid_spacing is not None else 0.375
        self.grid_size = grid_size if grid_size is not None else [40, 40, 40]
        if len(self.grid_size) == 1:
             self.grid_size *= 3  # Repeat the single value three times
        
        # Validate all parameters
        self.validate_parameters(grid_center, grid_size, grid_spacing, energy_range, exhaustiveness, py2, utils_path)
        
        # Updated call to locate_conversion_scripts
        self.locate_conversion_scripts()
        self.P_xyz_to_pdbqt()
        self.L_xyz_to_pdbqt(input_file=ligand)
    
    def validate_parameters(self, grid_center, grid_size, grid_spacing, energy_range, exhaustiveness, py2, utils_path):
        """ 
        Validates all parameters for the class, including file checks.
        """
        # Validate py2 and utils_path
        if not self.py2 or not self.utils_path:
            raise ValueError("Both py2 and utils_path must be provided either as parameters or environment variables.")

        # Validate grid_spacing
        if self.grid_spacing is not None:
            if not (0.2 <= grid_spacing <= 1.0):
                raise ValueError(f"Grid spacing must be between 0.2 and 1.0. Given value: {grid_spacing}")

        # Validate grid_center
        if grid_center is not None:
            if not isinstance(grid_center, list):
                raise ValueError("Grid center must be a list or None for the ligand's center of mass")
            if len(grid_center) != 3:
                raise ValueError("Grid center must be a list of 3 values (x, y, z) or None for the ligand's center of mass")

        #Validate grid_size 
        if grid_size is not None and len(grid_size) not in [1, 3]:
            raise ValueError("Grid size must be a list of 1 or 3 values.")

        # Validate energy_range
        if self.energy_range <= 0:
            raise ValueError(f"Energy range must be greater than zero. Given value: {energy_range}")
        if self.energy_range > 15:
            warnings.warn(f"Warning: Energy range value {energy_range} is quite high, which may lead to unrealistic docking results or longer computation times.")

        # Validate exhaustiveness
        if not isinstance(self.exhaustiveness, int):
            raise ValueError("Exhaustiveness must be an integer.")
        if self.exhaustiveness <= 0:
            raise ValueError("Exhaustiveness must be greater than zero.")
        if self.exhaustiveness < 8 or exhaustiveness > 32:
            warnings.warn(f"Warning: Exhaustiveness value {exhaustiveness} is outside the optimal range (8-32). It's still valid, but performance may vary.")

        # Validate grid_size
        if self.grid_size is not None and len(self.grid_size) not in [1, 3]:
            raise ValueError("Grid size must be a list of 1 or 3 values.")

        # Validate protein and ligand files
        if not self.protein.endswith("xyz"):
            raise ValueError("The protein file must be in XYZ format.")
        if not self.ligand.endswith("xyz"):
            raise ValueError("The ligand file must be in XYZ format.")

        # Check if the protein and ligand files exist
        if not Path(self.protein).exists():
            raise FileNotFoundError(f"The protein file '{self.protein}' does not exist.")
        if not Path(self.ligand).exists():
            raise FileNotFoundError(f"The ligand file '{self.ligand}' does not exist.")
    
    def locate_conversion_scripts(self):
        """
        Sets up paths to the utility scripts needed for preparing receptor and ligand files.
        
        This method locates the scripts prepare_receptor4.py and prepare_ligand4.py 
        in the provided utils_path directory, which are used for file conversions 
        required by the docking process.
        """
        self.fpr4 = Path(self.utils_path) / 'prepare_receptor4.py'
        self.fpl4 = Path(self.utils_path) / 'prepare_ligand4.py'

    def P_xyz_to_pdbqt(self):
        """
        Converts the protein file from XYZ to PDB and then to PDBQT format.
        Uses an external script (prepare_receptor4.py) to complete the PDBQT conversion.

        Steps:
            - Converts the protein XYZ file to PDB format.
            - Runs prepare_receptor4.py with Python 2 to convert PDB to PDBQT.
        """

        # Initialize Open Babel conversion for XYZ to PDB
        self.ppdb = Path(self.protein).with_suffix('.pdb')
        conv = ob.OBConversion()
        mol = ob.OBMol()
        conv.SetInAndOutFormats("xyz", "pdb")
        conv.ReadFile(mol, self.protein)
        conv.WriteFile(mol, str(self.ppdb))

        # Run external utility script to finalize PDBQT conversion
        try:
            result = subprocess.call([self.py2, str(self.fpr4), '-r', str(self.ppdb)])
            if result != 0:
                raise RuntimeError(f"Failed to run prepare_receptor4.py: {result}")
        except Exception as e:
            raise RuntimeError(f"An error occurred while trying to run prepare_receptor4.py: {e}")

        # Define the final PDBQT path for the protein
        self.protein_pdbqt = Path(self.ppdb).with_suffix('.pdbqt')
        return self.protein_pdbqt

    def L_xyz_to_pdbqt(self, input_file):
        """
        Converts the ligand file from XYZ to PDB, applying specific adjustments for atom naming 
        and formatting, then converts it to PDBQT format. Adjustments include ensuring unique, 
        four-character atom names for compatibility with downstream processing.

        Parameters:
            input_file (str): Path to the ligand file in XYZ format.

        Returns:
            str: Path to the generated PDBQT file for the ligand.
        """
    
        # Initialize Open Babel conversion for XYZ to PDB
        conv = ob.OBConversion()
        mol = ob.OBMol()
        conv.SetInAndOutFormats("xyz", "pdb")
        conv.ReadFile(mol, input_file)

        # Define the PDB output file path for the ligand
        self.lpdb = Path(input_file).with_suffix('.pdb').name
        conv.WriteFile(mol, str(self.lpdb))
        # Read the PDB file to adjust atom naming
        try:
            with open(self.lpdb, 'r') as file:
                pdb_lines = file.readlines()
        except FileNotFoundError:
            raise FileNotFoundError(f"Could not open the ligand PDB file: {self.lpdb}")

        adjusted_pdb_lines = []
        atom_counters = {}  # Dictionary to keep count of each atom type for unique naming

        # Process each line to ensure correct atom naming and formatting
        for line in pdb_lines:
            # Convert HETATM entries to ATOM for Vina compatibility
            if line.startswith("HETATM"):
                line = line.replace("HETATM", "ATOM  ")

            if line.startswith("ATOM  "):
                parts = list(line)  # Split line into a list of characters for easy modification
                atom_type = ''.join(parts[12:16]).strip()  # Extract atom type from the line
                
                # Count each atom type to create unique names if needed
                if atom_type not in atom_counters:
                    atom_counters[atom_type] = 1
                else:
                    atom_counters[atom_type] += 1

                # Generate a unique, four-character atom name
                if len(atom_type) > 2:
                    atom_name = f'{atom_type[:2]}{atom_counters[atom_type]:02d}'
                else:
                    atom_name = f'{atom_type}{atom_counters[atom_type]}'

                # Update atom name and molecule name (MOL) in the PDB line
                parts[12:16] = f'{atom_name:<4}'
                parts[17:20] = 'MOL'
                line = ''.join(parts)
            
            # Add the adjusted line to the list
            adjusted_pdb_lines.append(line)

        # Write the adjusted lines back to the PDB file
        with open(self.lpdb, 'w') as file:
            file.writelines(adjusted_pdb_lines)

        template = Chem.MolFromSmiles(self.smiles)

        # Load the adjusted PDB file
        pose = AllChem.MolFromPDBFile(self.lpdb)

        # Assign the bond order to force correct valence
        newMol = AllChem.AssignBondOrdersFromTemplate(template, pose)

        # Add hydrogens in 3D
        newMol_H = Chem.AddHs(newMol, addCoords=True)

        # Generate PDB block with customized formatting
        pdb_block = Chem.MolToPDBBlock(newMol_H)

        # Replace 'HETATM' with 'ATOM' and 'UNL' with 'MOL'
        pdb_block = pdb_block.replace("HETATM", "ATOM  ")
        pdb_block = pdb_block.replace(" UNL ", " MOL ")  # Ensure proper spacing
        adjusted_pdb_block = []
        for line in pdb_block.splitlines():
            if line.startswith("ATOM  ") or line.startswith("HETATM"):
                # Change chain ID to 'A' (21st character position in PDB format)
                line = line[:21] + "A" + line[22:]
            adjusted_pdb_block.append(line)

        # Join the adjusted lines back into a single block
        pdb_block = "\n".join(adjusted_pdb_block) + "\n"
        # Save the modified PDB file
        with open(self.lpdb, "w") as f:
            f.write(pdb_block)

        # Define the PDBQT output path and convert PDB to PDBQT using the external script
        self.ligand_pdbqt = Path(input_file).with_suffix('.pdbqt')
        try:
            result = subprocess.call([self.py2, str(self.fpl4), '-l', str(self.lpdb), '-U', 'nphs_lps', '-o', str(self.ligand_pdbqt)])
            if result != 0:
                raise RuntimeError(f"Failed to run prepare_ligand4.py: {result}")
        except Exception as e:
            raise RuntimeError(f"An error occurred while trying to run prepare_ligand4.py: {e}")

        return self.ligand_pdbqt
    
    def calculate_mass_center(self, input_file):
        """
        Calculates the center of mass of a ligand molecule from an XYZ file.

        Parameters:
            input_file (str): Path to the ligand XYZ file.

        Returns:
            ndarray: A numpy array representing the [x, y, z] coordinates of the ligand's center of mass.
        """
        # Define atomic weights for common elements
        atomic_weights = {
            'H': 1.00794,
            'C': 12.0107,
            'N': 14.00674,
            'O': 15.9994,
            'F': 18.9984,
            'P': 30.9738,
            'S': 32.066,
            'Cl': 35.453,
            'Br': 79.904,
            'I': 126.9045,
            'Mg': 24.305,
            'Fe': 55.845,
            'Zn': 65.38,
            'Cu': 63.546,
            'Mn': 54.938,
            'Ca': 40.078,
            'B': 10.81,
            'Si': 28.0855,
            'Se': 78.96
        }

        coords = []
        masses = []

        with open(input_file, 'r') as f:
            lines = f.readlines()

        for line in lines[2:]:
            elements = line.split()
            symbol = elements[0]
            x, y, z = float(elements[1]), float(elements[2]), float(elements[3])
            mass = atomic_weights.get(symbol, 0)

            coords.append([x, y, z])
            masses.append(mass)

        coords = array(coords)
        masses = array(masses)

        if masses.sum() == 0:
            raise ValueError("No valid atomic masses found to calculate the center of mass.")

        self.center_of_mass = dot(masses, coords) / masses.sum()

        return self.center_of_mass

    def vina_docking(self, output_file="docking_results.txt", output_prefix="pose"):
        """
        Performs molecular docking using the Vina module, sets up the receptor and ligand, defines 
        the grid, and executes the docking process. Outputs the best docking pose and, if specified, 
        all poses to PDB and XYZ formats.

        Parameters:
            output_file (str): Filename to save the docking results summary, including best energy.
            output_prefix (str): Prefix for the output pose files.
            output_all_poses (bool): If True, outputs all poses; otherwise, only outputs the best pose.

        Returns:
            float: Best docking energy (kcal/mol) if successful; otherwise, None.
        """
        self.seed = self.seed if self.seed is not None else 0
        vina = Vina(sf_name='vina', cpu=0, seed=self.seed, verbosity=1)

        # Set receptor and ligand files in PDBQT format
        vina.set_receptor(str(self.protein_pdbqt))
        vina.set_ligand_from_file(str(self.ligand_pdbqt))

        vina.compute_vina_maps(center=self.grid_center, box_size=self.grid_size, spacing=float(self.grid_spacing))
        vina.dock(exhaustiveness=self.exhaustiveness, n_poses=self.n_poses)
        
        # Retrieve the seed used by Vina
        vina_seed = vina._vina.seed()

        pdbqt_output_file = "output.pdbqt"
        if self.output_all_poses == True:
            vina.write_poses(pdbqt_output_file, n_poses=self.n_poses, energy_range=float(self.energy_range), overwrite=True)
            if os.path.exists(pdbqt_output_file) and os.path.getsize(pdbqt_output_file) > 0:
                energies = vina.energies(n_poses=self.n_poses, energy_range=self.energy_range)
                best_energy = energies[0][0] if energies.size > 0 else None
                with open(output_file, 'w') as f:
                    f.write(f"Best binding affinity: {best_energy}"+ " kcal/mol" + "\n")   
                    f.write(f"Protein: {self.protein}"+ "\n")   
                    f.write(f"Ligand: {self.ligand}"+ "\n")   
                    f.write(f"Grid center: {self.grid_center}"+ "\n")
                    f.write(f"Grid size: {self.grid_size}"+ "\n")   
                    f.write(f"Grid spacing: {self.grid_spacing}"+ "\n")   
                    f.write(f"Number of poses: {self.n_poses}"+ "\n")   
                    f.write(f"Energy range: {self.energy_range}"+ "\n")   
                    f.write(f"Exhaustiveness: {self.exhaustiveness}"+ "\n") 
                    f.write(f"Seed: {vina_seed}")     
            with open(pdbqt_output_file, 'r') as f:
                lines = f.readlines()

            poses, current_pose = [], []
            for line in lines:
                if line.startswith("REMARK"):
                    current_pose = []
                elif line.startswith("TORSDOF"):
                    poses.append(current_pose)
                else:
                    current_pose.append(line)
            
            for i, pose in enumerate(poses):
                pose_file = f"{output_prefix}_{i+1}.pdbqt"
                xyz_file = f"{output_prefix}_{i+1}.xyz"
                with open(pose_file, 'w') as f:
                    f.writelines(pose)
                self.pdbqt_to_xyz(pose_file,xyz_file)
        else:
            vina.write_pose(pdbqt_filename= pdbqt_output_file, overwrite=True)
            if os.path.exists(pdbqt_output_file) and os.path.getsize(pdbqt_output_file) > 0:
                energies = vina.energies(n_poses=1, energy_range=self.energy_range)
                best_energy = energies[0][0] if energies.size > 0 else None
                with open(output_file, 'w') as f:
                    f.write(f"Best binding affinity: {best_energy}"+ " kcal/mol" + "\n")   
                    f.write(f"Protein: {self.protein}"+ "\n")   
                    f.write(f"Ligand: {self.ligand}"+ "\n")   
                    f.write(f"Grid center: {self.grid_center}"+ "\n")
                    f.write(f"Grid size: {self.grid_size}"+ "\n")   
                    f.write(f"Grid spacing: {self.grid_spacing}"+ "\n")   
                    f.write(f"Number of poses: {self.n_poses}"+ "\n")   
                    f.write(f"Energy range: {self.energy_range}"+ "\n")   
                    f.write(f"Exhaustiveness: {self.exhaustiveness}"+ "\n") 
                    f.write(f"Seed: {vina_seed}")     
                self.pdbqt_to_xyz(pdbqt_output_file, "pose_1.xyz")
        
        return best_energy
    
    def pdbqt_to_xyz(self, input_file, output_xyz):
        conv = ob.OBConversion()
        mol = ob.OBMol()
        conv.SetInAndOutFormats("pdbqt", "pdb")
        conv.ReadFile(mol, input_file)
        template = Chem.MolFromSmiles(self.smiles)
        temp_pdb = Path(input_file).with_suffix('.pdb').name
        conv.WriteFile(mol, str(temp_pdb))
        # Load the docked pose as a PDB file
        docked_pose = AllChem.MolFromPDBFile(temp_pdb)

        # Assign the bond order to force correct valence
        newMol = AllChem.AssignBondOrdersFromTemplate(template, docked_pose)

        # Add hydrogens in 3D
        newMol_H = Chem.AddHs(newMol, addCoords=True)

        # Generate PDB block with customized formatting
        pdb_block = Chem.MolToPDBBlock(newMol_H)
        # Add hydrogens in 3D
        newMol_H = Chem.AddHs(newMol, addCoords=True)

        # Generate PDB block with customized formatting
        pdb_block = Chem.MolToPDBBlock(newMol_H)

        # Replace 'HETATM' with 'ATOM' and 'UNL' with 'MOL'
        pdb_block = pdb_block.replace("HETATM", "ATOM  ")
        pdb_block = pdb_block.replace(" UNL ", " MOL ")  # Ensure proper spacing

        # Save the modified PDB file
        with open(temp_pdb, "w") as f:
            f.write(pdb_block)
        conv = ob.OBConversion()
        conv.SetInAndOutFormats("pdb", "xyz")
        mol = ob.OBMol()
        if not conv.ReadFile(mol, temp_pdb):
            print(f"Error: Unable to read file {temp_pdb}")
            return
        if hasattr(self, 'lpdb') and self.lpdb:
            mol.SetTitle(str(self.lpdb))
        if not conv.WriteFile(mol, output_xyz):
            print(f"Error: Unable to write file {output_xyz}")
            return

   
def main(protein, ligand, smiles, grid_center=None, grid_size=None, grid_spacing=0.375, seed=None, n_poses=9, energy_range=7.0, exhaustiveness=8, output_all_poses=False, py2=None, utils_path=None):
    """
    Main function to run the docking process with Vina using specified parameters.

    Parameters:
        protein (str): Path to the protein file in XYZ format.
        ligand (str): Path to the ligand file in XYZ format.
        grid_center (list, optional): Center of the docking grid as [x, y, z]. If None, uses the ligand's center of mass.
        grid_size (list, optional): Grid size in one or three dimensions.
        grid_spacing (float, optional): Spacing between grid points for docking. Default is 0.375.
        seed (int, optional): Seed for reproducibility. Default is None.
        n_poses (int, optional): Number of docking poses to generate. Default is 9.
        energy_range (float, optional): Energy range for pose output. Default is 7.0.
        exhaustiveness (int, optional): Exhaustiveness of the search. Default is 8.
        output_all_poses (bool, optional): If True, outputs all poses. Default is False.
        py2 (str, optional): Path to Python 2 interpreter for external scripts.
        utils_path (str, optional): Path to utility scripts for file preparation.
    """
    # Initialize VinaDock class with provided parameters
    dock = Vina_dock(
        protein=protein,
        ligand=ligand,
        smiles=smiles,
        grid_center=grid_center,
        grid_size=grid_size,
        grid_spacing=grid_spacing,
        seed=seed,
        n_poses=n_poses,
        energy_range=energy_range,
        exhaustiveness=exhaustiveness,
        py2=py2,
        utils_path=utils_path,
        output_all_poses=output_all_poses
    )
    # Run the docking process and retrieve the best energy
    best_energy = dock.vina_docking(output_file="docking_results.txt")

if __name__ == '__main__':
    #Example parameters for testing
    # protein = "protein.xyz"  # Path to the protein file
    # ligand = "ligand.xyz"     # Path to the ligand file
    # smiles = "CC(=O)OC1=CC=CC=C1C(=O)O"
    # grid_center = [80.904, 64.305, 55.636]  # Example grid center
    # grid_size = [23]    # Example grid size
    # grid_spacing = 0.333  # Example grid spacing
    # seed = None
    # n_poses = 9
    # energy_range = 7
    # exhaustiveness = 8
    # py2 = None  # Path to Python 2 executable
    # utils_path = None  # Path to the utility scripts directory
    # output_all_poses = True

    # Run the main function with these parameters
    main(protein=protein, ligand=ligand, smiles=smiles, grid_center=grid_center, grid_size=grid_size, grid_spacing=grid_spacing, seed=seed, n_poses=n_poses, energy_range=energy_range, exhaustiveness=exhaustiveness, py2=py2, utils_path=utils_path, output_all_poses=output_all_poses)

    #Set up argument parser
    parser = argparse.ArgumentParser(description="Run molecular docking using Vina.")
    
    # Add arguments for all parameters
    parser.add_argument('--protein', type=str, required=True, help="Path to the protein file in XYZ format")
    parser.add_argument('--ligand', type=str, required=True, help="Path to the ligand file in XYZ format")
    parser.add_argument('--smiles', type=str, required=True, help="Smiles string of the ligand" )
    parser.add_argument('--grid_center', type=float, nargs=3, required=False, help="Grid center coordinates as x y z")
    parser.add_argument('--grid_size', type=float, nargs='*', required=False, help="Grid size: one number for cubic or three numbers for custom dimensions")
    parser.add_argument('--grid_spacing', type=float, default=0.375, help="Grid spacing (default is 0.375)")
    parser.add_argument('--seed', type=int, required=False, default=None, help="Seed number for reproducibility")
    parser.add_argument('--n_poses', type=int, default=9, help="Number of poses to generate (default is 9)")
    parser.add_argument('--energy_range', type=float, default=7.0, help="Energy range for output poses (default is 7.0)")
    parser.add_argument('--exhaustiveness', type=int, default=8, help="Exhaustiveness of the search (default is 8)")
    parser.add_argument('--py2', type=str, required=False, default=None, help="Path to Python 2 executable (required for running scripts)")
    parser.add_argument('--utils_path', type=str, required=False, default=None, help="Path to the utility scripts directory")
    parser.add_argument('--output_all_poses', type=bool, default=False, help="Set to True to output all poses as separate files. Default is False (outputs only the best pose).")

    # Parse the command-line arguments
    args = parser.parse_args()

     # Run the main function with the parsed arguments
    
    main(
        protein=args.protein, 
        ligand=args.ligand, 
        smiles=args.smiles,
        grid_center=args.grid_center, 
        grid_size=args.grid_size, 
        grid_spacing=args.grid_spacing, 
        seed=args.seed, 
        n_poses=args.n_poses, 
        energy_range=args.energy_range, 
        exhaustiveness=args.exhaustiveness, 
        py2=args.py2, 
        utils_path=args.utils_path,
        output_all_poses=args.output_all_poses
    )