#!/bin/bash
set -e
# Find the installed C compiler (gcc)
C_COMPILER=$(which gcc)
if [ -z "$C_COMPILER" ]; then
    echo "C compiler (gcc) not found!"
    exit 1
fi

# Find the installed C++ compiler (g++)
CXX_COMPILER=$(which g++)
if [ -z "$CXX_COMPILER" ]; then
    echo "C++ compiler (g++) not found!"
    exit 1
fi

# Set the CMake environment variables
export CMAKE_C_COMPILER=$C_COMPILER
export CMAKE_CXX_COMPILER=$CXX_COMPILER

# Optionally, print out the values for verification
echo "CMAKE_C_COMPILER is set to $CMAKE_C_COMPILER"
echo "CMAKE_CXX_COMPILER is set to $CMAKE_CXX_COMPILER"
# Prompt for environment name
read -p "Enter a name for the Conda environment (default: molpher-lib-build): " ENV_NAME
ENV_NAME=${ENV_NAME:-molpher-lib-build}
# Clone the repository
git clone https://github.com/lich-uct/molpher-lib.git
export REPOSITORY_ROOT=$(pwd)/molpher-lib
# Checkout development branch
cd "$REPOSITORY_ROOT"
git checkout dev
# Create and activate conda Molpher environment 
conda env create -n "$ENV_NAME" -f environment.yml
source "$(conda info --base)/etc/profile.d/conda.sh"
conda activate "$ENV_NAME"
pip install notebook
pip install ipycytoscape ipywidgets networkx requests beautifulsoup4 matplotlib pillow easygui
conda install -c conda-forge openbabel
conda install -c conda-forge matplotlib
# Build the library
mkdir -p cmake-build && cd cmake-build
cmake .. -DCMAKE_BUILD_TYPE=Debug -DPYTHON_EXECUTABLE=python3 -DRUN_SWIG=ON
make molpher_install_python

# Set environment variables
export LD_LIBRARY_PATH=$REPOSITORY_ROOT/dist/lib/:${CONDA_PREFIX}/lib/:${LD_LIBRARY_PATH}
export PYTHONPATH=$REPOSITORY_ROOT/dist/lib/python3.9/site-packages/mol
export PYTHONPATH=$REPOSITORY_ROOT/dist/lib/python3.9/site-packages/molpher-0.0.0b4.dev1-py3.9-linux-x86_64.egg/:${PYTHONPATH};

echo ""
echo " Molpher built and installed locally."
echo " Please add the following to your ~/.bashrc or ~/.zshrc:"
echo "export LD_LIBRARY_PATH=$REPOSITORY_ROOT/dist/lib/:${CONDA_PREFIX}/lib/:${LD_LIBRARY_PATH}"
echo "export PYTHONPATH=$REPOSITORY_ROOT/dist/lib/python3.9/site-packages/mol"
echo "export PYTHONPATH=$REPOSITORY_ROOT/dist/lib/python3.9/site-packages/molpher-0.0.0b4.dev1-py3.9-linux-x86_64.egg/:${PYTHONPATH};"
