module statistical_mechanics
include("parameters.jl")

using .parameters
using StaticArrays
using Interpolations
using SpecialFunctions  #For Gamma


export Q_translational, Q_rotational, Q_vibrational, Q_classical_vibrational, Q_laplace, average_over_J, Γ

function Q_translational(μ, β)                                                 #Translational partition function
    return (2π * μ / (h^2 * β))^1.5
end

function Q_rotational(Ia, Ib, Ic, β)                                          #Rotational partition function. Symmetry numbers NOT included. 
    if Ib ≈ 0.0
        return 1.0                                                   #atom 
    elseif Ia ≈ 0.0
        return 8π^2 * Ic / (h^2 * β)                                #linear molecule       
    else
        return 8π^2 * (2π / β)^1.5 * (Ia * Ib * Ic)^0.5 / h^3       #non-linear polyatomic molecule
    end
end

function Q_vibrational(ν::SVector{}, β)                                        #quantum-mechanical partition function for n harmonic oscillators 
    ν = ν * 100 * a0 * c                                            #convert to a.u.
    Q = 1.0
    for i = 1:length(ν)
        Q = Q * 1.0 / (1 - exp(-h * ν[i] * β))
    end
    return Q
end

function Q_classical_vibrational(ν::SVector{}, β)                                #Classical partition function for s harmonic oscillators 
    ν = ν * 100 * a0 * c                                           #convert to a.u. 
    Q = 1.0
    s = length(ν)
    for i = 1:s
        Q = Q / (h * ν[i] * β)
    end
    return Q
end

#Simple trapezoid integration to integrate N(E)exp(-Eβ) over E. 
function Q_laplace(E::SVector{}, state_sum_E::SVector{}, β)
    Q = 0.0
    for i = 1:(length(E)-1)
        Q = Q + (state_sum_E[i] * exp(-E[i] * β) + state_sum_E[i+1] * exp(-E[i+1] * β)) * (E[i+1] - E[i])
    end
    Q = Q / 2
    return Q
end

#Simple trapezoid integration to integrate N(E,J) over J. 
function average_over_J(J::SVector{}, state_sum_EJ::SVector{})       #Trapezoid integration with Boltzmann weighting
    state_sum_E = 0.0
    for i = 1:(length(J)-1)
        state_sum_E = state_sum_E + (state_sum_EJ[i] + state_sum_EJ[i+1]) * (J[i+1] - J[i])
    end
    state_sum_E = state_sum_E / 2
    return state_sum_E
end

function Γ(x::Real)
    return gamma(x)
end


end