#Dose-Responses from DPR Risk
#https://cawaterdatadive.shinyapps.io/DPRisk/
#See "Guidance Document" for more information on the below dose-responses
#additional file should be loaded for the manually coded hypergeometric1F1 that can work wtih mpfr (loaded in the BOR-QMRA-SI file)
#install.packages("gsl")  #for hypergeometric 1F1
#install.packages("Rmpfr")  #for high precision
library(gsl)
library(Rmpfr)

####------Enterovirus-----------------
#(Ward et al., 1986; alpha=0.253, beta=0.426)

#high precision dose-response
env_dr_hp <- function(dose) {
  env_alpha <- mpfr("0.253", precBits = 256)
  env_beta <- mpfr("0.426", precBits = 256)
  dose <- mpfr(dose, precBits = 256)
  response <- 1 - ((1 + (dose / env_beta))^(-1 * env_alpha))
  return(as(response,"mpfr"))
}

####------Giardia---------------------
#Exponential (Teunis et al., 1997; r=0.0199)

#high - precision dose-response
gia_dr_hp<-function(dose){
  gia_r <- mpfr("0.0199", precBits = 256)
  dose <- mpfr(dose, precBits = 256)
 response <- 1-exp(-1 * gia_r * dose)
 return(as(response,"mpfr"))
}

####------Cryptosporidium -------------

#Beta-Poisson (Messner and Berger, 2016; alpha=0.116, beta=0.121) 

#high - precision dose-response ---
cry_dr_hp<-function(dose){
  cry_alpha <- mpfr("0.116", precBits = 256)
  cry_beta <- mpfr("0.121", precBits = 256)
  dose <- mpfr(dose, precBits = 256)
  response <- 1 - ((1 + (dose / cry_beta))^(-1 * cry_alpha))
  return(response)
}

####------Adenovirus-----------------

## need both non-high precision and high precision for AdV
#"Exact Beta Poisson (Teunis et al., 2016; alpha=5.11, beta=2.8)"

adv_dr<-function(dose){
  adv_alpha <- 5.11
  adv_beta <- 2.8
  response <- 1 - hyperg_1F1(adv_alpha, (adv_alpha + adv_beta), -1 * dose) ###why did I add alpha and beta? because the arguments for this specific 1f1 function requires it. This was verified using wolfram alpha
  return(response)
}

##high - precision dose-response
adv_dr_hp<-function(dose){
  adv_alpha <- mpfr("5.11", precBits = 256)
  adv_beta <- mpfr("2.8", precBits = 256)
  sum_alpha_beta<-adv_alpha+adv_beta
    dose <- mpfr(dose, precBits = 256)
  response <- as(suppressWarnings(hypergeom1F1(z=-dose,a=adv_alpha, b=sum_alpha_beta)),"mpfr") #different function for 1F1 that works for high precision. manually coded in separate file.
  response<-1-response
  # response<-as(response,"mpfr")
  response<-as.numeric(response)
 return(response)
 # Wrap the return in tryCatch
  tryCatch(
    {
      return(response)
    },
    error = function(e) {
      cat("Error in returning response from function. Response is type:\n")
      class(response)
      stop("Error message: ", e$message)
    }
  )
}

####------Norovirus-----------------
# need both non-high precision and high precision dose-response for full code
#"Hypergeometric (Teunis et al., 2008; alpha=0.04, beta=0.055)" --- CHOSEN FOR DPR-----
nov_dr<-function(dose){
  nov_alpha <- 0.04
  nov_beta <- 0.055
  response <- 1 - hyperg_1F1(nov_alpha, (nov_alpha + nov_beta), -1 * dose)
  return(response)
}

# need both non-high precision and high precision dose-response for full code
nov_dr_hp<-function(dose){
  nov_alpha <- mpfr("0.04", precBits = 256)
  nov_beta <- mpfr("0.055", precBits = 256)
  sum_alpha_beta<-nov_alpha+nov_beta
  dose <- mpfr(dose, precBits = 256)
  response <- as(suppressWarnings(hypergeom1F1(z=-dose,a=nov_alpha, b=sum_alpha_beta)),"mpfr")#different function for 1F1 that works for high precision. manually coded in separate file.
  response<-1-response
  # response<-as(response,"mpfr")
  response<-as.numeric(response)
  return(response)
}

