#!/Users/mpol/opt/anaconda3/envs/qc/bin/python
# Don't forget to define the path to Python in your Anaconda environment
"""
Utility: Extract the Imaginary Mode Vector Components for Dimer Calculations

This script processes a vibrational output file (here, vib.xyz) generated by ASE.
It is designed to extract the vibrational mode vector components
associated with the imaginary vibrational mode (Mode #0) – which is essential for DIMER
calculations in VASP. Mode #0 is always imaginary (its frequency is displayed with an "i", e.g., "433.5i").
The script searches for a header line that contains "Mode #0", then reads the subsequent lines,
extracting the last three columns from each line (which represent the x, y, and z components of the
vibrational eigenvector for each atom). The extracted data are written, one line per atom, into a file
named "MODECAR" in the same directory as the input file.

Usage:
    /path/to/this/script vib.xyz

Replace "vib.xyz" with the path to your vibrational output file.
"""

import sys
import re
import os

def extract_mode0_lines(lines):
    """
    Given all lines from the input file, extract the last three columns
    for every line in the Mode #0 block.

    Parameters:
        lines (list of str): All lines from the input file.
    
    Returns:
        list of str: Extracted lines (each containing three columns).
    """
    mode0_data = []
    mode0_found = False

    for line in lines:
        # Look for the Mode #0 header; it should contain "Mode #0"
        if not mode0_found:
            if re.search(r'\bMode\s*#0\b', line):
                mode0_found = True
            continue  # skip lines until Mode #0 header is found

        # Stop if we encounter a new mode block (e.g., "Mode #1")
        if re.search(r'^\s*Mode\s*#', line):
            break

        # Skip empty lines
        if not line.strip():
            continue

        tokens = line.split()
        # Expecting at least 7 tokens (element, three coordinates, and three eigenvector components)
        if len(tokens) >= 7:
            # Append the last three tokens (the eigenvector components)
            mode0_data.append(" ".join(tokens[-3:]))

    return mode0_data

def main():
    if len(sys.argv) < 2:
        print("Usage: python extract_modecar.py input_file")
        sys.exit(1)
    
    input_file = sys.argv[1]

    try:
        with open(input_file, 'r') as f:
            lines = f.readlines()
    except IOError as e:
        print(f"Error reading file {input_file}: {e}")
        sys.exit(1)

    mode0_data = extract_mode0_lines(lines)
    if not mode0_data:
        print("No Mode #0 data found in the file.")
        sys.exit(1)

    output_file = os.path.join(os.path.dirname(os.path.abspath(input_file)), "MODECAR")
    try:
        with open(output_file, "w") as f_out:
            for entry in mode0_data:
                f_out.write(entry + "\n")
        print(f"Extracted Mode #0 data saved to {output_file}.")
    except IOError as e:
        print(f"Error writing to {output_file}: {e}")
        sys.exit(1)

if __name__ == '__main__':
    main()
