# ifndef _BONDED_INTERACTIONS_H_
# define _BONDED_INTERACTIONS_H_

/* bonded_interactions.h:
   MD functions for the energy and force of bonded interactions */

/* Math */
/* Vector cross product */
void cross(real U[], real V[], real UxV[]) {
  UxV[0] = U[1]*V[2] - U[2]*V[1];
  UxV[1] = U[2]*V[0] - U[0]*V[2];
  UxV[2] = U[0]*V[1] - U[1]*V[0];
  return;
}

/*** Load interactions from files ***/
void read_fixedbonds(struct system * sys, FILE * bfile)
{
  fprintf(stderr, LOG_INFO "Reading fixed bonds.\n");
  read_data(bfile, 1, INT, &(sys->nfixed));

  sys->fixed = calloc(sys->nfixed, sizeof(struct fixedbond));

  for(int n = 0; n < sys->nfixed; ++n) {
    read_data(bfile, 1, INT, &(sys->fixed[n].i));
    read_data(bfile, 3, REAL, sys->fixed[n].rfixed);
    read_data(bfile, 1, REAL, &(sys->fixed[n].kbond));
    read_data(bfile, 1, REAL, &(sys->fixed[n].r0));

    if(sys->fixed[n].i < 0 || sys->fixed[n].i >= sys->np
       || sys->fixed[n].kbond < 0|| sys->fixed[n].r0 < 0) {
      fprintf(stderr, LOG_ERROR "Error reading fixed bond number %d.\n", n + 1);
      exit(-1);
    }
  }

  return;
}

typedef enum {harmonic = 0, FENE} bondtype;

void read_pairbonds(struct system * sys, FILE * bfile, bondtype btype)
{
  int nbonds; /* Number of bonds to read from file*/
  read_data(bfile, 1, INT, &nbonds);
  struct pairbond * bond;

  switch(btype) { /* Select bond type */
    case FENE:
      fprintf(stderr, LOG_INFO "Reading FENE bonds.\n");
      sys->nfene = nbonds;
      sys->fene = calloc(nbonds, sizeof(struct pairbond));
      bond = sys->fene;
    default:
      fprintf(stderr, LOG_INFO "Reading harmonic bonds.\n");
      sys->nbonds = nbonds;
      sys->bond = calloc(nbonds, sizeof(struct pairbond));
      bond = sys->bond;
  }

  for(int n = 0; n < nbonds; ++n) {
    read_data(bfile, 1, INT, &(bond[n].i));
    read_data(bfile, 1, INT, &(bond[n].j));
    read_data(bfile, 1, REAL, &(bond[n].kbond));
    read_data(bfile, 1, REAL, &(bond[n].r0));

    if(bond[n].i < 0 || bond[n].i >= sys->np
      || bond[n].j < 0 || bond[n].j >= sys->np
      ||bond[n].kbond < 0 || bond[n].r0 < 0) {
      fprintf(stderr, LOG_ERROR "Error reading pair bond number %d.\n", n + 1);
      exit(-1);
    }
  }

  return;
}

void read_angular(struct system * sys, FILE * bfile)
{
  fprintf(stderr, LOG_INFO "Reading angular bonds.\n");
  read_data(bfile, 1, INT, &(sys->nangular));

  sys->angbond = calloc(sys->nangular, sizeof(struct angular));

  for(int n = 0; n < sys->nangular; ++n) {
    read_data(bfile, 1, INT, &(sys->angbond[n].i));
    read_data(bfile, 1, INT, &(sys->angbond[n].j));
    read_data(bfile, 1, INT, &(sys->angbond[n].k));
    read_data(bfile, 1, REAL, &(sys->angbond[n].kbond));
    read_data(bfile, 1, REAL, &(sys->angbond[n].theta0));

    if(sys->angbond[n].i < 0 || sys->angbond[n].i >= sys->np
       || sys->angbond[n].j < 0 || sys->angbond[n].j >= sys->np
       || sys->angbond[n].k < 0 || sys->angbond[n].k >= sys->np
       || sys->angbond[n].kbond < 0 || sys->angbond[n].theta0 < 0) {
      fprintf(stderr, LOG_ERROR "Error reading angular spring number %d.\n",
              n + 1);
      exit(-1);
    }
  }

  return;
}

void read_torsional(struct system * sys, FILE * bfile)
{
  fprintf(stderr, LOG_INFO "Reading torsional bonds.\n");
  read_data(bfile, 1, INT, &(sys->ntorsional));

  sys->torbond = calloc(sys->ntorsional, sizeof(struct torsional));

  for(int n = 0; n < sys->ntorsional; ++n) {
    read_data(bfile, 1, INT, &(sys->torbond[n].i));
    read_data(bfile, 1, INT, &(sys->torbond[n].j));
    read_data(bfile, 1, INT, &(sys->torbond[n].k));
    read_data(bfile, 1, INT, &(sys->torbond[n].l));
    read_data(bfile, 1, REAL, &(sys->torbond[n].kbond));
    read_data(bfile, 1, REAL, &(sys->torbond[n].phi0));

    if(sys->torbond[n].i < 0 || sys->torbond[n].i >= sys->np
       || sys->torbond[n].j < 0 || sys->torbond[n].j >= sys->np
       || sys->torbond[n].k < 0 || sys->torbond[n].k >= sys->np
       || sys->torbond[n].l < 0 || sys->torbond[n].l >= sys->np
       || sys->torbond[n].kbond < 0 || sys->torbond[n].phi0 < 0) {
      fprintf(stderr, LOG_ERROR "Error reading torsional potential number"
              " %d.\n", n + 1);
      exit(-1);
    }
  }

  return;
}

/*** Potentials and forces ***/
/* Fixed bonds */
real energy_fixed_bonds(struct system * sys)
{
  int n;            /* Bond index */
  int i;            /* Particle index */
  int l;            /* Coordinate index */
  real E = 0;       /* Energy */
  real * rfixed;    /* Fixed point coordinates */
  real kbond;       /* Harmonic bond spring constant */
  real r0;          /* Harmonic bond equilibrium length */
  real r, r2;       /* Distance and distance squared */
  real rij[MAXDIM] = {0}; /* Displacement from fixed point to particle */

  for(n = 0; n < sys->nfixed; ++n) {
    /* Particle indices */
    i = sys->fixed[n].i;

    /* Bond parameters */
    kbond = sys->fixed[n].kbond;
    rfixed = sys->fixed[n].rfixed;
    r0 = sys->fixed[n].r0;

    /* Relative separation vector */
    for(l = 0; l < sys->dim; ++l)
      rij[l] = rfixed[l] - sys->q[sys->dim*i + l];

    /* Distance squared */
    r2 = 0; for(l = 0; l < sys->dim; ++l) r2 += rij[l]*rij[l];

    /* Distance */
    r = sqrt(r2);

    /* Energy */
    E += real_val(0.5)*kbond*(r - r0)*(r - r0);
  }

  return E;
}

void force_fixed_bonds(struct system * sys)
{
  int n;            /* Bond index */
  int i;            /* Particle index */
  int l;            /* Coordinate index */
  real r, r2;       /* Distance and distance squared */
  real * rfixed;    /* Fixed point coordinates */
  real invr;        /* Inverse of distance */
  real kbond;       /* Harmonic bond spring constant */
  real r0;          /* Harmonic bond equilibrium length */
  real rij[MAXDIM] = {0}; /* Displacement from fixed point to particle */

  for(n = 0; n < sys->nfixed; ++n) {
    /* Particle index */
    i = sys->fixed[n].i;

    /* Bond parameters */
    kbond = sys->fixed[n].kbond;
    r0 = sys->fixed[n].r0;
    rfixed = sys->fixed[n].rfixed;

    /* Relative separation vector */
    for(l = 0; l < sys->dim; ++l)
      rij[l] = rfixed[l] - sys->q[sys->dim*i + l];

    /* Distance squared */
    r2 = 0; for(l = 0; l < sys->dim; ++l) r2 += rij[l]*rij[l];

    /* Distance */
    r = sqrt(r2);

    /* Inverse distance */
    invr = (r == 0)?0:1/r;

    /* Forces */
    for(l = 0; l < sys->dim; ++l) {
      sys->F[sys->dim*i + l] += kbond*(1 - r0*invr)*rij[l];
    }
  }

  return;
}

/* Harmonic bonds */
real energy_harmonic_bonds(struct system * sys)
{
  int n;            /* Bond index */
  int i, j;         /* Particle indices */
  int l;            /* Coordinate index */
  real E = 0;       /* Energy */
  real kbond;       /* Harmonic bond spring constant */
  real r0;          /* Harmonic bond equilibrium length */
  real r, r2;       /* Distance and distance squared */
  real rij[MAXDIM] = {0}; /* Displacement from fixed point to particle */

  for(n = 0; n < sys->nbonds; ++n) {
    /* Particle indices */
    i = sys->bond[n].i;
    j = sys->bond[n].j;

    /* Bond parameters */
    kbond = sys->bond[n].kbond;
    r0 = sys->bond[n].r0;

    /* Relative separation vector */
    for(l = 0; l < sys->dim; ++l)
      rij[l] = sys->q[sys->dim*j + l] - sys->q[sys->dim*i + l];

    /* Distance squared */
    r2 = 0; for(l = 0; l < sys->dim; ++l) r2 += rij[l]*rij[l];

    /* Distance */
    r = sqrt(r2);

    /* Energy */
    E += real_val(0.5)*kbond*(r - r0)*(r - r0);
  }

  return E;
}

void force_harmonic_bonds(struct system * sys)
{
  int n;            /* Bond index */
  int i, j;         /* Particle indices */
  int l;            /* Coordinate index */
  real r, r2;       /* Distance and distance squared */
  real invr;        /* Inverse of distance */
  real kbond;       /* Harmonic bond spring constant */
  real r0;          /* Harmonic bond equilibrium length */
  real rij[MAXDIM] = {0}; /* Displacement from fixed point to particle */

  for(n = 0; n < sys->nbonds; ++n) {
    /* Particle indices */
    i = sys->bond[n].i;
    j = sys->bond[n].j;

    /* Bond parameters */
    kbond = sys->bond[n].kbond;
    r0 = sys->bond[n].r0;

    /* Relative separation vector */
    for(l = 0; l < sys->dim; ++l)
      rij[l] = sys->q[sys->dim*j + l] - sys->q[sys->dim*i + l];

    /* Distance squared */
    r2 = 0; for(l = 0; l < sys->dim; ++l) r2 += rij[l]*rij[l];

    /* Distance */
    r = sqrt(r2);

    /* Inverse distance */
    invr = (r == 0)?0:1/r;

    /* Forces */
    for(l = 0; l < sys->dim; ++l) {
      sys->F[sys->dim*i + l] += kbond*(1 - r0*invr)*rij[l];
      sys->F[sys->dim*j + l] -= kbond*(1 - r0*invr)*rij[l];
    }
  }

  return;
}

/* Angular potentials */
real energy_angular_bonds(struct system * sys)
{
  real E = 0;          /* Energy */
  int n;               /* Bond index */
  int i, j, k;         /* Particle indices */
  int l;               /* Coordinate index */
  real kbond;          /* Potential strength constant */
  real theta0;         /* Equilibrium angle */
  real r2, rr2;        /* Squared distances */
  real costheta;       /* Cosine of bending angle */
  real rij[MAXDIM] = {0};    /* Vector connecting particle i to j */
  real rjk[MAXDIM] = {0};    /* Vector connecting particle j to k */

  for(n = 0; n < sys->nangular; ++n) {
    /* Particle indices */
    i = sys->angbond[n].i;
    j = sys->angbond[n].j;
    k = sys->angbond[n].k;

    /* Bond parameters */
    kbond  = sys->angbond[n].kbond;
    theta0 = sys->angbond[n].theta0;

    /* Relative separation vectors */
    for(l = 0; l < sys->dim; ++l)
      rij[l] = sys->q[sys->dim*j + l] - sys->q[sys->dim*i + l];
    for(l = 0; l < sys->dim; ++l)
      rjk[l] = sys->q[sys->dim*k + l] - sys->q[sys->dim*j + l];

    /* Distances */
    r2 = 0; rr2 = 0;
    for(l = 0; l < sys->dim; ++l) {
      r2 += rij[l]*rij[l];
      rr2 += rjk[l]*rjk[l];
    }

    /* Interaction energy */
    if(r2 > 0 && rr2 > 0) {

      /* Cosine of angle */
      costheta = 0;
      for(l = 0; l < sys->dim; ++l) costheta += rij[l]*rjk[l];
      costheta /= sqrt(r2*rr2);

      if(costheta*costheta < 1) {
        real sinthetao2 = sin(real_val(0.5)*acos(costheta)); /* sin(theta/2) */
        real deltar
          = sinthetao2 - sin(real_val(0.5)*theta0);
        E += real_val(2.0)*kbond*deltar*deltar;
      }
    }
  }

  return E;
}

void force_angular_bonds(struct system * sys)
{
  int n;               /* Bond index */
  int i, j, k;         /* Particle indices */
  int l;               /* Coordinate index */
  real kbond;          /* Potential strength constant */
  real theta0;         /* Equilibrium angle */
  real r2, rr2;        /* Squared distances */
  real invr, invrr;    /* Inverse distances */
  real costheta;       /* Cosine of bending angle */
  real F;              /* Magnitude of force */
  real rij[MAXDIM] = {0};    /* Vector connecting particle i to j */
  real rjk[MAXDIM] = {0};    /* Vector connecting particle j to k */

  for(n = 0; n < sys->nangular; ++n) {
    /* Particle indices */
    i = sys->angbond[n].i;
    j = sys->angbond[n].j;
    k = sys->angbond[n].k;

    /* Bond parameters */
    kbond  = sys->angbond[n].kbond;
    theta0 = sys->angbond[n].theta0;

    /* Relative separation vectors */
    for(l = 0; l < sys->dim; ++l)
      rij[l] = sys->q[sys->dim*j + l] - sys->q[sys->dim*i + l];
    for(l = 0; l < sys->dim; ++l)
      rjk[l] = sys->q[sys->dim*k + l] - sys->q[sys->dim*j + l];

    /* Distances */
    r2 = 0; rr2 = 0;
    for(l = 0; l < sys->dim; ++l) {
      r2 += rij[l]*rij[l];
      rr2 += rjk[l]*rjk[l];
    }

    /* Forces */
    if(r2 > 0 && rr2 > 0) {

      /* Inverse distances */
      invr = 1/sqrt(r2);
      invrr = 1/sqrt(rr2);

      /* Cosine of angle */
      costheta = 0;
      for(l = 0; l < sys->dim; ++l) costheta += rij[l]*rjk[l];
      costheta *= invr*invrr;

      /* Magnitude of force */
      F = 0;
      if(costheta*costheta < 1) {
        real sinthetao2 = sin(real_val(0.5)*acos(costheta)); /* sin(theta/2) */
        F = kbond*(sinthetao2 - sin(real_val(0.5)*theta0))/sinthetao2;
      }

      /* Normalise separation vectors */
      for(l = 0; l < sys->dim; ++l) {
        rij[l] *= invr;
        rjk[l] *= invrr;
      }

      /* Force (components) */
      for(l = 0; l < sys->dim; l++) {
        real Fij = F*(rjk[l] - costheta*rij[l])*invr;
        real Fjk = F*(rij[l] - costheta*rjk[l])*invrr;
        sys->F[sys->dim*i + l] += Fij;
        sys->F[sys->dim*j + l] -= (Fij - Fjk);
        sys->F[sys->dim*k + l] -= Fjk;
      }
    }
  }

  return;
}

/* Torsional potentials */
real energy_torsional_bonds(struct system * sys)
{
  real E = 0;          /* Energy */
  int n;               /* Bond index */
  int i, j, k, l;      /* Particle indices */
  int m;               /* Coordinate index */
  real kbond;          /* Potential strength constant */
  real phi0;           /* Equilibrium angle */
  real n2, nn2;        /* Squared distances */
  real cosphi;         /* Cosine of bending angle */
  real delta;          /* Measurement of angle deformation */
  real rij[MAXDIM] = {0};    /* Vector connecting particle i to j */
  real rjk[MAXDIM] = {0};    /* Vector connecting particle j to k */
  real rkl[MAXDIM] = {0};    /* Vector connecting particle k to l */

  for(n = 0; n < sys->nangular; ++n) {
    /* Particle indices */
    i = sys->torbond[n].i;
    j = sys->torbond[n].j;
    k = sys->torbond[n].k;
    l = sys->torbond[n].l;

    /* Bond parameters */
    kbond  = sys->torbond[n].kbond;
    phi0   = sys->torbond[n].phi0;

    /* Relative separation vectors */
    for(m = 0; m < sys->dim; ++m)
      rij[m] = sys->q[sys->dim*j + m] - sys->q[sys->dim*i + m];
    for(m = 0; m < sys->dim; ++m)
      rjk[m] = sys->q[sys->dim*k + m] - sys->q[sys->dim*j + m];
    for(m = 0; m < sys->dim; ++m)
      rkl[m] = sys->q[sys->dim*l + m] - sys->q[sys->dim*k + m];

    /* Direction vectors */
    real nijk[3]; cross(rij, rjk, nijk);
    real njkl[3]; cross(rjk, rkl, njkl);

    /* Distances */
    n2 = 0; nn2 = 0;
    for(m = 0; m < sys->dim; ++m) {
      n2  += nijk[m]*nijk[m];
      nn2 += njkl[m]*njkl[m];
    }

    /* Interaction energy */
    if(n2 > 0 && nn2 > 0) {
      cosphi = 0;
      for(m = 0; m < 3; ++m) cosphi += nijk[m]*njkl[m];
      cosphi /= sqrt(n2*nn2);

      if(cosphi*cosphi < 1) {
        delta = sin(0.5*acos(cosphi)) - sin(0.5*phi0);
        E += real_val(2.0)*kbond*delta*delta;
      }
    }
  }
  return E;
}

void force_torsional_bonds(struct system * sys)
{
  int n;               /* Bond index */
  int i, j, k, l;      /* Particle indices */
  int m;               /* Coordinate index */
  real kbond;          /* Potential strength constant */
  real phi0;           /* Equilibrium angle */
  real n2, nn2;        /* Squared distances */
  real invn, invnn;    /* Inverse distances */
  real cosphi;         /* Cosine of bending angle */
  real sinphio2;       /* sin(phi/2) */
  real F;              /* Magnitude of force */
  real rij[MAXDIM] = {0};    /* Vector connecting particle i to j */
  real rjk[MAXDIM] = {0};    /* Vector connecting particle j to k */
  real rkl[MAXDIM] = {0};    /* Vector connecting particle k to l */


 for(n = 0; n < sys->nangular; ++n) {
    /* Particle indices */
    i = sys->torbond[n].i;
    j = sys->torbond[n].j;
    k = sys->torbond[n].k;
    l = sys->torbond[n].l;

    /* Bond parameters */
    kbond  = sys->torbond[n].kbond;
    phi0   = sys->torbond[n].phi0;

    /* Relative separation vectors */
    for(m = 0; m < sys->dim; ++m)
      rij[m] = sys->q[sys->dim*j + m] - sys->q[sys->dim*i + m];
    for(m = 0; m < sys->dim; ++m)
      rjk[m] = sys->q[sys->dim*k + m] - sys->q[sys->dim*j + m];
    for(m = 0; m < sys->dim; ++m)
      rkl[m] = sys->q[sys->dim*l + m] - sys->q[sys->dim*k + m];

    /* Direction vectors */
    real nijk[3]; cross(rij, rjk, nijk);
    real njkl[3]; cross(rjk, rkl, njkl);

    /* Distances */
    n2 = 0; nn2 = 0;
    for(m = 0; m < sys->dim; ++m) {
      n2  += nijk[m]*nijk[m];
      nn2 += njkl[m]*njkl[m];
    }

    /* Forces */
    if(n2 > 0 && nn2 > 0) {

      /* Inverse distances */
      invn = 1/sqrt(n2);
      invnn = 1/sqrt(nn2);

      /* Normalise direction vectors */
      for(m = 0; m < sys->dim; ++m) nijk[m] *= invn;
      for(m = 0; m < sys->dim; ++m) njkl[m] *= invnn;

      /* Force (modulus) */
      F = 0;
      cosphi = 0;
      for(m = 0; m < 3; ++m) cosphi += nijk[m]*njkl[m];
      if(cosphi*cosphi < 1) {
        sinphio2 = sin(0.5*acos(cosphi));
        F = kbond*(sinphio2 - sin(0.5*phi0))/sinphio2;
      }

      /* Force (components) */
      real v1[3], v2[3]; /* Temporary vectors */
      for(m = 0; m < 3; ++m) {
        v1[m] = (nijk[m] - cosphi*nijk[m])*invn;
        v2[m] = (njkl[m] - cosphi*njkl[m])*invnn;
      }

      /* Force terms */
      real fi[3], fj[3], fk[3], fl[3];
      cross(v1, rjk, fi);
      cross(v2, rkl, fj);
      cross(v1, rij, fk);
      cross(v2, rjk, fl);

      for(int m = 0; m < sys->dim; ++m) sys->F[sys->dim*i + m] += F*fi[m];
      for(int m = 0; m < sys->dim; ++m) sys->F[sys->dim*j + m] += F*fj[m];
      for(int m = 0; m < sys->dim; ++m) sys->F[sys->dim*k + m] += F*fk[m];
      for(int m = 0; m < sys->dim; ++m) sys->F[sys->dim*l + m] += F*fl[m];
    }
  }

  return;
}

# endif
