# ifndef _FILE_H_
# define _FILE_H_

/***** Libraries *****/
# include <stdio.h>
# include <stdlib.h>
# include <string.h>

/***** Definitions *****/
# ifndef __bool_type__
# define __bool_type__
typedef enum {false = 0, true} bool;
# endif

# ifndef __vartype__
# define __vartype__
typedef enum {STRING, INT, FLOAT, DOUBLE} vartype;
# endif

enum {optional = 0, required};

/* Terminal colours, style and special characters */
# define ULINE     "\xe2\x94\x81" /* Special characters */
# define UTEE      "\xe2\x94\xaf"
# define NORMAL    "\x1B[0m"      /* Text style */
# define BRIGHT    "\x1B[1m"
# define FAINT     "\x1B[2m"
# define UNDERLINE "\x1B[4m"
# define BLINK     "\x1B[5m"
# define WHITE     "\x1B[37m"     /* Foreground colours */
# define RED       "\x1B[31m"
# define YELLOW    "\x1B[33m"
# define GREEN     "\x1B[32m"
# define CYAN      "\x1B[36m"
# define BLUE      "\x1B[34m"
# define MAGENTA   "\x1B[35m"
# define BGWHITE   "\x1B[47m"     /* Background colours */
# define BGRED     "\x1B[41m"
# define BGYELLOW  "\x1B[43m"
# define BGGREEN   "\x1B[42m"
# define BGCYAN    "\x1B[46m"
# define BGBLUE    "\x1B[44m"
# define BGMAGENTA "\x1B[45m"

/* Log message colour codes */
# define LOG_EMPTY   "   "                "         "        "   "
# define LOG_INFO    "  [" BRIGHT CYAN    " Info    " NORMAL "]  "
# define LOG_COMMENT "  [" BRIGHT BLUE    " Comment " NORMAL "]  "
# define LOG_STATUS  "  [" BRIGHT MAGENTA " Status  " NORMAL "]  "
# define LOG_DONE    "  [" BRIGHT GREEN   " Done    " NORMAL "]  "
# define LOG_WARNING "  [" BRIGHT YELLOW  " Warning " NORMAL "]  "
# define LOG_ERROR   "  [" BRIGHT RED     " Error   " NORMAL "]  "
# define LOG_PANIC   "  [" BGRED  BRIGHT  " PANIC!  " NORMAL "]  "

/***** Functions *****/

/*** Open file and check for errors and close file ***/
FILE * openfile(char * filename, char * mode)
{
  FILE * file = fopen(filename, mode);

  if(file == NULL) {
    fprintf(stderr, LOG_ERROR "Unable to open file (%s).\n",
                    filename);
    exit(-1);
  }

  return file;
}

#define closefile(file) fclose(file)

/*** Read the next N data points from a file and store them in data ***/
void read_data(FILE * datafile, int N, vartype type, void * data)
{
  for(int i = 0; i < N; ++i) {
    int nread = 0;
    switch(type) {
      case INT:
        nread = fscanf(datafile, "%d", (int *) data); break;
      case STRING:
        nread = fscanf(datafile, "%s", (char *) data); break;
      case FLOAT:
        nread = fscanf(datafile, "%f", (float *) data); break;
      case DOUBLE:
        nread = fscanf(datafile, "%lf", (double *) data); break;
    }

    if(nread > 0) {
      switch(type) {
        case INT:
          data = (int *) data + 1; break;
        case STRING:
          data = (char *) data + 1; break;
        case FLOAT:
          data = (float *) data + 1; break;
        case DOUBLE:
          data = (double *) data + 1; break;
      }
    } else {
      fprintf(stderr, LOG_ERROR "Reading data from file.\n");
      exit(-1);
    }

    if(feof(datafile)) {
      fprintf(stderr, LOG_ERROR "End of file reached prematurely.\n");
      exit(-1);
    }
  }

  return;
}

/*** Write N data points from data to a file ***/
void write_data(FILE * datafile, int N, vartype type, void * data)
{
  for(int i = 0; i < N; ++i) {
    switch(type) {
      case INT:
        fprintf(datafile, "%d", *((int *) data));
        data = (int *) data + 1;
        break;
      case STRING:
        fprintf(datafile, "%c", *((char *) data));
        data = (char *) data + 1;
        break;
      case FLOAT:
        fprintf(datafile, "%f", *((float *) data));
        data = (float *) data + 1;
        break;
      case DOUBLE:
        fprintf(datafile, "%f", *((double *) data));
        data = (double *) data + 1;
        break;
    }
    if(i < N - 1) fprintf(datafile, " ");
  }
  return;
}

/*** Read parameter name value pairs from a text file ***/
/* Read the value of a simulation parameter from a parameter file */
int read_parameter(char * filename, char * parametername,
                   vartype type, void * variable, bool required)
{
  FILE * parameters = openfile(filename, "r");

  char option[1000];
  char name[100];

  while(true) {
    if(fgets(option, 1000, parameters)) {
      sscanf(option, "%s", name);
      int nread = 0;
      if(!strcmp(name, parametername)) {
        switch(type) {
          case INT:
            nread = sscanf(option, "%*s %d", (int *) variable); break;
          case STRING:
            nread = sscanf(option, "%*s %s", (char *) variable); break;
          case FLOAT:
            nread = sscanf(option, "%*s %f", (float *) variable); break;
          case DOUBLE:
            nread = sscanf(option, "%*s %lf", (double *) variable); break;
        }

        if(nread < 1) {
          fprintf(stderr, LOG_ERROR "Unable to read value for %s.\n", name);
          exit(-1);
        }

        break;
      }
    } else if(required) {
      fprintf(stderr, LOG_ERROR "Option " YELLOW "%s " NORMAL
                      "not found in parameter file.\n",
                      parametername);
      return -1;
    } else {
      return 0;
    }
  }

  closefile(parameters);
  return 0;
}

/* Read the n values in a parameter array from a parameter file */
int read_nparameter(char * filename, char * parametername, int n,
                    vartype type, void * variable, bool required)
{
  FILE * parameters = openfile(filename, "r");

  char option[1000];
  char name[100];

  while(true) {
    if(fgets(option, 1000, parameters)) {
      sscanf(option, "%s", name);
      int nargs = 0;
      if(!strcmp(name, parametername)) {
        char * token;
        token = strtok(option, " ");
        while(nargs < n) {
          int nread = 0;
          token = strtok(NULL, ",");
          if(token == NULL) break;
          switch(type) {
            case INT:
              nread = sscanf(token, "%d", (int *) variable); break;
            case STRING:
              nread = sscanf(token, "%s", (char *) variable); break;
            case FLOAT:
              nread = sscanf(token, "%f", (float *) variable); break;
            case DOUBLE:
              nread = sscanf(token, "%lf", (double *) variable); break;
          }

          if(nread > 0) {
            ++nargs;
            switch(type) {
              case INT:
                variable = (int *) variable + 1; break;
              case STRING:
                variable = (char *) variable + 1; break;
              case FLOAT:
                variable = (float *) variable + 1; break;
              case DOUBLE:
                variable = (double *) variable + 1; break;
            }
          }
        }

        if(nargs < n) {
          fprintf(stderr, LOG_ERROR
                          "Unable to read %d values for %s.\n", n, name);
          exit(-1);
        }

        break;
      }
    } else if(required) {
      fprintf(stderr, LOG_ERROR "Option " YELLOW "%s " NORMAL
                      "not found in parameter file.\n",
                      parametername);
      return -1;
    } else {
      return 0;
    }
  }

  closefile(parameters);
  return 0;
}

/* Set described parameter with read_parameter */
void set_parameter(char * filename, char * parametername, char * description,
                   vartype type, void * variable, bool required)
{
  bool read_error
    = read_parameter(filename, parametername, type, variable, required);

  if(read_error) {
    fprintf(stderr, LOG_ERROR "Missing parameter. " YELLOW "%s" NORMAL ": "
                    "%s.\n", parametername, description);
    exit(-1);
  }
  return;
}

/* Set described parameter array with read_nparameter */
void set_nparameter(char * filename, char * parametername, int n,
                    char * description, vartype type, void * variable,
                    bool required)
{
  bool read_error
    = read_nparameter(filename, parametername, n, type, variable, required);

  if(read_error) {
    fprintf(stderr, LOG_ERROR "Missing parameter. " YELLOW "%s" NORMAL ": "
                    "%s\n", parametername, description);
    exit(-1);
  }
  return;
}

/*** Binary input and output ***/
void read_raw(FILE * datafile, int N, vartype type, void * data)
{
  switch(type) {
    case STRING:
      fread(data, sizeof(char), N, datafile); break;
    case INT:
      fread(data, sizeof(int), N, datafile); break;
    case FLOAT:
      fread(data, sizeof(float), N, datafile); break;
    case DOUBLE:
      fread(data, sizeof(double), N, datafile); break;
  }

  return;
}

void write_raw(FILE * datafile, int N, vartype type, void * data)
{
  switch(type) {
    case STRING:
      fwrite(data, sizeof(char), N, datafile); break;
    case INT:
      fwrite(data, sizeof(int), N, datafile); break;
    case FLOAT:
      fwrite(data, sizeof(float), N, datafile); break;
    case DOUBLE:
      fwrite(data, sizeof(double), N, datafile); break;
  }

  return;
}

# endif
