/* neighbour_list.h: linked lists for neighbours in MD simulations */
# ifndef _NEIGHBOUR_LIST_H
# define _NEIGHBOUR_LIST_H

# include "pbc.h"

/* Set neighbour list box dimensions */
void neighbour_list_box_default(struct system * sys, struct parameters * params)
{
  for(int l = 0; l < sys->dim; ++l) params->nlist.L[l] = sys->L[l];
  return;
}

__attribute__((weak, alias("neighbour_list_box_default")))
  void neighbour_list_box();

/* Fold coordinates into cell box */
void fold_generic(real * r, struct system * sys, struct parameters * params)
{
  applyPBC(r, sys->L, sys->dim);
  applyPBC_rectangular(r, params->nlist.L, sys->dim);
  return;
}

void fold_rectangular(real * r, struct system * sys, struct parameters * params)
{
  applyPBC_rectangular(r, params->nlist.L, sys->dim);
  return;
}

/* Default choice for fold */
__attribute__((weak, alias("fold_rectangular"))) void fold();


/* Find the cell containing particle i */
int getcell(int i, struct system * sys, struct parameters * params)
{
  int cellcoord[sys->dim]; /* Cell coordinates (in cells) */
  int celln; /* Cell number */
  real r[sys->dim]; /* Position vector */

  /* Get position and apply periodic boundary conditions */
  for(int l = 0; l < sys->dim; ++l) r[l] = sys->q[sys->dim*i + l];
  fold(r, sys, params);

  /* Work out cell coordinates */
  for(int l = 0; l < sys->dim; ++l) {
    cellcoord[l]
      = (int) floor(0.5f*params->nlist.side[l] + r[l]/params->nlist.csize[l]);
    if(cellcoord[l] >= params->nlist.side[l]) cellcoord[l] = 0;
  }

  /* Get cell number */
  celln = 0;
  for(int l = 0; l < sys->dim; ++l)
    celln += params->nlist.shift[l]*cellcoord[l];

  return celln;
}

/* Add a particle i to cell celln */
void addparticletocell(int i, int celln, struct parameters * params)
{
  if(celln < 0 || celln >= params->nlist.n) {
    fprintf(stderr, "\n" LOG_ERROR "Neighbour list: "
                    "particle %d sent to nonexistent cell %d.\n", i, celln);
    exit(-1);
  }
  params->nlist.next[i] = params->nlist.first[celln];
  params->nlist.first[celln] = i;

  return;
}

/* Update the neighbour list */
void update_links(struct system * sys, struct parameters * params)
{
  /* All cells begin empty */
  for(int n = 0; n < params->nlist.n; ++n) params->nlist.first[n] = -1;

  /* Add each particle to the cell it belongs to */
  for(int i = 0; i < sys->np; ++i)
    addparticletocell(i, getcell(i, sys, params), params);

  return;
}

/* Set up the list of links between particles and the list of cells */
void setup_links(struct system * sys, struct parameters * params)
{
  fprintf(stderr, LOG_INFO "Setting up neighbour lists.\n");

  neighbour_list_box(sys, params);

  /* Number of cells */
  params->nlist.n = 1;
  for(int l = 0; l < sys->dim; ++l) {
    params->nlist.side[l] = (int) floor(params->nlist.L[l]/params->cellsize);
    params->nlist.n *= params->nlist.side[l];
    params->nlist.csize[l] = sys->L[l]/params->nlist.side[l];
  }

  /* Allocate memory */
  params->nlist.next = (int *) calloc(sys->np, sizeof(int));
  params->nlist.first = (int *) calloc(params->nlist.n, sizeof(int));
  params->nlist.prev = NULL;
  params->nlist.ccell = NULL;

  /* Cell shifts */
  params->nlist.shift[0] = 1;
  for(int n = 1; n < sys->dim; ++n)
    params->nlist.shift[n]
      = params->nlist.shift[n - 1]*params->nlist.side[n - 1];
}

/*** Specialised versions for Monte Carlo sampling ***/

/* Set up doubly linked list for MC simulation */
void setup_links_MC(struct system * sys, struct parameters * params)
{
  fprintf(stderr, LOG_INFO "Setting up neighbour lists.\n");

  neighbour_list_box(sys, params);

  /* Number of cells */
  params->nlist.n = 1;
  for(int l = 0; l < sys->dim; ++l) {
    params->nlist.side[l] = (int) floor(params->nlist.L[l]/params->cellsize);
    params->nlist.n *= params->nlist.side[l];
    params->nlist.csize[l] = sys->L[l]/params->nlist.side[l];
  }

  /* Allocate memory */
  params->nlist.next = (int *) calloc(sys->np, sizeof(int));
  params->nlist.prev = (int *) calloc(sys->np, sizeof(int));
  params->nlist.ccell = (int *) calloc(sys->np, sizeof(int));
  params->nlist.first = (int *) calloc(params->nlist.n, sizeof(int));

  /* Cell shifts */
  params->nlist.shift[0] = 1;
  for(int n = 1; n < sys->dim; ++n)
    params->nlist.shift[n]
      = params->nlist.shift[n - 1]*params->nlist.side[n - 1];
}

/* MC version to add particle to a given cell */
void addparticletocell_MC(int i, int celln, struct parameters * params)
{
  if(celln < 0 || celln >= params->nlist.n) {
    fprintf(stderr, "\n" LOG_ERROR "Neighbour list: "
                    "particle %d sent to nonexistent cell %d.\n", i, celln);
    exit(-1);
  }

  params->nlist.prev[i] = -1;
  if(params->nlist.first[celln] >= 0)
    params->nlist.prev[params->nlist.first[celln]] = i;
  params->nlist.next[i] = params->nlist.first[celln];
  params->nlist.first[celln] = i;
  params->nlist.ccell[i] = celln;

  return;
}

/* MC version to update linked lists */
void update_links_MC(struct system * sys, struct parameters * params)
{
  /* All cells begin empty */
  for(int n = 0; n < params->nlist.n; ++n) params->nlist.first[n] = -1;

  /* Add each particle to the cell it belongs to */
  for(int i = 0; i < sys->np; ++i)
    addparticletocell_MC(i, getcell(i, sys, params), params);

  return;
}

/* MC version to update links for single particle */
void update_links_single_MC(int i,
                            struct system * sys, struct parameters * params)
{
  /* Check if particle has changed cell */
  int ccell = getcell(i, sys, params);
  if(ccell == params->nlist.ccell[i]) return;

  /* Remove particle from previous cell */
  if(params->nlist.first[params->nlist.ccell[i]] == i)
    params->nlist.first[params->nlist.ccell[i]] = params->nlist.next[i];
  else
    params->nlist.next[params->nlist.prev[i]] = params->nlist.next[i];
  if(params->nlist.next[i] >= 0)
    params->nlist.prev[params->nlist.next[i]] = params->nlist.prev[i];

  /* Add particle to new cell list */
  addparticletocell_MC(i, ccell, params);

  return;
}


# endif
