      /* optohydrodynamics.c: A generic multi-purpose simple MD program */

# ifdef SINGLE_PRECISION
# ifndef __real_type__
# define __real_type__
typedef float real; /* Real means single precision */
# define READ_REAL_VAL "%f "
# define REAL FLOAT
# define real_val(x) x##f
# endif
# endif

/*** Libraries ***/
# include <stdio.h>
# include <stdlib.h>
# include <math.h>
# ifndef __real_type__
# define __real_type__
typedef double real; /* Real means double precision */
# define READ_REAL_VAL "%lf "
# define REAL DOUBLE
# define real_val(x) x
# endif
# include "file.h"
# include "pbc.h"
# include <complex.h>
# include "random.h"
# include "mathtools.h"

/*** Definitions ***/
# ifndef __bool_type__
# define __bool_type__
typedef enum {false = 0, true} bool;
# endif

# ifndef MAXDIM
# define MAXDIM 3
# endif

struct neighbour_list {
  int n;                 /* Number of cells */
  real L[MAXDIM];        /* Neighbour list box dimensions */
  int side[MAXDIM];      /* Side length (measured in cells) */
  real csize[MAXDIM];    /* Cell size */
  int * prev;            /* Previous linked particle */
  int * next;            /* Next linked particle */
  int * ccell;           /* Current cell number for each particle */
  int * first;           /* Indices of the first particles in each cell */
  int shift[MAXDIM];     /* Shifts connecting neighbour cells */
};

# ifndef MAXDIM
# define MAXDIM 3
# endif
struct fixedbond {
  int i;               /* Particle index */
  real rfixed[MAXDIM]; /* Point to which particle i remains attached */
  real kbond;          /* Bond strength */
  real r0;             /* Equilibrium distance */
};
struct pairbond {
  int i, j;   /* Particle indices */
  real kbond; /* Bond strength parameter */
  real r0;    /* Bond distance parameter */
};
struct constraint {
  int i, j;   /* Particle indices */
  real r0;    /* Fixed distance */
};
struct angular {
  int i, j, k; /* Particle indices */
  real kbond;  /* Potential strength */
  real theta0; /* Equilibrium angle */
};
struct torsional {
  int i, j, k, l; /* Particle indices */
  real kbond;     /* Potential strength */
  real phi0;      /* Equilibrium angle */
};

struct system {
  real L[3];  /* System lengths */

  /* Particles */
  int np;   /* Number of particles */
  int dim;  /* System dimensionality */
  int N;    /* Number of degrees of freedom */
  real * q; /* Coordinates */

  real * p; /* Momenta */
  real * F; /* Force */
  real * m; /* Mass */
  real * r; /* Radius */
  int * c;  /* Colour (or type) */

  /* Bonds */
  int nfixed; /* Number of fixed bonds */
  struct fixedbond * fixed; /* List of particle indices defining fixed bonds */
  int nbonds; /* Number of harmonic bonds */
  struct pairbond * bond; /* List of harmonic bonds */
  int nfene; /* Number of FENE bonds */
  struct pairbond * fene; /* List of FENE bonds */
  int nangular; /* Number of angular springs */
  struct angular * angbond; /* List of angular potentials */
  int ntorsional; /* Number of torsional springs */
  struct torsional * torbond; /* List of torsional potentials */

  real kT; /* Thermal energy */
  real gamma; /* Friction coefficient for a unit radius sphere */

  /* DLVO interaction */
  real Zeta;    /* Zeta potential */
  real kappa;   /* Screening constant */
  real elayer;  /* Ad hoc expanded exclusion layer */
  real ljrc;    /* Cut off radius for Lennard-Jones potential */
  real epsilon; /* Lennard-Jones energy parameter */
  real H_121; /* Hamaker constant */

  /* External force fields */
  real me;      /* Excess mass of particles*/
  real g;       /* Acceleration of gravity */
  real k;       /* Harmonic potential centred at the origin */
  real Kq;      /* Electrostatic attraction towards the origin due to lamp */
  real lampr0;  /* Lamp area radius */

  /* Optical field */
  char incident_field[256]; /* Incident field components file */
  int nactive;  /* Number of optically active particles */
  int nwaves;   /* Number of plane waves in the field decomposition */
  real F0;      /* Incident laser force intensity */
  real kwave;   /* Incident laser wave length */
  real focus_z; /* Height of the laser focus above the origin */
  bool icone;   /* Supress optical field outside of irradiation cone */

  /* Scattering algorithm parameters */
  int SCATTERING_MAXITER;    /* Maximum number of iterations */
  real SCATTERING_TOLERANCE; /* Error tolerance */

  complex * alpha0; /* Particle polarisabilities */
  complex * alpha;  /* Active particle polarisabilities */
};

# include "bonded_interactions.h"

struct parameters {
  real dt;           /* Integration time step size */
  real trelax;       /* Relaxation time (no output till t > trelax) */
  real tmax;         /* Simulation runs from t = 0 to t = tmax */
  long int nsteps;   /* Number of simulation steps */
  long int step;     /* Simulation step number */
  int sampling_step; /* Sample system every sampling_step steps */
  char initial_configuration[256]; /* Initial configuration file name */
  char output_configurations[256]; /* Sampled configurations file name */
  char final_state[256]; /* Final configuration */
  char measurements_file[256]; /* Macroscopic measurements */
  char bondfilename[256]; /* Bond interactions file name */
  real cellsize; /* Desired neighbour list cell size */
  struct neighbour_list nlist;
  real rc; /* Cut off radius for short-ranged interactions */
  real maxradius; /* Largest particle radius */
  real interaction_list_radius;
  int ninteractions;
  int * interaction_list;
};

# include "neighbour_list.h"

/*** Auxiliary functions ***/
/* Usage message */
void usage_message(int argc, char * argv[])
{
  if(argc < 2) {
    fprintf(stderr,
            RED "  º" BLUE "\xe2\xac\xa4 " GREEN " nanometre\n"
            YELLOW "  "
            "\xe2\x94\x8f" ULINE ULINE ULINE UTEE ULINE ULINE ULINE
            "\xe2\x94\xb3" ULINE ULINE ULINE UTEE
            "\n  \xe2\x94\x83 " NORMAL "nm   1" YELLOW "\xe2\x95\xb9\n"
            NORMAL "\n");

    fprintf(stderr, "  Usage: %s <parameter file>\n", argv[0]);
    exit(0);
  }
  return;
}

/* Log information header */
void output_log_header()
{
  fprintf(stderr, RED "  º" BLUE "\xe2\xac\xa4 " GREEN " nanometre\n"
                  YELLOW "  \xe2\x94\x8f");
  for(int i = 0; i < 4; ++i)
      fprintf(stderr, ULINE ULINE ULINE UTEE ULINE ULINE ULINE "\xe2\x94\xb3");
  fprintf(stderr, ULINE ULINE);
  for(int i = 0; i < 20; ++i)
    fprintf(stderr, " " ULINE);
  fprintf(stderr, "\n  \xe2\x94\x83 " NORMAL "nm   ");
  for(int i = 1; i < 5; ++i)
      fprintf(stderr, "%d" YELLOW "\xe2\x95\xb9      " NORMAL, i);
  fprintf(stderr, "\n\n");
  return;
}

/* Log information end */
void output_log_end()
{
  fprintf(stderr, "\n");
  fprintf(stderr, "  " YELLOW);
  for(int i = 0; i < 38; ++i) fprintf(stderr, ULINE " ");
  fprintf(stderr, NORMAL "\n");
  return;
}

/* Allocate memory */
void interaction_list_allocate_memory(struct system * sys,
                                      struct parameters * params)
{
    if(sys->np <= 0) {
        fprintf(stderr, LOG_ERROR "No particles in the system (np = %d).",
                        sys->np);
        exit(-1);
    }

    float minradius = sys->r[0];
    params->maxradius = 0;

    for(int i = 0; i < sys->np; ++i) {
      if(sys->r[i] < minradius) minradius = sys->r[i];
      if(sys->r[i] > params->maxradius) params->maxradius = sys->r[i];
      if(sys->r[i] == 0) {
        fprintf(stderr, LOG_ERROR "Radius value not allowed r[%d] = %f",
                        i, sys->r[i]);
        exit(-1);
      }
    }

    params->interaction_list_radius = 2*params->rc*params->maxradius;
    int maxneighbours = pow(params->interaction_list_radius, 3)
                          /pow(minradius, 3);
    params->interaction_list = calloc(2*sys->np*maxneighbours, sizeof(int));

}

void memory_allocation(struct system * sys, struct parameters * params)
{
  fprintf(stderr, LOG_INFO "Allocating memory (%d degrees of freedom).\n",
                  sys->N);

  sys->q = (real *) calloc(sys->N,  sizeof(real));
  sys->p = (real *) calloc(sys->N,  sizeof(real));
  sys->F = (real *) calloc(sys->N,  sizeof(real));
  sys->m = (real *) calloc(sys->np, sizeof(real));
  sys->r = (real *) calloc(sys->np, sizeof(real));
  sys->c = (int *)  calloc(sys->np, sizeof(int));
  sys->alpha0 = calloc(sys->N, sizeof(complex));

  return;
}

/* Free memory */
void free_memory(struct system * sys, struct parameters * params)
{
  free(sys->q);
  free(sys->p);
  free(sys->F);
  free(sys->m);
  free(sys->r);
  free(sys->c);
  free(params->interaction_list);
  free(params->nlist.next);
  free(params->nlist.prev);
  free(params->nlist.first);
  free(params->nlist.ccell);
  free(sys->alpha);
  free(sys->alpha0);

  return;
}

/* Load state from file */
void load_system(struct system * sys, char * filename)
{
  FILE * statefile = openfile(filename, "r");
  fprintf(stderr, LOG_INFO "Reading initial conditions file (%s).\n", filename);
  read_data(statefile, 1, INT, &(sys->np));
  real alpha_re = 0, alpha_im = 0;
  sys->N = sys->np*sys->dim;
  for(int i = 0; i < sys->np; ++i) {
    read_data(statefile, sys->dim, REAL, &(sys->q[sys->dim*i]));
    read_data(statefile, 1, REAL, &(sys->r[i]));
    read_data(statefile, 1, REAL, &alpha_re);
    read_data(statefile, 1, REAL, &alpha_im);
    sys->alpha0[i] = alpha_re + I*alpha_im;
  }
  closefile(statefile);
}

/* Save state to file */
void save_system(struct system * sys, char * filename)
{
  FILE * statefile = openfile(filename, "w");
  fprintf(stderr, LOG_INFO "Writing state to file (%s).\n", filename);
  write_data(statefile, 1, INT, &(sys->np));
  fprintf(statefile, "\n");
  for(int i = 0; i < sys->np; ++i) {
    write_data(statefile, sys->dim, REAL, &(sys->q[sys->dim*i]));
    fprintf(statefile, " ");
    write_data(statefile, 1, REAL, &(sys->r[i]));
    fprintf(statefile, " %f %f\n", creal(sys->alpha0[i]),
                                   cimag(sys->alpha0[i]));
  }
  closefile(statefile);
}

/* Read simulation parameters from file */
void read_parameter_file(char * filename, struct system * sys,
                                          struct parameters * params) {
  /* Default simulation parameter values (parameter definitions below) */
  sys->dim = 3;
  sys->np  = 0;

  /* Set values from parameter file */
  set_parameter(filename, "dim", "system dimensionality (1, 2 or 3)",
                INT, &(sys->dim), optional);

  real L[sys->dim];
  set_nparameter(filename, "L", sys->dim, "Simulation box lengths, "
                 "separated by commas (for example: L 10.5, 15.0, 12.0)",
                 REAL, L, required);
  for(int k = 0; k < sys->dim; ++k) sys->L[k] = L[k];
  /* Defaults for algorithm */
  params->rc = real_val(1.12246204830937); /* Cut off radius */
  params->trelax = 0.0;
  params->tmax   = 0.0;
  params->final_state[0] = '\0';
  params->measurements_file[0] = '\0';
  params->bondfilename[0] = '\0';

  sys->nfixed = sys->nbonds = sys->nangular = 0;
  sys->fixed = NULL; sys->bond = NULL; sys->angbond = NULL;

  set_parameter(filename, "dt", "Numerical integration time step size",
                REAL, &(params->dt), required);
  set_parameter(filename, "tmax", "Total simulation time",
                REAL, &(params->tmax), optional);
  params->nsteps = (int) round(params->tmax / params->dt);
  set_parameter(filename, "rcut", "Lennard-Jones cut off distance "
                          "(default: WCA)", REAL, &(params->rc), optional);
  set_parameter(filename, "sampling_step", "sampling rate for data output",
                INT, &(params->sampling_step), required);
  set_parameter(filename, "output_configurations",
                         "output sampled configurations to this file "
                         "(use /dev/null to discard)",
                STRING, &(params->output_configurations), required);
  set_parameter(filename, "initial_configuration",
                         "list of initial coordinates, "
                         "momenta, radius and mass",
                STRING, &(params->initial_configuration), required);
  set_parameter(filename, "bonds",
                         "bond interactions file",
                STRING, &(params->bondfilename), optional);
  set_parameter(filename, "final_state",
                         "output final configuration to this file",
                STRING, &(params->final_state), optional);
  set_parameter(filename, "macro_file",
                         "output macro measurements to this file",
                STRING, &(params->measurements_file), optional);

  /* Read number of particles from initial conditions file */
  FILE * statefile = openfile(params->initial_configuration, "r");
  read_data(statefile, 1, INT, &(sys->np));
  fprintf(stderr, LOG_INFO "%d particles in the system.\n", sys->np);
  sys->N = sys->np*sys->dim;
  closefile(statefile);

  /* Allocate memory for system */
  memory_allocation(sys, params);
  set_parameter(filename, "cellsize", "desired neighbour list cell size",
                REAL, &(params->cellsize), required);
  set_parameter(filename, "rc", "Cut off radius",
                REAL, &(params->rc), optional);
  set_parameter(filename, "kT", "Thermal energy",
                REAL, &(sys->kT), required);
  set_parameter(filename, "H_121",
                "Hamaker constant",
                REAL, &(sys->H_121), required);
  set_parameter(filename, "Zeta",
                "Double-layer electrostatic potential Zeta parameter",
                REAL, &(sys->Zeta), required);
  set_parameter(filename, "kappa", "Double-layer electrostatic potential kappa "
                                   "parameter (inverse of Debye length)",
                REAL, &(sys->kappa), required);
  sys->elayer = 0;
  set_parameter(filename, "elayer", "Ad hoc expanded exclusion layer",
                REAL, &(sys->elayer), optional);
  sys->ljrc = 1.1224620;
  set_parameter(filename, "ljrc", "Lennard-Jones potential cut off radius",
                REAL, &(sys->ljrc), optional);
  sys->epsilon = 1.0;
  set_parameter(filename, "epsilon", "Lennard-Jones energy parameter",
                REAL, &(sys->epsilon), optional);
  set_parameter(filename, "gamma", "Friction for a unit-radius sphere\n"
                         LOG_EMPTY "(typically 6 pi times the shear viscosity)",
                REAL, &(sys->gamma), required);
  set_parameter(filename, "F0", "Incident laser intensity",
                REAL, &(sys->F0), required);
  set_parameter(filename, "kwave", "Incident laser wave length",
                REAL, &(sys->kwave), required);
  set_parameter(filename, "incident_field", "incident laser data file",
                STRING, sys->incident_field, required);
  set_parameter(filename, "focus_z",
                "height of the laser focus above the origin",
                REAL, &(sys->focus_z), required);
  sys->g = 0;
  set_parameter(filename, "g",
                "acceleration of gravity",
                REAL, &(sys->g), optional);
  sys->me = 0;
  set_parameter(filename, "me",
                "particle excess mass",
                REAL, &(sys->me), optional);
  sys->k = 0;
  set_parameter(filename, "k",
                "harmonic potential strength",
                REAL, &(sys->k), optional);
  sys->Kq = 0;
  set_parameter(filename, "Kq",
                "electric attraction strength (due to lamp)",
                REAL, &(sys->Kq), optional);
  sys->lampr0 = 1.0;
  set_parameter(filename, "lampr0",
                "lamp area radius",
                REAL, &(sys->lampr0), optional);
  sys->icone = 1;
  set_parameter(filename, "icone",
                "supress optical field outside irradiation cone (boolean)",
                INT, &(sys->icone), optional);
  if(sys->icone)
    fprintf(stderr, LOG_INFO "Optical field limited to irradiation cone\n");
  sys->SCATTERING_MAXITER = 100;
  set_parameter(filename, "SCATTERING_MAXITER",
                "Maximum number of iterations in scattering calculation",
                INT, &(sys->SCATTERING_MAXITER), optional);
  sys->SCATTERING_TOLERANCE = 1e-8;
  set_parameter(filename, "SCATTERING_TOLERANCE",
                "Error tolerance in scattering calculation",
                REAL, &(sys->SCATTERING_TOLERANCE), optional);
}

/* Interaction list */
void update_interaction_list(struct system * sys, struct parameters * params)
{

  static bool init = true; /* Initialise interaction list? */

  /* Memory allocation */
  static real max_displacement; /* Max displacement since last list rewrite */
  if(init) {
    interaction_list_allocate_memory(sys, params);
    max_displacement = 0;
    init = false;
  }

  /* Determine whether we need to rewrite the list */
  static bool list_built = false; /* List already built from scratch? */
  real v[sys->dim], speed2; /* Velocity vector and speed squared */
  real max_speed2 = 0; /* Maximum squared speed */

  /* Find the largest velocity */
  for(int i = 0; i < sys->np; ++i) {
    speed2 = 0;
    for(int l = 0; l < sys->dim; ++l) {
      v[l] = sys->p[sys->dim*i + l]/sys->m[i]; /* Velocity vector */
      speed2 += v[l]*v[l]; /* Squared speed */
    }
    if(speed2 > max_speed2) max_speed2 = speed2;
  }

  /* Maximum possible displacement */
  max_displacement += sqrt(max_speed2)*params->dt;

  /* Check max_displacement to see if the list needs to be rewritten */
  if((max_displacement < 0.4*(params->interaction_list_radius
                               - params->maxradius*params->rc))
     && list_built)
    return;

  /* Rewrite list */
  update_links(sys, params); /* Update neighbour list */
  params->ninteractions = 0; /* Start interactions list */
  int i, j; /* Particle indices */

  /* Number of neighbour cells of a cell */
  int cellcoord[sys->dim];
  int nneighcells = 1;
  for(int l = 0; l < sys->dim; ++l)
    nneighcells *= (params->nlist.side[l] < 3)? 1 : 3;

  /* Folded coordinates */
  real foldedq[sys->dim];

  /* For every cell */
  for(int n = 0; n < params->nlist.n; ++n) {
    i = params->nlist.first[n]; /* Start with particle i */
    if(i < 0) continue;

    /* For every particle in the cell */
    while(i >= 0) {
      /* Fold coordinates into cell box */
      for(int l = 0; l < sys->dim; ++l) foldedq[l] = sys->q[sys->dim*i + l];
      fold(foldedq, sys, params);

      /* Calculate the current cell coordinates */
      for(int l = 0; l < sys->dim; ++l)
        cellcoord[l]
          = (int) floor((foldedq[l] + .5*params->nlist.L[l])
                        /params->nlist.csize[l]);

      /* Loop over neighbour cells */
      int ncellcoord[sys->dim];
      int nm[sys->dim];
      for(int l = 0; l < sys->dim; ++l)
        nm[l] = (params->nlist.side[l] < 3)? 0 : -1;
      for(int m = 0; m < nneighcells; ++m) {
        /* Neighbour cell coordinates */
        for(int l = 0; l < sys->dim; ++l) ncellcoord[l] = cellcoord[l] + nm[l];

        /* Periodic boundary conditions */
        for(int l = 0; l < sys->dim; ++l) {
          if(params->nlist.side[l] > 2) {
            if(ncellcoord[l] < 0)
              ncellcoord[l] += params->nlist.side[l];
            else if(ncellcoord[l] >= params->nlist.side[l])
              ncellcoord[l] -= params->nlist.side[l];
          }
        }

        /* Determine neighbour cell index */
        int cellidx = 0; /* Clear cell index */
        for(int l = sys->dim; l > 0; --l) {
          int tmpidx = ncellcoord[l - 1];
          for(int k = l - 1; k > 0; --k)
            tmpidx *= params->nlist.side[k - 1];
          cellidx += tmpidx;
        }

        /* Carry out an operation for every neighbour */
        j = params->nlist.first[cellidx];
        while(j >= 0) {
          if(j <= i) { /* Count each pair only once */
            j = params->nlist.next[j];
            continue;
          }


          /* Displacement vector */
          real rij[sys->dim]; /* Displacement vector */
          real r2 = 0; /* Distance squared */

          for(int l = 0; l < sys->dim; ++l)
            rij[l] = sys->q[sys->dim*j + l] - sys->q[sys->dim*i + l];

          applyPBC(rij, sys->L, sys->dim);

          for(int l = 0; l < sys->dim; ++l) r2 += rij[l]*rij[l];

          /* Add interactions within radius */
          if(r2 <= params->interaction_list_radius
                   *params->interaction_list_radius) {
            params->interaction_list[2*params->ninteractions]     = i;
            params->interaction_list[2*params->ninteractions + 1] = j;
            ++params->ninteractions;
          }

          /* Next neighbour */
          j = params->nlist.next[j];
        }

        /* Next neighbour cell */
        nm[0]++;
        for(int l = 0; l < sys->dim - 1; ++l) {
          if(nm[l] > ((params->nlist.side[l] < 3)? 0 : 1)) {
            nm[l] = (params->nlist.side[l] < 3)? 0 : -1;
            nm[l + 1]++;
          }
        }
      }

      /* Next particle */
      i = params->nlist.next[i];
    }
  }

  /* Reset max_displacement variable */
  max_displacement = 0;

  list_built = true;

  return;
}

/* Short range potential and force */
real LennardJones(real r, real epsilon, real sigma, real rc)
{
  static real shift = -1;
  if(shift < 0)
    shift = 4*epsilon*pow(sigma/rc, 6)*(pow(sigma/rc, 6) - real_val(1.0));
  if(r >= sigma*rc || r == 0) return real_val(0.0);
  real sigmaor = (sigma/r);
  real invr2 = sigmaor*sigmaor;
  real invr6 = invr2*invr2*invr2;
  return 4*epsilon*invr6*(invr6 - real_val(1.0)) + shift;
}

real LennardJones_force(real r, real epsilon, real sigma, real rc)
{
  if(r >= sigma*rc || r == 0) return real_val(0.0);
  real sigmaor = (sigma/r);
  real invr2 = sigmaor*sigmaor;
  real invr6 = invr2*invr2*invr2;
  return 24*epsilon*invr6*(real_val(1.0) - 2*invr6)/r;
}

/* Force calculation */

/* Single reflection scattering of field on particles at positions qactive */
double scatter(double * qactive, double complex * E0, double complex * E,
               struct system * sys)
{
  double rji[3], r, r2; /* Displacement vector, distance and distance squared */
  double kr; /* Wave number times distance */
  /* Total field from the previous reflection */
  double complex Eold[sys->dim*sys->nactive];
  double complex G[sys->dim*sys->dim]; /* Green's tensor */
  double complex f, g; /* Components of the green tensor in the direction of the
                          incident field (f) and separation (g) */
  double DeltaE2 = 0;  /* Difference between E and E0 squared */

  /* Copy electric field to incident field vector */
  for(int i = 0; i < 3*sys->nactive; ++i) {
    Eold[i] = E[i]; /* Store previous value of the total field */
    E[i] = E0[i]; /* Set E field equal to incident field */
  }

  /* For every pair of particles affected by the field */
  for(int i = 0; i < sys->nactive - 1; ++i) {
    for(int j = i + 1; j < sys->nactive; ++j) {

      /* Separation between particles and distance */
      for(int k = 0; k < 3; ++k)
        rji[k] = qactive[3*i + k] - qactive[3*j + k];
      r2 = 0; for(int k = 0; k < 3; k++) r2 += rji[k]*rji[k]; r = sqrt(r2);

      /* Calculate Green's tensor */
      kr = sys->kwave*r;
      f = 1 + I/kr - 1/(kr*kr);
      g = 1 + 3*I/kr - 3/(kr*kr);

      for(int k = 0; k < 3; ++k)
        for(int l = 0; l < 3; ++l)
          G[3*k + l]
            = cexp(I*kr)*(f*delta(k, l) - g*rji[k]*rji[l]/r2)/(4*M_PI*r);

      /* Add field due to the scattering of particle j at the position of
         particle i and viceversa */
      for(int k = 0; k < 3; ++k) {
        for(int l = 0; l < 3; ++l) {
          E[3*i + k]
            += sys->kwave*sys->kwave*G[3*k + l]*sys->alpha[j]*Eold[3*j + l];
          E[3*j + k]
            += sys->kwave*sys->kwave*G[3*k + l]*sys->alpha[i]*Eold[3*i + l];
        }
      }
    }
  }

  /* Return the magnitude of the difference between E and Eold */
  for(int l = 0; l < 3*sys->nactive; ++l)
    DeltaE2 += creal(E[l] - Eold[l])*creal(E[l] - Eold[l])
               + cimag(E[l] - Eold[l])*cimag(E[l] - Eold[l]);

  return DeltaE2;
}

void optical_forces(real * qactive, real * Factive,
                    struct system * sys, struct parameters * params)
{
  /* Incident field data */
  static real * k = NULL;                /* Wave vectors */
  static double complex * E0 = NULL;     /* Incident field components */
  static double complex * E = NULL;      /* Field at the coordinate positions */
  static double complex * Ei = NULL;     /* Incident field at these points */
  static double complex * gradE = NULL;  /* Gradient of field at these points */

  /* Initialise field */
  static bool init = true;
  if(init) {
    FILE * incident_field_file = openfile(sys->incident_field, "r");
    read_data(incident_field_file, 1, INT, &(sys->nwaves));
    k     = calloc(sys->dim*sys->nwaves,  sizeof(real));
    E0    = calloc(sys->dim*sys->nwaves,  sizeof(double complex));
    Ei    = calloc(sys->dim*sys->nactive, sizeof(double complex));
    E     = calloc(sys->dim*sys->nactive, sizeof(double complex));
    gradE = calloc(sys->dim*sys->dim*sys->nactive, sizeof(double complex));

    for(int i = 0; i < sys->nwaves; ++i) {
      read_data(incident_field_file, sys->dim, REAL, &k[sys->dim*i]);
      for(int l = 0; l < sys->dim; ++l) {
        real E_comp[2]; /* Real and imaginary components of field value */
        read_data(incident_field_file, 2, REAL, &E_comp[0]);
        E0[sys->dim*i + l] = E_comp[0] + I*E_comp[1];
      }
    }

    closefile(incident_field_file);
    init = false;
  }

  /* Calculate incident field and gradient */
  for(int l = 0; l < sys->dim*sys->nactive; ++l) Ei[l] = 0;
  for(int l = 0; l < sys->dim*sys->dim*sys->nactive; ++l) gradE[l] = 0;
  for(int i = 0; i < sys->nactive; ++i) {
    for(int j = 0; j < sys->nwaves; ++j) {
      if(sys->icone &&
         (qactive[sys->dim*i]*qactive[sys->dim*i]
           + qactive[sys->dim*i + 1]*qactive[sys->dim*i + 1]
           >= (qactive[sys->dim*i + 2] + 25)*(qactive[sys->dim*i + 2] + 25)))
         continue;
      double complex expikr
        = cexp(I*sys->kwave
                *(  k[sys->dim*j]     * qactive[sys->dim*i]
                  + k[sys->dim*j + 1] * qactive[sys->dim*i + 1]
                  + k[sys->dim*j + 2] * qactive[sys->dim*i + 2]));
      for(int l = 0; l < sys->dim; ++l)
        Ei[sys->dim*i + l] += E0[sys->dim*j + l]*expikr;
      for(int l = 0; l < sys->dim; ++l)
        for(int m = 0; m < sys->dim; ++m)
          gradE[sys->dim*sys->dim*i + sys->dim*l + m]
            += I*sys->kwave*k[sys->dim*j + l]*E0[sys->dim*j + m]*expikr;
    }
  }

  /* Total field (incident + scattered) */
  if(sys->nactive > 1) {
    real DeltaE2 = 2*sys->SCATTERING_TOLERANCE;
    for(int n = 0; DeltaE2 > sys->SCATTERING_TOLERANCE
                   && n < sys->SCATTERING_MAXITER; ++n)
      DeltaE2 = scatter(qactive, Ei, E, sys);
  } else
    for(int l = 0; l < sys->dim; ++l) E[l] = Ei[l];

  /* Force due to incident field */
  for(int i = 0; i < sys->nactive; ++i) {
    for(int l = 0; l < sys->dim; ++l) {
      Factive[sys->dim*i + l] = 0;
      for(int m = 0; m < sys->dim; ++m)
        Factive[sys->dim*i + l]
          += sys->F0*creal(conj(sys->alpha[i]*E[sys->dim*i + m])
                                             *gradE[sys->dim*sys->dim*i
                                                    + sys->dim*l + m]);
    }
  }

/* Scattering */
  double rji[3], r, r2; /* Displacement vector, distance and distance squared */
  double kr; /* Wave number times distance */
  /* Temporary values involving the gradient of the Green tensor */
  double complex gradG[sys->dim*sys->dim*sys->dim], g, dG1dr, dG2dr,
                 gradGdotalphaE[sys->dim*sys->dim];

  /* Add force due to light scattering */
  if(sys->nactive > 1) {
    for(int i = 0; i < sys->nactive; ++i) {
      for(int k = 0; k < sys->dim*sys->dim; ++k)
        gradGdotalphaE[k] = 0; /* Clear temporary vector */

      /* Sum of the products of the gradient of the Green tensor times the
         polarization of particle j */
      for(int j = 0; j < sys->nactive; ++j) {
        if(j != i) {
          /* Separation between particles */
          for(int k = 0; k < sys->dim; ++k)
            rji[k] = qactive[sys->dim*i + k] - qactive[sys->dim*j + k];
          r2 = 0;
          for(int k = 0; k < sys->dim; ++k) r2 += rji[k]*rji[k];
          r = sqrt(r2);

          /* Analytical gradient of the Green tensor */
          kr = sys->kwave*r;
          dG1dr = sys->kwave
                  *cexp(I*kr)*(I - 2/kr - 3*I/(kr*kr) + 3/(kr*kr*kr))
                  /(4*M_PI*r);
          dG2dr = sys->kwave
                  *cexp(I*kr)*(I - 6/kr - 15*I/(kr*kr) + 15/(kr*kr*kr))
                  /(4*M_PI*r2);
          g = cexp(I*kr)*(1 + 3*I/kr - 3/(kr*kr))/(4*M_PI*r);

          for(int m = 0; m < sys->dim; ++m)
            for(int k = 0; k < sys->dim; ++k)
              for(int l = 0; l < sys->dim; ++l)
                gradG[sys->dim*sys->dim*m + sys->dim*k + l]
                  = dG1dr*(qactive[sys->dim*i + m]
                           - qactive[sys->dim*j + m])*delta(k, l)/r
                    - dG2dr*(qactive[sys->dim*i + m] - qactive[sys->dim*j + m])
                           *rji[k]*rji[l]/r2
                    - g*((m == l && m == k)?2:((m != l && m != k)?0:1))
                      *((m == k)?
                          qactive[sys->dim*i + l] - qactive[sys->dim*j + l]
                          : qactive[sys->dim*i + k]
                            - qactive[sys->dim*j + k])/r2;

          /* Contract the gradient of the Green tensor with alpha E */
          for(int m = 0; m < sys->dim; ++m)
            for(int k = 0; k < sys->dim; ++k)
              for(int l = 0; l < sys->dim; ++l)
                gradGdotalphaE[sys->dim*m + k]
                  += gradG[sys->dim*sys->dim*m + sys->dim*k + l]*sys->alpha[j]
                     *E[sys->dim*j + l];
        }
      }

      /* Force on particle i due to the scattering */
      for(int m = 0; m < sys->dim; ++m)
        for(int k = 0; k < sys->dim; ++k)
          Factive[sys->dim*i + m]
            += sys->F0*creal(conj(sys->alpha[i]*E[sys->dim*i + k])
                                  *sys->kwave*sys->kwave
                                  *gradGdotalphaE[sys->dim*m + k]);
    }
  }

  /* Free memory at the end of the simulation */
  if(params->step + 1 >= params->nsteps) {
    free(k);
    free(E0);
    free(Ei);
    free(E);
    free(gradE);
    init = true;
  }

  return;
}

void force_external(struct system * sys, struct parameters * params)
{
  /* Wall repulsion and gravity */
  for(int i = 0; i < sys->np; ++i) {
    sys->F[3*i + 2]
      += ((sys->q[3*i + 2] <= 0)? -1 : 1)
         *LennardJones(-sys->q[3*i + 2], real_val(1.0),
                       sys->r[i], real_val(1.12246204830937))
         - sys->me*sys->g;
  }

  /* Harmonic potential */
  for(int i = 0; i < sys->np; ++i) {
    double r = sqrt(sys->q[3*i]*sys->q[3*i]
                    + sys->q[3*i + 1]*sys->q[3*i + 1]);
    sys->F[3*i]     -= sys->k*r*sys->q[3*i];
    sys->F[3*i + 1] -= sys->k*r*sys->q[3*i + 1];
  }

  /* Electric potential due to lamp */
  for(int i = 0; i < sys->np; ++i) {
    double r = sqrt(sys->q[3*i]*sys->q[3*i]
                    + sys->q[3*i + 1]*sys->q[3*i + 1]
                    + sys->q[3*i + 2]*sys->q[3*i + 2]);
    double Fmod = (r>sys->lampr0)?
                    sys->Kq*(sys->lampr0*sys->lampr0)/(r*r*r)
                    :sys->Kq/sys->lampr0;
    sys->F[3*i]     -= Fmod*sys->q[3*i];
    sys->F[3*i + 1] -= Fmod*sys->q[3*i + 1];
    sys->F[3*i + 2] -= Fmod*sys->q[3*i + 2];
  }

  static bool init = true; /* Initialise optical forces? */
  static real * qactive, * Factive; /* Positions and forces */
  if(init) {
    /* Count active particles */
    sys->nactive = 0;
    for(int i = 0; i < sys->np; ++i)
      if(cabs(sys->alpha0[i]) > real_val(0.0)) ++(sys->nactive);

    /* Allocate memory */
    qactive = malloc(sys->dim*sys->nactive*sizeof(real));
    Factive = malloc(sys->dim*sys->nactive*sizeof(real));
    sys->alpha = malloc(sys->dim*sys->nactive*sizeof(complex));

    /* Copy active particles polarisabilities */
    int j = 0;
    for(int i = 0; i < sys->np; ++i) {
      if(cabs(sys->alpha0[i]) > real_val(0.0)) {
        sys->alpha[j] = sys->alpha0[i];
        ++j;
      }
    }

    init = false;
  }

  if(sys->nactive <= 0) return;

  /* Get active particle positions */
  int j = 0;
  for(int i = 0; i < sys->np; ++i) {
    if(cabs(sys->alpha0[i]) > real_val(0.0)) {
      for(int l = 0; l < sys->dim; ++l)
        qactive[sys->dim*j + l] = sys->q[sys->dim*i + l];
      qactive[sys->dim*j + 2] -= sys->focus_z;
      ++j;
    }
  }

  optical_forces(qactive, Factive, sys, params);

  /* Add optical forces to active particles */
  j = 0;
  for(int i = 0; i < sys->np; ++i) {
    if(cabs(sys->alpha0[i]) > real_val(0.0)) {
      for(int l = 0; l < sys->dim; ++l)
        sys->F[sys->dim*i + l] += Factive[sys->dim*j + l];
      ++j;
    }
  }

  /* Free memory */
  if(params->step + 1 >= params->nsteps) {
    free(qactive);
    free(Factive);
    init = true;
  }

  return;
}

real Hamaker_force(real r, real r1, real r2, real H_121)
{
  real d = r - (r1 + r2);
  return H_121*r1*r2/(real_val(6.0)*(r1 + r2)*d*d);
}

/* Force due to a double-layer electrostatic repulsion */
real double_layer_electrostatic_force(real r, real r1, real r2,
                                  real Zeta, real kappa)
{
  if(r < r1 + r2)
    return -(Zeta/((1 + kappa*r1)*(1 + kappa*r2)))
            *((kappa*(r1 + r2) + 1)/((r1 + r2)*(r1 + r2)))
            *pow((r1 + r2)/r, 12);
  else
    return -(Zeta/((1 + kappa*r1)*(1 + kappa*r2)))
            *exp(-kappa*(r - (r1 + r2)))
            *(kappa/r + 1/(r*r));
}

void force_short_range(struct system * sys, struct parameters * params)
{
  real rij[sys->dim]; /* Displacement vector */
  real r, r2; /* Distance and distance squared */
  real F; /* Modulus of the force */
  real sigma; /* Lennard-Jones sigma parameter */
  int i, j; /* Particle indices */

  /* Update interaction list */
  update_interaction_list(sys, params);

  /* For every interaction in the list */
  for(int n = 0; n < params->ninteractions; ++n) {
    /* Displacement vector */
    i = params->interaction_list[2*n];
    j = params->interaction_list[2*n + 1];
    for(int l = 0; l < sys->dim; ++l)
      rij[l] = sys->q[sys->dim*j + l] - sys->q[sys->dim*i + l];

    /* Periodic boundary conditions */
    applyPBC(rij, sys->L, sys->dim);

    /* Distance */
    r2 = 0; for(int l = 0; l < sys->dim; ++l) r2 += rij[l]*rij[l];
    r = sqrt(r2);

    /* Force */
    sigma = sys->r[i] + sys->r[j] + 2*sys->elayer;
    F = LennardJones_force(r, sys->epsilon, sigma, sys->ljrc);
    if(sys->H_121 != 0)
      F += Hamaker_force(r, sys->r[i], sys->r[j], sys->H_121);
    if(sys->Zeta != 0)
      F += double_layer_electrostatic_force(r, sys->r[i], sys->r[j], sys->Zeta,
                                            sys->kappa);
    for(int l = 0; l < sys->dim; ++l) {
      sys->F[sys->dim*i + l] += F*rij[l]/r;
      sys->F[sys->dim*j + l] -= F*rij[l]/r;
    }
  }

  return;
}

void force(struct system * sys, struct parameters * params) {
  /* Clear force vector */
  for(int l = 0; l < sys->N; ++l) sys->F[l] = real_val(0.0);

  /* External force fields */
  force_external(sys, params);

  /* Bonded forces */
  force_fixed_bonds(sys);
  force_harmonic_bonds(sys);
  force_angular_bonds(sys);
  force_torsional_bonds(sys);

  /* Short-range forces */
  force_short_range(sys, params);
  return;
}

/* Mobility */
void update_mobility(real * M, struct system * sys, struct parameters * params)
{
  real Rij[sys->dim]; /* Displacement vector between particles i and j */
  real distRij; /* Distance between particles i and j */
  real Rij2; /* distRij squared */
  real c1, c2; /* Coefficients in the Rotne-Prager-Yamakawa tensor */
  real ri2, rj2; /* Hydrodynamic radii squared */
  real rimrj2; /* Difference of squared radii */

  /* Calculate the Rotne-Prager-Yamakawa mobility tensor */
  for(int i = 0; i < sys->np; ++i) {
    for(int j = i + 1; j < sys->np; ++j) {
      /* Calculate the vector from R_i to R_j */
      for(int k = 0; k < sys->dim; ++k)
        Rij[k] = sys->q[sys->dim*j + k] - sys->q[sys->dim*i + k];

      /* Calculate the distance between R_i and R_j */
      Rij2 = 0;
      for(int k = 0; k < sys->dim; ++k)
        Rij2 += Rij[k]*Rij[k];
      distRij = sqrt(Rij2);

      /* Calculate the factors c1 and c2
         (taking into account whether particles overlap) */
      if(distRij > sys->r[i] + sys->r[j]) {
        ri2 = sys->r[i]*sys->r[i];
        rj2 = sys->r[j]*sys->r[j];
        c1 = 0.75*(1 + (ri2 + rj2)/(3*Rij2))/distRij;
        c2 = 0.75*(1 - (ri2 + rj2)/Rij2)/distRij;
      }
      else if(distRij <= fabs(sys->r[i] - sys->r[j])) {
        c1 = (sys->r[i] > sys->r[j])?(1/sys->r[i]):(1/sys->r[j]);
        c2 = 0;
      }
      else {
        rimrj2 = (sys->r[i] - sys->r[j])*(sys->r[i] - sys->r[j]);
        c1 = (0.5*(sys->r[i] + sys->r[j])
               - (rimrj2 + 3*Rij2)*(rimrj2 + 3*Rij2)/(32*Rij2*distRij))
             /(sys->r[i]*sys->r[j]);
        c2 = 3.0*(rimrj2 - Rij2)*(rimrj2 - Rij2)
             /(32*Rij2*distRij*sys->r[i]*sys->r[j]);
      }

      /* Fill in the RPY tensor components */
      for(int k = 0; k < sys->dim; ++k) {
        for(int l = 0; l < sys->dim; ++l) {
          M[sys->N*(sys->dim*i + k) + sys->dim*j + l]
            = (c1*delta(k,l) + c2*Rij[k]*Rij[l]/((Rij2>0)?Rij2:1))
              /sys->gamma;
        }
      }
    }
  }

  /* With wall correction, you need to reset diagonal blocks */
  for(int i = 0; i < sys->np; ++i)
    for(int k = 0; k < sys->dim; ++k)
      for(int l = 0; l < sys->dim; ++l)
        M[sys->N*(sys->dim*i + k) + sys->dim*i + l]
          = delta(k, l)/(sys->gamma*sys->r[l/3]);

  /* Correction due to a slip wall at z = 0 */
  real sign[sys->dim]; /* Sign of mirror force */
  for(int l = 0; l < sys->dim; ++l) sign[l] = 0;
  sign[sys->dim - 1] = -1;
  for(int i = 0; i < sys->np; ++i) {
    for(int j = i; j < sys->np; ++j) {
      /* Calculate the vector from R_i to R_j */
      Rij[0] = sys->q[sys->dim*j] - sys->q[sys->dim*i];
      Rij[1] = sys->q[sys->dim*j + 1] - sys->q[sys->dim*i + 1];
      Rij[2] = -sys->q[sys->dim*j + 2] - sys->q[sys->dim*i + 2];

      /* Calculate the distance between R_i and R_j */
      Rij2 = 0;
      for(int k = 0; k < sys->dim; ++k)
        Rij2 += Rij[k]*Rij[k];
      distRij = sqrt(Rij2);

      /* Calculate the factors c1 and c2
         (taking into account whether particles overlap) */
      if(distRij > sys->r[i] + sys->r[j]) {
        ri2 = sys->r[i]*sys->r[i];
        rj2 = sys->r[j]*sys->r[j];
        c1 = 0.75*(1 + (ri2 + rj2)/(3*Rij2))/distRij;
        c2 = 0.75*(1 - (ri2 + rj2)/Rij2)/distRij;
      }
      else {
        rimrj2 = (sys->r[i] - sys->r[j])*(sys->r[i] - sys->r[j]);
        c1 = (0.5*(sys->r[i] + sys->r[j])
               - (rimrj2 + 3*Rij2)*(rimrj2 + 3*Rij2)/(32*Rij2*distRij))
             /(sys->r[i]*sys->r[j]);
        c2 = 3.0*(rimrj2 - Rij2)*(rimrj2 - Rij2)
             /(32*Rij2*distRij*sys->r[i]*sys->r[j]);
      }

      /* Fill in the RPY tensor components */
      for(int k = 0; k < sys->dim; ++k) {
        for(int l = 0; l < sys->dim; ++l) {
          M[sys->N*(sys->dim*i + k) + sys->dim*j + l]
            += sign[l]*(c1*delta(k,l) + c2*Rij[k]*Rij[l]/((Rij2>0)?Rij2:1))
               /sys->gamma;
        }
      }
    }
  }

  /* The RPY tensor is symmetrical */
  for(int i = 1; i < sys->N; ++i)
    for(int j = 0; j < i; ++j)
      M[sys->N*i + j]  = M[sys->N*j + i];
}

/* Integration scheme */
void integration_step(struct system * sys, struct parameters * params)
{
  /*** Integrate the stochastic differential equation ***/
  /*           dR = M F dt + sqrt(2 kT) B dW            */

  /* Allocate memory for integration scheme */
  static bool init = true;
  static real * dW, * dW2; /* Wiener displacements */
  static real * M, * B; /* Mobility and Cholesky factor of mobility */
  static real * MF, * BdW; /* Intermediate vector results */
  if(init) {
    dW   = malloc(sys->N*sizeof(real));
    dW2  = malloc(sys->N*sizeof(real));
    M    = calloc(sys->N*sys->N, sizeof(real));
    MF   = malloc(sys->N*sizeof(real));
    B    = malloc(sys->N*sys->N*sizeof(real));
    BdW  = calloc(sys->N, sizeof(real));
    init = false;

    /* Diagonal boxes remain unchanged during execution */
    for(int l = 0; l < sys->N; ++l)
      M[sys->N*l + l] = 1/(sys->gamma*sys->r[l/3]);

    /* Fill dW with Gaussian random numbers */
    for(int l = 0; l < sys->N; ++l)
      dW[l] = Gaussian(0,1)*sqrt(params->dt);
  }

  /* Update mobility tensor */
  update_mobility(M, sys, params);

  /* Find B through Cholesky decomposition of M */
  if(sys->kT > 0)
    Cholesky(M, B, sys->N);

  /* Calculate the forces */
  force(sys, params);

  /* Save old positions */
  for(int l = 0; l < sys->N; ++l) sys->p[l] = sys->q[l];

  /* Calculate the Wiener displacement */
  if(sys->kT > 0) {
    for(int l = 0; l < sys->N; ++l) {
      dW2[l] = Gaussian(0,1)*sqrt(params->dt);
      dW[l] = real_val(0.5)*(dW2[l] + dW[l]);
    }
  }

  /* Euler scheme integration step */
  for(int i = 0; i < sys->np; ++i)
    prod(M, sys->F, MF, sys->N); /* M·F */

  if(sys->kT > 0) {
    prod(B, dW, BdW, sys->N); /* B·dW */
  }

  /* dR = M F dt + sqrt(2 kT) B·dW */
  for(int l = 0; l < sys->N; ++l)
    sys->q[l] += MF[l]*params->dt + sqrt(2.0*sys->kT)*BdW[l];

  if(sys->kT > 0) {
    for(int l = 0; l < sys->N; ++l)
      dW[l] = dW2[l];
  }

  /* Calculate "momenta" (used for interaction lists) */
  for(int l = 0; l < sys->N; ++l)
    sys->p[l] = sys->m[l/sys->dim]*(sys->q[l] - sys->p[l]);

  /* Free memory at last step */
  if(params->step + 1 >= params->nsteps) {
    free(dW);
    free(dW2);
    free(M);
    free(MF);
    free(B);
    free(BdW);
    init = true;
  }

  return;
}

/* Measurements */
double energy_short_range(){ return 0.0; }

double total_energy(struct system * sys, struct parameters * params)
{
  double Etot = 0.0;

  /* Kinetic energy */
  for(int l = 0; l < sys->N; ++l)
    Etot += 0.5*sys->p[l]*sys->p[l]/sys->m[l/sys->dim];

  /* Internal energy */
  Etot += energy_fixed_bonds(sys);
  Etot += energy_harmonic_bonds(sys);
  Etot += energy_angular_bonds(sys);
  Etot += energy_torsional_bonds(sys);
  Etot += energy_short_range(sys, params);

  return Etot;
}

void total_momentum(struct system * sys, double * ptot)
{
  /* Clear total momentum */
  for(int l = 0; l < sys->dim; ++l) ptot[l] = real_val(0.0);

  /* Sum momenta */
  for(int i = 0; i < sys->np; ++i) {
    for(int l = 0; l < sys->dim; ++l)
      ptot[l] += sys->p[sys->dim*i + l];
  }

  return;
}

/*** Main function ***/
int main(int argc, char * argv[]) {
  /* Usage message */
  usage_message(argc, argv);

  /* nanometre header for information sent to stderr */
  output_log_header();
  fprintf(stderr,
          LOG_INFO  "Generic Molecular Dynamics\n");

  /* Set up the system */
  struct parameters params; /* Algorithm information */
  struct system sys;        /* System information */

  /* Read the simulation parameters from external file */
  read_parameter_file(argv[1], &sys, &params);

  /* Open data files for output */
  FILE * coordinates = openfile(params.output_configurations, "w");

  /* Load system information from files */
  /* Load positions from configuration file */
  if(params.initial_configuration[0] == '\0')
    fprintf(stderr,
            LOG_WARNING "No positions file specified in %s.\n", argv[1]);
  else load_system(&sys, params.initial_configuration);

  /* Load bond potential parameters from files */
  if(params.bondfilename[0] == '\0')
    fprintf(stderr,
            LOG_WARNING "No bonds file specified in %s.\n", argv[1]);
  else {
    fprintf(stderr, LOG_INFO "Reading bond file (%s).\n", params.bondfilename);
    FILE * bondfile = openfile(params.bondfilename, "r");
    read_fixedbonds(&sys, bondfile);
    read_pairbonds(&sys, bondfile, harmonic);
    read_angular(&sys, bondfile);
    read_torsional(&sys, bondfile);
    closefile(bondfile);
  }

  /* Exit if there are no particles */
  if(sys.N == 0) {
    fprintf(stderr, LOG_DONE "No system present.\n");
    output_log_end();
    return 0;
  }

  /* Set up neighbour lists */
  setup_links(&sys, &params);

  /* Seed random number generator */
  prng_srand();

  /* Integrate the equations of motion */
  for(params.step = 0; params.step < params.nsteps; ++params.step) {
    integration_step(&sys, &params);

    /* Configuration sampling */
    if(params.step % params.sampling_step == 0) {
      fprintf(stderr, "\r"
                      LOG_STATUS "Progress: [ " YELLOW "%.1f%%" NORMAL " ]",
                      (((float) params.step)*100.0)
                      /((float) params.nsteps));


      /* Coordinates */
      fprintf(coordinates, "# t = %f\n", params.step*params.dt);
      for(int i = 0; i < sys.np; ++i)
        fprintf(coordinates, "%f  %f  %f  %f\n",
                             sys.q[sys.dim*i],
                             sys.q[sys.dim*i + 1],
                             sys.q[sys.dim*i + 2],
                             sys.r[i]);
      fprintf(coordinates, "\n");

    }
  }

  /* Clean up and exit */
  fprintf(stderr, "\r" LOG_STATUS "Progress: [ " GREEN "DONE" NORMAL " ]   \n");

  if(params.final_state[0] != '\0') {
    fprintf(stderr, LOG_INFO "Saving final state to %s.\n", params.final_state);
    save_system(&sys, params.final_state);
  }

  fprintf(stderr, LOG_INFO "Cleaning up and exiting.\n");

  free_memory(&sys, &params);
  closefile(coordinates);
  output_log_end();

  return 0;
}
