# ifndef _RANDOM_H_
# define _RANDOM_H_
# include <math.h>
# include <stdio.h>

/* Data type definitions */
# include <stdint.h>
# ifndef __bool_type__
# define __bool_type__
typedef enum {false, true} bool; /* Boolean type */
# endif

/* Pseudorandom number generator seed */
/* Default seed */
uint64_t rngseed[] = {12679825035178159220ull, 15438657923749336752ull};

/* Seed the PRNG by reading from /dev/urandom */
void prng_srand() {
  FILE * urandom = fopen("/dev/urandom", "r");
  fread(rngseed, sizeof(uint64_t), 2, urandom);
  fclose(urandom);
  return;
}

/* Random integer */
# define PRNG() xorshift128plus()
# define RANDOM_MAX 18446744073709551615ull

/* 64-bit (pseudo)random integer */
uint64_t xorshift128plus(void)
{
  uint64_t x = rngseed[0];
  uint64_t const y = rngseed[1];
  rngseed[0] = y;
  x ^= x << 23; // a
  x ^= x >> 17; // b
  x ^= y ^ (y >> 26); // c
  rngseed[1] = x;
  return x + y;
}

/* Random number from a uniform distribution */
double uniform(double min, double max)
{
  return min + (PRNG()/((double) RANDOM_MAX))*(max - min);
}

/* Random number from a Gaussian distribution */
float Gaussian(float mean, float stddev)
{
  double u1, u2, s0 = 2;
  static bool saved_value = false; /* Flag indicating stored number */
  static double Gnum; /* Stored Gaussian number */

  if(saved_value) {
    saved_value = false;
    return mean + stddev*Gnum;
  }

  while(s0 >= 1)
  {
    u1 = uniform(-1, 1);
    u2 = uniform(-1, 1);
    s0 = u1*u1 + u2*u2;
  }

  Gnum = u2*sqrt(-2*logf(s0)/s0);
  saved_value = true;

  return mean + stddev*u1*sqrt(-2*logf(s0)/s0);
}

/* Marsaglia's algorithm for random point on a unit sphere */
void random_axis(float * n)
{
  float x, y, s = 2;

  while(s > 1) {
    x = uniform(-1, 1);
    y = uniform(-1, 1);
    s = x*x + y*y;
  }

  n[0] = 2*x*sqrt(1 - s);
  n[1] = 2*y*sqrt(1 - s);
  n[2] = 1 - 2*s;

  return;
}

# endif
