package ec.evolutionary_extraction_procedure.models.draincurrentmodel.Hysteresis;

import ec.evolutionary_extraction_procedure.exception.*;
import ec.evolutionary_extraction_procedure.fitness.Fitness;

import static ec.evolutionary_extraction_procedure.constraints.ContactVoltage.*;
import static ec.evolutionary_extraction_procedure.constraints.Current.infCurrent;
import static ec.evolutionary_extraction_procedure.constraints.Current.nanCurrent;
import ec.evolutionary_extraction_procedure.utils.StandardDeviation;
/**
 * @author Adrián Romero <adrianromerocaceres @ gmail.com>
 * @version 1.0 (current version number of program)
 * @since 26 (the version of the package this class was first added to)
 */
public class Subthreshold {

    public static double contactConductance(double alfa, double VSS, double VG, double VT, double gamma) {
        double MC;

        MC = alfa * Math.pow(VSS * Math.log(1 + Math.exp((VG - VT) / VSS)), 1 + gamma);

        return MC;
    }

    public static double genericContactConductance(double alfa, double VG, double VT, double gamma) {
        double MC;

        MC = alfa * Math.pow(VG - VT, 1 + gamma);

        return MC;
    }

    public static double fIV(double ID, double MM, double VGS, double VT, double VDS, double gamma, double mu0, double W, double L, double Cox, double ms, double VSS) throws fIVnoSolution {

        // Variables
        double VS, y;
        double Veosr, Veodr;

        /* Source contact voltage */
        VS = Math.pow(ID / MM, 1 / ms);

        if (nanContactVoltage(VS)) {
            throw new NaNConctactVoltage("contactEffects: NaN value for contacts");
        } else if (negativeContactVoltage(VS)) {
            throw new NegativeConctactVoltage("contactEffects: Conctact effects positive");
        } else if (infContactVoltage(VS)) {
            throw new InfConctactVoltage("contactEffects: Conctact effects positive");
        }
        else
        {

        }

        Veosr = VSS * Math.log(1 + Math.exp(((VGS - VT - VS)) / VSS));
        Veodr = VSS * Math.log(1 + Math.exp(((VGS - VT - VDS)) / VSS));

        y = (mu0 * Cox * (W / L) * (Math.pow(Veosr, gamma + 2) - Math.pow(Veodr, gamma + 2)) / (gamma + 2)) - ID;

        // If the solution is NaN means that we do not have a valid solution
        if (Double.isNaN(y) || Double.isInfinite(y)) {
            throw new fIVnoSolution("fIV: NaN value for individual");
        }

        return y;

    }

    public static double busquedaceros(double IA, double IB, double MM, double VG, double VTC, double VD, double ghamma, double mu0, double W, double L, double Cox, double mk, double VSS) throws ParameterSetNoConvergence {

        // declaration of variables
        double fa, fb, fr, IR;
        int solution, iteration;

        // initialization of variables
        IR = 0;
        solution = 0;
        iteration = 0;

        // compute differences in points IA and IB
        fa = fIV(IA, MM, VG, VTC, VD, ghamma, mu0, W, L, Cox, mk, VSS);
        fb = fIV(IB, MM, VG, VTC, VD, ghamma, mu0, W, L, Cox, mk, VSS);

        // search for solution
        while ((solution == 0) && (iteration < 1000)) {

            IR = (IA + IB) / 2; // corriente media

            fr = fIV(IR, MM, VG, VTC, VD, ghamma, mu0, W, L, Cox, mk, VSS); // f(corriente media)

            if (Math.abs(fr) < 1.0E-15) {
                //System.out.println("Solution found");
                solution = 1;
            }
            if (fa * fr < 0) {
                IB = IR;
            } else if (fa * fr > 0) {
                IA = IR;
                fa = fr;
            }
            iteration += 1;
        }

        if (solution == 0) {
            //System.out.println("Solution not found");
            throw new ParameterSetNoConvergence("busquedaceros: Solution not found");
        }

        return IR;

    }

    public static double IdVd(double[][][] IVexp,
                                  double[] VGS,
                                  double[] VDS,
                                  double[] IAA,
                                  double W,
                                  double L,
                                  double Cox,
                                  double mu0,
                                  double gamma,
                                  double alphaS,
                                  double vt_t0,
                                  double vss_t0,
                                  double ms,
                                  double tao,
                                  double QA,
                                  double[] alphaQ,
                                  double[] FactorQA,
                                  double thold,
                                  double tdelay,
                                  double tmeasure) {

        int points, incurvas, numSets;
        points = IVexp[1].length;
        incurvas = IVexp[0][1].length;
        numSets = IVexp.length;
        double IA, IB, Q_measure, Q_delay, IVsimA, IVsimB;
        int n = 0;
        int IZ, IS, direction;
        int iterations;

        double[] outputs = new double[points * incurvas * numSets];
        double[] targets = new double[points * incurvas * numSets];

        double[][][] IVsim = new double[numSets][points][incurvas];
        double[][][] vt = new double[numSets][points][incurvas];
        double[][][] vss = new double[numSets][points][incurvas];
        double[][][] qt = new double[numSets][points][incurvas];
        double[][][] ccs = new double[numSets][points][incurvas];
        double[][][] t = new double[numSets][points][incurvas];

        double ccs_tdelay, vt_tdelay, vss_tdelay;

        /* Analyze every curve */
        for (int i2 = 0; i2 < incurvas; i2++) {

            /* going forward and backward */
            for (int i3 = 0; i3 < 2; i3++) {

                if (i3 == 0) /* FORDWARD */ {
                    IZ = 0;
                    IS = 1;
                    direction = 1;
                } else /* BAKWARD */ {
                    IZ = points - 1;
                    direction = -1; /* para el bakward cambiamos la direccion ya que retrocedemos */
                    IS = -1;
                }

                /* Iterate for all VDS values */
                for (int i1 = IZ; (i1 >= 0) && (i1 < points); i1 += IS) {
                    //System.out.println("i1: "+ i1 + " i2: " + i2 + " i3: " + i3);
                    if (i3 == 0) {
                        if (i1 <= 1) {
                            IA = IAA[i2];
                            IB = 0;
                        } else {
                            IA = IVsim[i3][i1 - 1][i2] / 2; // I assign the value of the previous point
                            IB = IVsim[i3][i1 - 1][i2] * 2; // I assign the value of the previous point
                        }
                    } else /* BACKWARD */ {
                        if (i1 == points - 1) {
                            IA = IVsim[i3 - 1][i1 - 1][i2] / 2; // I assign the value of the previous point
                            IB = IVsim[i3 - 1][i1 - 1][i2] * 2; // I assign the value of the previous point
                        } else if (i1 <= 1) {
                            IA = IAA[i2];
                            IB = 0;
                        } else //((i1 < (points - 1)) && (i1>1))
                        {
                            IA = IVsim[i3][i1 + 1][i2] / 2; // I assign the value of the previous point
                            IB = IVsim[i3][i1 + 1][i2] * 2; // I assign the value of the previous point
                        }
                    }

                    /* Take into consideration thold - New VG value */
                    if ((i1 == 0) && (i3 == 0)) {
                        /* First point of each curve - VDS = 0 -> thold */
                        t[i3][i1][i2] = thold;

                        /* First curve. Initial charge is Qa */ /* t=0 Unknown trapped charge */
                        if (i2 == 0) {
                            vt[i3][i1][i2] = vt_t0;
                            vss[i3][i1][i2] = vss_t0;
                            ccs[i3][i1][i2] = contactConductance(alphaS, vss[i3][i1][i2], VGS[i2], vt[i3][i1][i2], gamma);
                            IVsim[i3][i1][i2] = busquedaceros(IA, IB, ccs[i3][i1][i2], VGS[i2], vt[i3][i1][i2], VDS[i1], gamma, mu0, W, L, Cox, ms, vss[i3][i1][i2]);
                            qt[i3][i1][i2] = -QA;
                        } else /* There is already charge */ {
                            qt[i3][i1][i2] = qt[i3 + 1][i1][i2 - 1] * Math.exp(-t[i3][i1][i2] / tao) - FactorQA[i2] * QA * (1 - Math.exp(-t[i3][i1][i2] / tao));
                            vt[i3][i1][i2] = vt[i3 + 1][i1][i2 - 1] - ((qt[i3][i1][i2] - qt[i3 + 1][i1][i2 - 1]) / Cox);
                            vss[i3][i1][i2] = vss[i3 + 1][i1][i2 - 1] - (0.02586 * Math.log(10) * (qt[i3][i1][i2] - qt[i3 + 1][i1][i2 - 1]) / Cox);
                            ccs[i3][i1][i2] = ccs[i3 + 1][i1][i2 - 1];
                            IVsim[i3][i1][i2] = IVsim[i3 + 1][i1][i2 - 1];
                        }


                    } else if ((i3 == 1) && (i1 == points - 1)) {
                        /* cuando pasamos de fordward a backward queremos en el primer punto del backward la tension
                         umbral, la corriente y la carga atrapada sea todo lo mismo que el ultimo punto del VGSforward */
                        vt[i3][i1][i2] = vt[i3 - 1][i1][i2];
                        vss[i3][i1][i2] = vss[i3 - 1][i1][i2];
                        ccs[i3][i1][i2] = ccs[i3 - 1][i1][i2];
                        IVsim[i3][i1][i2] = IVsim[i3 - 1][i1][i2];
                        qt[i3][i1][i2] = qt[i3 - 1][i1][i2];
                        t[i3][i1][i2] = 0;

                    } else {

                        /* evaluate at tdelay */
                        vt_tdelay = vt[i3][i1 - direction][i2];
                        vss_tdelay = vss[i3][i1 - direction][i2];
                        ccs_tdelay = ccs[i3][i1 - direction][i2];

                        /* evaluate current */
                        IVsimB = busquedaceros(IA, IB, ccs_tdelay, VGS[i2], vt_tdelay, VDS[i1], gamma, mu0, W, L, Cox, ms, vss_tdelay);

                        iterations = 0;

                        do {

                            IVsimA = IVsimB;

                            /* update charge and parameters */
                            Q_delay = qt[i3][i1 - direction][i2] * Math.exp(-tdelay / tao) + alphaQ[i3] * IVsimA * tao * (1 - Math.exp(-tdelay / tao));
                            vt_tdelay = vt[i3][i1 - direction][i2] - ((Q_delay - qt[i3][i1 - direction][i2]) / Cox);
                            vss_tdelay = vss[i3][i1 - direction][i2] - (0.02586 * Math.log(10) * (Q_delay - qt[i3][i1 - direction][i2]) / Cox);
                            ccs_tdelay = contactConductance(alphaS, vss_tdelay, VGS[i2], vt_tdelay, gamma);

                            /* Update current */
                            IVsimB = busquedaceros(IA, IB, ccs_tdelay, VGS[i2], vt_tdelay, VDS[i1], gamma, mu0, W, L, Cox, ms, vss_tdelay);

                            iterations++;

                        } while ((Math.abs((IVsimB - IVsimA) / IVsimB) > 0.001) && (iterations < 1000));

                        if (iterations >= 1000) {
                            throw new ParameterSetNoConvergence("outputCharacteristics: Solution not found for tdelay");
                        }

                        /* update current value at tdaly */
                        IVsim[i3][i1][i2] = IVsimB;

                        /* initially previous values are taken */
                        vt[i3][i1][i2] = vt[i3][i1 - direction][i2];
                        vss[i3][i1][i2] = vss[i3][i1 - direction][i2];
                        ccs[i3][i1][i2] = ccs[i3][i1 - direction][i2];

                        t[i3][i1][i2] = tmeasure;
                        /* Update current */
                        IVsimB = busquedaceros(IA, IB, ccs[i3][i1][i2], VGS[i2], vt[i3][i1][i2], VDS[i1], gamma, mu0, W, L, Cox, ms, vss[i3][i1][i2]);

                        iterations = 0;

                        do {

                            IVsimA = IVsimB;
                            Q_measure = qt[i3][i1 - direction][i2] * Math.exp(-t[i3][i1][i2] / tao) + alphaQ[i3] * IVsimA * tao * (1 - Math.exp(-t[i3][i1][i2] / tao));
                            vt[i3][i1][i2] = vt[i3][i1 - direction][i2] - ((Q_measure - qt[i3][i1 - direction][i2]) / Cox);
                            vss[i3][i1][i2] = vss[i3][i1 - direction][i2] - (0.02586 * Math.log(10) * (Q_measure - qt[i3][i1 - direction][i2]) / Cox);
                            ccs[i3][i1][i2] = contactConductance(alphaS, vss[i3][i1][i2], VGS[i2], vt[i3][i1][i2], gamma);
                            IVsimB = busquedaceros(IA, IB, ccs[i3][i1][i2], VGS[i2], vt[i3][i1][i2], VDS[i1], gamma, mu0, W, L, Cox, ms, vss[i3][i1][i2]);
                            iterations++;

                        } while ((Math.abs((IVsimB - IVsimA) / IVsimB) > 0.001) && (iterations < 1000));

                        if (iterations >= 1000) {
                            throw new ParameterSetNoConvergence("outputCharacteristics: Solution not found for tmeasure");
                        }

                        /* update trapped charge value at tmeasure */
                        qt[i3][i1][i2] = Q_measure;

                    }

                    if (qt[i3][i1][i2] >= 0) {
                        throw new ParameterSetNoConvergence("outputCharacteristics: Q>=0");
                    }

                    targets[n] = IVexp[i3][i1][i2];
                    outputs[n] = IVsim[i3][i1][i2];
                    n++;
                }
            }
        }

        //plotMatrixt;
        //plotMatrix(outputs, targets);
        /* Fitting error */
        return Fitness.NRMSE(targets, outputs);
    }

    public static double IdVd(double[][][] IVexp,
                              double[] VGS,
                              double[] VDS,
                              double[] IAA,
                              double W,
                              double L,
                              double Cox,
                              double mu0,
                              double gamma,
                              double alphaS,
                              double vt_t0,
                              double vss_t0,
                              double ms,
                              double tao,
                              double QA,
                              double[] beta,
                              double tmeasure) {

        int points, incurvas, numSets;
        points = IVexp[1].length;
        incurvas = IVexp[0][1].length;
        numSets = IVexp.length;

        //System.out.println("points: "+ points + " incurvas: " + incurvas + " numSets: " + numSets);

        double IA, IB, Q_measure, IVsimA, IVsimB;
        int n = 0;
        int IZ, IS, direction;
        int iterations;

        double[] outputs = new double[points * incurvas * numSets];
        double[] targets = new double[points * incurvas * numSets];

        double[][][] IVsim = new double[numSets][points][incurvas];
        double[][][] vt = new double[numSets][points][incurvas];
        double[][][] vss = new double[numSets][points][incurvas];
        double[][][] qt = new double[numSets][points][incurvas];
        double[][][] ccs = new double[numSets][points][incurvas];
        double[][][] t = new double[numSets][points][incurvas];

        /* Analyze every curve */
        for (int i2 = 0; i2 < incurvas; i2++) {

            /* going forward and backward */
            for (int i3 = 0; i3 < 2; i3++) {

                if (i3 == 0) /* FORDWARD */ {
                    IZ = 0;
                    IS = 1;
                    direction = 1;
                } else /* BAKWARD */ {
                    IZ = points - 1;
                    direction = -1; /* para el bakward cambiamos la direccion ya que retrocedemos */
                    IS = -1;
                }

                /* Iterate for all VDS values */
                for (int i1 = IZ; (i1 >= 0) && (i1 < points); i1 += IS) {

                    //System.out.println("i1: "+ i1 + " i2: " + i2 + " i3: " + i3);
                    //System.out.println("i1: "+ i1 + " i2: " + i2 + " i3: " + i3);

                    if (i3 == 0) {
                        if (i1 == 0){
                            IA = IAA[i2];
                            IB = 0;
                        }
                        else if ((IVexp[i3][i1][i2] == 0) ||
                                ((IVexp[i3][i1 - 1][i2] == 0) && (IVexp[i3][i1][i2] != 0)) ||
                                (IVsim[i3][i1 - 1][i2] == 0) ){
                            IA = IAA[i2];
                            IB = 0;
                        } else {
                            IA = IVsim[i3][i1 - 1][i2] / 2; // I assign the value of the previous point
                            IB = IVsim[i3][i1 - 1][i2] * 2; // I assign the value of the previous point
                        }
                    } else /* BACKWARD */ {
                        if (i1 == points - 1) {
                            IA = IVsim[i3 - 1][i1 - 1][i2] / 4; // I assign the value of the previous point
                            IB = IVsim[i3 - 1][i1 - 1][i2] * 4; // I assign the value of the previous point
                        } else if (i1 <= 1) {
                            IA = IAA[i2];
                            IB = 0;
                        } else //((i1 < (points - 1)) && (i1>1))
                        {
                            IA = IVsim[i3][i1 + 1][i2] / 2; // I assign the value of the previous point
                            IB = IVsim[i3][i1 + 1][i2] * 2; // I assign the value of the previous point
                        }
                    }

                    /* Take into consideration thold - New VG value */
                    if ((i1 == 0) && (i3 == 0)) {

                        vt[i3][i1][i2] = vt_t0;
                        vss[i3][i1][i2] = vss_t0;
                        ccs[i3][i1][i2] = contactConductance(alphaS, vss[i3][i1][i2], VGS[i2], vt[i3][i1][i2], gamma);
                        IVsim[i3][i1][i2] = busquedaceros(IA, IB, ccs[i3][i1][i2], VGS[i2], vt[i3][i1][i2], VDS[i1], gamma, mu0, W, L, Cox, ms, vss[i3][i1][i2]);
                        qt[i3][i1][i2] = -QA;
                        //System.out.println("Aquí llego 1");

                    } else if ((i3 == 1) && (i1 == points - 1)) {
                        /* cuando pasamos de fordward a backward queremos en el primer punto del backward la tension
                         umbral, la corriente y la carga atrapada sea todo lo mismo que el ultimo punto del VGSforward */
                        vt[i3][i1][i2] = vt[i3 - 1][i1][i2];
                        vss[i3][i1][i2] = vss[i3 - 1][i1][i2];
                        ccs[i3][i1][i2] = ccs[i3 - 1][i1][i2];
                        IVsim[i3][i1][i2] = IVsim[i3 - 1][i1][i2];
                        qt[i3][i1][i2] = qt[i3 - 1][i1][i2];
                        t[i3][i1][i2] = 0;

                    } else {

                        //System.out.println("Aquí llego 6");

                        /* initially previous values are taken */
                        vt[i3][i1][i2] = vt[i3][i1 - direction][i2];
                        vss[i3][i1][i2] = vss[i3][i1 - direction][i2];
                        ccs[i3][i1][i2] = ccs[i3][i1 - direction][i2];

                        t[i3][i1][i2] = tmeasure;
                        /* Update current */
                        IVsimB = busquedaceros(IA, IB, ccs[i3][i1][i2], VGS[i2], vt[i3][i1][i2], VDS[i1], gamma, mu0, W, L, Cox, ms, vss[i3][i1][i2]);

                        //System.out.println("Aquí llego2");

                        iterations = 0;

                        do {

                            IVsimA = IVsimB;
                            Q_measure = qt[i3][i1 - direction][i2] * Math.exp(-t[i3][i1][i2] / tao) + beta[i3] * IVsimA * tao * (1 - Math.exp(-t[i3][i1][i2] / tao));
                            vt[i3][i1][i2] = vt[i3][i1 - direction][i2] - ((Q_measure - qt[i3][i1 - direction][i2]) / Cox);
                            vss[i3][i1][i2] = vss[i3][i1 - direction][i2] - (0.02586 * Math.log(10) * (Q_measure - qt[i3][i1 - direction][i2]) / Cox);
                            ccs[i3][i1][i2] = contactConductance(alphaS, vss[i3][i1][i2], VGS[i2], vt[i3][i1][i2], gamma);
                            IVsimB = busquedaceros(IA, IB, ccs[i3][i1][i2], VGS[i2], vt[i3][i1][i2], VDS[i1], gamma, mu0, W, L, Cox, ms, vss[i3][i1][i2]);
                            iterations++;

                        } while ((Math.abs((IVsimB - IVsimA) / IVsimB) > 0.001) && (iterations < 1000));
                        //System.out.println("Aquí llego 3");

                        if (iterations >= 1000) {
                            //System.out.println("Se han excedido las iteraciones");
                            throw new ParameterSetNoConvergence("outputCharacteristics: Solution not found for tmeasure");
                        }

                        IVsim[i3][i1][i2] = IVsimB;
                        /* update trapped charge value at tmeasure */
                        qt[i3][i1][i2] = Q_measure;

                    }

                    if (qt[i3][i1][i2] > 0) {
                        throw new ParameterSetNoConvergence("outputCharacteristics: Q>0");
                    }

                    //System.out.println("Qt: " + qt[i3][i1][i2] + " -- Beta: " + beta[i3]);
                    //System.out.println("IVexp: " + IVexp[i3][i1][i2] + " -- IVsim:" + IVsim[i3][i1][i2]);

                    targets[n] = IVexp[i3][i1][i2];
                    outputs[n] = IVsim[i3][i1][i2];
                    n++;

                }
            }
        }

        //plotMatrixt;
        //plotMatrix(outputs, targets);
        /* Fitting error */
        return Fitness.NRMSE(targets, outputs);
    }

    public static double IdVd(double[][][] IVexp,
                              double[] VGS,
                              double[] VDS,
                              double[] IAA,
                              double W,
                              double L,
                              double Cox,
                              double mu0,
                              double gamma,
                              double alphaS,
                              double vt_t0,
                              double vss_t0,
                              double ms,
                              double tao,
                              double QA,
                              double[][] beta,
                              double tmeasure) {

        int points, incurvas, numSets;
        points = IVexp[1].length;
        incurvas = IVexp[0][1].length;
        numSets = IVexp.length;

        //System.out.println("points: "+ points + " incurvas: " + incurvas + " numSets: " + numSets);

        double IA, IB, Q_measure, IVsimA, IVsimB;
        int n = 0;
        int IZ, IS, direction;
        int iterations;

        double[] outputs = new double[points * incurvas * numSets];
        double[] targets = new double[points * incurvas * numSets];

        double[][][] IVsim = new double[numSets][points][incurvas];
        double[][][] vt = new double[numSets][points][incurvas];
        double[][][] vss = new double[numSets][points][incurvas];
        double[][][] qt = new double[numSets][points][incurvas];
        double[][][] ccs = new double[numSets][points][incurvas];
        double[][][] t = new double[numSets][points][incurvas];

        /* Analyze every curve */
        for (int i2 = 0; i2 < incurvas; i2++) {

            /* going forward and backward */
            for (int i3 = 0; i3 < 2; i3++) {

                if (i3 == 0) /* FORDWARD */ {
                    IZ = 0;
                    IS = 1;
                    direction = 1;
                } else /* BAKWARD */ {
                    IZ = points - 1;
                    direction = -1; /* para el bakward cambiamos la direccion ya que retrocedemos */
                    IS = -1;
                }

                /* Iterate for all VDS values */
                for (int i1 = IZ; (i1 >= 0) && (i1 < points); i1 += IS) {

                    //System.out.println("i1: "+ i1 + " i2: " + i2 + " i3: " + i3);
                    //System.out.println("i1: "+ i1 + " i2: " + i2 + " i3: " + i3);

                    if (i3 == 0) {
                        if (i1 == 0){
                            IA = IAA[i2];
                            IB = 0;
                        }
                        else if ((IVexp[i3][i1][i2] == 0) ||
                                ((IVexp[i3][i1 - 1][i2] == 0) && (IVexp[i3][i1][i2] != 0)) ||
                                (IVsim[i3][i1 - 1][i2] == 0) ){
                            IA = IAA[i2];
                            IB = 0;
                        } else {
                            IA = IVsim[i3][i1 - 1][i2] / 2; // I assign the value of the previous point
                            IB = IVsim[i3][i1 - 1][i2] * 2; // I assign the value of the previous point
                        }
                    } else /* BACKWARD */ {
                        if (i1 == points - 1) {
                            IA = IVsim[i3 - 1][i1 - 1][i2] / 4; // I assign the value of the previous point
                            IB = IVsim[i3 - 1][i1 - 1][i2] * 4; // I assign the value of the previous point
                        } else if (i1 <= 1) {
                            IA = IAA[i2];
                            IB = 0;
                        } else //((i1 < (points - 1)) && (i1>1))
                        {
                            IA = IVsim[i3][i1 + 1][i2] / 2; // I assign the value of the previous point
                            IB = IVsim[i3][i1 + 1][i2] * 2; // I assign the value of the previous point
                        }
                    }

                    /* Take into consideration thold - New VG value */
                    if ((i1 == 0) && (i3 == 0)) {

                        vt[i3][i1][i2] = vt_t0;
                        vss[i3][i1][i2] = vss_t0;
                        ccs[i3][i1][i2] = contactConductance(alphaS, vss[i3][i1][i2], VGS[i2], vt[i3][i1][i2], gamma);
                        IVsim[i3][i1][i2] = busquedaceros(IA, IB, ccs[i3][i1][i2], VGS[i2], vt[i3][i1][i2], VDS[i1], gamma, mu0, W, L, Cox, ms, vss[i3][i1][i2]);
                        qt[i3][i1][i2] = -QA;
                        //System.out.println("Aquí llego 1");

                    } else if ((i3 == 1) && (i1 == points - 1)) {
                        /* cuando pasamos de fordward a backward queremos en el primer punto del backward la tension
                         umbral, la corriente y la carga atrapada sea todo lo mismo que el ultimo punto del VGSforward */
                        vt[i3][i1][i2] = vt[i3 - 1][i1][i2];
                        vss[i3][i1][i2] = vss[i3 - 1][i1][i2];
                        ccs[i3][i1][i2] = ccs[i3 - 1][i1][i2];
                        IVsim[i3][i1][i2] = IVsim[i3 - 1][i1][i2];
                        qt[i3][i1][i2] = qt[i3 - 1][i1][i2];
                        t[i3][i1][i2] = 0;

                    } else {

                        //System.out.println("Aquí llego 6");

                        /* initially previous values are taken */
                        vt[i3][i1][i2] = vt[i3][i1 - direction][i2];
                        vss[i3][i1][i2] = vss[i3][i1 - direction][i2];
                        ccs[i3][i1][i2] = ccs[i3][i1 - direction][i2];

                        t[i3][i1][i2] = tmeasure;
                        /* Update current */
                        IVsimB = busquedaceros(IA, IB, ccs[i3][i1][i2], VGS[i2], vt[i3][i1][i2], VDS[i1], gamma, mu0, W, L, Cox, ms, vss[i3][i1][i2]);

                        //System.out.println("Aquí llego2");

                        iterations = 0;

                        do {

                            IVsimA = IVsimB;
                            Q_measure = qt[i3][i1 - direction][i2] * Math.exp(-t[i3][i1][i2] / tao) + beta[i1][i3] * IVsimA * tao * (1 - Math.exp(-t[i3][i1][i2] / tao));
                            vt[i3][i1][i2] = vt[i3][i1 - direction][i2] - ((Q_measure - qt[i3][i1 - direction][i2]) / Cox);
                            vss[i3][i1][i2] = vss[i3][i1 - direction][i2] - (0.02586 * Math.log(10) * (Q_measure - qt[i3][i1 - direction][i2]) / Cox);
                            ccs[i3][i1][i2] = contactConductance(alphaS, vss[i3][i1][i2], VGS[i2], vt[i3][i1][i2], gamma);
                            IVsimB = busquedaceros(IA, IB, ccs[i3][i1][i2], VGS[i2], vt[i3][i1][i2], VDS[i1], gamma, mu0, W, L, Cox, ms, vss[i3][i1][i2]);
                            iterations++;

                        } while ((Math.abs((IVsimB - IVsimA) / IVsimB) > 0.001) && (iterations < 1000));
                        //System.out.println("Aquí llego 3");

                        if (iterations >= 1000) {
                            //System.out.println("Se han excedido las iteraciones");
                            throw new ParameterSetNoConvergence("outputCharacteristics: Solution not found for tmeasure");
                        }

                        IVsim[i3][i1][i2] = IVsimB;
                        /* update trapped charge value at tmeasure */
                        qt[i3][i1][i2] = Q_measure;

                    }

                    if (qt[i3][i1][i2] > 0) {
                        throw new ParameterSetNoConvergence("outputCharacteristics: Q>0");
                    }

                    //System.out.println("Qt: " + qt[i3][i1][i2] + " -- Beta: " + beta[i3]);
                    //System.out.println("IVexp: " + IVexp[i3][i1][i2] + " -- IVsim:" + IVsim[i3][i1][i2]);

                    targets[n] = IVexp[i3][i1][i2];
                    outputs[n] = IVsim[i3][i1][i2];
                    n++;

                    //System.out.println("Aquí llego 5");

                }
            }
        }

        //plotMatrixt;
        //plotMatrix(outputs, targets);
        /* Fitting error */
        return Fitness.NRMSE(targets, outputs);
    }


    public static double IdVd(double[][][] IVexp,
                              double VGS,
                              double[] VDS,
                              double[] IAA,
                              double W,
                              double L,
                              double Cox,
                              double mu0,
                              double gamma,
                              double alphaS,
                              double vt_t0,
                              double vss_t0,
                              double ms,
                              double tao,
                              double QA,
                              double[] alphaQ,
                              double thold,
                              double tmeasure) {

        int points, incurvas, numSets;
        points = VDS.length;
        incurvas = 1;
        numSets = 1;
        double IA, IB, Q_measure, Q_delay, IVsimA, IVsimB;
        int n = 0;
        int IZ, IS, direction;
        int iterations;

        double[] outputs = new double[points * incurvas * numSets];
        double[] targets = new double[points * incurvas * numSets];

        double[][][] IVsim = new double[numSets][points][incurvas];
        double[][][] vt = new double[numSets][points][incurvas];
        double[][][] vss = new double[numSets][points][incurvas];
        double[][][] qt = new double[numSets][points][incurvas];
        double[][][] ccs = new double[numSets][points][incurvas];
        double[][][] t = new double[numSets][points][incurvas];

        /* Analyze every curve */
        for (int i2 = 0; i2 < incurvas; i2++) {

            /* going forward and backward */
            for (int i3 = 0; i3 < 2; i3++) {

                if (i3 == 0) /* FORDWARD */ {
                    IZ = 0;
                    IS = 1;
                    direction = 1;
                } else /* BAKWARD */ {
                    IZ = points - 1;
                    direction = -1; /* para el bakward cambiamos la direccion ya que retrocedemos */
                    IS = -1;
                }

                /* Iterate for all VDS values */
                for (int i1 = IZ; (i1 >= 0) && (i1 < points); i1 += IS) {
                    //System.out.println("i1: "+ i1 + " i2: " + i2 + " i3: " + i3);
                    if (i3 == 0) {
                        if (i1 <= 1) {
                            IA = IAA[i2];
                            IB = 0;
                        } else {
                            IA = IVsim[i3][i1 - 1][i2] / 4; // I assign the value of the previous point
                            IB = IVsim[i3][i1 - 1][i2] * 4; // I assign the value of the previous point
                        }
                    } else /* BACKWARD */ {
                        if (i1 == points - 1) {
                            IA = IVsim[i3 - 1][i1 - 1][i2] / 4; // I assign the value of the previous point
                            IB = IVsim[i3 - 1][i1 - 1][i2] * 4; // I assign the value of the previous point
                        } else if (i1 <= 1) {
                            IA = IAA[i2];
                            IB = 0;
                        } else //((i1 < (points - 1)) && (i1>1))
                        {
                            IA = IVsim[i3][i1 + 1][i2] / 4; // I assign the value of the previous point
                            IB = IVsim[i3][i1 + 1][i2] * 4; // I assign the value of the previous point
                        }
                    }

                    /* Take into consideration thold - New VG value */
                    if ((i1 == 0) && (i3 == 0)) {
                        /* First point of each curve - VDS = 0 -> thold */
                        t[i3][i1][i2] = thold;

                        /* First curve. Initial charge is Qa */ /* t=0 Unknown trapped charge */
                        vt[i3][i1][i2] = vt_t0;
                        vss[i3][i1][i2] = vss_t0;
                        ccs[i3][i1][i2] = contactConductance(alphaS, vss[i3][i1][i2], VGS, vt[i3][i1][i2], gamma);
                        IVsim[i3][i1][i2] = busquedaceros(IA, IB, ccs[i3][i1][i2], VGS, vt[i3][i1][i2], VDS[i1], gamma, mu0, W, L, Cox, ms, vss[i3][i1][i2]);
                        qt[i3][i1][i2] = -QA;


                    } else if ((i3 == 1) && (i1 == points - 1)) {

                        /* cuando pasamos de fordward a backward queremos en el primer punto del backward la tension
                         umbral, la corriente y la carga atrapada sea todo lo mismo que el ultimo punto del VGSforward */
                        vt[i3][i1][i2] = vt[i3 - 1][i1][i2];
                        vss[i3][i1][i2] = vss[i3 - 1][i1][i2];
                        ccs[i3][i1][i2] = ccs[i3 - 1][i1][i2];
                        IVsim[i3][i1][i2] = IVsim[i3 - 1][i1][i2];
                        qt[i3][i1][i2] = qt[i3 - 1][i1][i2];
                        t[i3][i1][i2] = 0;

                    } else {

                        t[i3][i1][i2] = tmeasure;
                        /* Update current */
                        IVsimB = busquedaceros(IA, IB, ccs[i3][i1][i2], VGS, vt[i3][i1][i2], VDS[i1], gamma, mu0, W, L, Cox, ms, vss[i3][i1][i2]);

                        iterations = 0;

                        do {

                            IVsimA = IVsimB;
                            Q_measure = qt[i3][i1 - direction][i2] * Math.exp(-t[i3][i1][i2] / tao) + alphaQ[i3] * IVsimA * tao * (1 - Math.exp(-t[i3][i1][i2] / tao));
                            vt[i3][i1][i2] = vt[i3][i1 - direction][i2] - ((Q_measure - qt[i3][i1 - direction][i2]) / Cox);
                            vss[i3][i1][i2] = vss[i3][i1 - direction][i2] - (0.02586 * Math.log(10) * (Q_measure - qt[i3][i1 - direction][i2]) / Cox);
                            ccs[i3][i1][i2] = contactConductance(alphaS, vss[i3][i1][i2], VGS, vt[i3][i1][i2], gamma);
                            IVsimB = busquedaceros(IA, IB, ccs[i3][i1][i2], VGS, vt[i3][i1][i2], VDS[i1], gamma, mu0, W, L, Cox, ms, vss[i3][i1][i2]);
                            iterations++;

                        } while ((Math.abs((IVsimB - IVsimA) / IVsimB) > 0.001) && (iterations < 1000));

                        if (iterations >= 1000) {
                            throw new ParameterSetNoConvergence("outputCharacteristics: Solution not found for tmeasure");
                        }

                        /* update trapped charge value at tmeasure */
                        qt[i3][i1][i2] = Q_measure;

                    }

                    if (qt[i3][i1][i2] >= 0) {
                        throw new ParameterSetNoConvergence("outputCharacteristics: Q>=0");
                    }

                    targets[n] = IVexp[i3][i1][i2];
                    outputs[n] = IVsim[i3][i1][i2];
                    n++;
                }
            }
        }

        //plotMatrixt;
        //plotMatrix(outputs, targets);
        /* Fitting error */
        return Fitness.NRMSE(targets, outputs);
    }

    public static double IdVd_All(double[][][] IVexp,
                                  double[] VGS,
                                  double[] VDS,
                                  double[] IAA,
                                  double W,
                                  double L,
                                  double Cox,
                                  double mu0,
                                  double gamma,
                                  double alphaS,
                                  double VT,
                                  double VSS0,
                                  double ms,
                                  double tao,
                                  double QA,
                                  double[] alphaQ,
                                  double[] tt,
                                  double[] FactorQA) {

        int points, incurvas, numSets;
        points = IVexp[1].length;
        incurvas = IVexp[0][1].length;
        numSets = IVexp.length;

        double IA, IB, Q, IVsimA, IVsimB;
        int n = 0;
        int IZ, IS, direction;
        double speed;
        int iterations;

        double[] outputs = new double[points * incurvas * numSets];
        double[] targets = new double[points * incurvas * numSets];

        double[][][] IVsim = new double[numSets][points][incurvas];
        double[][][] vt = new double[numSets][points][incurvas];
        double[][][] vss = new double[numSets][points][incurvas];
        double[][][] qt = new double[numSets][points][incurvas];
        double[][][] ccs = new double[numSets][points][incurvas];
        double[][][] t = new double[numSets][points][incurvas];

        /* Analyze every curve */
        for (int i2 = 0; i2 < incurvas; i2++) {

            /* going forward and backward */
            for (int i3 = 0; i3 < 2; i3++) {

                if (i3 == 0) /* FORDWARD */ {
                    IZ = 0;
                    IS = 1;
                    direction = 1;
                } else /* BAKWARD */ {
                    IZ = points - 1;
                    direction = -1; /* para el bakward cambiamos la direccion ya que retrocedemos */
                    IS = -1;
                }

                /* Iterate for all VDS values */
                for (int i1 = IZ; i1 >= 0 && i1 < points; i1 += IS) {
                    //System.out.println("i1: "+ i1 + " i2: " + i2 + " i3: " + i3);

                    if (i3 == 0) {
                        if (i1 > 1) {
                            IA = IVsim[i3][i1 - 1][i2] / 2; // I assign the value of the previous point
                            IB = IVsim[i3][i1 - 1][i2] * 2; // I assign the value of the previous point
                        } else {
                            IA = IAA[i2];
                            IB = 0;
                        }
                        if (i1 == 0) {
                            speed = tt[2]; // Ucurum = 187.5;
                        } else {
                            speed = tt[1]; // Ucurum = 1;
                        }
                    } else /* BACKWARD */ {
                        if ((i1 < (points - 1)) && (i1 > 1)) {
                            IA = IVsim[i3][i1 + 1][i2] / 2; // I assign the value of the previous point
                            IB = IVsim[i3][i1 + 1][i2] * 2; // I assign the value of the previous point
                            speed = tt[1]; // Ucurum = 1;
                        } else if (i1 == points - 1) {
                            IA = IVsim[i3 - 1][i1 - 1][i2] / 2; // I assign the value of the previous point
                            IB = IVsim[i3 - 1][i1 - 1][i2] * 2; // I assign the value of the previous point
                            speed = tt[0]; // Ucurum = 0;
                        } else {
                            IA = IAA[i2];
                            IB = 0;
                            speed = tt[1]; // Ucurum = 1;
                        }
                    }

                    t[i3][i1][i2] = speed;
                    if ((i1 == 0) && (i3 == 0)) /* First point of every curve in forward direction */ {

                        if (i2 == 0)  /* First curve. Initial charge is Qa */ {
                            vt[i3][i1][i2] = VT;
                            vss[i3][i1][i2] = VSS0;
                            ccs[i3][i1][i2] = contactConductance(alphaS, vss[i3][i1][i2], VGS[i2], vt[i3][i1][i2], gamma);
                            IVsim[i3][i1][i2] = busquedaceros(IA, IB, ccs[i3][i1][i2], VGS[i2], vt[i3][i1][i2], VDS[i1], gamma, mu0, W, L, Cox, ms, vss[i3][i1][i2]);
                            qt[i3][i1][i2] = -QA;
                        } else /* There is already charge */ {
                            qt[i3][i1][i2] = qt[i3 + 1][i1][i2 - 1] * Math.exp(-t[i3][i1][i2] / tao) - FactorQA[i2] * QA * (1 - Math.exp(-t[i3][i1][i2] / tao));
                            vt[i3][i1][i2] = vt[i3 + 1][i1][i2 - 1] - ((qt[i3][i1][i2] - qt[i3 + 1][i1][i2 - 1]) / Cox);
                            vss[i3][i1][i2] = vss[i3 + 1][i1][i2 - 1] - (0.02586 * Math.log(10) * (qt[i3][i1][i2] - qt[i3 + 1][i1][i2 - 1]) / Cox);
                            ccs[i3][i1][i2] = ccs[i3 + 1][i1][i2 - 1];
                            IVsim[i3][i1][i2] = IVsim[i3 + 1][i1][i2 - 1];
                        }

                    } else if ((i3 == 1) && (i1 == points - 1)) {
                        /* cuando pasamos de fordward a backward queremos en el primer punto del backward la tension
                         umbral, la corriente y la carga atrapada sea todo lo mismo que el ultimo punto del VGSforward */
                        vt[i3][i1][i2] = vt[i3 - 1][i1][i2];
                        vss[i3][i1][i2] = vss[i3 - 1][i1][i2];
                        ccs[i3][i1][i2] = ccs[i3 - 1][i1][i2];
                        IVsim[i3][i1][i2] = IVsim[i3 - 1][i1][i2];
                        qt[i3][i1][i2] = qt[i3 - 1][i1][i2];
                    } else {
                        /* en cualquier otro caso la tension umbral (en principio la misma q la anterior) */
                        vt[i3][i1][i2] = vt[i3][i1 - direction][i2];
                        vss[i3][i1][i2] = vss[i3][i1 - direction][i2];

                        ccs[i3][i1][i2] = contactConductance(alphaS, ccs[i3][i1][i2], VGS[i2], vt[i3][i1][i2], gamma);

                        /* Y evaluamos la corriente con esa tension umbral */
                        IVsimA = busquedaceros(IA, IB, ccs[i3][i1][i2], VGS[i2], vt[i3][i1][i2], VDS[i1], gamma, mu0, W, L, Cox, ms, vss[i3][i1][i2]);

                        /* Qe carga atrapada tendriamos con esa corriente? */
                        Q = qt[i3][i1 - direction][i2] * Math.exp(-t[i3][i1][i2] / tao) + alphaQ[i3] * IVsimA * tao * (1 - Math.exp(-t[i3][i1][i2] / tao));

                        /* y por lo tanto que tensio umbral? */
                        vt[i3][i1][i2] = vt[i3][i1 - direction][i2] - ((Q - qt[i3][i1 - direction][i2]) / Cox); /* gamma1*Q */

                        vss[i3][i1][i2] = vss[i3][i1 - direction][i2] - (0.02586 * Math.log(10) * (Q - qt[i3][i1 - direction][i2]) / Cox);

                        ccs[i3][i1][i2] = contactConductance(alphaS, vss[i3][i1][i2], VGS[i2], vt[i3][i1][i2], gamma);

                        /* Volvemos a evaluar la corriente con esta nueva tensio umbral */
                        IVsimB = busquedaceros(IA, IB, ccs[i3][i1][i2], VGS[i2], vt[i3][i1][i2], VDS[i1], gamma, mu0, W, L, Cox, ms, ccs[i3][i1][i2]);

                        iterations = 0;

                        /* Y repetimos el proceso hasta que tengamos un error menor del 1 */
                        while ((Math.abs((IVsimB - IVsimA) / IVsimB) > 0.001) && (iterations < 1000)) {
                            IVsimA = IVsimB;
                            Q = qt[i3][i1 - direction][i2] * Math.exp(-t[i3][i1][i2] / tao) + alphaQ[i3] * IVsimA * tao * (1 - Math.exp(-t[i3][i1][i2] / tao));
                            vt[i3][i1][i2] = vt[i3][i1 - direction][i2] - ((Q - qt[i3][i1 - direction][i2]) / Cox); /* gamma1*Q */
                            vss[i3][i1][i2] = vss[i3][i1 - direction][i2] - (0.02586 * Math.log(10) * (Q - qt[i3][i1 - direction][i2]) / Cox);
                            ccs[i3][i1][i2] = contactConductance(alphaS, vss[i3][i1][i2], VGS[i2], vt[i3][i1][i2], gamma);
                            IVsimB = busquedaceros(IA, IB, ccs[i3][i1][i2], VGS[i2], vt[i3][i1][i2], VDS[i1], gamma, mu0, W, L, Cox, ms, vss[i3][i1][i2]);
                            iterations++;
                        }

                        if (iterations >= 1000) {
                            throw new ParameterSetNoConvergence("outputCharacteristics: Solution not found");
                        }

                        /* Cuando tenemos menos error del 1% actualizamos el valor de la corriente en ese punto asi como la
                        carga atrapada */
                        IVsim[i3][i1][i2] = IVsimB;
                        qt[i3][i1][i2] = Q;
                    }

                    if (qt[i3][i1][i2] >= 0) {
                        throw new ParameterSetNoConvergence("outputCharacteristics: Q>=0");
                    }

                    targets[n] = IVexp[i3][i1][i2];
                    outputs[n] = IVsim[i3][i1][i2];
                    n++;
                }
            }
        }

        //plotMatrix(MS);
        //plotMatrix(outputs, targets);
        /* Fitting error */
        return Fitness.NRMSE(targets, outputs);
    }

    public static double IdVg_All(double[][] IVexp, double[] VGS, double VDS, double IAA, double W, double L, double Cox, double mu0, double gamma, double alphaS, double VT, double VSS0, double ms, double tao, double[] alphaQ, double QA, double[] tt, double[] Ioff, double m2) {

        int numsets, incurvas;
        numsets = IVexp[0].length;
        incurvas = IVexp.length;

        double IA, IB, Q, IVsimA, IVsimB;
        int n = 0;
        int IZ, direction;
        double speed;
        int iterations;

        double[] outputs = new double[numsets * incurvas];
        double[] targets = new double[numsets * incurvas];

        double[][] IVsim = new double[incurvas][numsets];
        double[][] IVslope = new double[incurvas][numsets];
        double[][] VTH = new double[incurvas][numsets];
        double[][] Vss = new double[incurvas][numsets];
        double[][] Qt = new double[incurvas][numsets];
        double[][] MS = new double[incurvas][numsets];
        double[][] t = new double[incurvas][numsets];

        for (int i3 = 0; i3 < numsets; i3++) {
            if (i3 == 0) /* FORDWARD */ {
                IZ = 0;
                direction = 1;
            } else /* BAKWARD */ {
                IZ = incurvas - 1;
                direction = -1; /* para el bakward cambiamos la direccion ya que retrocedemos */
            }

            /* Iterate for all VGS values */
            for (int i1 = IZ; i1 >= 0 && i1 < incurvas; i1 += direction) {
                if (i3 == 0) {
                    if (i1 > 0) {
                        IA = IVsim[i1 - 1][i3] / 2; // I assign the value of the previous point
                        IB = IVsim[i1 - 1][i3] * 2; // I assign the value of the previous point
                        speed = tt[1]; // Ucurum 1
                    } else {
                        IA = IAA;
                        IB = 0;
                        speed = tt[2]; // Ucurum 187.5;
                    }
                } else /* BACKWARD */ {
                    if ((i1 < (incurvas - 1)) && (i1 > 1)) {
                        IA = IVsim[i1 + 1][i3] / 2; // I assign the value of the previous point
                        IB = IVsim[i1 + 1][i3] * 2; // I assign the value of the previous point
                        speed = tt[1]; // Ucurum 1;
                    } else if (i1 == incurvas - 1) {
                        IA = IVsim[i1 - 1][i3] / 2; // I assign the value of the previous point
                        IB = IVsim[i1 - 1][i3] * 2; // I assign the value of the previous point
                        speed = tt[0]; // Ucurum = 0;
                    } else {
                        IA = IAA;
                        IB = 0;
                        speed = tt[1]; // Ucurum 1;
                    }
                }

                t[i1][i3] = speed;
                //System.out.println("i3: "+ i3 + "i1: "+i1+ " V_G: "+ VGS[i1] + "  I_D_EXP:" +IVexp[i1][i3]);

                if ((i1 == 0) && (i3 == 0)) {
                    VTH[i1][i3] = VT;
                    Vss[i1][i3] = VSS0;
                    MS[i1][i3] = contactConductance(alphaS, Vss[i1][i3], VGS[i1], VTH[i1][i3], gamma);
                    IVsim[i1][i3] = busquedaceros(IA, IB, MS[i1][i3], VGS[i1], VTH[i1][i3], VDS, gamma, mu0, W, L, Cox, ms, Vss[i1][i3]);
                    Qt[i1][i3] = -QA;
                } else if ((i3 == 1) && (i1 == incurvas - 1)) {
                    VTH[i1][i3] = VTH[i1][i3 - 1];
                    Vss[i1][i3] = Vss[i1][i3 - 1];
                    MS[i1][i3] = MS[i1][i3 - 1];
                    IVsim[i1][i3] = IVsim[i1][i3 - 1];
                    Qt[i1][i3] = Qt[i1][i3 - 1];
                } else {
                    VTH[i1][i3] = VTH[i1 - direction][i3];
                    Vss[i1][i3] = Vss[i1 - direction][i3];
                    MS[i1][i3] = contactConductance(alphaS, Vss[i1][i3], VGS[i1], VTH[i1][i3], gamma);
                    IVsimA = busquedaceros(IA, IB, MS[i1][i3], VGS[i1], VTH[i1][i3], VDS, gamma, mu0, W, L, Cox, ms, Vss[i1][i3]);

                    Q = ((Qt[i1 - direction][i3]) * Math.exp(-t[i1][i3] / tao)) + (alphaQ[i3] * IVsimA * tao * (1 - Math.exp(-t[i1][i3] / tao)));
                    Vss[i1][i3] = Vss[i1 - direction][i3] - (0.02586 * Math.log(10) * (Q - Qt[i1 - direction][i3]) / Cox);
                    VTH[i1][i3] = VTH[i1 - direction][i3] - ((Q - Qt[i1 - direction][i3]) / Cox);
                    MS[i1][i3] = contactConductance(alphaS, Vss[i1][i3], VGS[i1], VTH[i1][i3], gamma);
                    IVsimB = busquedaceros(IA, IB, MS[i1][i3], VGS[i1], VTH[i1][i3], VDS, gamma, mu0, W, L, Cox, ms, Vss[i1][i3]);

                    iterations = 0;
                    while ((Math.abs((IVsimB - IVsimA) / IVsimB) > 0.001) && iterations < 1000) {
                        iterations++;
                        IVsimA = IVsimB;
                        Q = ((Qt[i1 - direction][i3]) * Math.exp(-t[i1][i3] / tao)) + (alphaQ[i3] * IVsimA * tao * (1 - Math.exp(-t[i1][i3] / tao)));
                        Vss[i1][i3] = Vss[i1 - direction][i3] - (0.02586 * Math.log(10) * (Q - Qt[i1 - direction][i3]) / Cox);
                        VTH[i1][i3] = VTH[i1 - direction][i3] - ((Q - Qt[i1 - direction][i3]) / Cox);
                        MS[i1][i3] = contactConductance(alphaS, Vss[i1][i3], VGS[i1], VTH[i1][i3], gamma);
                        IVsimB = busquedaceros(IA, IB, MS[i1][i3], VGS[i1], VTH[i1][i3], VDS, gamma, mu0, W, L, Cox, ms, Vss[i1][i3]);
                    }
                    IVsim[i1][i3] = IVsimB;
                    Qt[i1][i3] = Q;

                    if (iterations >= 1000) {
                        throw new ParameterSetNoConvergence("iterations >= 1000");
                    }
                }

                if (Qt[i1][i3] > 0) {
                    throw new ParameterSetNoConvergence("transferCharacteristics: Q>0");
                }

                // Flat the curve at low VG values
                IVslope[i1][i3] = Ioff[i3] * Math.pow(1 + Math.pow(IVsim[i1][i3] / Ioff[i3], m2), 1 / m2);

                if ((infCurrent(IVslope[i1][i3]) == true) || (nanCurrent(IVslope[i1][i3]) == true)) {
                    throw new ParameterSetNoConvergence("IdVg: Solution not found");
                }
                // System.out.println("i3: "+i3+ "i1: "+i1+ " V_G: "+ VGS[i1] + "  I_D_SIM:" +IVsim[i1][i3]+ "  I_D_EXP:" +IVexp[i1][i3]);
                targets[n] = IVexp[i1][i3];
                outputs[n] = IVslope[i1][i3];
                n++;
            }
        }
        //plotMatrix(IVsim);
        return Fitness.NRMSE(targets, outputs);
    }

    public static double IdVg(double[][] IVexp,
                              double[] VGS,
                              double VDS,
                              double []IAA,
                              double W,
                              double L,
                              double Cox,
                              double mu0,
                              double gamma,
                              double alphaS,
                              double VT,
                              double VSS0,
                              double ms,
                              double tao,
                              double[] alphaQ,
                              double QA,
                              double[] Ioff,
                              double m2,
                              double tdelay,
                              double tmeasure) {

        int numsets, incurvas;
        numsets = IVexp[0].length;
        incurvas = IVexp.length;

        double IA, IB, Q, IVsimA, IVsimB;
        int n = 0;
        int IZ, direction;
        int iterations;

        double[] outputs = new double[numsets * incurvas];
        double[] targets = new double[numsets * incurvas];

        double[][] IVsim = new double[incurvas][numsets];
        double[][] IVslope = new double[incurvas][numsets];
        double[][] vt = new double[incurvas][numsets];
        double[][] vss = new double[incurvas][numsets];
        double[][] qt = new double[incurvas][numsets];
        double[][] qt_delay = new double[incurvas][numsets];
        double[][] ccs = new double[incurvas][numsets];
        double[][] t = new double[incurvas][numsets];

        for (int i3 = 0; i3 < numsets; i3++) {
            if (i3 == 0) /* FORDWARD */ {
                IZ = 0;
                direction = 1;
            } else /* BAKWARD */ {
                IZ = incurvas - 1;
                direction = -1; /* para el bakward cambiamos la direccion ya que retrocedemos */
            }

            /* Iterate for all VGS values */
            for (int i1 = IZ; i1 >= 0 && i1 < incurvas; i1 += direction) {
                if (i3 == 0) {
                    if (i1 > 1) {
                        IA = IVsim[i1 - 1][i3] / 10; // I assign the value of the previous point
                        IB = IVsim[i1 - 1][i3] * 10; // I assign the value of the previous point
                    } else {
                        IA = IAA[0];
                        IB = IAA[1];
                    }
                } else /* BACKWARD */ {
                    if ((i1 < (incurvas - 1)) && (i1 > 1)) {
                        IA = IVsim[i1 + 1][i3] / 10; // I assign the value of the previous point
                        IB = IVsim[i1 + 1][i3] * 10; // I assign the value of the previous point
                    } else if (i1 == incurvas - 1) {
                        IA = IVsim[i1 - 1][i3] / 10; // I assign the value of the previous point
                        IB = IVsim[i1 - 1][i3] * 10; // I assign the value of the previous point
                    } else {
                        IA = IAA[0];
                        IB = IAA[1];
                    }
                }

                //System.out.println("i3: "+ i3 + "i1: "+i1+ " V_G: "+ VGS[i1] + "  I_D_EXP:" +IVexp[i1][i3]+ " IA: "+ IA + "IB: "+ IB);
                if ((i1 == 0) && (i3 == 0)) {
                    vt[i1][i3] = VT;
                    vss[i1][i3] = VSS0;
                    ccs[i1][i3] = contactConductance(alphaS, vss[i1][i3], VGS[i1], vt[i1][i3], gamma);
                    IVsim[i1][i3] = busquedaceros(IA, IB, ccs[i1][i3], VGS[i1], vt[i1][i3], VDS, gamma, mu0, W, L, Cox, ms, vss[i1][i3]);
                    qt[i1][i3] = -QA;
                    t[i1][i3] = 0;
                } else if ((i3 == 1) && (i1 == incurvas - 1)) {
                    vt[i1][i3] = vt[i1][i3 - 1];
                    vss[i1][i3] = vss[i1][i3 - 1];
                    ccs[i1][i3] = ccs[i1][i3 - 1];
                    IVsim[i1][i3] = IVsim[i1][i3 - 1];
                    qt[i1][i3] = qt[i1][i3 - 1];
                    t[i1][i3] = 0;
                } else {

                    t[i1][i3] = tdelay;
                    vt[i1][i3] = vt[i1 - direction][i3];
                    vss[i1][i3] = vss[i1 - direction][i3];
                    ccs[i1][i3] = contactConductance(alphaS, vss[i1][i3], VGS[i1], vt[i1][i3], gamma);
                    IVsimB = busquedaceros(IA, IB, ccs[i1][i3], VGS[i1], vt[i1][i3], VDS, gamma, mu0, W, L, Cox, ms, vss[i1][i3]);
                    iterations = 0;
                    do {
                        iterations++;
                        IVsimA = IVsimB;

                        Q = ((qt[i1 - direction][i3]) * Math.exp(-t[i1][i3] / tao)) + (alphaQ[i3] * IVsimA * tao * (1 - Math.exp(-t[i1][i3] / tao)));

                        vss[i1][i3] = vss[i1 - direction][i3] - (0.02586 * Math.log(10) * (Q - qt[i1 - direction][i3]) / Cox);
                        vt[i1][i3] = vt[i1 - direction][i3] - ((Q - qt[i1 - direction][i3]) / Cox);
                        ccs[i1][i3] = contactConductance(alphaS, vss[i1][i3], VGS[i1], vt[i1][i3], gamma);
                        IVsimB = busquedaceros(IA, IB, ccs[i1][i3], VGS[i1], vt[i1][i3], VDS, gamma, mu0, W, L, Cox, ms, vss[i1][i3]);
                    } while ((Math.abs((IVsimB - IVsimA) / IVsimB) > 0.001) && iterations < 1000);

                    if (iterations >= 1000)
                    {
                        throw new ParameterSetNoConvergence("iterations >= 1000");
                    }

                    IVsim[i1][i3] = IVsimB;
                    qt_delay[i1][i3] = Q;

                    t[i1][i3] = tmeasure;
                    vt[i1][i3] = vt[i1 - direction][i3];
                    vss[i1][i3] = vss[i1 - direction][i3];
                    ccs[i1][i3] = contactConductance(alphaS, vss[i1][i3], VGS[i1], vt[i1][i3], gamma);
                    IVsimB = busquedaceros(IA, IB, ccs[i1][i3], VGS[i1], vt[i1][i3], VDS, gamma, mu0, W, L, Cox, ms, vss[i1][i3]);
                    iterations = 0;

                    do
                    {
                        iterations++;
                        IVsimA = IVsimB;
                        Q = ((qt[i1 - direction][i3]) * Math.exp(-t[i1][i3] / tao)) + (alphaQ[i3] * IVsimA * tao * (1 - Math.exp(-t[i1][i3] / tao)));
                        vss[i1][i3] = vss[i1 - direction][i3] - (0.02586 * Math.log(10) * (Q - qt[i1 - direction][i3]) / Cox);
                        vt[i1][i3] = vt[i1 - direction][i3] - ((Q - qt[i1 - direction][i3]) / Cox);
                        ccs[i1][i3] = contactConductance(alphaS, vss[i1][i3], VGS[i1], vt[i1][i3], gamma);
                        IVsimB = busquedaceros(IA, IB, ccs[i1][i3], VGS[i1], vt[i1][i3], VDS, gamma, mu0, W, L, Cox, ms, vss[i1][i3]);
                    }
                    while ((Math.abs((IVsimB - IVsimA) / IVsimB) > 0.001) && iterations < 1000);

                    qt[i1][i3] = Q;

                    if (iterations >= 1000)
                    {
                        throw new ParameterSetNoConvergence("iterations >= 1000");
                    }
                }

                if (qt[i1][i3] > 0) {
                    throw new ParameterSetNoConvergence("transferCharacteristics: Q>0");
                }

                // Flat the curve at low VG values
                IVslope[i1][i3] = Ioff[i3] * Math.pow(1 + Math.pow(IVsim[i1][i3] / Ioff[i3], m2), 1 / m2);

                if ((infCurrent(IVslope[i1][i3]) == true) || (nanCurrent(IVslope[i1][i3]) == true)) {
                    throw new ParameterSetNoConvergence("IdVg: Solution not found");
                }
                // System.out.println("i3: "+i3+ "i1: "+i1+ " V_G: "+ VGS[i1] + "  I_D_SIM:" +IVsim[i1][i3]+ "  I_D_EXP:" +IVexp[i1][i3]);
                targets[n] = IVexp[i1][i3];
                outputs[n] = IVslope[i1][i3];
                n++;
            }
        }
        //plotMatrix(IVsim);
        return Fitness.NRMSE(targets, outputs);
    }

    public static double IdVg(double[][] IVexp,
                              double[] iaa,
                              double[] vg,
                              double vd,
                              double vt0,
                              double vss0,
                              double alpha,
                              double ms,
                              double gamma,
                              double mu0,
                              double cox,
                              double w,
                              double l,
                              double va_prima,
                              double tao,
                              double[] alphaQ,
                              double qa,
                              double[] ioff,
                              double m2,
                              double tmeasure)
    {

        int numsets, incurvas;
        numsets = IVexp[0].length;
        incurvas = IVexp.length;

        double IA, IB, Q, IVsimA, IVsimB;
        int n = 0;
        int IZ, direction;
        int iterations;

        double[] outputs = new double[numsets * incurvas];
        double[] targets = new double[numsets * incurvas];

        double[][] IVsim = new double[incurvas][numsets];
        double[][] vt = new double[incurvas][numsets];
        double[][] vss = new double[incurvas][numsets];
        double[][] qt = new double[incurvas][numsets];
        double[][] ccs = new double[incurvas][numsets];
        double[][] t = new double[incurvas][numsets];

        for (int i3 = 0; i3 < numsets; i3++)
        {
            if (i3 == 0) /* FORDWARD */
            {
                IZ = 0;
                direction = 1;
            }
            else /* BAKWARD */
            {
                IZ = incurvas - 1;
                direction = -1; /* para el bakward cambiamos la direccion ya que retrocedemos */
            }

            /* Iterate for all VGS values */
            for (int i1 = IZ; i1 >= 0 && i1 < incurvas; i1 += direction)
            {
                if (i3 == 0)
                {
                    if (i1 > 1)
                    {
                        IA = IVsim[i1 - 1][i3] / 2; // I assign the value of the previous point
                        IB = IVsim[i1 - 1][i3] * 2; // I assign the value of the previous point
                    }
                    else
                    {
                        IA = iaa[0];
                        IB = iaa[1];
                    }
                }
                else /* BACKWARD */
                {
                    if ((i1 < (incurvas - 1)) && (i1 > 1))
                    {
                        IA = IVsim[i1 + 1][i3] / 2; // I assign the value of the previous point
                        IB = IVsim[i1 + 1][i3] * 2; // I assign the value of the previous point
                    }
                    else if (i1 == incurvas - 1)
                    {
                        IA = IVsim[i1 - 1][i3] / 2; // I assign the value of the previous point
                        IB = IVsim[i1 - 1][i3] * 2; // I assign the value of the previous point
                    }
                    else
                    {
                        IA = iaa[0];
                        IB = iaa[1];
                    }
                }

                //System.out.println("i3: "+ i3 + "i1: "+i1+ " V_G: "+ vg[i1] + "  I_D_EXP:" +IVexp[i1][i3]+ " IA: "+ IA + " IB: "+ IB);

                if ((i1 == 0) && (i3 == 0))
                {
                    vt[i1][i3] = vt0;
                    vss[i1][i3] = vss0;
                    ccs[i1][i3] = contactConductance(alpha, vss[i1][i3], vg[i1], vt[i1][i3], gamma);
                    IVsim[i1][i3] = ec.evolutionary_extraction_procedure.models.draincurrentmodel.MarinovJimenezTejada.Subthreshold.busquedaceros(IA, IB, vg[i1], vd, vt[i1][i3], vss[i1][i3], ccs[i1][i3], ms, gamma, mu0, cox, w, l, va_prima);
                    qt[i1][i3] = -qa;
                    t[i1][i3] = 0;
                }
                else if ((i3 == 1) && (i1 == incurvas - 1))
                {
                    vt[i1][i3] = vt[i1][i3 - 1];
                    vss[i1][i3] = vss[i1][i3 - 1];
                    ccs[i1][i3] = ccs[i1][i3 - 1];
                    IVsim[i1][i3] = IVsim[i1][i3 - 1];
                    qt[i1][i3] = qt[i1][i3 - 1];
                    t[i1][i3] = 0;
                }
                else
                {
                    t[i1][i3] = tmeasure;
                    vt[i1][i3] = vt[i1 - direction][i3];
                    vss[i1][i3] = vss[i1 - direction][i3];
                    ccs[i1][i3] = contactConductance(alpha, vss[i1][i3], vg[i1], vt[i1][i3], gamma);
                    IVsimB =  ec.evolutionary_extraction_procedure.models.draincurrentmodel.MarinovJimenezTejada.Subthreshold.busquedaceros(IA, IB, vg[i1], vd, vt[i1][i3], vss[i1][i3], ccs[i1][i3], ms, gamma, mu0, cox, w, l, va_prima);
                    iterations = 0;

                    do
                    {
                        iterations++;
                        IVsimA = IVsimB;
                        Q = ((qt[i1 - direction][i3]) * Math.exp(-t[i1][i3] / tao)) + (alphaQ[i3] * IVsimA * tao * (1 - Math.exp(-t[i1][i3] / tao)));
                        vss[i1][i3] = vss[i1 - direction][i3] - (0.02586 * Math.log(10) * (Q - qt[i1 - direction][i3]) / cox);
                        vt[i1][i3] = vt[i1 - direction][i3] - ((Q - qt[i1 - direction][i3]) / cox);
                        ccs[i1][i3] = contactConductance(alpha, vss[i1][i3], vg[i1], vt[i1][i3], gamma);
                        IVsimB =  ec.evolutionary_extraction_procedure.models.draincurrentmodel.MarinovJimenezTejada.Subthreshold.busquedaceros(IA, IB, vg[i1], vd, vt[i1][i3], vss[i1][i3], ccs[i1][i3], ms, gamma, mu0, cox, w, l, va_prima);
                    }
                    while ((Math.abs((IVsimB - IVsimA) / IVsimB) > 0.001) && iterations < 1000);

                    qt[i1][i3] = Q;
                    IVsim[i1][i3] = IVsimB;

                    if (iterations >= 1000)
                    {
                        throw new ParameterSetNoConvergence("iterations >= 1000");
                    }
                }

                if (qt[i1][i3] > 0)
                {
                    throw new ParameterSetNoConvergence("transferCharacteristics: Q>0");
                }

                if (m2 != 0)
                {
                    // Flat the curve at low VG values
                    IVsim[i1][i3] = ioff[i3] * Math.pow(1 + Math.pow(IVsim[i1][i3] / ioff[i3], m2), 1 / m2);
                }
                //System.out.println("i3: "+i3+ "i1: "+i1+ " V_G: "+ vg[i1] + "  I_D_SIM:" +IVsim[i1][i3]+ "  I_D_EXP:" +IVexp[i1][i3]);

                if ((infCurrent(IVsim[i1][i3]) == true) || (nanCurrent(IVsim[i1][i3]) == true)) {
                    throw new ParameterSetNoConvergence("IdVg: Solution not found");
                }
                targets[n] = IVexp[i1][i3];
                outputs[n] = IVsim[i1][i3];
                n++;
            }
        }
        //plotMatrix(IVsim);
        return Fitness.NRMSE(targets, outputs);
    }

    public static double IdVg(double[][] IVexp,
                              double[] iaa,
                              double[] vg,
                              double vd,
                              double vt0,
                              double vss0,
                              double[][] ccs,
                              double ms,
                              double gamma,
                              double mu0,
                              double cox,
                              double w,
                              double l,
                              double va_prima,
                              double tao,
                              double[] alphaQ,
                              double qa,
                              double[] ioff,
                              double m2,
                              double tmeasure)
    {

        int numsets, incurvas;
        numsets = IVexp[0].length;
        incurvas = IVexp.length;

        double IA, IB, Q, IVsimA, IVsimB;
        int n = 0;
        int IZ, direction;
        int iterations;

        double[] outputs = new double[numsets * incurvas];
        double[] targets = new double[numsets * incurvas];

        double[][] IVsim = new double[incurvas][numsets];
        double[][] vt = new double[incurvas][numsets];
        double[][] vss = new double[incurvas][numsets];
        double[][] qt = new double[incurvas][numsets];
        double[][] t = new double[incurvas][numsets];

        for (int i3 = 0; i3 < numsets; i3++)
        {
            if (i3 == 0) /* FORDWARD */
            {
                IZ = 0;
                direction = 1;
            }
            else /* BAKWARD */
            {
                IZ = incurvas - 1;
                direction = -1; /* para el bakward cambiamos la direccion ya que retrocedemos */
            }

            /* Iterate for all VGS values */
            for (int i1 = IZ; i1 >= 0 && i1 < incurvas; i1 += direction)
            {
                if (i3 == 0)
                {
                    if (i1 > 1)
                    {
                        IA = IVsim[i1 - 1][i3] / 2; // I assign the value of the previous point
                        IB = IVsim[i1 - 1][i3] * 2; // I assign the value of the previous point
                    }
                    else
                    {
                        IA = iaa[0];
                        IB = iaa[1];
                    }
                }
                else /* BACKWARD */
                {
                    if ((i1 < (incurvas - 1)) && (i1 > 1))
                    {
                        IA = IVsim[i1 + 1][i3] / 2; // I assign the value of the previous point
                        IB = IVsim[i1 + 1][i3] * 2; // I assign the value of the previous point
                    }
                    else if (i1 == incurvas - 1)
                    {
                        IA = IVsim[i1 - 1][i3] / 2; // I assign the value of the previous point
                        IB = IVsim[i1 - 1][i3] * 2; // I assign the value of the previous point
                    }
                    else
                    {
                        IA = iaa[0];
                        IB = iaa[1];
                    }
                }

                //System.out.println("i3: "+ i3 + "i1: "+i1+ " V_G: "+ vg[i1] + "  I_D_EXP:" +IVexp[i1][i3]+ " IA: "+ IA + " IB: "+ IB);

                if ((i1 == 0) && (i3 == 0))
                {
                    vt[i1][i3] = vt0;
                    vss[i1][i3] = vss0;
                    IVsim[i1][i3] = ec.evolutionary_extraction_procedure.models.draincurrentmodel.MarinovJimenezTejada.Subthreshold.busquedaceros(IA, IB, vg[i1], vd, vt[i1][i3], vss[i1][i3], ccs[i1][i3], ms, gamma, mu0, cox, w, l, va_prima);
                    qt[i1][i3] = -qa;
                    t[i1][i3] = 0;
                }
                else if ((i3 == 1) && (i1 == incurvas - 1))
                {
                    vt[i1][i3] = vt[i1][i3 - 1];
                    vss[i1][i3] = vss[i1][i3 - 1];
                    ccs[i1][i3] = ccs[i1][i3 - 1];
                    IVsim[i1][i3] = IVsim[i1][i3 - 1];
                    qt[i1][i3] = qt[i1][i3 - 1];
                    t[i1][i3] = 0;
                }
                else
                {
                    t[i1][i3] = tmeasure;
                    vt[i1][i3] = vt[i1 - direction][i3];
                    vss[i1][i3] = vss[i1 - direction][i3];
                    IVsimB =  ec.evolutionary_extraction_procedure.models.draincurrentmodel.MarinovJimenezTejada.Subthreshold.busquedaceros(IA, IB, vg[i1], vd, vt[i1][i3], vss[i1][i3], ccs[i1][i3], ms, gamma, mu0, cox, w, l, va_prima);
                    iterations = 0;

                    do
                    {
                        iterations++;
                        IVsimA = IVsimB;
                        Q = ((qt[i1 - direction][i3]) * Math.exp(-t[i1][i3] / tao)) + (alphaQ[i3] * IVsimA * tao * (1 - Math.exp(-t[i1][i3] / tao)));
                        vss[i1][i3] = vss[i1 - direction][i3] - (0.02586 * Math.log(10) * (Q - qt[i1 - direction][i3]) / cox);
                        vt[i1][i3] = vt[i1 - direction][i3] - ((Q - qt[i1 - direction][i3]) / cox);
                        IVsimB =  ec.evolutionary_extraction_procedure.models.draincurrentmodel.MarinovJimenezTejada.Subthreshold.busquedaceros(IA, IB, vg[i1], vd, vt[i1][i3], vss[i1][i3], ccs[i1][i3], ms, gamma, mu0, cox, w, l, va_prima);
                    }
                    while ((Math.abs((IVsimB - IVsimA) / IVsimB) > 0.001) && iterations < 1000);

                    qt[i1][i3] = Q;
                    IVsim[i1][i3] = IVsimB;

                    if (iterations >= 1000)
                    {
                        throw new ParameterSetNoConvergence("iterations >= 1000");
                    }
                }

                if (qt[i1][i3] > 0)
                {
                    throw new ParameterSetNoConvergence("transferCharacteristics: Q>0");
                }

                if (m2 != 0)
                {
                    // Flat the curve at low VG values
                    IVsim[i1][i3] = ioff[i3] * Math.pow(1 + Math.pow(IVsim[i1][i3] / ioff[i3], m2), 1 / m2);
                }
                //System.out.println("i3: "+i3+ "i1: "+i1+ " V_G: "+ vg[i1] + "  I_D_SIM:" +IVsim[i1][i3]+ "  I_D_EXP:" +IVexp[i1][i3]);

                if ((infCurrent(IVsim[i1][i3]) == true) || (nanCurrent(IVsim[i1][i3]) == true)) {
                    throw new ParameterSetNoConvergence("IdVg: Solution not found");
                }
                targets[n] = IVexp[i1][i3];
                outputs[n] = IVsim[i1][i3];
                n++;
            }
        }
        //plotMatrix(IVsim);
        return Fitness.NRMSE(targets, outputs);
    }

    public static double IdVg(double[][] IVexp,
                              double[] iaa,
                              double[] vg,
                              double vd,
                              double vt0_IdVg_ini,
                              double vt0_IdVg_fin,
                              double vss0,
                              double[][] ccs,
                              double ms,
                              double gamma,
                              double mu0,
                              double cox,
                              double w,
                              double l,
                              double va_prima,
                              double tao,
                              double[] alphaQ,
                              double[] ioff,
                              double m2,
                              double tmeasure,
                              double time_for_trap_2,
                              double tao2)
    {

        int numsets, incurvas;
        numsets = IVexp[0].length;
        incurvas = IVexp.length;

        double A = vt0_IdVg_fin;
        double B = vt0_IdVg_ini - A;
        double q0 = -B * cox;

        //System.out.println("A: "+ A + "\nB: "+B + "\nvt0_IdVg_fin: "+ vt0_IdVg_fin + "\nvt0_IdVg_ini: "+ vt0_IdVg_ini + "\ntrap_duration_time: "+ trap_duration_time + "\ntao2: " + tao2 + "\n\n\n");

        double IA, IB, Q, IVsimA, IVsimB;
        int n = 0;
        int IZ, direction;
        int iterations;

        double[] outputs = new double[numsets * incurvas];
        double[] targets = new double[numsets * incurvas];

        double[][] IVsim = new double[incurvas][numsets];
        double[][] vt = new double[incurvas][numsets];
        double[][] vss = new double[incurvas][numsets];
        double[][] qt = new double[incurvas][numsets];
        double[][] t = new double[incurvas][numsets];
        double[][] time = new double[incurvas][numsets];

        for (int i3 = 0; i3 < numsets; i3++)
        {
            if (i3 == 0) /* FORDWARD */
            {
                IZ = 0;
                direction = 1;
            }
            else /* BAKWARD */
            {
                IZ = incurvas - 1;
                direction = -1; /* para el bakward cambiamos la direccion ya que retrocedemos */
            }

            /* Iterate for all VGS values */
            for (int i1 = IZ; i1 >= 0 && i1 < incurvas; i1 += direction)
            {
                if (i3 == 0)
                {
                    if (i1 > 1)
                    {
                        IA = IVsim[i1 - 1][i3] / 2; // I assign the value of the previous point
                        IB = IVsim[i1 - 1][i3] * 2; // I assign the value of the previous point
                    }
                    else
                    {
                        IA = iaa[0];
                        IB = iaa[1];
                    }
                }
                else /* BACKWARD */
                {
                    if ((i1 < (incurvas - 1)) && (i1 > 1))
                    {
                        IA = IVsim[i1 + 1][i3] / 2; // I assign the value of the previous point
                        IB = IVsim[i1 + 1][i3] * 2; // I assign the value of the previous point
                    }
                    else if (i1 == incurvas - 1)
                    {
                        IA = IVsim[i1 - 1][i3] / 2; // I assign the value of the previous point
                        IB = IVsim[i1 - 1][i3] * 2; // I assign the value of the previous point
                    }
                    else
                    {
                        IA = iaa[0];
                        IB = iaa[1];
                    }
                }

                //System.out.println("i3: "+ i3 + " i1: "+i1+ " V_G: "+ vg[i1] + "  I_D_EXP:" +IVexp[i1][i3]+ " IA: "+ IA + " IB: "+ IB);

                if ((i1 == 0) && (i3 == 0))
                {
                    time[i1][i3] = 0;
                    vt[i1][i3] = A+B*Math.exp(-time[i1][i3]/tao2);
                    vss[i1][i3] = vss0;
                    IVsim[i1][i3] = ec.evolutionary_extraction_procedure.models.draincurrentmodel.MarinovJimenezTejada.Subthreshold.busquedaceros(IA, IB, vg[i1], vd, vt[i1][i3], vss[i1][i3], ccs[i1][i3], ms, gamma, mu0, cox, w, l, va_prima);
                    qt[i1][i3] = q0 * Math.exp(-time[i1][i3]/tao2);
                    t[i1][i3] = 0;
                }
                else if ((i3 == 1) && (i1 == incurvas - 1))
                {
                    vt[i1][i3] = vt[i1][i3 - 1];
                    vss[i1][i3] = vss[i1][i3 - 1];
                    ccs[i1][i3] = ccs[i1][i3 - 1];
                    IVsim[i1][i3] = IVsim[i1][i3 - 1];
                    qt[i1][i3] = qt[i1][i3 - 1];
                    time[i1][i3] = time[i1][i3 - 1];
                    t[i1][i3] = 0;
                }
                else
                {
                    t[i1][i3] = tmeasure;
                    vt[i1][i3] = vt[i1 - direction][i3];
                    vss[i1][i3] = vss[i1 - direction][i3];
                    IVsimB =  ec.evolutionary_extraction_procedure.models.draincurrentmodel.MarinovJimenezTejada.Subthreshold.busquedaceros(IA, IB, vg[i1], vd, vt[i1][i3], vss[i1][i3], ccs[i1][i3], ms, gamma, mu0, cox, w, l, va_prima);
                    iterations = 0;

                    time[i1][i3] = time[i1 - direction][i3] + tmeasure;

                    do
                    {
                        iterations++;
                        IVsimA = IVsimB;

                        if (time[i1][i3] <= time_for_trap_2)
                        {
                            Q = q0 * Math.exp(-time[i1][i3]/tao2);
                        }
                        else
                        {
                            Q = ((qt[i1 - direction][i3]) * Math.exp(-t[i1][i3] / tao)) + (alphaQ[i3] * IVsimA * tao * (1 - Math.exp(-t[i1][i3] / tao)));
                        }

                        vss[i1][i3] = vss[i1 - direction][i3] - (0.02586 * Math.log(10) * (Q - qt[i1 - direction][i3]) / cox);

                        if (time[i1][i3] <= time_for_trap_2)
                        {
                            vt[i1][i3] = A+B*Math.exp(-time[i1][i3]/tao2);
                        }
                        else
                        {
                            vt[i1][i3] = vt[i1 - direction][i3] - ((Q - qt[i1 - direction][i3]) / cox);
                        }
                        IVsimB =  ec.evolutionary_extraction_procedure.models.draincurrentmodel.MarinovJimenezTejada.Subthreshold.busquedaceros(IA, IB, vg[i1], vd, vt[i1][i3], vss[i1][i3], ccs[i1][i3], ms, gamma, mu0, cox, w, l, va_prima);
                    }
                    while ((Math.abs((IVsimB - IVsimA) / IVsimB) > 0.001) && iterations < 1000);

                    qt[i1][i3] = Q;
                    IVsim[i1][i3] = IVsimB;

                    if (iterations >= 1000)
                    {
                        throw new ParameterSetNoConvergence("iterations >= 1000");
                    }
                }

                /*
                if (!((vg[i1] <= umbral_vt) && (i3 == 1)))
                {
                    if (qt[i1][i3] > 0)
                    {
                        throw new ParameterSetNoConvergence("transferCharacteristics: Q>0");
                    }
                }
                */

                if (m2 != 0)
                {
                    // Flat the curve at low VG values
                    IVsim[i1][i3] = ioff[i3] * Math.pow(1 + Math.pow(IVsim[i1][i3] / ioff[i3], m2), 1 / m2);
                }

                //System.out.println("i3: "+i3+ "i1: "+i1+ " V_G: "+ vg[i1] + "  I_D_SIM:" +IVsim[i1][i3]+ "  I_D_EXP:" +IVexp[i1][i3]);

                if ((infCurrent(IVsim[i1][i3]) == true) || (nanCurrent(IVsim[i1][i3]) == true)) {
                    throw new ParameterSetNoConvergence("IdVg: Solution not found");
                }
                targets[n] = IVexp[i1][i3];
                outputs[n] = IVsim[i1][i3];
                n++;
            }
        }
        //plotMatrix(IVsim);
        return Fitness.NRMSE(targets, outputs);
    }
    public static double IdVg(double[][] IVexp,
                              double[] iaa,
                              double[] vg,
                              double vd,
                              double vt0_IdVg_ini,
                              double vt0_IdVg_fin,
                              double vss0,
                              double alphaS,
                              double ms,
                              double gamma,
                              double mu0,
                              double cox,
                              double w,
                              double l,
                              double va_prima,
                              double tao,
                              double[] alphaQ,
                              double[] ioff,
                              double m2,
                              double tmeasure,
                              double time_for_trap_2,
                              double tao2)
    {

        int numsets, incurvas;
        numsets = IVexp[0].length;
        incurvas = IVexp.length;

        double A = vt0_IdVg_fin;
        double B = vt0_IdVg_ini - A;
        double q0 = B * cox;

        //System.out.println("A: "+ A + "\nB: "+B + "\nvt0_IdVg_fin: "+ vt0_IdVg_fin + "\nvt0_IdVg_ini: "+ vt0_IdVg_ini + "\ntrap_duration_time: "+ trap_duration_time + "\ntao2: " + tao2 + "\n\n\n");

        double IA, IB, Q, IVsimA, IVsimB;
        int n = 0;
        int IZ, direction;
        int iterations;

        double[] outputs = new double[numsets * incurvas];
        double[] targets = new double[numsets * incurvas];

        double[][] IVsim = new double[incurvas][numsets];
        double[][] vt = new double[incurvas][numsets];
        double[][] vss = new double[incurvas][numsets];
        double[][] ccs = new double[incurvas][numsets];
        double[][] qt = new double[incurvas][numsets];
        double[][] t = new double[incurvas][numsets];
        double[][] time = new double[incurvas][numsets];

        for (int i3 = 0; i3 < numsets; i3++)
        {
            if (i3 == 0) /* FORDWARD */
            {
                IZ = 0;
                direction = 1;
            }
            else /* BAKWARD */
            {
                IZ = incurvas - 1;
                direction = -1; /* para el bakward cambiamos la direccion ya que retrocedemos */
            }

            /* Iterate for all VGS values */
            for (int i1 = IZ; i1 >= 0 && i1 < incurvas; i1 += direction)
            {
                if (i3 == 0)
                {
                    if (i1 > 1)
                    {
                        IA = IVsim[i1 - 1][i3] / 2; // I assign the value of the previous point
                        IB = IVsim[i1 - 1][i3] * 2; // I assign the value of the previous point
                    }
                    else
                    {
                        IA = iaa[0];
                        IB = iaa[1];
                    }
                }
                else /* BACKWARD */
                {
                    if ((i1 < (incurvas - 1)) && (i1 > 1))
                    {
                        IA = IVsim[i1 + 1][i3] / 2; // I assign the value of the previous point
                        IB = IVsim[i1 + 1][i3] * 2; // I assign the value of the previous point
                    }
                    else if (i1 == incurvas - 1)
                    {
                        IA = IVsim[i1 - 1][i3] / 2; // I assign the value of the previous point
                        IB = IVsim[i1 - 1][i3] * 2; // I assign the value of the previous point
                    }
                    else
                    {
                        IA = iaa[0];
                        IB = iaa[1];
                    }
                }

                //System.out.println("i3: "+ i3 + " i1: "+i1+ " V_G: "+ vg[i1] + "  I_D_EXP:" +IVexp[i1][i3]+ " IA: "+ IA + " IB: "+ IB);

                if ((i1 == 0) && (i3 == 0))
                {
                    time[i1][i3] = 0;
                    vt[i1][i3] = A+B*Math.exp(-time[i1][i3]/tao2);
                    vss[i1][i3] = vss0;
                    ccs[i1][i3] = contactConductance(alphaS, vss[i1][i3], vg[i1], vt[i1][i3], gamma);
                    IVsim[i1][i3] = ec.evolutionary_extraction_procedure.models.draincurrentmodel.MarinovJimenezTejada.Subthreshold.busquedaceros(IA, IB, vg[i1], vd, vt[i1][i3], vss[i1][i3], ccs[i1][i3], ms, gamma, mu0, cox, w, l, va_prima);
                    qt[i1][i3] = q0 * Math.exp(-time[i1][i3]/tao2);
                    t[i1][i3] = 0;
                }
                else if ((i3 == 1) && (i1 == incurvas - 1))
                {
                    vt[i1][i3] = vt[i1][i3 - 1];
                    vss[i1][i3] = vss[i1][i3 - 1];
                    ccs[i1][i3] = ccs[i1][i3 - 1];
                    IVsim[i1][i3] = IVsim[i1][i3 - 1];
                    qt[i1][i3] = qt[i1][i3 - 1];
                    time[i1][i3] = time[i1][i3 - 1];
                    t[i1][i3] = 0;
                }
                else
                {
                    t[i1][i3] = tmeasure;
                    vt[i1][i3] = vt[i1 - direction][i3];
                    vss[i1][i3] = vss[i1 - direction][i3];
                    ccs[i1][i3] = contactConductance(alphaS, vss[i1][i3], vg[i1], vt[i1][i3], gamma);
                    IVsimB =  ec.evolutionary_extraction_procedure.models.draincurrentmodel.MarinovJimenezTejada.Subthreshold.busquedaceros(IA, IB, vg[i1], vd, vt[i1][i3], vss[i1][i3], ccs[i1][i3], ms, gamma, mu0, cox, w, l, va_prima);
                    iterations = 0;

                    time[i1][i3] = time[i1 - direction][i3] + tmeasure;

                    do
                    {
                        iterations++;
                        IVsimA = IVsimB;

                        if (time[i1][i3] <= time_for_trap_2)
                        {
                            Q = q0 * Math.exp(-time[i1][i3]/tao2);
                        }
                        else
                        {
                            Q = ((qt[i1 - direction][i3]) * Math.exp(-t[i1][i3] / tao)) + (alphaQ[i3] * IVsimA * tao * (1 - Math.exp(-t[i1][i3] / tao)));
                        }

                        vss[i1][i3] = vss[i1 - direction][i3] - (0.02586 * Math.log(10) * (Q - qt[i1 - direction][i3]) / cox);

                        if (time[i1][i3] <= time_for_trap_2)
                        {
                            vt[i1][i3] = A+B*Math.exp(-time[i1][i3]/tao2);
                        }
                        else
                        {
                            vt[i1][i3] = vt[i1 - direction][i3] - ((Q - qt[i1 - direction][i3]) / cox);
                        }

                        ccs[i1][i3] = contactConductance(alphaS, vss[i1][i3], vg[i1], vt[i1][i3], gamma);
                        IVsimB =  ec.evolutionary_extraction_procedure.models.draincurrentmodel.MarinovJimenezTejada.Subthreshold.busquedaceros(IA, IB, vg[i1], vd, vt[i1][i3], vss[i1][i3], ccs[i1][i3], ms, gamma, mu0, cox, w, l, va_prima);
                    }
                    while ((Math.abs((IVsimB - IVsimA) / IVsimB) > 0.001) && iterations < 1000);

                    qt[i1][i3] = Q;
                    IVsim[i1][i3] = IVsimB;

                    if (iterations >= 1000)
                    {
                        throw new ParameterSetNoConvergence("iterations >= 1000");
                    }
                }

                if (m2 != 0)
                {
                    // Flat the curve at low VG values
                    IVsim[i1][i3] = ioff[i3] * Math.pow(1 + Math.pow(IVsim[i1][i3] / ioff[i3], m2), 1 / m2);
                }

                //System.out.println("i3: "+i3+ "i1: "+i1+ " V_G: "+ vg[i1] + "  I_D_SIM:" +IVsim[i1][i3]+ "  I_D_EXP:" +IVexp[i1][i3]);

                if ((infCurrent(IVsim[i1][i3]) == true) || (nanCurrent(IVsim[i1][i3]) == true)) {
                    throw new ParameterSetNoConvergence("IdVg: Solution not found");
                }
                targets[n] = IVexp[i1][i3];
                outputs[n] = IVsim[i1][i3];
                n++;
            }
        }
        //plotMatrix(IVsim);
        return Fitness.NRMSE(targets, outputs);
    }
    public static double IdVg(double[][] IVexp,
                              double[] VGS,
                              double VDS,
                              double []IAA,
                              double W,
                              double L,
                              double Cox,
                              double mu0,
                              double gamma,
                              double alphaS,
                              double VT,
                              double VSS0,
                              double ms,
                              double tao,
                              double[] alphaQ,
                              double QA,
                              double[] Ioff,
                              double m2,
                              double tmeasure) {

        int numsets, incurvas;
        numsets = IVexp[0].length;
        incurvas = IVexp.length;

        double IA, IB, Q, IVsimA, IVsimB;
        int n = 0;
        int IZ, direction;
        int iterations;

        double[] outputs = new double[numsets * incurvas];
        double[] targets = new double[numsets * incurvas];

        double[][] IVsim = new double[incurvas][numsets];
        double[][] IVslope = new double[incurvas][numsets];
        double[][] vt = new double[incurvas][numsets];
        double[][] vss = new double[incurvas][numsets];
        double[][] qt = new double[incurvas][numsets];
        double[][] ccs = new double[incurvas][numsets];
        double[][] t = new double[incurvas][numsets];

        for (int i3 = 0; i3 < numsets; i3++) {
            if (i3 == 0) /* FORDWARD */ {
                IZ = 0;
                direction = 1;
            } else /* BAKWARD */ {
                IZ = incurvas - 1;
                direction = -1; /* para el bakward cambiamos la direccion ya que retrocedemos */
            }

            /* Iterate for all VGS values */
            for (int i1 = IZ; i1 >= 0 && i1 < incurvas; i1 += direction) {
                if (i3 == 0) {

                    if (i1 == 0) {
                        IA = IAA[0];
                        IB = IAA[1];
                    }else if ((IVexp[i1][i3] == 0) ||
                                ((IVexp[i1 - 1][i3] == 0) && (IVexp[i1][i3] != 0)) ||
                                (IVsim[i1 - 1][i3] == 0) ){
                        IA = IAA[0];
                        IB = IAA[1];
                    } else {
                        IA = IVsim[i1 - 1][i3] / 2; // I assign the value of the previous point
                        IB = IVsim[i1 - 1][i3] * 2; // I assign the value of the previous point
                    }
                } else /* BACKWARD */ {
                    if ((i1 < (incurvas - 1)) && (i1 > 1)) {
                        IA = IVsim[i1 + 1][i3] / 2; // I assign the value of the previous point
                        IB = IVsim[i1 + 1][i3] * 2; // I assign the value of the previous point
                    } else if (i1 == incurvas - 1) {
                        IA = IVsim[i1 - 1][i3] / 2; // I assign the value of the previous point
                        IB = IVsim[i1 - 1][i3] * 2; // I assign the value of the previous point
                    } else {
                        IA = IAA[0];
                        IB = IAA[1];
                    }
                }

                //System.out.println("i3: "+ i3 + "i1: "+i1+ " V_G: "+ VGS[i1] + "  I_D_EXP:" +IVexp[i1][i3]+ " IA: "+ IA + "IB: "+ IB);
                if ((i1 == 0) && (i3 == 0)) {

                    vt[i1][i3] = VT;
                    vss[i1][i3] = VSS0;
                    ccs[i1][i3] = contactConductance(alphaS, vss[i1][i3], VGS[i1], vt[i1][i3], gamma);
                    IVsim[i1][i3] = busquedaceros(IA, IB, ccs[i1][i3], VGS[i1], vt[i1][i3], VDS, gamma, mu0, W, L, Cox, ms, vss[i1][i3]);
                    qt[i1][i3] = -QA;
                    t[i1][i3] = 0;

                } else if ((i3 == 1) && (i1 == incurvas - 1)) {
                    vt[i1][i3] = vt[i1][i3 - 1];
                    vss[i1][i3] = vss[i1][i3 - 1];
                    ccs[i1][i3] = ccs[i1][i3 - 1];
                    IVsim[i1][i3] = IVsim[i1][i3 - 1];
                    qt[i1][i3] = qt[i1][i3 - 1];
                    t[i1][i3] = 0;

                } else {

                    t[i1][i3] = tmeasure;
                    vt[i1][i3] = vt[i1 - direction][i3];
                    vss[i1][i3] = vss[i1 - direction][i3];
                    ccs[i1][i3] = contactConductance(alphaS, vss[i1][i3], VGS[i1], vt[i1][i3], gamma);
                    IVsimB = busquedaceros(IA, IB, ccs[i1][i3], VGS[i1], vt[i1][i3], VDS, gamma, mu0, W, L, Cox, ms, vss[i1][i3]);
                    iterations = 0;

                    do
                    {
                        iterations++;
                        IVsimA = IVsimB;
                        Q = ((qt[i1 - direction][i3]) * Math.exp(-t[i1][i3] / tao)) + (alphaQ[i3] * IVsimA * tao * (1 - Math.exp(-t[i1][i3] / tao)));
                        vss[i1][i3] = vss[i1 - direction][i3] - (0.02586 * Math.log(10) * (Q - qt[i1 - direction][i3]) / Cox);
                        vt[i1][i3] = vt[i1 - direction][i3] - ((Q - qt[i1 - direction][i3]) / Cox);
                        ccs[i1][i3] = contactConductance(alphaS, vss[i1][i3], VGS[i1], vt[i1][i3], gamma);
                        IVsimB = busquedaceros(IA, IB, ccs[i1][i3], VGS[i1], vt[i1][i3], VDS, gamma, mu0, W, L, Cox, ms, vss[i1][i3]);
                    }
                    while ((Math.abs((IVsimB - IVsimA) / IVsimB) > 0.001) && iterations < 1000);

                    qt[i1][i3] = Q;

                    if (iterations >= 1000)
                    {
                        throw new ParameterSetNoConvergence("iterations >= 1000");
                    }

                    IVsim[i1][i3] = IVsimB;
                }


                if (qt[i1][i3] > 0) {
                    throw new ParameterSetNoConvergence("transferCharacteristics: Q>0");
                }

                // Flat the curve at low VG values
                //IVslope[i1][i3] = Ioff[i3] * Math.pow(1 + Math.pow(IVsim[i1][i3] / Ioff[i3], m2), 1 / m2);

                if ((infCurrent(IVslope[i1][i3]) == true) || (nanCurrent(IVslope[i1][i3]) == true)) {
                    throw new ParameterSetNoConvergence("IdVg: Solution not found");
                }
                // System.out.println("i3: "+i3+ "i1: "+i1+ " V_G: "+ VGS[i1] + "  I_D_SIM:" +IVsim[i1][i3]+ "  I_D_EXP:" +IVexp[i1][i3]);
                targets[n] = IVexp[i1][i3];
                outputs[n] = IVsim[i1][i3];
                n++;
            }
        }
        //plotMatrix(IVsim);
        return Fitness.NRMSE(targets, outputs);
    }


    public static double monitoring(double[][] IVexp,
                                    double[][] VGS,
                                    double VDS,
                                    double [][]IAA,
                                    double W,
                                    double L,
                                    double Cox,
                                    double mu0,
                                    double []ghamma,
                                    double alphaS,
                                    double []Vtt0,
                                    double [] Vsst0,
                                    double ms,
                                    double tao,
                                    double[] alphaQ,
                                    double QA,
                                    double []t) {

        int ipuntos = IVexp.length;
        int curvas = IVexp[0].length;

        //System.out.println("i1: "+ipuntos+ " V_G: "+ curvas);

        double IA, IB, Q, IVsimA, IVsimB;
        int n = 0;
        int iterations;

        double[] outputs = new double[ipuntos*curvas];
        double[] targets = new double[ipuntos*curvas];

        double[][] IVsim = new double[ipuntos][curvas];
        double[][] vt = new double[ipuntos][curvas];
        double[][] vss = new double[ipuntos][curvas];
        double[][] qt = new double[ipuntos][curvas];
        double[][] ccs = new double[ipuntos][curvas];

        for (int i2 = 0; i2 < curvas; i2++) {

            /* Iterate for all VGS values */
            for (int i1 = 0; i1 < ipuntos; i1++) {

                IA = IAA[i2][0];
                IB = IAA[i2][1];

                if (i1 == 0)
                {
                    vt[i1][i2] = Vtt0[i2];
                    vss[i1][i2] = Vsst0[i2];
                    ccs[i1][i2] = contactConductance(alphaS, vss[i1][i2], VGS[i1][i2], vt[i1][i2], ghamma[i2]);
                    IVsim[i1][i2] = busquedaceros(IA, IB, ccs[i1][i2], VGS[i1][i2], vt[i1][i2], VDS, ghamma[i2], mu0, W, L, Cox, ms, vss[i1][i2]);
                    qt[i1][i2] = -QA;
                }
                else
                {
                    vt[i1][i2] = vt[i1 - 1][i2];
                    vss[i1][i2] = vss[i1 - 1][i2];
                    ccs[i1][i2] = contactConductance(alphaS, vss[i1][i2], VGS[i1][i2], vt[i1][i2], ghamma[i2]);
                    IVsimB = busquedaceros(IA, IB, ccs[i1][i2], VGS[i1][i2], vt[i1][i2], VDS, ghamma[i2], mu0, W, L, Cox, ms, vss[i1][i2]);
                    iterations = 0;

                    do {
                        iterations++;
                        IVsimA = IVsimB;
                        Q = ((qt[i1 - 1][i2]) * Math.exp(-t[i1] / tao)) + (alphaQ[i2] * IVsimA * tao * (1 - Math.exp(-t[i1] / tao)));
                        vss[i1][i2] = vss[i1 - 1][i2] - (0.02586 * Math.log(10) * (Q - qt[i1 - 1][i2]) / Cox);
                        vt[i1][i2] = vt[i1 - 1][i2] - ((Q - qt[i1 - 1][i2]) / Cox);
                        ccs[i1][i2] = contactConductance(alphaS, vss[i1][i2], VGS[i1][i2], vt[i1][i2], ghamma[i2]);
                        IVsimB = busquedaceros(IA, IB, ccs[i1][i2], VGS[i1][i2], vt[i1][i2], VDS, ghamma[i2], mu0, W, L, Cox, ms, vss[i1][i2]);
                    }
                    while ((Math.abs((IVsimB - IVsimA) / IVsimB) > 0.001) && iterations < 1000);

                    qt[i1][i2] = Q;
                    IVsim[i1][i2] = IVsimB;

                    if (iterations >= 1000) {
                        throw new ParameterSetNoConvergence("iterations >= 1000");
                    }
                }

                //System.out.println("i1: "+i1+ " V_G: "+ VGS[i1] + "  I_D_SIM:" +IVsim[i1]+ "  I_D_EXP:" +IVexp[i1]);

                if (qt[i1][i2] > 0) {
                    throw new ParameterSetNoConvergence("transferCharacteristics: Q>0");
                }

                for (int i3 = 0; i3 < i2; i3++)
                {
                    if (qt[i1][i2] > qt[i1][i3])
                    {
                        throw new ParameterSetNoConvergence("transferCharacteristics: Q>0");
                    }
                }

                //System.out.println("i1: "+i1+ " V_G: "+ VGS[i1] + "  I_D_SIM:" +IVsim[i1]+ "  I_D_EXP:" +IVexp[i1]);
                targets[n] = IVexp[i1][i2];
                outputs[n] = IVsim[i1][i2];
                n++;
            }
        }

        //plotMatrix(IVsim);
        return Fitness.NRMSE(targets, outputs);
    }

    public static double monitoring(double[][] IVexp,
                                    double[][] VGS,
                                    double VDS,
                                    double [][]IAA,
                                    double W,
                                    double L,
                                    double Cox,
                                    double mu0,
                                    double ghamma,
                                    double alphaS,
                                    double Vtt0,
                                    double Vsst0,
                                    double ms,
                                    double tao1,
                                    double tao2,
                                    double[] beta1,
                                    double[] beta2,
                                    double QA,
                                    double []t,
                                    double scale_factor,
                                    double q1_VG20,
                                    double q2_VG20,
                                    double vt_VG20,
                                    double vss_VG20) {

        int ipuntos = IVexp.length;
        int curvas = IVexp[0].length;

        //System.out.println("i1: "+ipuntos+ " V_G: "+ curvas);

        double IA, IB, Q, Q1, Q2, IVsimA, IVsimB;
        int n = 0;
        int iterations;

        double[] outputs = new double[ipuntos*curvas];
        double[] targets = new double[ipuntos*curvas];

        double[][] IVsim = new double[ipuntos][curvas];
        double[][] vt = new double[ipuntos][curvas];
        double[][] vss = new double[ipuntos][curvas];

        double[][] qt = new double[ipuntos][curvas];
        double[][] qt1 = new double[ipuntos][curvas];
        double[][] qt2 = new double[ipuntos][curvas];

        double[][] ccs = new double[ipuntos][curvas];

        double flag = 0;
        for (int i2 = 0; i2 < curvas; i2++) {

            /* Iterate for all VGS values */
            for (int i1 = 0; i1 < ipuntos; i1++)
            {

                //System.out.println("i1: "+ i1 + " i2: "+ i2 +  " V_G: "+ VGS[i1][i2] + "  I_D_EXP:" +IVexp[i1][i2]);

                IA = IAA[i2][0];
                IB = IAA[i2][1];

                if ((i2 == 0) && (flag == 0))
                {

                    if (IVexp[i1][i2] != 0.0)
                    {
                        qt1[i1][i2] = -q1_VG20;
                        qt2[i1][i2] = -q2_VG20;
                        qt[i1][i2] = qt1[i1][i2] + qt2[i1][i2];

                        vt[i1][i2]  = vt_VG20;
                        vss[i1][i2] = vss_VG20;

                        ccs[i1][i2] = contactConductance(alphaS, vss[i1][i2], VGS[i1][i2], vt[i1][i2], ghamma);
                        IVsim[i1][i2] = busquedaceros(IA, IB, ccs[i1][i2], VGS[i1][i2], vt[i1][i2], VDS, ghamma, mu0, W, L, Cox, ms, vss[i1][i2]);

                        flag = 1;
                    }
                    else
                    {
                        qt1[i1][i2] = 0;
                        qt2[i1][i2] = 0;
                        qt[i1][i2] = qt1[i1][i2] + qt2[i1][i2];

                        vt[i1][i2]  = 0;
                        vss[i1][i2] = 0;

                        IVsim[i1][i2] = 0;
                    }

                    targets[n] = IVexp[i1][i2];
                    outputs[n] = IVsim[i1][i2]/scale_factor;
                    n++;

                    continue;
                }


                if (i1 == 0)
                {
                    vt[i1][i2] = Vtt0;
                    vss[i1][i2] = Vsst0;
                    ccs[i1][i2] = contactConductance(alphaS, vss[i1][i2], VGS[i1][i2], vt[i1][i2], ghamma);
                    IVsim[i1][i2] = busquedaceros(IA, IB, ccs[i1][i2], VGS[i1][i2], vt[i1][i2], VDS, ghamma, mu0, W, L, Cox, ms, vss[i1][i2]);

                    qt1[i1][i2] = -QA;
                    qt2[i1][i2] = 0;
                }
                else
                {
                    vt[i1][i2] = vt[i1 - 1][i2];
                    vss[i1][i2] = vss[i1 - 1][i2];
                    ccs[i1][i2] = contactConductance(alphaS, vss[i1][i2], VGS[i1][i2], vt[i1][i2], ghamma);
                    IVsimB = busquedaceros(IA, IB, ccs[i1][i2], VGS[i1][i2], vt[i1][i2], VDS, ghamma, mu0, W, L, Cox, ms, vss[i1][i2]);
                    iterations = 0;

                    do
                    {
                        iterations++;
                        IVsimA = IVsimB;

                        Q1 = ((qt1[i1 - 1][i2]) * Math.exp(-t[i1] / tao1)) + (beta1[i2] * IVsimA * tao1 * (1 - Math.exp(-t[i1] / tao1)));
                        Q2 = ((qt2[i1 - 1][i2]) * Math.exp(-t[i1] / tao2)) + (beta2[i2] * IVsimA * tao2 * (1 - Math.exp(-t[i1] / tao2)));
                        Q = Q1 + Q2;

                        vss[i1][i2] = vss[i1 - 1][i2] - (0.02586 * Math.log(10) * (Q - qt[i1 - 1][i2]) / Cox);
                        vt[i1][i2] = vt[i1 - 1][i2] - ((Q - qt[i1 - 1][i2]) / Cox);
                        ccs[i1][i2] = contactConductance(alphaS, vss[i1][i2], VGS[i1][i2], vt[i1][i2], ghamma);
                        IVsimB = busquedaceros(IA, IB, ccs[i1][i2], VGS[i1][i2], vt[i1][i2], VDS, ghamma, mu0, W, L, Cox, ms, vss[i1][i2]);
                    }
                    while ((Math.abs((IVsimB - IVsimA) / IVsimB) > 0.001) && iterations < 1000);

                    qt1[i1][i2] = Q1;
                    qt2[i1][i2] = Q2;
                    IVsim[i1][i2] = IVsimB;

                    if (iterations >= 1000)
                    {
                        throw new ParameterSetNoConvergence("iterations >= 1000");
                    }
                }

                qt[i1][i2] = qt1[i1][i2] + qt2[i1][i2];

                if (qt[i1][i2] > 0)
                {
                    throw new ParameterSetNoConvergence("transferCharacteristics: Q>0");
                }

                //System.out.println("i1: "+i1+ " V_G: "+ VGS[i1][i2] + "  I_D_SIM:" +IVsim[i1][i2]/scale_factor + "  I_D_EXP:" +IVexp[i1][i2]);

                for (int i3 = 1; i3 < i2; i3++)
                {
                    if (Math.abs(qt[i1][i2]) < Math.abs(qt[i1][i3]))
                    {
                        throw new ParameterSetNoConvergence("transferCharacteristics: Q>0");
                    }
/*
                    if (Math.abs(vt[i1][i2]) < Math.abs(vt[i1][i3]))
                    {
                        throw new ParameterSetNoConvergence("transferCharacteristics: Q>0");
                    }

                    if (Math.abs(vss[i1][i2]) < Math.abs(vss[i1][i3]))
                    {
                        throw new ParameterSetNoConvergence("transferCharacteristics: Q>0");
                    }

 */
                }


                //System.out.println("i1: "+i1+ " V_G: "+ VGS[i1] + "  I_D_SIM:" +IVsim[i1]+ "  I_D_EXP:" +IVexp[i1]);
                targets[n] = IVexp[i1][i2];
                outputs[n] = IVsim[i1][i2]/scale_factor;
                n++;
            }
        }

        //plotMatrix(IVsim);
        return Fitness.NRMSE(targets, outputs);
    }

    public static double monitoring(double[][] IVexp,
                                    double[][] VGS,
                                    double VDS,
                                    double [][]IAA,
                                    double W,
                                    double L,
                                    double Cox,
                                    double mu0,
                                    double ghamma,
                                    double alphaS,
                                    double Vtt0,
                                    double Vsst0,
                                    double ms,
                                    double tao1,
                                    double tao2,
                                    double[] beta1,
                                    double[] beta2,
                                    double QA,
                                    double []t) {

        int ipuntos = IVexp.length;
        int curvas = IVexp[0].length;

        //System.out.println("i1: "+ipuntos+ " V_G: "+ curvas);

        double IA, IB, Q, Q1, Q2, IVsimA, IVsimB;
        int n = 0;
        int iterations;

        double[] outputs = new double[ipuntos*curvas];
        double[] outputsAux = new double[ipuntos*curvas];
        double[] targets = new double[ipuntos*curvas];

        double[][] IVsim = new double[ipuntos][curvas];
        double[][] vt = new double[ipuntos][curvas];
        double[][] vss = new double[ipuntos][curvas];

        double[][] qt = new double[ipuntos][curvas];
        double[][] qt1 = new double[ipuntos][curvas];
        double[][] qt2 = new double[ipuntos][curvas];

        double[][] ccs = new double[ipuntos][curvas];

        double []scale_factors = {3, 3.25, 3.5, 3.75, 4, 4.25, 4.5, 4.75, 5};

        for (int i2 = 0; i2 < curvas; i2++) {

            /* Iterate for all VGS values */
            for (int i1 = 0; i1 < ipuntos; i1++)
            {

                //System.out.println("i1: "+ i1 + " i2: "+ i2 +  " V_G: "+ VGS[i1][i2] + "  I_D_EXP:" +IVexp[i1][i2]);

                IA = IAA[i2][0];
                IB = IAA[i2][1];

                if (i1 == 0)
                {
                    vt[i1][i2] = Vtt0;
                    vss[i1][i2] = Vsst0;
                    ccs[i1][i2] = contactConductance(alphaS, vss[i1][i2], VGS[i1][i2], vt[i1][i2], ghamma);
                    IVsim[i1][i2] = busquedaceros(IA, IB, ccs[i1][i2], VGS[i1][i2], vt[i1][i2], VDS, ghamma, mu0, W, L, Cox, ms, vss[i1][i2]);

                    qt1[i1][i2] = -QA;
                    qt2[i1][i2] = 0;
                }
                else
                {
                    vt[i1][i2] = vt[i1 - 1][i2];
                    vss[i1][i2] = vss[i1 - 1][i2];
                    ccs[i1][i2] = contactConductance(alphaS, vss[i1][i2], VGS[i1][i2], vt[i1][i2], ghamma);
                    IVsimB = busquedaceros(IA, IB, ccs[i1][i2], VGS[i1][i2], vt[i1][i2], VDS, ghamma, mu0, W, L, Cox, ms, vss[i1][i2]);
                    iterations = 0;

                    do
                    {
                        iterations++;
                        IVsimA = IVsimB;

                        Q1 = ((qt1[i1 - 1][i2]) * Math.exp(-t[i1] / tao1)) + (beta1[i2] * IVsimA * tao1 * (1 - Math.exp(-t[i1] / tao1)));
                        Q2 = ((qt2[i1 - 1][i2]) * Math.exp(-t[i1] / tao2)) + (beta2[i2] * IVsimA * tao2 * (1 - Math.exp(-t[i1] / tao2)));
                        Q = Q1 + Q2;

                        vss[i1][i2] = vss[i1 - 1][i2] - (0.02586 * Math.log(10) * (Q - qt[i1 - 1][i2]) / Cox);
                        vt[i1][i2] = vt[i1 - 1][i2] - ((Q - qt[i1 - 1][i2]) / Cox);
                        ccs[i1][i2] = contactConductance(alphaS, vss[i1][i2], VGS[i1][i2], vt[i1][i2], ghamma);
                        IVsimB = busquedaceros(IA, IB, ccs[i1][i2], VGS[i1][i2], vt[i1][i2], VDS, ghamma, mu0, W, L, Cox, ms, vss[i1][i2]);
                    }
                    while ((Math.abs((IVsimB - IVsimA) / IVsimB) > 0.001) && iterations < 1000);

                    qt1[i1][i2] = Q1;
                    qt2[i1][i2] = Q2;
                    IVsim[i1][i2] = IVsimB;

                    if (iterations >= 1000)
                    {
                        throw new ParameterSetNoConvergence("iterations >= 1000");
                    }
                }

                qt[i1][i2] = qt1[i1][i2] + qt2[i1][i2];

                if (qt[i1][i2] > 0)
                {
                    throw new ParameterSetNoConvergence("transferCharacteristics: Q>0");
                }

                //System.out.println("i1: "+i1+ " V_G: "+ VGS[i1][i2] + "  I_D_SIM:" +IVsim[i1][i2]/scale_factor + "  I_D_EXP:" +IVexp[i1][i2]);

                for (int i3 = 1; i3 < i2; i3++)
                {
                    if (Math.abs(qt[i1][i2]) < Math.abs(qt[i1][i3]))
                    {
                        throw new ParameterSetNoConvergence("transferCharacteristics: Q>0");
                    }
                }

                //System.out.println("i1: "+i1+ " V_G: "+ VGS[i1] + "  I_D_SIM:" +IVsim[i1]+ "  I_D_EXP:" +IVexp[i1]);
                targets[n] = IVexp[i1][i2];
                outputs[n] = IVsim[i1][i2];
                n++;
            }
        }

        double best_nrmse = 1e200;
        double nrmse_calculated;

        for (int index = 0; index < scale_factors.length; index++)
        {
            for (int o = 0; o < outputs.length; o++)
            {
                outputsAux[o] = outputs[o]/scale_factors[index];
            }

            nrmse_calculated = Fitness.NRMSE(targets, outputsAux);

            if (nrmse_calculated < best_nrmse)
            {
                best_nrmse = nrmse_calculated;
            }
        }

        //plotMatrix(IVsim);
        return best_nrmse;
    }

    public static double monitoring_t0(double[] IVexp,
                                    double[] VGS,
                                    double VDS,
                                    double[] IAA,
                                    double W,
                                    double L,
                                    double Cox,
                                    double mu0,
                                    double []ghamma,
                                    double alphaS,
                                    double []Vtt0,
                                    double []Vsst0,
                                    double ms) {

        int ipuntos = IVexp.length;

        double IA, IB, ccs;
        int n = 0;

        double[] outputs = new double[ipuntos];
        double[] targets = new double[ipuntos];

        double[] IVsim = new double[ipuntos];

        /* Iterate for all VGS values */
        for (int i1 = 0; i1 < ipuntos; i1++) {

            IA = IAA[0];
            IB = IAA[1];

            ccs = contactConductance(alphaS, Vsst0[i1], VGS[i1], Vtt0[i1], ghamma[i1]);
            IVsim[i1] = busquedaceros(IA, IB, ccs, VGS[i1], Vtt0[i1], VDS, ghamma[i1], mu0, W, L, Cox, ms, Vsst0[i1]);

            targets[n] = IVexp[i1];
            outputs[n] = IVsim[i1];
            n++;
        }

        return Fitness.NRMSE(targets, outputs);
    }

    public static double monitoring(double[][] IVexp,
                                    double[][] VGS,
                                    double VDS,
                                    double []IAA,
                                    double W,
                                    double L,
                                    double Cox,
                                    double mu0,
                                    double gamma,
                                    double alphaS,
                                    double[] VT,
                                    double[] VSS0,
                                    double ms,
                                    double tao,
                                    double[] alphaQ,
                                    double QA,
                                    double []t) {

        int ipuntos = IVexp.length;
        int curvas = IVexp[0].length;


        double IA, IB, Q, IVsimA, IVsimB;
        int n = 0;
        int iterations;

        double[] outputs = new double[ipuntos*curvas];
        double[] targets = new double[ipuntos*curvas];

        double[][] IVsim = new double[ipuntos][curvas];
        double[][] vt = new double[ipuntos][curvas];
        double[][] vss = new double[ipuntos][curvas];
        double[][] qt = new double[ipuntos][curvas];
        double[][] ccs = new double[ipuntos][curvas];

        for (int i2 = 0; i2 < curvas; i2++) {

            /* Iterate for all VGS values */
            for (int i1 = 0; i1 < ipuntos; i1++) {

                if (i1 > 0) {
                    IA = IVsim[i1 - 1][i2] / 10; // I assign the value of the previous point
                    IB = IVsim[i1 - 1][i2] * 10; // I assign the value of the previous point
                } else {
                    IA = IAA[0];
                    IB = IAA[1];
                }

                if (i1 == 0)
                {
                    vt[i1][i2] = VT[i2];
                    vss[i1][i2] = VSS0[i2];
                    ccs[i1][i2] = contactConductance(alphaS, vss[i1][i2], VGS[i2][i1], vt[i1][i2], gamma);
                    IVsim[i1][i2] = busquedaceros(IA, IB, ccs[i1][i2], VGS[i2][i1], vt[i1][i2], VDS, gamma, mu0, W, L, Cox, ms, vss[i1][i2]);
                    qt[i1][i2] = -QA;
                }
                else
                {
                    vt[i1][i2] = vt[i1 - 1][i2];
                    vss[i1][i2] = vss[i1 - 1][i2];
                    ccs[i1][i2] = contactConductance(alphaS, vss[i1][i2], VGS[i2][i1], vt[i1][i2], gamma);
                    IVsimB = busquedaceros(IA, IB, ccs[i1][i2], VGS[i2][i1], vt[i1][i2], VDS, gamma, mu0, W, L, Cox, ms, vss[i1][i2]);
                    iterations = 0;

                    do {
                        iterations++;
                        IVsimA = IVsimB;
                        Q = ((qt[i1 - 1][i2]) * Math.exp(-t[i1] / tao)) + (alphaQ[i2] * IVsimA * tao * (1 - Math.exp(-t[i1] / tao)));
                        vss[i1][i2] = vss[i1 - 1][i2] - (0.02586 * Math.log(10) * (Q - qt[i1 - 1][i2]) / Cox);
                        vt[i1][i2] = vt[i1 - 1][i2] - ((Q - qt[i1 - 1][i2]) / Cox);
                        ccs[i1][i2] = contactConductance(alphaS, vss[i1][i2], VGS[i2][i1], vt[i1][i2], gamma);
                        IVsimB = busquedaceros(IA, IB, ccs[i1][i2], VGS[i2][i1], vt[i1][i2], VDS, gamma, mu0, W, L, Cox, ms, vss[i1][i2]);
                    }
                    while ((Math.abs((IVsimB - IVsimA) / IVsimB) > 0.001) && iterations < 1000);

                    qt[i1][i2] = Q;
                    IVsim[i1][i2] = IVsimB;

                    if (iterations >= 1000)
                    {
                        throw new ParameterSetNoConvergence("iterations >= 1000");
                    }
                }

                //System.out.println("i1: "+i1+ " V_G: "+ VGS[i1] + "  I_D_SIM:" +IVsim[i1]+ "  I_D_EXP:" +IVexp[i1]);

                if (qt[i1][i2] > 0) {
                    throw new ParameterSetNoConvergence("transferCharacteristics: Q>0");
                }

                //System.out.println("i1: "+i1+ " V_G: "+ VGS[i1] + "  I_D_SIM:" +IVsim[i1]+ "  I_D_EXP:" +IVexp[i1]);
                targets[n] = IVexp[i1][i2];
                outputs[n] = IVsim[i1][i2];
                n++;
            }
        }

        //plotMatrix(IVsim);
        return Fitness.NRMSE(targets, outputs);
    }

    public static void plotMatrix(double[][][] matrix) {
        int points, incurvas, numSets;
        points = matrix[1].length;
        incurvas = matrix[0][1].length;
        numSets = matrix.length;

        for (int i3 = 0; i3 < numSets; i3++) {
            if (i3 == 0) {
                System.out.println("FORWARD-------------------------------------------");
            } else {
                System.out.println("BACKWARD-------------------------------------------");
            }
            for (int i1 = 0; i1 < points; i1++) {
                for (int i2 = 0; i2 < incurvas; i2++) {
                    System.out.print(matrix[i3][i1][i2] + " ");
                }
                System.out.println();
            }
        }
    }


    public static void plotMatrix(double[][] matrix) {
        int incurvas, numSets;
        incurvas = matrix[0].length;
        numSets = matrix.length;

        for (int i3 = 0; i3 < numSets; i3++) {
            if (i3 == 0) {
                System.out.println("FORWARD-------------------------------------------");
            } else {
                System.out.println("BACKWARD-------------------------------------------");
            }
            for (int i1 = 0; i1 < incurvas; i1++) {
                System.out.print(matrix[i1][i3] + " ");
            }
            System.out.println();
        }
    }

    public static void plotMatrix(double[] output, double[] targets) {
        int points;
        points = output.length;

        System.out.println("Outputs-------------------------------------------");
        for (int i1 = 0; i1 < points; i1++) {
            System.out.print(output[i1] + " ");
        }
        System.out.println();
        System.out.println("Targets-------------------------------------------");
        for (int i1 = 0; i1 < points; i1++) {
            System.out.print(targets[i1] + " ");
        }
        System.out.println();

    }

}
