package ec.evolutionary_extraction_procedure.models.draincurrentmodel.MarinovJimenezTejada;

import static ec.evolutionary_extraction_procedure.constraints.ContactVoltage.*;
import ec.evolutionary_extraction_procedure.exception.NaNConctactVoltage;
import ec.evolutionary_extraction_procedure.exception.NegativeConctactVoltage;
import ec.evolutionary_extraction_procedure.exception.ParameterSetNoConvergence;
import ec.evolutionary_extraction_procedure.exception.PositiveConctactVoltage;
import ec.evolutionary_extraction_procedure.exception.fIVnoSolution;
import ec.evolutionary_extraction_procedure.fitness.Fitness;

public class GenericDrift {

    /**
     * Plot the individual.
     * <p>
     * Longer description. If there were any, it would be here.
     *
     * @param W Transistor width
     * @param L Transistor length
     * @param k k'
     * @param VGS Gate voltage
     * @param VDS Drain voltage
     * @param VT Threshold voltage
     * @param gamma gamma
     * @param ilinear tren description of the triode region
     * @param mk mk
     */
    public static void toStringIndividual(double W, double L, double k, double VGS, double VDS, double VT, double gamma, int ilinear, double mk) {

        System.out.println("Individual: \n\tVDS := " + VDS
                + "\n\tVGS := " + VGS
                + "\n\tVT := " + VT
                + "\n\tgamma := " + gamma
                + "\n\tk := " + k
                + "\n\tW := " + W
                + "\n\tL_2_5um := " + L
                + "\n\tilinear := " + ilinear
                + "\n\tmk := " + mk);

    }

    /**
     * Find the drain current root using the parameters coded in the individual
     * representation.
     * <p>
     * Longer description. If there were any, it would be here.
     *
     * @param ID Estimated drain current
     * @param MM Contact conductance
     * @param VGS Gate voltage
     * @param VT Threshold voltage
     * @param VDS Drain voltage
     * @param gamma gamma
     * @param k k'
     * @param W Transistor width
     * @param L Transistor length
     * @param ilinear trend description of the triode region
     * @param mk mk
     * @throws fIVnoSolution if not solution is found
     * @return Root between the estimated ID and the drain current generated
     * using the model.
     */
    public double fIV(double ID, double MM, double VGS, double VT, double VDS, double gamma, double k, double W, double L, int ilinear, double mk) {

        double VSS, y;

        if (ilinear == 0) {
            VSS = -(Math.pow(-ID / MM, 1 / 2));
        } else if (ilinear == 1) {
            VSS = ID * MM;
        } else {
            VSS = -(Math.pow(-ID / MM, 1 / mk));
        }

        if (Math.abs(VGS - VT) >= Math.abs(VDS)) {
            y = -k * (W / L) * ((Math.pow(Math.abs(VGS - VT) - Math.abs(VSS), gamma + 2)) - (Math.pow(Math.abs(VGS - VT) - Math.abs(VDS), gamma + 2))) / (gamma + 2) - ID;
        } else {
            y = -k * (W / L) * (Math.pow(Math.abs(VGS - VT) - Math.abs(VSS), gamma + 2)) / (gamma + 2) - ID;
        }

        // If the solution is NaN means that we do not have a valid solution
        if (Double.isNaN(y)) {
            throw new fIVnoSolution("fIV: NaN value for individual");
        }

        return y;
    }

    /**
     *
     * @param IA
     * @param IB
     * @param MR
     * @param VG
     * @param VTC
     * @param VD
     * @param ghamma
     * @param k
     * @param W
     * @param L
     * @param ilinear
     * @param mk
     * @return simulated drain current
     * @throws ParameterSetNoConvergence
     */
    public double busquedaceros(double IA, double IB, double MR, double VG, double VTC, double VD, double ghamma, double k, double W, double L, int ilinear, double mk) {

        double fa, fb, fr, IR;
        int solution, iteration;

        IR = 0;
        fa = fIV(IA, MR, VG, VTC, VD, ghamma, k, W, L, ilinear, mk);
        fb = fIV(IB, MR, VG, VTC, VD, ghamma, k, W, L, ilinear, mk);

        solution = 0;
        iteration = 0;

        while ((solution == 0) && (iteration < 1000)) {

            IR = (IA + IB) / 2; // corriente media
            fr = fIV(IR, MR, VG, VTC, VD, ghamma, k, W, L, ilinear, mk); // f(corriente media)

            if (Math.abs(fr) < 1e-15) {
                solution = 1;
            }
            if (fa * fr < 0) {
                IB = IR;
            } else if (fa * fr > 0) {
                IA = IR;
                fa = fr;
            }
            iteration += 1;
        }

        if (solution == 0) {
            throw new ParameterSetNoConvergence("busquedaceros: Solution not found");
        }

        return IR;

    }

    /**
     *
     * @param IVexp
     * @param VGS
     * @param VDS
     * @param IAA
     * @param W
     * @param L
     * @param k
     * @param gamma
     * @param MC
     * @param VT
     * @param mk
     * @param ilinear
     * @return
     */
    public double outputCharacteristics(double[][] IVexp, double[] IAA, double[] MC, double[] VGS, double VT, double[] VDS, double gamma, double k, double W, double L, int ilinear, double mk) {

        int n, points, incurvas;
        points = VDS.length;
        incurvas = VGS.length;
        double IA, IB, error;

        double[] outputs = new double[points * incurvas];
        double[] targets = new double[points * incurvas];
        double[][] IVsim = new double[points][incurvas];
        n = 0;

        //System.out.print("\n\n\n\n\n");
        for (int i2 = 0; i2 < incurvas; i2++) {
            //System.out.print("Curva NUEVA VGS: " + VGSP[i2] +"\n");
            for (int i1 = 0; i1 < points; i1++) {
                if (i1 > 1) {
                    IA = (IVsim[i1 - 1][i2]) / 2; // I assign the value of the previous point
                    IB = (IVsim[i1 - 1][i2]) * 2; // I assign the value of the previous point
                } else {
                    IA = IAA[i2];
                    IB = 0;
                }
                IVsim[i1][i2] = busquedaceros(IA, IB, MC[i2], VGS[i2], VT, VDS[i1], gamma, k, W, L, ilinear, mk); // NEW  
                //System.out.print(IDS[i1][i2]+" ");
                targets[n] = IVexp[i1][i2];
                outputs[n] = IVsim[i1][i2];
                n += 1;
            } // fin de i1
        }   // fin de i2

        error = Fitness.NRMSE(targets, outputs);

        return error;
    }

    /**
     *
     * @param IVexp
     * @param VGS
     * @param VDS
     * @param W
     * @param L
     * @param k
     * @param gamma
     * @param MC
     * @param VT
     * @param mk
     * @param ilinear
     * @return
     * @throws NegativeConctactVoltage
     * @throws NaNConctactVoltage
     */
    public double contactEffects(double[][] IVexp, double[] MC, double[] VGS, double VT, double[] VDS, double gamma, double k, double W, double L, int ilinear, double mk) {

        int n, points, incurvas;
        double error;
        points = VDS.length;
        incurvas = VGS.length;
        double[][] Vcon = new double[points][incurvas];
        double[][] Vcon2 = new double[points][incurvas];
        double[] outputs = new double[points * incurvas];
        double[] targets = new double[points * incurvas];

        n = 0;
        for (int i2 = 0; i2 < incurvas; i2++) {
            for (int i1 = 0; i1 < points; i1++) {
                if (Math.abs((VGS[i2]) - (VT)) >= Math.abs(VDS[i1])) {
                    //Vcon[i1][i2]=-((Math.abs(VGS[i2]-VTC))-Math.pow(Math.pow(Math.abs(VGS[i2]-VTC)-Math.abs(VDS[i1]), ghamma+2)+Math.abs(ID[i1][i2])*(ghamma+2)*L_2_5um/k/W, 1/(2+ghamma)));
                    Vcon[i1][i2] = -((Math.abs(VGS[i2] - VT)) - Math.pow((Math.pow((Math.abs(VGS[i2] - VT) - Math.abs(VDS[i1])), (gamma + 2)) + Math.abs(IVexp[i1][i2]) * (gamma + 2) * L / k / W), (1 / (2 + gamma))));
                } else {
                    Vcon[i1][i2] = Vcon[i1 - 1][i2];
                }

                if (ilinear == 0) {
                    Vcon2[i1][i2] = -Math.pow(-IVexp[i1][i2] / MC[i2], 1 / 2);
                } else if (ilinear == 1) {
                    Vcon2[i1][i2] = IVexp[i1][i2] * MC[i2];
                } else {
                    Vcon2[i1][i2] = -Math.pow(-IVexp[i1][i2] / MC[i2], 1 / mk);
                }

                targets[n] = Vcon2[i1][i2];
                outputs[n] = Vcon[i1][i2];
                n += 1;

                if (nanContactVoltage(Vcon2[i1][i2]) || nanContactVoltage(Vcon[i1][i2])) {
                    throw new NaNConctactVoltage("contactEffects: NaN value for contacts");
                } else if (postiveContactVoltage(Vcon[i1][i2]) || postiveContactVoltage(Vcon2[i1][i2])) {
                    throw new PositiveConctactVoltage("contactEffects: Conctac effects positive");
                }

            }
        }

        error = Fitness.NRMSE(targets, outputs);

        return error;

    }

}
