% Main computation function for the APEX program.

% Copyright by Davide Vione and Marco Bodrato
% This free software is released with a BY-NC licence. You are free to Share (to copy, distribute and transmit the work) and 
% to Remix (to adapt the work), under the following conditions:
% Attribution   You must attribute the work in the manner specified by the author or licensor 
%                (but not in any way that suggests that they endorse you or your use of the work). 
% Noncommercial  You may not use this work for commercial purposes.

% This is part of the APEX program
% Aqueous Photochemistry of Environmentally occurring Xenobiotics

function [t_OH, t_CO3, t_1O2, t_3DOM, t_Phot, t_tot, k_OH, k_CO3, k_1O2, k_3DOM, k_Phot, k_tot, coOH, coCO3, co1O2, co3DOM, f_OH, f_CO3, f_1O2, f_3DOM, f_Phot, f_tot] = apex (file_prefix, d, CNO3, CNO2, NPOC, CCO3, CHCO3,kP_OH, kP_CO3, kP_DOM, kP_1O2, fi_P, y_OH, y_CO3, y_1O2, y_3DOM, y_Phot)

	IC = CCO3 + CHCO3;
	CP = 1e-8;
	V = 1.26*d;

	ALL = csvread(strcat(file_prefix,".csv"),1,0);
	LL = ALL(:,1);
	ENO3 = ALL(:,2);
	ENO2 = ALL(:,3);
	p0sun = ALL(:,4);
	phi = ALL(:,5);
	EP = ALL(:,6);

	RL = 1:length(LL);
	paTOT = RL;
	paDOM = RL;
	paNO3 = RL;
	paNO2 = RL;
	paP = RL;

	for I = 1:length(LL),
	  ANO3 = ENO3(I)*d*CNO3*100;
	  ANO2 = ENO2(I)*d*CNO2*100;
	  ADOM = NPOC*0.45*exp(-0.015*LL(I))*d*100;
	  if (EP(I) > 0)
	    AP = EP(I)*d*CP*100;
	  else
	    AP = 0;
	  endif

	  ATOT = ANO3+ANO2+ADOM+AP;
	  paTOT(I) = p0sun(I)*(1-exp(-2.303*ATOT));
	  paDOM(I) = paTOT(I)*ADOM/ATOT;
	  paNO3(I) = paTOT(I)*ANO3/ATOT;
	  paNO2(I) = paTOT(I)*ANO2/ATOT;
	  paP(I)   = paTOT(I)*  AP/ATOT;
	  RL(I)    = paP(I)*phi(I);
	endfor;

	Pa_NO3 = integral(LL, paNO3);
	Pa_NO2 = integral(LL, paNO2);
	Pa_DOM = integral(LL, paDOM);
	Pa_TOT = integral(LL, paTOT);
	RPhot  = integral(LL, RL);

	ROH_NO3 = Pa_NO3*(4.3e-2)*(IC+0.0075)/(2.25*IC+0.0075);
	ROH_NO2 = Pa_NO2*(0.12);
	ROH_DOM = Pa_DOM*(3e-5);
	ROH_TOT = ROH_NO3+ROH_NO2+ROH_DOM;
	Scav = (5.0e4)*NPOC+(8.5e6)*CHCO3+(3.9e8)*CCO3+(1e10)*CNO2;
	if (kP_OH > 0)
	  t_OH = (2.39e-5)*d*Scav/(ROH_TOT*kP_OH);
	  k_OH = 0.693/t_OH;
	endif;

	RCO3NO3 = ROH_NO3*11.3*IC/(IC+0.061);
	RCO3 = RCO3NO3+((6.5e-3)*CCO3*Pa_DOM)+ROH_TOT*((8.5e6)*CHCO3+(3.9e8)*CCO3)/Scav;
	if (kP_CO3 > 0)
	  t_CO3 = 0.002394*d*NPOC/(RCO3*kP_CO3);
	  k_CO3 = 0.693/t_CO3;
	endif;

	R3DOM = (1.28e-3)*Pa_DOM;
	if (kP_DOM > 0)
	  t_3DOM = (2.43e-5)*d*(5e5)/(R3DOM*kP_DOM);
	  k_3DOM = 0.693/t_3DOM;
	endif;
	
	R1O2 = (1.25e-3)*Pa_DOM;
	if (kP_1O2 > 0)
	  t_1O2 = (2.43e-5)*d*(2.5e5)/(R1O2*kP_1O2);
	  k_1O2 = 0.693/t_1O2;
	endif;

	if ((fi_P > 0) || (fi_P == -1))
	  t_Phot = 0.693*V*CP/(36000*RPhot);
	  k_Phot = 0.693/t_Phot;
	else
	  k_Phot = 0;
	endif;

	k_tot = k_OH+k_CO3+k_3DOM+k_1O2+k_Phot;
	t_tot = 0.693/k_tot;

	coOH = ROH_TOT/(V*Scav);
	coCO3 = RCO3/(V*100*NPOC);
	co1O2 = R1O2/(V*2.5e5);
	co3DOM = R3DOM/(V*5e5);

	f_OH = y_OH * k_OH;
	f_CO3 = y_CO3 * k_CO3;
	f_1O2 = y_1O2 * k_1O2;
	f_3DOM = y_3DOM * k_3DOM;
	f_Phot = y_Phot * k_Phot;
	f_tot = f_OH + f_CO3 + f_1O2 + f_3DOM + f_Phot;

endfunction
