function oSVMStruct = JzSVMTrain(iXTrain,iYTrain,varargin)
pArguName = {'KernelFunction','Weight','Order','Bias','OptimMethod', ...
    'MarginCost','Autoscale'};
pDefaultValue = {'gaussian',1,3,-1,'qp',1,false};
error(nargchk(1,(2+length(pArguName)*2),nargin,'struct'));
pErrId = ['Matlab:',mfilename];
[ptErrH,ptUnRecogName,pKernelFun,pKFOpts.Weight,pKFOpts.Order, ...
    pKFOpts.Bias,pOptimMethod,pMarginCost,pAutoScale] =  ...
    GetVarArgins(pArguName,pDefaultValue,varargin{:});
if ~isempty(ptErrH)
    if any(eq(ptErrH.Id,[1,2,3]))
        error(pErrId,['Incorrect Name/Value pairs: ',ptErrH.Msg]);
    else
        error(pErrId,['Internal error: ',ptErrH.Msg]);
    end
end
if ~isempty(ptUnRecogName)
    error(pErrId,['Unrecognized parameter names: ',ptUnRecogName]);
end
if nargin == 1
    iYTrain = [];
end
if ischar(pKernelFun)
    pKernelFun = lower(pKernelFun);
    if ~ismember(pKernelFun,{'gaussian','htrbf','laplacian', ...
            'multiquadr','inverquadr','invermq','polynomial', ...
            'polyhomo','sigmoid','linear'})
        error(pErrId,'Unrecognized argument value of ''KernelFunction''.');
    end
else
    error(pErrId, ['''KernelFunction'' argument must be a string' ...
        'or a function handle with the form of "KM = KFUN(U,V)".']);
end
if ischar(pOptimMethod)
    pOptimMethod = lower(pOptimMethod);
    if ~ismember(pOptimMethod,{'qp','smo','ls'})
        error(pErrId,'Unrecognized argument value of ''OptimMethod''.');
    end
else
    error(pErrId,'''OptimMethod'' argument must be a string.');
end
if isempty(iYTrain)
    iYTrain = ones(size(iXTrain,1),1);
elseif ~isvector(iYTrain) && ~iscellstr(iYTrain) && ~ischar(iYTrain)
    error(pErrId,['iYTrain must be a vector, a char array, or ' ...
        'a cell array of string.']);
end
if size(iYTrain,2) ~= 1 || size(iYTrain,1) <= 1
    error(pErrId,'iYTrain must be a m-by-1 vector or array.');
end
if ~isnumeric(iXTrain) || ndims(iXTrain) ~= 2
    error(pErrId,'iXTrain must be a two dimensional numeric matrix.');
end
if size(iXTrain,1) ~= size(iYTrain,1)
    error(pErrId,'iXTrain and iYTrain should have the same row number.');
end
[pYTrainIndex,pYTrainGroup] = grp2idx(iYTrain);
pCkTrain = ~(isnan(pYTrainIndex) | any(isnan(iXTrain),2));
if all(pCkTrain)
    pXTrain = iXTrain;
    pYTrain = pYTrainIndex;
elseif any(pCkTrain)
    pXTrain = iXTrain(pCkTrain,:);
    pYTrain = pYTrainIndex(pCkTrain);
else
    error(pErrId,['Either iXTrain or iYTrain should not "all" ' ...
        'contain NaN or empty (i.e. [],'''',{}) values.']);
end
pYTrain = 1 - (2 * (pYTrain - 1));
pGroupNo = length(pYTrainGroup);
if pGroupNo > 2
    error(pErrId,['JzSVMTrain only supports two-class ' ...
        'and one-class SVM classification.']);
end
pObserveNo = length(pYTrain);
pFeatureNo = size(pXTrain,2);
if ~isnumeric(pMarginCost)
    error(pErrId,'''MarginCost'' must be a numeric scalar or vector.');
end
if isscalar(pMarginCost) && pMarginCost > 0
    if pGroupNo == 2
        ptCk1 = pYTrain == 1;
        ptCk2 = pYTrain == -1;
        ptC1 = 0.5 * pMarginCost * pObserveNo / sum(ptCk1);
        ptC2 = 0.5 * pMarginCost * pObserveNo / sum(ptCk2);
        pMarginCost = ones(pObserveNo,1);
        pMarginCost(ptCk1) = ptC1;
        pMarginCost(ptCk2) = ptC2;
    else
        pMarginCost = repmat(pMarginCost,pObserveNo,1);
    end
elseif isvector(pMarginCost) && all(pMarginCost > 0)
    if size(pMarginCost,1) == size(iXTrain,1)
        if any(pCkTrain)
            pMarginCost = pMarginCost(pCkTrain);
        end
    else
        error(pErrId,['If ''MarginCost'' is a m-by-1 vector, its ' ...
            'length must equal to the row number of iXTrain.']);
    end
else
    error(pErrId,['The ''MarginCost'' must be a positive numeric ' ...
        'scalar or vector.']);
end
pMarginCost = min(pMarginCost, ...
    repmat(1/sqrt(eps(class(pMarginCost))),size(pMarginCost)));
if pAutoScale
    pScaleData.Shift = - mean(pXTrain);
    pScaleData.ScaleFactor = 1./std(pXTrain);
    pScaleData.ScaleFactor(~isfinite(pScaleData.ScaleFactor)) = 1;
    for pI1 = 1:pFeatureNo
        pXTrain(:,pI1) = pScaleData.ScaleFactor(pI1) * ...
            (pXTrain(:,pI1) + pScaleData.Shift(pI1));
    end
else
    pScaleData = [];
end

if ismember(pOptimMethod,{'qp','ls','monqp'})
    try
        pKFMatrix = JzSVMKernel(pXTrain,[],pKernelFun,pKFOpts);
        pKFMatrix = pKFMatrix + diag(1./pMarginCost);
        H = (pYTrain * pYTrain') .* pKFMatrix;
    catch ME
        throw(addCause(MException(pErrId, ...
            'Error in calculating the kernel function matrix.'),ME));
    end
end

switch pOptimMethod
    case 'smo'
        if pGroupNo == 1
            error(pErrId,'Currently not support one-class using SMO.');
        end
        pSMO_KernelFun = @(x,y)JzSVMKernel(x,y,pKernelFun,pKFOpts);
        pSMO_Opts = struct('Display','off','MaxIter',15000, ...
            'TolKKT',1e-3,'KKTViolationLevel',0,'KernelCacheLimit',5000);
        [pAlpha,pBias] = JzSeqMinOpt(pXTrain, pYTrain, ...
            pMarginCost, pSMO_KernelFun, pSMO_Opts);
        pSuppVectCk = pAlpha > sqrt(eps);
        pAlphaH = pYTrain(pSuppVectCk) .* pAlpha(pSuppVectCk);
        pOutput = [];
    case 'qp'
        pQP_Opts = optimset('Display','off','MaxIter',500, ...
            'Algorithm','active-set','LargeScale','off');
        if pGroupNo == 2
            pQP_F = -ones(pObserveNo,1);
            pQP_Beq = 0;
        else
            pQP_F = zeros(pObserveNo,1);
            pQP_Beq = 1;
        end
        [pAlpha,~,pFlag,pOutput] = quadprog(H,pQP_F,[],[],pYTrain', ...
            pQP_Beq,zeros(pObserveNo,1),Inf(pObserveNo,1),[],pQP_Opts);
        pOutput.exitflag = pFlag;
        if pOutput.exitflag <= 0
            warning(pErrId,['Unable to solve the quadratic ' ...
                'optimization problem: ',pOutput.message]);
        end
        pSuppVectCk = pAlpha > sqrt(eps);
        [~,pMaxLoc] = max(pAlpha);
        pAlphaH = pYTrain(pSuppVectCk) .* pAlpha(pSuppVectCk);
        pBias = pYTrain(pMaxLoc) - ...
            sum(pAlphaH .* pKFMatrix(pSuppVectCk,pMaxLoc));
    case 'ls'
        if pGroupNo == 1
            error(pErrId,'Currently not support one-class using LS.');
        end
        pLSMatrix = [0,pYTrain';pYTrain,H]\[0;ones(pObserveNo,1)];
        pSuppVectCk = 1:pObserveNo;
        pBias = pLSMatrix(1);
        pAlphaH = pYTrain .* pLSMatrix(2:end);
        pOutput = [];
end

oSVMStruct.SupportVector = pXTrain(pSuppVectCk,:);
pXYTrainLoc = find(pCkTrain);
oSVMStruct.SupportVectorIdx = pXYTrainLoc(pSuppVectCk);
oSVMStruct.ScaleData = pScaleData;
oSVMStruct.AlphaH = pAlphaH;
oSVMStruct.Bias = pBias;
oSVMStruct.KernelFunction = pKernelFun;
oSVMStruct.KernelFunctionArgs = pKFOpts;
oSVMStruct.OptimizeMethod = pOptimMethod;
oSVMStruct.OptimizeResult = pOutput;
oSVMStruct.GroupNames = pYTrainGroup;

end
